/*
    object.c -- reiserfs files and directories access common code
    Copyright (C) 2001, 2002 Yury Umanets <torque@ukrpost.net>, see COPYING for 
    licensing and copyright details.
*/

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <string.h>
#include <limits.h>
#include <unistd.h>

#if !defined(__GNUC__) && (defined(__sparc__) || defined(__sparc_v9__))
#	include <reiserfs/strsep.h>
#endif

#include <reiserfs/reiserfs.h>
#include <reiserfs/debug.h>

#define N_(String) (String)
#if ENABLE_NLS
#	include <libintl.h>
#	define _(String) dgettext (PACKAGE, String)
#else
#	define _(String) (String)
#endif

reiserfs_path_node_t *reiserfs_object_seek_by_offset(reiserfs_object_t *object, 
    uint64_t offset, uint64_t type, reiserfs_comp_func_t comp_func)
{
	if (reiserfs_fs_format(object->fs) == FS_FORMAT_3_6) {
		set_key_v2_offset(&object->key, offset);
		set_key_v2_type(&object->key, type);
	} else {
		set_key_v1_offset(&object->key, (uint32_t)offset);
		set_key_v1_type(&object->key, reiserfs_key_type2uniq((uint32_t)type));
	}
    
    return reiserfs_tree_lookup_leaf(object->fs->tree, 
		reiserfs_tree_root(object->fs->tree), comp_func, &object->key, object->path);
}

int reiserfs_object_test(reiserfs_fs_t *fs, uint32_t objectid) {
    int i;
    uint32_t *objectids;
    
    ASSERT(fs != NULL, return 0);
    
    objectids = get_sb_objectid_map(fs->super);
  
    for (i = 0; i < get_sb_oid_cursize(fs->super); i += 2) {
		if (objectid == LE32_TO_CPU(objectids[i]))
			return 1;
		
		if (objectid > LE32_TO_CPU(objectids[i]) &&	
				objectid < LE32_TO_CPU(objectids[i + 1]))
			return 1;
		
		if (objectid < LE32_TO_CPU(objectids[i]))
			break;
    }
    
    return 0;
}

int reiserfs_object_use(reiserfs_fs_t *fs, uint32_t objectid) {
    int i, cursize;
    uint32_t *objectids;

    ASSERT(fs != NULL, return 0);
    
    if (reiserfs_object_test(fs, objectid))
		return 1;

    objectids = get_sb_objectid_map(fs->super);
    cursize = get_sb_oid_cursize(fs->super);

    for (i = 0; i < cursize; i += 2) {
		if (objectid >= LE32_TO_CPU(objectids[i]) && 
				objectid < LE32_TO_CPU(objectids[i + 1]))
	    	return 1;
	
		if (objectid + 1 == LE32_TO_CPU(objectids[i])) {
	    	objectids[i] = CPU_TO_LE32(objectid);
	    	goto mark_super;
		}
	
		if (objectid == LE32_TO_CPU(objectids[i + 1])) {
	    	objectids[i + 1] = CPU_TO_LE32(LE32_TO_CPU(objectids[i + 1]) + 1);

	    	if (i + 2 < cursize) {
				if (objectids[i + 1] == objectids[i + 2]) {
		    		memmove(objectids + i + 1, objectids + i + 1 + 2, 
			 			(cursize - (i + 2 + 2 - 1)) * sizeof(uint32_t));
		    		set_sb_oid_cursize(fs->super, cursize - 2);
				}
			}
			goto mark_super;
		}
	
		if (objectid < LE32_TO_CPU(objectids[i])) {
			if (cursize == get_sb_oid_maxsize(fs->super)) {
				objectids[i] = CPU_TO_LE32(objectid);
				goto mark_super;
			} else {
				memmove(objectids + i + 2, objectids + i, (cursize - i) * 
					sizeof(uint32_t));
				set_sb_oid_cursize(fs->super, cursize + 2);
			}
	    
			objectids[i] = CPU_TO_LE32(objectid);
			objectids[i + 1] = CPU_TO_LE32(objectid + 1);
			goto mark_super;
		}
    }
    
    if (i < get_sb_oid_maxsize(fs->super)) {
		objectids[i] = CPU_TO_LE32(objectid);
		objectids[i + 1] = CPU_TO_LE32(objectid + 1);
		set_sb_oid_cursize(fs->super, cursize + 2);
    } else if (i == get_sb_oid_maxsize(fs->super))
		objectids[i - 1] = CPU_TO_LE32(objectid + 1);
    else
		return 0;

mark_super:
    mark_super_dirty(fs);	
    return 1;	    
}

static void object_fill_stat(reiserfs_object_t *object, int format, void *sd) {
    reiserfs_sd_v1_t *sd_v1;
    reiserfs_sd_v2_t *sd_v2;

    memset(&object->stat, 0, sizeof(object->stat));
    
    object->stat.st_dev = dal_dev(object->fs->host_dal);
    object->stat.st_ino = get_key_objectid(&object->key);
    object->stat.st_blksize = reiserfs_fs_block_size(object->fs);
    
    if (format == ITEM_FORMAT_1) {
		sd_v1 = (reiserfs_sd_v1_t *)sd;

		object->stat.st_mode = get_sd_v1_mode(sd_v1);
		object->stat.st_nlink = get_sd_v1_nlink(sd_v1);
		object->stat.st_uid = get_sd_v1_uid(sd_v1);
		object->stat.st_gid = get_sd_v1_gid(sd_v1);
		object->stat.st_rdev = get_sd_v1_rdev(sd_v1);
		object->stat.st_size = get_sd_v1_size(sd_v1);
		object->stat.st_blocks = get_sd_v1_blocks(sd_v1);
		object->stat.st_atime = get_sd_v1_atime(sd_v1);
		object->stat.st_mtime = get_sd_v1_mtime(sd_v1);
		object->stat.st_ctime = get_sd_v1_ctime(sd_v1);
    } else {
		sd_v2 = (reiserfs_sd_v2_t *)sd;
		
		object->stat.st_mode = get_sd_v2_mode(sd_v2);
		object->stat.st_nlink = get_sd_v2_nlink(sd_v2);
		object->stat.st_uid = get_sd_v2_uid(sd_v2);
		object->stat.st_gid = get_sd_v2_gid(sd_v2);
		object->stat.st_rdev = get_sd_v2_rdev(sd_v2);
		object->stat.st_size = get_sd_v2_size(sd_v2);
		object->stat.st_atime = get_sd_v2_atime(sd_v2);
		object->stat.st_mtime = get_sd_v2_mtime(sd_v2);
		object->stat.st_ctime = get_sd_v2_ctime(sd_v2);
    }
}

int reiserfs_object_find_stat(reiserfs_object_t *object) {
    void *sd;
    reiserfs_path_node_t *leaf;
    reiserfs_item_head_t *item;
    
    if (!(leaf = reiserfs_object_seek_by_offset(object, SD_OFFSET, 
		KEY_TYPE_SD, reiserfs_key_comp_four_components)))
    {	
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Couldn't find stat data of object (%lu %lu)."), 
			get_key_dirid(&object->key), get_key_objectid(&object->key));
		return 0;
    }
    
    item = GET_ITEM_HEAD(leaf->node, leaf->pos);
    sd = GET_ITEM_BODY(leaf->node, item);
    
    object_fill_stat(object, get_ih_item_format(item), sd);
    
    return 1;
}

static int reiserfs_object_link(reiserfs_path_node_t *leaf, char *link) {
    reiserfs_item_head_t *item;

    ASSERT(leaf != NULL, return 0);
    ASSERT(link != NULL, return 0);

    if (leaf->pos >= get_blkh_nr_items(GET_BLOCK_HEAD(leaf->node)))
		return 0;
    
    item = GET_ITEM_HEAD(leaf->node, leaf->pos + 1);

    if (reiserfs_key_type(&item->ih_key) != KEY_TYPE_DT)
		return 0;
    
    memcpy(link, GET_ITEM_BODY(leaf->node, item), get_ih_item_len(item));
    
    return 1;
}

int reiserfs_object_find_entry(reiserfs_path_node_t *leaf, uint32_t entry_hash, 
    struct key *entry_key)
{
    uint32_t entry_pos;
    reiserfs_item_head_t *item;
    
    item = GET_ITEM_HEAD(leaf->node, leaf->pos);
    
    if (reiserfs_key_type(&item->ih_key) != KEY_TYPE_DR) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Invalid key type detected %d."), reiserfs_key_type(&item->ih_key));
		return 0;
    }
    
    if (reiserfs_tools_fast_search(&entry_hash, GET_ITEM_BODY(leaf->node, item), 
        get_ih_entry_count(item), DE_SIZE, reiserfs_tools_comp_generic, &entry_pos)) 
    {
        reiserfs_de_head_t *deh = ((reiserfs_de_head_t *)GET_ITEM_BODY(leaf->node, 
	    item)) + entry_pos;
	
		set_key_dirid(entry_key, get_de_dirid(deh));
		set_key_objectid(entry_key, get_de_objectid(deh));
	
		return 1;
    }

    return 0;
}

int reiserfs_object_find_path(reiserfs_object_t *object, const char *name, 
    struct key *dirkey, int as_link) 
{
    uint32_t hash_value;
    char track[_POSIX_PATH_MAX];
    char dirpath[_POSIX_PATH_MAX], *pointer, *dirname = NULL;
    
    uint32_t hash;
    uint16_t *sd_mode;
    reiserfs_path_node_t *leaf;
    
    ASSERT(name != NULL, return 0);
    
    strncpy(dirpath, name, sizeof(dirpath));
    memset(track, 0, sizeof(track));
    
    if (dirpath[0] != '.' || dirpath[0] == '/')
        strncat(track, "/", 1);
    
    pointer = &dirpath[0];
    while (1) {
	
		/* Looking for stat data */
		if (!(leaf = reiserfs_object_seek_by_offset(object, SD_OFFSET, 
			KEY_TYPE_SD, reiserfs_key_comp_four_components)))
		{
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Couldn't find stat data of directory %s."), track);
			return 0;
		}
		
		/*
			Checking whether found item is a link. 
		*/
		sd_mode = (uint16_t *)GET_ITEM_BODY(leaf->node, GET_ITEM_HEAD(leaf->node, 
			leaf->pos));
		
		if (!S_ISLNK(LE16_TO_CPU(*sd_mode)) && !S_ISDIR(LE16_TO_CPU(*sd_mode)) && 
			!S_ISREG(LE16_TO_CPU(*sd_mode))) 
		{
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("%s has invalid object type."), track);
			return 0;
		}
		
		if (S_ISLNK(LE16_TO_CPU(*sd_mode))) {
			int is_terminator = dirname && 
			!strchr((dirname + strlen(dirname) + 1), '/');
			
			if (!as_link || !is_terminator) {
				char link[MAX_DIRECT_ITEM_LEN(reiserfs_fs_block_size(object->fs))];
				
				memset(link, 0, sizeof(link));
				
				if (!reiserfs_object_link(leaf, link) || strlen(link) == 0)
					return 0;
				
				set_key_dirid(&object->key, 
					(link[0] == '/' ? ROOT_DIR_ID : get_key_dirid(dirkey)));
				set_key_objectid(&object->key, 
					(link[0] == '/' ? ROOT_OBJ_ID : get_key_objectid(dirkey)));
				
				if (!reiserfs_object_find_path(object, link, dirkey, 1)) {
					libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					_("Couldn't follow link %s."), link);
					return 0;
				}
			}
		}

		set_key_dirid(dirkey, get_key_dirid(&object->key));
		set_key_objectid(dirkey, get_key_objectid(&object->key));

		if (!(dirname = strsep(&pointer, "/")))
			break;
		
		if (!strlen(dirname))
			continue;
		
		strncat(track, dirname, strlen(dirname));
		
		hash_value = reiserfs_fs_hash_value(object->fs, dirname);
		
		/* Finding directory item */	
		if (!(leaf = reiserfs_object_seek_by_offset(object, hash_value, 
			KEY_TYPE_DR, reiserfs_key_comp_four_components)))
		{
			leaf = reiserfs_path_last(object->path);
			leaf->pos--;
		}

		/* Finding corresponding dir entry */
		if (!reiserfs_object_find_entry(leaf, hash_value, &object->key)) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Couldn't find entry %s."), track);
			return 0;
		}
		
		strncat(track, "/", 1);
    }
    
    return 1;
}

static void reiserfs_object_make_absolute_name(const char *name, 
    char *absolute) 
{
    if (name[0] != '/') {
		char cwd[_POSIX_PATH_MAX];
		
		memset(cwd, 0, sizeof(cwd));
		getcwd(cwd, sizeof(cwd));
		
		strncpy(absolute, cwd, _POSIX_PATH_MAX);
		strncat(absolute, "/", 1);
		strncat(absolute, name, strlen(name));
    } else
		strncpy(absolute, name, strlen(name));
}

reiserfs_object_t *reiserfs_object_create(reiserfs_fs_t *fs, 
    const char *name, int as_link) 
{
    struct key dirkey; 
    reiserfs_object_t *object;
    char absolute[_POSIX_PATH_MAX];
    
    ASSERT(fs != NULL, return NULL);
    ASSERT(name != NULL, return NULL);
    ASSERT(strlen(name) > 0, return NULL);
    
    memset(absolute, 0, sizeof(absolute));
    reiserfs_object_make_absolute_name(name, absolute);
    
    if (!(object = libreiserfs_calloc(sizeof(*object), 0)))
		return NULL;
    
    if (!(object->path = reiserfs_path_create(MAX_HEIGHT)))
		goto error_free_object;
    
    object->fs = fs;
    
    reiserfs_key_form(&dirkey, ROOT_DIR_ID - 1, ROOT_OBJ_ID - 1, 
		SD_OFFSET, KEY_TYPE_SD, reiserfs_fs_format(fs));
    
    reiserfs_key_form(&object->key, ROOT_DIR_ID, ROOT_OBJ_ID, 
		SD_OFFSET, KEY_TYPE_SD, reiserfs_fs_format(fs));
	
    if (!reiserfs_object_find_path(object, absolute, &dirkey, as_link))
		goto error_free_path;
    
    if (!reiserfs_object_find_stat(object))
		goto error_free_path;
    
    return object;
    
error_free_path:
    reiserfs_path_free(object->path);    
error_free_object:
    libreiserfs_free(object);
error:
    return NULL;    
}

int reiserfs_object_is_reg(reiserfs_object_t *object) {
    return S_ISREG(object->stat.st_mode);
}

int reiserfs_object_is_dir(reiserfs_object_t *object) {
    return S_ISDIR(object->stat.st_mode);
}

int reiserfs_object_is_lnk(reiserfs_object_t *object) {
    return S_ISLNK(object->stat.st_mode);
}

void reiserfs_object_free(reiserfs_object_t *object) {
    
    ASSERT(object != NULL, return);
    
    reiserfs_path_free(object->path);
    libreiserfs_free(object);
}

