/*
 * Based upon file which is:
 *  arch/ppc64/kernel/maple_setup.c
 *
 *  (c) Copyright 2004 Benjamin Herrenschmidt (benh@kernel.crashing.org),
 *                     IBM Corp.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version
 *  2 of the License, or (at your option) any later version.
 *
 *****
 *
 * Some part of this file is taken from:
 * arch/ppc64/kernel/lvn_prom.c
 *
 * Author: MontaVista Software, Inc.
 *         source@mvista.com
 *
 * 2003 (c) MontaVista, Software, Inc.
 *
 * Based upon prom.c which is:
 ***
 *** Paul Mackerras	August 1996.
 *** Copyright (C) 1996 Paul Mackerras.
 ***
 ***  Adapted for 64bit PowerPC by Dave Engebretsen and Peter Bergner.
 ***    {engebret|bergner}@us.ibm.com
 ***
 *
 *****
 *
 * Some part of this file is taken from:
 *  linux/arch/ppc64/kernel/lvn_setup.c
 *  copied from linux/arch/ppc64/kernel/chrp_setup.c
 *  and modified for BPB.
 *
 *  linux/arch/ppc/kernel/setup.c
 *
 *  Copyright (C) 1995  Linus Torvalds
 *  Adapted from 'alpha' version by Gary Thomas
 *  Modified by Cort Dougan (cort@cs.nmt.edu)
 *  Modified by PPC64 Team, IBM Corp
 *  Modified by Frank Rowand, MontaVista Software
 *
 *****
 *
 * Some part of this file is taken from:
 * arch/ppc64/kernel/idle.c
 * Idle daemon for PowerPC.  Idle daemon will handle any action
 * that needs to be taken when the system becomes idle.
 *
 * Originally Written by Cort Dougan (cort@cs.nmt.edu)
 *
 * iSeries supported added by Mike Corrigan <mikejc@us.ibm.com>
 *
 * Additional shared processor, SMT, and firmware support
 *    Copyright (c) 2003 Dave Engebretsen <engebret@us.ibm.com>
 *
 *****
 *
 * Some part of this file is taken from:
 * arch/ppc64/kernel/lvn_ipc.c
 *
 * LVN Control signal port, event ports, message ports
 *  (Inter-Partition Communication, but also somewhat available
 *   within a single partition)
 *
 * Author: Frank Rowand frowand@mvista.com or source@mvista.com
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 *****
 *
 * Copyright (C) 2006  Sony Computer Entertainment Inc.
 *
 */

#undef DEBUG

#include <linux/config.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/stddef.h>
#include <linux/unistd.h>
#include <linux/ptrace.h>
#include <linux/slab.h>
#include <linux/user.h>
#include <linux/a.out.h>
#include <linux/tty.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/major.h>
#include <linux/initrd.h>
#include <linux/vt_kern.h>
#include <linux/console.h>
#include <linux/ide.h>
#include <linux/pci.h>
#include <linux/adb.h>
#include <linux/cuda.h>
#include <linux/pmu.h>
#include <linux/irq.h>
#include <linux/seq_file.h>
#include <linux/root_dev.h>
#include <linux/serial.h>
#include <linux/smp.h>
#include <linux/rtc.h>
#include <linux/linux_logo.h>
#include <linux/syscalls.h>

#include <asm/processor.h>
#include <asm/sections.h>
#include <asm/prom.h>
#include <asm/system.h>
#include <asm/pgtable.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/pci-bridge.h>
#include <asm/iommu.h>
#include <asm/machdep.h>
#include <asm/dma.h>
#include <asm/cputable.h>
#include <asm/kexec.h>
#include <asm/time.h>
#include <asm/of_device.h>
#include <asm/lmb.h>
#include <asm/lv1call.h>
#include <asm/abs_addr.h>
#include <asm/byteorder.h>

#include "spu_priv.h"

#ifdef DEBUG
#include <asm/udbg.h>
#define DBG(fmt...) udbg_printf(fmt)
#else
#define DBG(fmt...)
#endif

extern void ps3pf_init_IRQ(void);
extern int ps3pf_get_irq(struct pt_regs *regs);

extern void ps3pf_hpte_init(void);
extern void ps3pf_alloc_htab_bitmap(long htab_size, long rm_limit);
extern void ps3pf_map_htab(void);

static void ps3pf_show_cpuinfo(struct seq_file *m)
{
	seq_printf(m, "machine\t\t: PS3PF\n");
}

#if defined(CONFIG_SPU_FS) || defined(CONFIG_SPU_FS_MODULE)
struct spu_priv_data spu_data[CONFIG_NR_SPUS];
EXPORT_SYMBOL(spu_data);

/* ps3pf_spumem_init_page() is ported from cell_spuprop_present() */
static void __init ps3pf_spumem_init_page(unsigned long address,
					unsigned int len, int early)
{
	unsigned long start_pfn, end_pfn, pfn;
	int node_id;

	node_id = 0;

	start_pfn = address >> PAGE_SHIFT;
	end_pfn = (address + len + PAGE_SIZE - 1) >> PAGE_SHIFT;

	/* We need to call memory_present *before* the call to sparse_init,
	   but we can initialize the page structs only *after* that call.
	   Thus, we're being called twice. */
	if (early)
		memory_present(node_id, start_pfn, end_pfn);
	else {
		/* As the pages backing SPU LS and I/O are outside the range
		   of regular memory, their page structs were not initialized
		   by free_area_init. Do it here instead. */
		for (pfn = start_pfn; pfn < end_pfn; pfn++) {
			struct page *page = pfn_to_page(pfn);
			set_page_links(page, ZONE_DMA, node_id, pfn);
			set_page_count(page, 1);
			reset_page_mapcount(page);
			SetPageReserved(page);
			INIT_LIST_HEAD(&page->lru);
		}
	}
}

void __init ps3pf_spumem_init(int early)
{
	int i;

	for (i = 0; i < CONFIG_NR_SPUS; i++) {
		if (spu_data[i].lspu_id == 0)
			continue;

		ps3pf_spumem_init_page(spu_data[i].ls_phys,
				     LS_SIZE, early);
		ps3pf_spumem_init_page(spu_data[i].problem_phys,
				     sizeof(struct spu_problem), early);
		ps3pf_spumem_init_page(spu_data[i].priv2_phys,
				     sizeof(struct spu_priv2), early);
	}
}
EXPORT_SYMBOL_GPL(ps3pf_spumem_init);

static inline unsigned long __init read_repos(
	unsigned long lpar_id,
	const char *s1, unsigned long n1,
	const char *s2, unsigned long n2,
	const char *s3, unsigned long n3,
	const char *s4, unsigned long n4,
	unsigned long *v1,
	unsigned long *v2)
{
	strncat(((char*)&n1)+4, s1, 4);
	strncat( (char*)&n2,    s2, 8);
	strncat( (char*)&n3,    s3, 8);
	strncat( (char*)&n4,    s4, 8);
	return lv1_get_repository_node_value(lpar_id, n1, n2, n3, n4, v1, v2);
}

static int __init get_spu_resource_id(unsigned long *ids)
{
	unsigned long status, v1, v2;
	unsigned long lpid;
	int spursvn;
	int spun = 0;
	int i;

	lv1_get_logical_partition_id(&lpid);

	/* read bi.spursvn.#0,#0 */
	status = read_repos(lpid, "bi",0, "spursvn",0, "",0, "",0,
			    &v1, &v2);
	if (unlikely(status != 0)) {
		return 0;
	}
	spursvn = v1;

	for (i = 0, spun = 0; i < spursvn && spun < CONFIG_NR_SPUS ; i++) {
		status = read_repos(lpid, "bi",0, "spursv",0, "",i, "",0,
				    &v1, &v2);
		if (likely(status == 0)) {
			if (v1 & 0x8000000000000000UL) {
				ids[spun] = v2;
				spun++;
			}
		}
	}

	return spun;
}

static void __init spu_setup_early(void)
{
	int i;
	long rc;
	unsigned long pu_id, vas_id;
	unsigned long resource_id[CONFIG_NR_SPUS];
	int spun;
	extern struct linux_logo logo_spu_clut224;

	/* get virtual address space id */
	lv1_get_logical_pu_id(&pu_id);
	lv1_get_virtual_address_space_id_of_pu(pu_id, &vas_id);

	/* get spu resource id */
	spun = get_spu_resource_id(&resource_id[0]);

	/* construct logical spu */
	for (i = 0; i < spun; i++) {
		unsigned long lspu_id;
		unsigned long ctxt_addr, ls_addr, pbs_addr, ps2_addr;
		unsigned long align = PAGE_SHIFT;
		unsigned long shadow_addr;

		spu_data[i].lspu_id = 0;

		rc = lv1_construct_logical_spu(align, align, align,
					       align, align,
					       vas_id, 0,
					       &ps2_addr,
					       &pbs_addr,
					       &ls_addr,
					       &ctxt_addr,
					       &shadow_addr,
					       &lspu_id);
		if (rc) {
			printk("%s : Failed to construct logical spu\n",
			       __func__);
			printk("  vasid=0x%lx\n", vas_id);
			printk("  rc=%ld\n", rc);
			continue;
		}
		rc = lv1_enable_logical_spu(lspu_id, resource_id[i]);
		if (rc) {
			printk("%s : Failed to enable logical spu\n",
			       __func__);
			printk("  lspu_id=%ld\n", lspu_id);
			printk("  resource_id=0x%lx\n", resource_id[i]);
			printk("  rc=%ld\n", rc);
			lv1_destruct_logical_spu(lspu_id);
			continue;
		}

		spu_data[i].lspu_id = lspu_id;
		spu_data[i].ls_phys = ls_addr;
		spu_data[i].problem_phys = pbs_addr;
		spu_data[i].priv2_phys = ps2_addr;
		spu_data[i].shadow_phys = shadow_addr;

		spu_data[i].irq_mask[0]		= 0x0000000000000000;
		spu_data[i].irq_mask[1]		= 0x0000000000000000;
		spu_data[i].irq_mask[2]		= 0x0000000000000000;
		spu_data[i].mfc_sr1		= 0x0000000000000033;
		spu_data[i].mfc_tclass_id	= 0x0000000000000000;
		spin_lock_init(&spu_data[i].regs_lock);

		snprintf(spu_data[i].name, sizeof(spu_data[i].name),
			 "spe%016lx", lspu_id);

#ifdef CONFIG_BRINGUP_MSG
		printk("spu[%d]: ", i);
		printk("lspu_id=%02lx ", spu_data[i].lspu_id);
		printk("pb=%016lx ", spu_data[i].problem_phys);
		printk("p1=%016lx ", spu_data[i].shadow_phys);
		printk("p2=%016lx\n", spu_data[i].priv2_phys);
#endif
	}

#ifdef CONFIG_LOGO
	fb_append_extra_logo(&logo_spu_clut224, spun);
#endif
}

void ps3pf_spu_release_all(void)
{
	int i;
	long rc;

	for (i = 0; i < CONFIG_NR_SPUS; i++) {
		if (spu_data[i].lspu_id == 0)
			continue;

		rc = lv1_destruct_logical_spu(spu_data[i].lspu_id);
		spu_data[i].lspu_id = 0;
	}
}
EXPORT_SYMBOL_GPL(ps3pf_spu_release_all);

#else /* !CONFIG_SPU_FS && !CONFIG_SPU_FS_MODULE */
void __init ps3pf_spumem_init(int early)
{
}
EXPORT_SYMBOL_GPL(ps3pf_spumem_init);

static void __init spu_setup_early(void)
{
}

void ps3pf_spu_release_all(void)
{
}
EXPORT_SYMBOL_GPL(ps3pf_spu_release_all);
#endif /* !CONFIG_SPU_FS && !CONFIG_SPU_FS_MODULE */

/*
 * Taken from arch/ppc64/kernel/idle.c and modified by
 * Sony Computer Entertainment Inc.
 */
DECLARE_PER_CPU(unsigned long, smt_snooze_delay);

static void ps3pf_idle(void)
{
	long oldval;
	unsigned long start_snooze;
	unsigned long *smt_snooze_delay = &__get_cpu_var(smt_snooze_delay);

	while (1) {
		oldval = test_and_clear_thread_flag(TIF_NEED_RESCHED);

		if (!oldval) {
			set_thread_flag(TIF_POLLING_NRFLAG);
			start_snooze = get_tb() +
				*smt_snooze_delay * tb_ticks_per_usec;
			while (!need_resched()) {
				if (*smt_snooze_delay > 0 &&
				    get_tb() < start_snooze) {
					HMT_low(); /* Low thread priority */
					continue;
				}
				clear_thread_flag(TIF_POLLING_NRFLAG);
				smp_mb__after_clear_bit();
				lv1_pause(0);
				set_thread_flag(TIF_POLLING_NRFLAG);
				start_snooze = get_tb() +
					*smt_snooze_delay * tb_ticks_per_usec;
			}

			HMT_medium();
			clear_thread_flag(TIF_POLLING_NRFLAG);
		} else {
			set_need_resched();
		}

		schedule();
	}
}

/*
 * Taken from arch/ppc64/kernel/maple_setup.c and modified by
 * Sony Computer Entertainment Inc.
 */

#ifdef CONFIG_PS3PF_SYSMGR
extern void sysmgr_reboot_ready(void);
extern void sysmgr_shutdown_ready(void);
extern void sysmgr_spin_forever(void);
#else
static inline void sysmgr_reboot_ready(void) {}
static inline void sysmgr_shutdown_ready(void) {}
static inline void sysmgr_spin_forever(void) { while (1) ; }
#endif

#ifdef CONFIG_PPC_PS3PF_SB
extern void ps3pf_pci_shutdown(void);
#else
static inline void ps3pf_pci_shutdown(void) {}
#endif
#ifdef CONFIG_FB_PS3
extern void ps3fb_cleanup(void);
#else
extern void ps3fb_cleanup(void) {}
#endif

static void ps3pf_prepare_shutdown(void)
{
#ifdef CONFIG_SMP
	smp_send_stop();
#endif

	local_irq_disable();

	ps3pf_pci_shutdown();
	ps3fb_cleanup();
}

static void ps3pf_restart(char *cmd)
{
	ps3pf_prepare_shutdown();
	sysmgr_reboot_ready();
}

static void ps3pf_power_off(void)
{
	ps3pf_prepare_shutdown();
	sysmgr_shutdown_ready();
}

static void ps3pf_halt(void)
{
	ps3pf_prepare_shutdown();
	sysmgr_shutdown_ready();
}

static void ps3pf_panic(char *str)
{
#ifdef CONFIG_SMP
	smp_send_stop();
#endif

	local_irq_disable();

	printk("\n");
	printk("   System does not reboot automatically.\n");
	printk("   Please press POWER button.\n");
	printk("\n");

	sysmgr_spin_forever();
}

/*
 * Taken from arch/ppc64/kernel/lvn_ipc.c and modified by
 * Sony Computer Entertainment Inc.
 */

#ifdef CONFIG_SMP
static unsigned int ipi0_irq_no   [NR_CPUS];
static unsigned int ipi1_irq_no   [NR_CPUS];
static unsigned int ipi2_irq_no   [NR_CPUS];
static unsigned int ipi3_irq_no   [NR_CPUS];

static unsigned long ipi0_outlet   [NR_CPUS];
static unsigned long ipi1_outlet   [NR_CPUS];
static unsigned long ipi2_outlet   [NR_CPUS];
static unsigned long ipi3_outlet   [NR_CPUS];

static int ipi0_outlet_exists      [NR_CPUS];
static int ipi1_outlet_exists      [NR_CPUS];
static int ipi2_outlet_exists      [NR_CPUS];
static int ipi3_outlet_exists      [NR_CPUS];

void
ps3pf_cause_ipi(int ipi, int procno)
{
	unsigned long status;

	status = L1_ST_SUCCESS;
	switch (ipi) {
	case PPC_MSG_CALL_FUNCTION:
		if (ipi0_outlet_exists[procno])
			status = lv1_send_event_locally(ipi0_outlet[procno]);
		break;
	case PPC_MSG_RESCHEDULE:
		if (ipi1_outlet_exists[procno])
			status = lv1_send_event_locally(ipi1_outlet[procno]);
		break;
#ifdef PPC_MSG_MIGRATE_TASK
	case PPC_MSG_MIGRATE_TASK:
		if (ipi2_outlet_exists[procno])
			status = lv1_send_event_locally(ipi2_outlet[procno]);
		break;
#endif
#ifdef CONFIG_DEBUGGER
	case PPC_MSG_DEBUGGER_BREAK:
		if (ipi3_outlet_exists[procno])
			status = lv1_send_event_locally(ipi3_outlet[procno]);
		break;
#endif
	default:
		printk(KERN_ERR "%s: unknown ipi %d\n", __FUNCTION__, ipi);
		return;
	}
	if (status) {
		printk(KERN_ERR "%s(): lv1_send_event_locally() failed, "
		       "status %ld\n", __FUNCTION__, status);
	}
}

static irqreturn_t
ipi0_handler(int irq, void *dev_id, struct pt_regs *regs)
{

	smp_message_recv(PPC_MSG_CALL_FUNCTION, regs);

	return IRQ_HANDLED;
}

static irqreturn_t
ipi1_handler(int irq, void *dev_id, struct pt_regs *regs)
{

	smp_message_recv(PPC_MSG_RESCHEDULE, regs);

	return IRQ_HANDLED;
}

static irqreturn_t
ipi2_handler(int irq, void *dev_id, struct pt_regs *regs)
{

#ifdef PPC_MSG_MIGRATE_TASK
	smp_message_recv(PPC_MSG_MIGRATE_TASK, regs);
#else
	smp_message_recv(2, regs);
#endif

	return IRQ_HANDLED;
}

static irqreturn_t
ipi3_handler(int irq, void *dev_id, struct pt_regs *regs)
{

	smp_message_recv(PPC_MSG_DEBUGGER_BREAK, regs);

	return IRQ_HANDLED;
}

/*
 * ps3pf_hookup_ipi() must be called from the hardware thread (processor
 * number) that is the target of the ipi
 */
void __init
ps3pf_hookup_ipi(int nr)
{
	unsigned int			cpu;
	int				ret;

	ret = ps3pf_connect_irq(&ipi0_outlet[nr], &ipi0_irq_no[nr], &cpu,
				ipi0_handler,
				SA_INTERRUPT,
				"IPI0 (call function)",
				0);
	if (unlikely(ret)) {
		printk(KERN_ERR "%s(): failed to create ipi0 port[%d]: %d\n",
		       __FUNCTION__, nr, ret);
	} else {
		ipi0_outlet_exists[nr] = 1;
	}

	ret = ps3pf_connect_irq(&ipi1_outlet[nr], &ipi1_irq_no[nr], &cpu,
				ipi1_handler,
				SA_INTERRUPT,
				"IPI1 (reschedule)",
				0);
	if (unlikely(ret)) {
		printk(KERN_ERR "%s(): failed to create ipi1 port[%d]: %d\n",
		       __FUNCTION__, nr, ret);
	} else {
		ipi1_outlet_exists[nr] = 1;
	}

	ret = ps3pf_connect_irq(&ipi2_outlet[nr], &ipi2_irq_no[nr], &cpu,
				ipi2_handler,
				SA_INTERRUPT,
				"IPI2 (unused)",
				0);
	if (unlikely(ret)) {
		printk(KERN_ERR "%s(): failed to create ipi2 port[%d]: %d\n",
		       __FUNCTION__, nr, ret);
	} else {
		ipi2_outlet_exists[nr] = 1;
	}

	ret = ps3pf_connect_irq(&ipi3_outlet[nr], &ipi3_irq_no[nr], &cpu,
				ipi3_handler,
				SA_INTERRUPT,
				"IPI3 (debugger break)",
				0);
	if (unlikely(ret)) {
		printk(KERN_ERR "%s(): failed to create ipi3 port[%d]: %d\n",
		       __FUNCTION__, nr, ret);
	} else {
		ipi3_outlet_exists[nr] = 1;
	}
}

#ifdef CONFIG_KEXEC
static void ps3pf_cleanup_ipi(int cpu_no)
{
	lv1_destruct_irq_plug(ipi0_irq_no[cpu_no]);
	lv1_destruct_irq_plug(ipi1_irq_no[cpu_no]);
	lv1_destruct_irq_plug(ipi2_irq_no[cpu_no]);
	lv1_destruct_irq_plug(ipi3_irq_no[cpu_no]);
}
#endif /* CONFIG_KEXEC */
#endif /* CONFIG_SMP */

static void
smp_ps3pf_message_pass(int target, int msg)
{
	if (target < NR_CPUS)
		ps3pf_cause_ipi(msg, target);
	else {
		int i;

		for_each_online_cpu(i) {
			if (target == MSG_ALL_BUT_SELF
			    && i == smp_processor_id())
				continue;
			ps3pf_cause_ipi(msg, i);
		}
	}
}

static int
smp_ps3pf_probe(void)
{
	int nr_cpus;

	nr_cpus = cpus_weight(cpu_possible_map);

	return nr_cpus;
}

static void smp_ps3pf_setup_cpu(int nr)
{
	ps3pf_hookup_ipi(nr);
}

static struct smp_ops_t ps3pf_smp_ops = {
	.probe		= smp_ps3pf_probe,
	.message_pass	= smp_ps3pf_message_pass,
	.kick_cpu	= smp_generic_kick_cpu,
	.setup_cpu	= smp_ps3pf_setup_cpu,
};

/* This is called very early */
static void smp_init_ps3pf(void)
{
	smp_ops = &ps3pf_smp_ops;
}

/*
 * Taken from arch/ppc64/kernel/lvn_setup.c and modified by
 * Sony Computer Entertainment Inc.
 */
void __init ps3pf_setup_arch(void)
{
	unsigned long id, status;

	/* init to some ~sane value until calibrate_delay() runs */
	loops_per_jiffy = 50000000;

#ifdef CONFIG_BLK_DEV_INITRD
	if (initrd_start)
		ROOT_DEV = Root_RAM0;
	else
#endif
		ROOT_DEV = Root_HDA1;

	/* Setup SMP callback */
#ifdef CONFIG_SMP
	smp_init_ps3pf();			/* sets up ppc_md.smp_* */
#endif

#ifdef CONFIG_DUMMY_CONSOLE
	conswitchp = &dummy_con;
#endif
	/*
	 * BE performance bug workaround
	 */
	init_task.thread.fpexc_mode = 0;

	/* set idle loop function*/
	ppc_md.idle_loop = ps3pf_idle;

	/* initialize spu memory area */
	ps3pf_spumem_init(0);

	/* map HTAB into LPAR address space */
	ps3pf_map_htab();
}

/*
 * Written by Sony Computer Entertainment Inc.
 */
unsigned long ps3pf_rm_limit, ps3pf_mem_total, ps3pf_2nd_mem_base, ps3pf_2nd_mem_size;
EXPORT_SYMBOL(ps3pf_rm_limit);
EXPORT_SYMBOL(ps3pf_2nd_mem_base);
EXPORT_SYMBOL(ps3pf_mem_total);
EXPORT_SYMBOL(ps3pf_2nd_mem_size);
static unsigned long ps3pf_boot_data_addr, ps3pf_boot_data_size;
static char *ps3pf_boot_data;

#define PS3PF_LP_SIZE	(1024*1024*16)	/* large page size = 16MB */

long ps3pf_get_memsize(void)
{
	long		status;
	unsigned long	lpar_id, n1, n2, n3, n4, v1, v2;
	unsigned long	puid, real_mode_limit, mem_total;
	unsigned long	lpar_addr, muid;

	status = lv1_get_logical_pu_id(&puid);
	status = lv1_get_logical_partition_id(&lpar_id);

	/*
	 * get boot data address
	 * name  : bi.boot_dat.address.#0
	 */
	n1 = 0x0000000062690000;		/* "bi" */
	n2 = n3 = 0;
	strncpy((char *)&n2, "boot_dat", 8);	/* "boot_dat" */
	strncpy((char *)&n3, "address", 8);	/* "address" */
	n4 = 0;					/* #0 */
	status = lv1_get_repository_node_value(lpar_id, n1, n2, n3, n4,
					       &ps3pf_boot_data_addr, &v2);
	ps3pf_boot_data = abs_to_virt(ps3pf_boot_data_addr);

	/*
	 * get boot data size
	 * name  : bi.boot_dat.size.#0
	 */
	n1 = 0x0000000062690000;		/* "bi" */
	n2 = n3 = 0;
	strncpy((char *)&n2, "boot_dat", 8);	/* "boot_dat" */
	strncpy((char *)&n3, "size", 8);	/* "size" */
	n4 = 0;					/* #0 */
	status = lv1_get_repository_node_value(lpar_id, n1, n2, n3, n4,
					       &ps3pf_boot_data_size, &v2);

	/*
	 * get real mode limit.
	 * name  : bi.pu.#<puid>.rm_size
	 */
	n1 = 0x0000000062690000;		/* "bi" */
	n2 = 0x7075000000000000;		/* "pu" */
	n3 = puid;				/* #<puid> */
	n4 = 0x726d5f73697a6500;		/* "rm_size" */
	status = lv1_get_repository_node_value(lpar_id, n1, n2, n3, n4,
					       &v1, &v2);
	if (status == L1_ST_NOT_IMPLEMENTED || status == L1_ST_NO_ENTRY) {
		real_mode_limit = 128 * 1024 * 1024;	/* 128MB */
	} else {
		real_mode_limit = v1;
	}

	/*
	 * get max memory size.
	 * name  : bi.rgntotal.#0.#0
	 */
	n1 = 0x0000000062690000;		/* "bi" */
	strncpy((char *)&n2, "rgntotal", 8);	/* "rgntotal" */
	n3 = 0;					/* #0 */
	n4 = 0;					/* #0 */
	status = lv1_get_repository_node_value(lpar_id, n1, n2, n3, n4,
					       &v1, &v2);
	if (status == L1_ST_NOT_IMPLEMENTED || status == L1_ST_NO_ENTRY) {
		mem_total = real_mode_limit;
	} else {
		mem_total = v1;
		ps3pf_2nd_mem_size = mem_total - real_mode_limit;
		ps3pf_2nd_mem_size /= PS3PF_LP_SIZE;
		if (ps3pf_2nd_mem_size < 2) {
			mem_total = real_mode_limit;
			ps3pf_2nd_mem_size = 0;
			lpar_addr = 0;
			goto out;
		}
		/* reserve 16MB for dynamic allocation */
		ps3pf_2nd_mem_size -= 1;
		ps3pf_2nd_mem_size *= PS3PF_LP_SIZE;
		status = lv1_allocate_memory(ps3pf_2nd_mem_size,
					     24,	/* 2^24 = 16MB */
					     0,
					     LV1_AM_TF_NO | LV1_AM_DS_ANYTIME |
					     LV1_AM_FA_ALTERNATIVE |
					     LV1_AM_ADDR_ANY,
					     &lpar_addr, &muid);
		mem_total = real_mode_limit + ps3pf_2nd_mem_size;
	}

 out:
	ps3pf_rm_limit = real_mode_limit;
	ps3pf_mem_total = mem_total;
	ps3pf_2nd_mem_base = lpar_addr;

	return ps3pf_mem_total;
}

unsigned long ps3pf_vas_id;

/*
 * Taken from arch/ppc64/kernel/lvn_prom.c and modified by
 * Sony Computer Entertainment Inc.
 */
static void __init as_initialize(void)
{
	unsigned long access_right;
	unsigned long actual_size;
	unsigned long logical_partition_address;
	unsigned long max_page_size;
	unsigned long number_of_page_sizes;
	unsigned long ownership_transferability;
	unsigned long page_sizes;
	long          status;
	unsigned long size;
	unsigned long size_at_least;
	unsigned long start_address;
	int i;
	unsigned long rnd_mem_size, pteg_count;

	logical_partition_address = 0;

	status = lv1_query_logical_partition_address_region_info(
			logical_partition_address,
			&start_address,
			&size,
			&access_right,
			&max_page_size,
			&ownership_transferability);

	if (status) {
		printk("%s():\n", __FUNCTION__);
		printk("  lv1_query_logical_partition_address_region_info() "
		       "status %ld\n", status);
		panic("lv1_query_logical_partition_address_region_info() "
		      "failed");
	}

	if (max_page_size < 24) {
		printk("%s():\n", __FUNCTION__);
		printk("  max_page_size %ld\n", max_page_size);
		printk("  Unable to use large page mappings for kernel "
		       "mappings\n");
		panic("max_page_size < 24");
	}

	/*
	 * calculate HTAB size
	 */
	rnd_mem_size = 1UL << __ilog2(ps3pf_mem_total);
	if (rnd_mem_size < ps3pf_mem_total)
		rnd_mem_size <<= 1;
	pteg_count = max(rnd_mem_size >> (12 + 1), 1UL << 11);
	/*
	 * Lv-1 restricts HTAB size to 1MB(max).
	 */
	pteg_count = min(1024UL * 1024UL >> 7, pteg_count);

	ppc64_pft_size = __ilog2(pteg_count << 7);

	/*
	 * unsigned char[4] page_sizes contains log2(page sizes):
	 *       16(64K), 20(1M), 24(16M)
	 */
	page_sizes           = (24UL << 56) |
			       (16UL << 48);
	number_of_page_sizes = 2;	/* maximum value for BPA is 2 */
	size_at_least        = ppc64_pft_size;

#ifdef CONFIG_BRINGUP_MSG
	printk("%s():\n", __FUNCTION__);
	printk("  page_sizes 0x%016lx\n", page_sizes);
	printk("  number_of_page_sizes %ld\n", number_of_page_sizes);
	printk("  size_at_least        %ld\n", size_at_least);
#endif

	status = lv1_construct_virtual_address_space(size_at_least,
			number_of_page_sizes, page_sizes,
			&ps3pf_vas_id, &actual_size);

	if (status) {
		printk("%s():\n", __FUNCTION__);
		printk("  lv1_construct_virtual_address_space() status "
		       "%ld\n", status);
		panic("lv1_construct_virtual_address_space() failed");
	}

	if (ps3pf_boot_data_size)
		lmb_reserve(ps3pf_boot_data_addr,
			    ps3pf_boot_data_size);
	ps3pf_alloc_htab_bitmap(actual_size, ps3pf_rm_limit);

	for (i = 0; ;i++){
		if ((1UL << i) == actual_size)
			break;
	}
#ifdef CONFIG_BRINGUP_MSG
	printk("  lv1_construct_virtual_address_space() status %ld\n", status);
	printk("  virtual_address_space_id 0x%lx\n", ps3pf_vas_id);
	printk("  actual_size 0x%lx\n", actual_size);
	if (ppc64_pft_size != i) {
		printk("  ppc64_pft_size was changed from %ld to %d\n",
		       ppc64_pft_size, i);
		ppc64_pft_size = i;
	}
#endif
	status = lv1_select_virtual_address_space(ps3pf_vas_id);
	if (status) {
		printk("%s():\n", __FUNCTION__);
		printk("  lv1_select_virtual_address_space() status %ld\n",
		       status);
		panic("lv1_select_virtual_address_space() failed");
	}
#ifdef CONFIG_BRINGUP_MSG
	printk("ps3pf_rm_limit=%ld, ps3pf_mem_total=%ld\n",
	       ps3pf_rm_limit, ps3pf_mem_total);
	printk("ps3pf_2nd_mem_base=%p, ps3pf_2nd_mem_size=%ld\n",
	       (void *)ps3pf_2nd_mem_base, ps3pf_2nd_mem_size);
	printk("lmb.rmo_size = %ld\n", lmb.rmo_size);
#endif
	lmb.rmo_size = ps3pf_rm_limit;
}

/*
 * Taken from arch/ppc64/kernel/maple_setup.c and modified by
 * Sony Computer Entertainment Inc.
 */
/*
 * Early initialization.
 */
static void __init ps3pf_init_early(void)
{
#ifdef CONFIG_EMBED_INITRD
	extern int __initrd_start;
	extern int __initrd_end;
#endif

	DBG(" -> ps3pf_init_early\n");

	/* Initialize hash table, from now on, we can take hash faults
	 * and call ioremap
	 */
	ps3pf_hpte_init();

#ifdef CONFIG_PCI
	pci_direct_iommu_init();
#endif

	/* Setup interrupt mapping options */
	ppc64_interrupt_controller = IC_OPEN_PIC;

#ifdef CONFIG_EMBED_INITRD
	initrd_start = (unsigned long)&__initrd_start;
	initrd_end = (unsigned long)&__initrd_end;
#endif

	DBG(" <- ps3pf_init_early\n");
}

/*
 * Taken from arch/ppc64/kernel/maple_setup.c and modified by
 * Sony Computer Entertainment Inc.
 */
static void __init ps3pf_progress(char *s, unsigned short hex)
{
	printk("*** %04x : %s\n", hex, s ? s : "");
}


/*
 * Called very early, MMU is off, device-tree isn't unflattened
 */
static int __init ps3pf_probe(int platform)
{
	if (platform != PLATFORM_PS3PF)
		return 0;

	as_initialize();
	spu_setup_early();
	return 1;
}

/*
 * Taken from arch/ppc64/kernel/lvn_setup.c and modified by
 * Sony Computer Entertainment Inc.
 */
extern unsigned long ppc_tb_freq;
extern unsigned long ppc_proc_freq;

void __init ps3pf_calibrate_decr(void)
{
	struct div_result divres;
	unsigned long freq, processor_freq;
	long status;
	unsigned long nodeid, n1, n2, n3, n4, v1, v2;

	/*
	 * get nodeid of BE#0
	 * name  : be#0.#0.#0,#0
	 */
	n1 = 0x0000000062650000;		/* "be" */
	n2 = 0;
	strncpy((char *)&n2, "be", 8);		/* be#0 */
	n3 = 0;					/* #0 */
	n4 = 0;					/* #0 */
	status = lv1_get_repository_node_value(1, n1, n2, n3, n4,
					       &v1, &v2);
	nodeid = v1;
	DBG("%s: nodeid = %ld\n", __FUNCTION__, nodeid);

	n1 = 0x0000000062650000;		/* "be" */
	n2 = nodeid;
	n3 = 0;
	strncpy((char *)&n3, "clock", 8);	/* "clock" */
	n4 = 0;					/* #0 */
	status = lv1_get_repository_node_value(1, n1, n2, n3, n4,
					       &v1, &v2);
	DBG("%s: TB clock = %ld\n", __FUNCTION__, v1);
	freq = v1;
	processor_freq = freq * 40;

	ppc_tb_freq = freq;
	ppc_proc_freq = processor_freq;

        printk("ps3pf_calibrate_decr: decrementer frequency = %lu.%.6lu MHz\n",
	       freq/1000000, freq%1000000 );
	printk("ps3pf_calibrate_decr: processor frequency   = %lu.%.6lu MHz\n",
		processor_freq/1000000, processor_freq%1000000 );

	tb_ticks_per_jiffy = freq / HZ;
	tb_ticks_per_sec = tb_ticks_per_jiffy * HZ;
	tb_ticks_per_usec = freq / 1000000;
	div128_by_32( 1024*1024, 0, tb_ticks_per_sec, &divres );
	tb_to_xs = divres.result_low;
}

static int ps3pf_flash_check_format(char *buf,
				    unsigned long buf_size,
				    unsigned long flash_size)
{
	int offset1, offset2;

	if (buf_size < 0x20) {
		return 0;
	}

	if (memcmp(buf, "cell_ext_os_area\0\0\0\x01", 0x14) != 0) {
		return 0;
	}

	offset1 = be32_to_cpu(*(__be32 *)(buf + 0x14));
	offset2 = be32_to_cpu(*(__be32 *)(buf + 0x18));

	if (offset1 != 2 || offset2 < offset1 + 1 ||
	    flash_size < (offset1 + 1) * 0x200) {
		return 0;
	}

	return offset1;
}

static int ps3pf_flash_try_open_check(char *path)
{
	int fd = 0;
	size_t len;
	off_t size;
	char buf[0x20];
	int offset;

	fd = sys_open(path, O_RDWR, 0);
	if (fd < 0) {
		return fd;
	}

	if ((len = sys_read(fd, buf, 0x20)) < 0x20) {
		goto fail_out;
	}

	if ((size = sys_lseek(fd, 0, 2)) < 0) {
		goto fail_out;
	}

	offset = ps3pf_flash_check_format(buf, len, size);
	if (offset == 0) {
		goto fail_out;
	}

	if (sys_lseek(fd, offset * 0x200, 0) != offset * 0x200) {
		goto fail_out;
	}

	return fd;

fail_out:
	sys_close(fd);
	return -EIO;
}

static int ps3pf_flash_find_open(void)
{
	int fd = 0;
	char path[32];
	char d;

	for (d = 'a'; d <= 'z'; d++) {
		sprintf(path, "/dev/sd%c", d);
		fd = ps3pf_flash_try_open_check(path);
		if (fd >= 0) {
			return fd;
		}
	}
	return -EIO;
}

static unsigned long ps3pf_boot_data_get_rtc_diff(void)
{
	int offset;

	offset = ps3pf_flash_check_format(ps3pf_boot_data,
					  ps3pf_boot_data_size,
					  ps3pf_boot_data_size);
	if (offset == 0) {
		return 0;
	}
	return be64_to_cpu(*(__be64 *)(ps3pf_boot_data + (offset * 0x200)));
}

int ps3pf_boot_data_get_video_mode(void)
{
	int offset;

	offset = ps3pf_flash_check_format(ps3pf_boot_data,
					  ps3pf_boot_data_size,
					  ps3pf_boot_data_size);
	if (offset == 0) {
		return -EIO;
	}
	if (be32_to_cpu(*(__be32 *)(ps3pf_boot_data + 0x210)) < 1){
		return -EIO;
	}

	return *(u8 *)(ps3pf_boot_data + 0x228);
}
EXPORT_SYMBOL(ps3pf_boot_data_get_video_mode);

static void set_rtc_diff_fn(void *param)
{
	long diff_val = *(long *)param;
	ssize_t len;
	int fd;
	char buf[8];

	fd = ps3pf_flash_find_open();
	if (fd >= 0) {
		*(__be64 *)(buf) = cpu_to_be64((__u64)diff_val);
		len = sys_write(fd, buf, 8);
		sys_close(fd);
		if (len == 8) {
			return;
		}
	}

	printk(KERN_ERR "PS3PF: Flash : Failed to save rtc\n");
	return;
}

static int ps3pf_boot_data_set_rtc_diff(long diff_val)
{
	int offset;

	offset = ps3pf_flash_check_format(ps3pf_boot_data,
					  ps3pf_boot_data_size,
					  ps3pf_boot_data_size);
	if (offset == 0) {
		return -EIO;
	}

	*(__be64 *)(ps3pf_boot_data + (offset * 0x200)) =
		cpu_to_be64((__u64)diff_val);

	return 0;
}

static long set_rtc_diff_val;
DECLARE_WORK(set_rtc_diff_work, set_rtc_diff_fn, &set_rtc_diff_val);

static int ps3pf_write_rtc_diff_to_flash(long diff_val)
{
	set_rtc_diff_val = diff_val;
	mb();
	schedule_work(&set_rtc_diff_work);
	return 0;
}

static int ps3pf_set_rtc_time(struct rtc_time *time)
{
	long status;
	unsigned long rtc_val, tb_val;
	long diff_val;

	status = lv1_get_rtc(&rtc_val, &tb_val);

	diff_val = mktime(time->tm_year + 1900, time->tm_mon + 1,
			  time->tm_mday, time->tm_hour,
			  time->tm_min, time->tm_sec)
		- (365 * 23 + 366 * 7) * 24 * 60 * 60
		   /* seconds from 1970 to 2000 */
		- rtc_val;

	ps3pf_boot_data_set_rtc_diff(diff_val);
	ps3pf_write_rtc_diff_to_flash(diff_val);

	return 0;
}

static unsigned int ps3pf_get_time(void)
{
	long status;
	unsigned long rtc_val, tb_val;

	status = lv1_get_rtc(&rtc_val, &tb_val);

	rtc_val += ps3pf_boot_data_get_rtc_diff();

	return rtc_val;
}

static void ps3pf_get_rtc_time(struct rtc_time *time)
{
	int days, tmp, sec, min, hour, years;
	int mon, mday, is_leap_year;
	static int mdays[2][12] = {
		{31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
		{31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
	};
	static int ydays[2] = {365, 366};
	unsigned long rtc_val;

	rtc_val = ps3pf_get_time();

	/*
	 * RTC holds seconds since 2000-01-01 00:00:00 UTC
	 */
	days = rtc_val / 86400;
	tmp = rtc_val % 86400;
	sec = tmp % 60;
	tmp /= 60;
	min = tmp % 60;
	hour = tmp / 60;

	years = 0;
	is_leap_year = 1;

	while (days >= ydays[is_leap_year]) {
		days -= ydays[is_leap_year];
		years++;
		is_leap_year = years % 4 == 0;
	}

	for (mon = 0; mon < 12; mon++) {
		if (days < mdays[is_leap_year][mon])
			break;
		days -= mdays[is_leap_year][mon];
	}
	/* tm_mon starts at zero */
	mon = mon + 0;
	/* tm_mday starts at one */
	mday = days + 1;

	time->tm_sec   = sec;
	time->tm_min   = min;
	time->tm_hour  = hour;
	time->tm_mday  = mday;
	time->tm_mon   = mon;
	time->tm_year  = 100 + years;	/* relative to 2000 */
	time->tm_wday  = 1;
	time->tm_yday  = 1;
}

static unsigned long __init ps3pf_get_boot_time(void)
{
	return ps3pf_get_time() + (365 * 23 + 366 * 7) * 24 * 60 * 60;
}

#ifdef CONFIG_KEXEC
static void ps3pf_kexec_cpu_down(int crash_shutdown, int secondary)
{
	ps3pf_cleanup_ipi(!!secondary);
}
#endif

/*
 * Taken from arch/ppc64/kernel/maple_setup.c and modified by
 * Sony Computer Entertainment Inc.
 */
struct machdep_calls __initdata ps3pf_md = {
	.probe			= ps3pf_probe,
	.setup_arch		= ps3pf_setup_arch,
	.init_early		= ps3pf_init_early,
	.show_cpuinfo		= ps3pf_show_cpuinfo,
	.init_IRQ		= ps3pf_init_IRQ,
	.get_irq		= ps3pf_get_irq,
	.restart		= ps3pf_restart,
	.power_off		= ps3pf_power_off,
	.halt			= ps3pf_halt,
	.panic			= ps3pf_panic,
	.get_boot_time		= ps3pf_get_boot_time,
	.set_rtc_time		= ps3pf_set_rtc_time,
	.get_rtc_time		= ps3pf_get_rtc_time,
	.calibrate_decr		= ps3pf_calibrate_decr,
	.progress		= ps3pf_progress,
#ifdef CONFIG_KEXEC
	.kexec_cpu_down		= ps3pf_kexec_cpu_down,
	.machine_kexec		= default_machine_kexec,
	.machine_kexec_prepare	= default_machine_kexec_prepare,
	.machine_crash_shutdown	= default_machine_crash_shutdown,
#endif
};

/*
 * export all lv1 calls
 */
#define LV1_CALL(name, in, out, num) \
	EXPORT_SYMBOL( _lv1_##name );

#include <asm/lv1calltab.h>

#undef LV1_CALL
