/*
 * Copyright (C) 2006 Sony Computer Entertainment Inc.
 * storage support for PS3PF
 *
 * based on scsi_debug.c
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/platform_device.h>

#include <linux/list.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/blkdev.h>
#include <linux/kthread.h>
#include <scsi/scsi.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#include <linux/cdrom.h>
#include <asm/lv1call.h>
#include <asm/abs_addr.h>
#include "ps3pf_storage.h"

#undef _DEBUG
#if defined(_DEBUG)
#define DPRINTK(x...) printk(x)
#else
#define DPRINTK(x...) do {} while(0)
#endif

#undef _DEBUG_CALLTREE
#if defined(_DEBUG_CALLTREE)
static int func_level;
#define FUNC_START     printk(KERN_ERR "%s:%d start\n", __FUNCTION__, func_level++)
#define FUNC_STEP_C(x) printk(KERN_ERR "%s:%d step %s\n", __FUNCTION__, func_level, x)
#define FUNC_END       printk(KERN_ERR "%s:%d end\n", __FUNCTION__, --func_level)
#define FUNC_END_C(x)  printk(KERN_ERR "%s:%d end %s\n", __FUNCTION__, --func_level, x)
#else
#define FUNC_START     do {} while(0)
#define FUNC_END       FUNC_START
#define FUNC_STEP_C(x) FUNC_START
#define FUNC_END_C(x)  FUNC_START
#endif

#define FLASH_ALIGN    (0x00040000) /* flash safe write size (256KB); should be powers of 2 */

static int ps3pf_stor_add_host = 2;
static int ps3pf_stor_wait_time = CONFIG_PS3PF_STORAGE_MAX_SPINUP_WAIT_TIME;
static int ps3pf_stor_wait_num_storages = CONFIG_PS3PF_STORAGE_EXPECTED_NUM_DRIVES + 1;

#define CEIL_ALIGN_16M(mem)  ((((mem - 1) >> 24) + 1 ) << 24) /* 2^24=16M */
#define CEIL_ALIGN_1M(mem)   ((((mem - 1) >> 20) + 1 ) << 20) /* 2^20=1M */
#define CEIL_ALIGN_64K(mem)  ((((mem - 1) >> 16) + 1 ) << 16) /* 2^16=64K */
#define CEIL_ALIGN_4K(mem)   ((((mem - 1) >> 12) + 1 ) << 12) /* 2^12=4K */

module_param_named(wait_num_storages, ps3pf_stor_wait_num_storages, int, 0);
module_param_named(wait_time, ps3pf_stor_wait_time, int, 0);
MODULE_PARM_DESC(wait_num_storages, "Number of expected (wanted) drives to wait spin up (default=3 drives)");
MODULE_PARM_DESC(wait_time, "Maximum time to wait spinup (default=10sec)");

static struct ps3pf_stor_lv1_bus_info ps3pf_stor_lv1_bus_info_array[PS3PF_STORAGE_NUM_OF_BUS_TYPES];

static struct ps3pf_stor_lv1_dev_info * ps3pf_stor_lv1_dev_info_array;
static int                            ps3pf_stor_lv1_devnum ; /* number of configured(used) lv1 devices */

static LIST_HEAD(ps3pf_stor_host_list);
static DEFINE_SPINLOCK(ps3pf_stor_host_list_lock);
static char ps3pf_stor_proc_name[] = "ps3pf_stor";


static DEVICE_ATTR(max_sectors,
		   S_IRUGO | S_IWUSR,
		   ps3pf_stor_get_max_sectors,
		   ps3pf_stor_set_max_sectors);

static struct device_attribute *ps3pf_stor_sysfs_device_attr_list[] = {
	&dev_attr_max_sectors,
	NULL,
};

static struct scsi_host_template ps3pf_stor_driver_template = {
	.name =			"ps3pf",
	.slave_alloc =		ps3pf_stor_slave_alloc,
	.slave_configure =	ps3pf_stor_slave_configure,
	.slave_destroy =	ps3pf_stor_slave_destroy,
	.queuecommand =		ps3pf_stor_queuecommand,
	.eh_host_reset_handler = ps3pf_stor_host_reset,
	.can_queue =		PS3PF_STOR_CANQUEUE,
	.this_id =		7,
	.sg_tablesize =		SG_ALL,
	.cmd_per_lun =		1,
	.emulated =             1,   /* only sg driver uses this       */
	.max_sectors =		128, /* multiple of pagesize, reset later */
	.unchecked_isa_dma = 	0,
	.use_clustering = 	ENABLE_CLUSTERING,
	.sdev_attrs =           ps3pf_stor_sysfs_device_attr_list,
	.module =		THIS_MODULE,
};

static struct platform_driver ps3pf_stor_platform_driver = {
	.driver = {
		.name = "ps3pf_stor"
	},
	.probe          = ps3pf_stor_driver_probe,
	.remove         = ps3pf_stor_driver_remove,
	.shutdown       = ps3pf_stor_driver_shutdown
};

const static struct platform_device ps3pf_stor_platform_device = {
	.name           = "ps3pf_stor",
	.dev            = {
		.release        = ps3pf_stor_device_release
	}
};

/*
 * fill buf with MODE SENSE page 8 (caching parameter)
 * changable: 0 fills current value, otherwise fills 0
 * returns length of this page
 */
const static unsigned char page_data_6[] =
{
	0x06,    2, /* page 6, length =2                         */
	0x01,       /* 0: write cache disabled                   */
	0x00        /* reserved                                  */
};
const static unsigned char page_data_8[] =
{
	0x08,   10, /* page 8, length =10                        */
	0x04,       /* 0:read cache, 1:mult factor, 2:write cache*/
	0x00,       /* 0..3:write retantion, 4..7:read retantion */
	0xff, 0xff, /* disable prefech block length              */
	0x00, 0x00, /* minimum prefech                           */
	0xff, 0xff, /* maximum prefech                           */
	0xff, 0xff  /* maximum prefech ceiling                   */
};

const struct scsi_command_handler_info scsi_cmnd_info_table_hdd[256] =
{
	[INQUIRY]                 = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_inquiry},
	[REQUEST_SENSE]           = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_request_sense},
	[TEST_UNIT_READY]         = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_just_ok},
	[READ_CAPACITY]           = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_read_capacity},
	[MODE_SENSE_10]           = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_mode_sense},
	[SYNCHRONIZE_CACHE]       = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_sync_cache},
	[READ_10]                 = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_read},
	[READ_6]                  = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_read},
	[WRITE_10]                = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_write},
	[WRITE_6]                 = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_write}
};

const struct scsi_command_handler_info scsi_cmnd_info_table_flash[256] =
{
	[INQUIRY]                 = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_inquiry},
	[REQUEST_SENSE]           = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_request_sense},
	[TEST_UNIT_READY]         = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_just_ok},
	[READ_CAPACITY]           = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_read_capacity},
	[MODE_SENSE_10]           = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_mode_sense},
	[SYNCHRONIZE_CACHE]       = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_hdd_handle_sync_cache},
	[READ_10]                 = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_read},
	[READ_6]                  = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_read},
	[WRITE_10]                = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_handle_write_flash},
	[WRITE_6]                 = { NOT_AVAIL, NOT_AVAIL, NOT_AVAIL, ps3pf_stor_handle_write_flash}
};

const struct scsi_command_handler_info scsi_cmnd_info_table_atapi[256] =
{
	[INQUIRY]                 = {  USE_SRB_6, PIO_DATA_IN_PROTO,  DIR_READ, ps3pf_stor_atapi_handle_simple},
	[REQUEST_SENSE]           = {  USE_SRB_6, PIO_DATA_IN_PROTO,  DIR_READ, ps3pf_stor_atapi_handle_request_sense},
	[START_STOP]              = {          0,    NON_DATA_PROTO, NOT_AVAIL, ps3pf_stor_atapi_handle_simple},
	[ALLOW_MEDIUM_REMOVAL]    = {          0,    NON_DATA_PROTO, NOT_AVAIL, ps3pf_stor_atapi_handle_simple},
	[TEST_UNIT_READY]         = {          0,    NON_DATA_PROTO, NOT_AVAIL, ps3pf_stor_atapi_handle_simple},
	[READ_CAPACITY]           = {          8, PIO_DATA_IN_PROTO,  DIR_READ, ps3pf_stor_atapi_handle_simple},
	[MODE_SENSE_10]           = { USE_SRB_10, PIO_DATA_IN_PROTO,  DIR_READ, ps3pf_stor_atapi_handle_simple},
	[READ_TOC]                = { USE_SRB_10, PIO_DATA_IN_PROTO,  DIR_READ, ps3pf_stor_atapi_handle_simple},
	[GPCMD_GET_CONFIGURATION] = { USE_SRB_10, PIO_DATA_IN_PROTO,  DIR_READ, ps3pf_stor_atapi_handle_simple},
	[GPCMD_READ_DISC_INFO]    = { USE_SRB_10, PIO_DATA_IN_PROTO,  DIR_READ, ps3pf_stor_atapi_handle_simple},
	[READ_10]                 = {  NOT_AVAIL,         NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_read},
	[READ_6]                  = {  NOT_AVAIL,         NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_read},
	[WRITE_10]                = {  NOT_AVAIL,         NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_write},
	[WRITE_6]                 = {  NOT_AVAIL,         NOT_AVAIL, NOT_AVAIL, ps3pf_stor_common_handle_write},
	[GPCMD_READ_CD]           = {USE_CDDA_FRAME_RAW,  DMA_PROTO,  DIR_READ, ps3pf_stor_atapi_handle_simple}
};

/*
 * lv1 repository function
 */
static void make_first_key(const char *name, uint64_t index, uint64_t *n)
{
	strncpy((char *)n, name, 8);
	*n = (*n >> 32) + index;
}

static void make_key(const char *name, uint64_t index, uint64_t *n)
{
	strncpy((char *)n, name, 8);
	*n += index;
}

static int ps3pf_read_repository_bus_str(uint64_t bus_index, const char * second_key, uint64_t * value)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;

	n1 = n2 = n3 = n4 = v1 = v2 = 0;
	make_first_key("bus", bus_index, &n1);
	make_key(second_key, 0, &n2);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	if (v2 != 0) {
		printk(KERN_ERR "%s:v2 not zero:%#lx %#lx %#lx %#lx, v1=%#lx, v2=%#lx\n", __FUNCTION__,
		       n1, n2, n3, n4,
		       v1, v2);
	}
	*value = v1;
	return 0;
}

static int ps3pf_read_repository_bus_id(uint64_t bus_index, uint64_t *system_bus_id)
{
	return ps3pf_read_repository_bus_str(bus_index, "id", system_bus_id);
}

static int ps3pf_read_repository_bus_type(uint64_t bus_index, uint64_t *bus_type)
{
	return ps3pf_read_repository_bus_str(bus_index, "type", bus_type);
}

static int ps3pf_read_repository_num_of_dev(uint64_t bus_index, uint64_t *num_of_dev)
{
	return ps3pf_read_repository_bus_str(bus_index, "num_dev", num_of_dev);
}

static int ps3pf_read_repository_bus_dev_str(uint64_t bus_index, uint64_t device_index,
					   const char * third_key, uint64_t *value)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key("bus", bus_index, &n1);
	make_key("dev", device_index, &n2);
	make_key(third_key, 0, &n3);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	if (v2 != 0) {
		printk(KERN_ERR "%s:v2 not zero:%#lx %#lx %#lx %#lx, v1=%#lx, v2=%#lx\n", __FUNCTION__,
		       n1, n2, n3, n4,
		       v1, v2);
	}
	*value = v1;
	return 0;
}

static int ps3pf_read_repository_bus_dev_intr(uint64_t bus_index, uint64_t device_index,
					      uint64_t intr_index, uint64_t *value0, uint64_t *value1)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key("bus", bus_index, &n1);
	make_key("dev", device_index, &n2);
	make_key("intr", intr_index, &n3);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*value0 = v1;
	*value1 = v2;
	return 0;
}

static int ps3pf_read_repository_device_type(uint64_t bus_index, uint64_t device_index,
					uint64_t *device_type)
{
	return ps3pf_read_repository_bus_dev_str(bus_index, device_index, "type", device_type);
}

static int ps3pf_read_repository_device_port(uint64_t bus_index, uint64_t device_index,
					   uint64_t *port)
{
	return ps3pf_read_repository_bus_dev_str(bus_index, device_index, "port", port);
}

static int ps3pf_read_repository_device_id(uint64_t bus_index, uint64_t device_index,
					   uint64_t *id)
{
	return ps3pf_read_repository_bus_dev_str(bus_index, device_index, "id", id);
}
static int ps3pf_read_repository_device_blksize(uint64_t bus_index, uint64_t device_index,
					      uint64_t *blksize)
{
	return ps3pf_read_repository_bus_dev_str(bus_index, device_index, "blk_size", blksize);
}

static int ps3pf_read_repository_device_blocks(uint64_t bus_index, uint64_t device_index,
				uint64_t *blocks)
{
	return ps3pf_read_repository_bus_dev_str(bus_index, device_index, "n_blocks", blocks);
}

static int ps3pf_read_repository_device_regions(uint64_t bus_index, uint64_t device_index,
				uint64_t *regions)
{
	return ps3pf_read_repository_bus_dev_str(bus_index, device_index, "n_regs", regions);
}

static int ps3pf_read_repository_bus_dev_reg_str(uint64_t bus_index, uint64_t device_index, uint64_t reg_index,
					       const char * fourth_key, uint64_t *value)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key("bus", bus_index, &n1);
	make_key("dev", device_index, &n2);
	make_key("region", reg_index, &n3);
	make_key(fourth_key, 0, &n4);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	if (v2 != 0) {
		printk(KERN_ERR "%s:v2 not zero:%#lx %#lx %#lx %#lx, v1=%#lx, v2=%#lx\n", __FUNCTION__,
		       n1, n2, n3, n4,
		       v1, v2);
	}
	*value = v1;
	return 0;
}

static int ps3pf_read_repository_device_region_id(uint64_t bus_index,
						uint64_t device_index,
						uint64_t region_index,
						uint64_t *region_id)
{
	return ps3pf_read_repository_bus_dev_reg_str(bus_index,
						   device_index,
						   region_index,
						   "id",
						   region_id);
}

static int ps3pf_read_repository_device_region_size(uint64_t bus_index,
						  uint64_t device_index,
						  uint64_t region_index,
						  uint64_t *region_size)
{
	return ps3pf_read_repository_bus_dev_reg_str(bus_index,
						   device_index,
						   region_index,
						   "size",
						   region_size);
}

static int ps3pf_read_repository_device_region_start(uint64_t bus_index,
						  uint64_t device_index,
						  uint64_t region_index,
						  uint64_t *start)
{
	return ps3pf_read_repository_bus_dev_reg_str(bus_index,
						   device_index,
						   region_index,
						   "start",
						   start);
}

static int ps3pf_get_device_interrupt_id(uint64_t bus_index,
					 uint64_t device_index,
					 uint64_t *interrupt_id)
{
	int i, ret;
	uint64_t v1 = 0;
	uint64_t v2 = 0;

	for (i = 0; i < PS3PF_MMIO_REGION_MAX; i++) {
		ret = ps3pf_read_repository_bus_dev_intr(bus_index, device_index, i, &v1, &v2);
		if (!ret)
			continue;
		/* check whether infotype is event port */
		if (v1 == 2) {
			*interrupt_id = v2;
			return 0;
		} else {
			ret = -1;
		}
	}
	return ret;
}

/*
 * returns 0: decoded
 *        -1: not sense info, issue REQUEST_SENSE needed
 */
static int decode_lv1_status(uint64_t status,
			     unsigned char * sense_key,
			     unsigned char * asc,
			     unsigned char * ascq)
{
	if (((status >> 24) & 0xff) != 0x02)
		return -1;

	*sense_key = (status >> 16) & 0xff;
	*asc       = (status >>  8) & 0xff;
	*ascq      =  status        & 0xff;
	return 0;
}
/*
 * main thread to process srb.
 * thread is created per device basis.
 * srb are often passed in interrupt context (softirq), so
 * we can't sleep at queuecommand().  just receive it
 * at queucommand(), then passed it to other thread
 * to process it under non-interrupt context.
 */
static int ps3pf_stor_main_thread(void * parm)
{
	struct ps3pf_stor_dev_info * dev_info = (struct ps3pf_stor_dev_info *) parm;
	int reason = 0;

	current->flags |= PF_NOFREEZE; /* jugemu jugemu */

	while (!reason) {
		down_interruptible(&(dev_info->thread_sema));
		switch (dev_info->thread_wakeup_reason) {
		case SRB_QUEUED:
			ps3pf_stor_process_srb(dev_info->srb);
			break;
		case THREAD_TERMINATE:
			reason =  THREAD_TERMINATE;
			break;
		default:
			printk(KERN_ERR "%s: unknown wakeup reason %d\n", __FUNCTION__,
			       dev_info->thread_wakeup_reason);
			break;
		}
	}

	complete_and_exit(&(dev_info->thread_terminated), reason);
}

static int ps3pf_stor_queuecommand(struct scsi_cmnd * srb, void (*done)(struct scsi_cmnd *))
{
	struct ps3pf_stor_dev_info * dev_info;
	unsigned long flags;
	int ret = 0;
	dev_info = (struct ps3pf_stor_dev_info *)srb->device->hostdata;

	spin_lock_irqsave(&dev_info->srb_lock, flags);
	{
		if (dev_info->srb) {
			/* no more than one can be processed */
			printk(KERN_ERR "%s: more than 1 SRB queued %d %d\n", __FUNCTION__,
			       srb->device->host->host_no, srb->device->id);
			ret = SCSI_MLQUEUE_HOST_BUSY;
		} else {
			srb->scsi_done = done;
			dev_info->srb = srb;

			dev_info->thread_wakeup_reason = SRB_QUEUED;
			up(&(dev_info->thread_sema));
			ret = 0;
		};
	}
	spin_unlock_irqrestore(&(dev_info->srb_lock), flags);
	return ret;
}

static void ps3pf_stor_srb_done(struct ps3pf_stor_dev_info * dev_info)
{
	struct scsi_cmnd * srb = dev_info->srb;
	unsigned long flags;

	spin_lock_irqsave(&(dev_info->srb_lock), flags);
	{
		dev_info->srb = NULL;
		srb->scsi_done(srb);
	}
	spin_unlock_irqrestore(&(dev_info->srb_lock), flags);
}

static void ps3pf_stor_process_srb(struct scsi_cmnd * srb)
{
	struct ps3pf_stor_dev_info * dev_info;
	int (*command_handler)(struct ps3pf_stor_dev_info *, struct scsi_cmnd *);

	dev_info = (struct ps3pf_stor_dev_info*) srb->device->hostdata;
	command_handler = dev_info->handler_info[srb->cmnd[0]].cmnd_handler;

	if (command_handler) {
		(*command_handler)(dev_info, srb);
	} else {
		srb->result = (DID_ERROR << 16);
		memset(srb->sense_buffer, 0, SCSI_SENSE_BUFFERSIZE);
		srb->sense_buffer[0] = 0x70;
		srb->sense_buffer[2] = ILLEGAL_REQUEST;
		ps3pf_stor_srb_done(dev_info);
	}
}

/*
 * copy data from device into scatter/gather buffer
 */
static int fill_from_dev_buffer(struct scsi_cmnd * srb,
				const unsigned char * arr,
				int arr_len)
{
	int k, req_len, act_len, len, active;
	void * kaddr;
	void * kaddr_off;
	struct scatterlist * sgpnt;

	if (0 == srb->request_bufflen)
		return 0;
	if (NULL == srb->request_buffer)
		return (DID_ERROR << 16);
	if (! ((srb->sc_data_direction == DMA_BIDIRECTIONAL) ||
	      (srb->sc_data_direction == DMA_FROM_DEVICE)))
		return (DID_ERROR << 16);
	if (0 == srb->use_sg) {
		req_len = srb->request_bufflen;
		act_len = (req_len < arr_len) ? req_len : arr_len;
		memcpy(srb->request_buffer, arr, act_len);
		srb->resid = req_len - act_len;
		return 0;
	}
	sgpnt = (struct scatterlist *)srb->request_buffer;
	active = 1;
	for (k = 0, req_len = 0, act_len = 0; k < srb->use_sg; ++k, ++sgpnt) {
		if (active) {
			kaddr = (unsigned char *)
				kmap_atomic(sgpnt->page, KM_USER0);
			if (NULL == kaddr)
				return (DID_ERROR << 16);
			kaddr_off = (unsigned char *)kaddr + sgpnt->offset;
			len = sgpnt->length;
			if ((req_len + len) > arr_len) {
				active = 0;
				len = arr_len - req_len;
			}
			memcpy(kaddr_off, arr + req_len, len);
			kunmap_atomic(kaddr, KM_USER0);
			act_len += len;
		}
		req_len += sgpnt->length;
	}
	srb->resid = req_len - act_len;
	return 0;
}

/*
 * copy data from scatter/gather into device's buffer
 */
static int fetch_to_dev_buffer(struct scsi_cmnd * srb,
			       unsigned char * arr,
			       int max_arr_len)
{
	int k, req_len, len, fin;
	void * kaddr;
	void * kaddr_off;
	struct scatterlist * sgpnt;

	if (0 == srb->request_bufflen)
		return 0;
	if (NULL == srb->request_buffer)
		return -1;
	if (! ((srb->sc_data_direction == DMA_BIDIRECTIONAL) ||
	      (srb->sc_data_direction == DMA_TO_DEVICE)))
		return -1;
	if (0 == srb->use_sg) {
		req_len = srb->request_bufflen;
		len = (req_len < max_arr_len) ? req_len : max_arr_len;
		memcpy(arr, srb->request_buffer, len);
		return len;
	}

	sgpnt = (struct scatterlist *)srb->request_buffer;
	for (k = 0, req_len = 0, fin = 0; k < srb->use_sg; ++k, ++sgpnt) {
		kaddr = (unsigned char *)kmap_atomic(sgpnt->page, KM_USER0);
		if (NULL == kaddr)
			return -1;
		kaddr_off = (unsigned char *)kaddr + sgpnt->offset;
		len = sgpnt->length;
		if ((req_len + len) > max_arr_len) {
			len = max_arr_len - req_len;
			fin = 1;
		}
		memcpy(arr + req_len, kaddr_off, len);
		kunmap_atomic(kaddr, KM_USER0);
		if (fin)
			return req_len + len;
		req_len += sgpnt->length;
	}
	return req_len;
}

/*
 * copy data into device buffer to write.
 * byte offset 'from' until byte offset 'to'
 * data always copied into 'arr'
 */
static off_t fetch_to_dev_buffer_abs(struct scsi_cmnd * srb,
				     unsigned char * arr,
				     off_t from,
				     off_t to)
{
	int i, fin;
	void * kaddr;
	void * kaddr_off;
	off_t cur_pos, end_pos, start_pos, len;
	struct scatterlist * sg;

	if (0 == srb->request_bufflen)
		return 0;
	if (NULL == srb->request_buffer)
		return -1;
	if (! ((srb->sc_data_direction == DMA_BIDIRECTIONAL) ||
	      (srb->sc_data_direction == DMA_TO_DEVICE)))
		return -1;
	if (to < from)
		return -1;

	DPRINTK(KERN_ERR "%s: from=%#lx(%ld) to=%#lx(%ld) sg=%d\n", __FUNCTION__,
		from, from , to, to, srb->use_sg);

	if (0 == srb->use_sg) {
		len = (srb->request_bufflen < to) ? (srb->request_bufflen - from) : (to - from);
		memcpy(arr, srb->request_buffer + from, len);
		return len;
	}


	len = 0;
	sg = (struct scatterlist *)srb->request_buffer;
	for (i = 0, cur_pos = 0, fin = 0;
	     (i < srb->use_sg) && !fin;
	     cur_pos += sg->length, i++, sg++) {
		kaddr = (unsigned char *)kmap_atomic(sg->page, KM_USER0);
		kaddr_off = (unsigned char *)kaddr + sg->offset;

		//DPRINTK(KERN_ERR "%s: cur_pos=%ld, sglen=%d kadoff=%p\n", __FUNCTION__,
		//cur_pos, sg->length, kaddr_off);
		if (NULL == kaddr)
			return -1;

		if (from <= cur_pos) {
			start_pos = cur_pos;
		}  else {
			if (from < (cur_pos + sg->length)) {
				/* copy start with middle of this segment */
				start_pos = from;
				kaddr_off += from - start_pos;
			} else {
				/* this segment does not have any desired data */
				kunmap_atomic(kaddr, KM_USER0);
				continue;
			}
		}

		if (to < (cur_pos + sg->length)) {
			/* copy end with middle of this segment */
			end_pos = to;
			fin = 1;
		} else {
			end_pos = cur_pos + sg->length;
		}

		if (start_pos < end_pos) {
			//DPRINTK(KERN_ERR "%s: COPY start=%ld end=%ld kaddoff=%p\n", __FUNCTION__,
			//start_pos, end_pos, kaddr_off);
			memcpy(arr + len, kaddr_off, end_pos - start_pos);
			len += end_pos - start_pos;
		}
		kunmap_atomic(kaddr, KM_USER0);

	}
	DPRINTK(KERN_ERR "%s: return ren=%ld\n", __FUNCTION__, len);
	return len;
}


/*
 * issue PACKET command according to passed SRB
 * caller will block until the command completed.
 * returns 0 if command sucessfully done,
 * otherwise error detected.
 * if auto_sense is on, request_sense is automatically issued and
 * return the sense data into srb->sensebuffer[SCSI_SENSE_BUFFERSIZE]
 * srb->result will be set.
 * caller should call done(srb) to inform mid layer the command completed.
 */
static int issue_atapi_by_srb(struct ps3pf_stor_dev_info * dev_info, int auto_sense)
{
	struct lv1_atapi_cmnd_block atapi_cmnd;
	const struct scsi_command_handler_info * handler_info;
	unsigned char * cmnd = dev_info->srb->cmnd;
	int len = 0;
	unsigned char * bounce_buf = NULL;
	uint64_t ret64;
	unsigned char keys[4];

	DPRINTK(KERN_ERR "start %#x\n", cmnd[0]);
	handler_info = &(dev_info->handler_info[cmnd[0]]);

	/* calc buffer length */
	switch (handler_info->buflen) {
	case USE_SRB_6:
		len = cmnd[4];
		break;
	case USE_SRB_10:
 		len = (cmnd[7] << 8) | cmnd[8];
		break;
	case USE_CDDA_FRAME_RAW:
		len = ((cmnd[6] << 16) |
		       (cmnd[7] <<  8) |
		       (cmnd[8] <<  0)) * CD_FRAMESIZE_RAW;
		break;
	default:
		len = handler_info->buflen;
	}

	if ((dev_info->bounce_type == DYNAMIC_BOUNCE) && len) {
		/* alloc bounce buffer if needed */
		bounce_buf = kmalloc(len, GFP_NOIO | __GFP_DMA);
		if (!bounce_buf) {
			printk(KERN_ERR "%s: no memory \n", __FUNCTION__);
			dev_info->srb->result = DID_ERROR <<16;
			return -1;
		}
	} else {
		/* check buffer length */
		if (dev_info->dedicated_bounce_size < len) {
			printk(KERN_ERR "%s: requested length exceeded dedicated buffer %d < %d\n",
			       __FUNCTION__, dev_info->dedicated_bounce_size, len);
			dev_info->srb->result = DID_ERROR <<16;
			return -1;
		}
		bounce_buf = dev_info->bounce_buf;
	}

	memset(&atapi_cmnd, 0, sizeof(struct lv1_atapi_cmnd_block));
	memcpy(&(atapi_cmnd.pkt), cmnd, 12);
	atapi_cmnd.pktlen = 12;
	atapi_cmnd.proto = handler_info->proto;
	if (handler_info->in_out != NOT_AVAIL)
		atapi_cmnd.in_out = handler_info->in_out;

	if (atapi_cmnd.in_out == DIR_WRITE) {
		fetch_to_dev_buffer(dev_info->srb, bounce_buf, len);
	}

	atapi_cmnd.block_size = 1; /* transfer size is block_size * blocks */
	atapi_cmnd.blocks = atapi_cmnd.arglen = len;

	if (len)
		atapi_cmnd.buffer = ps3pf_stor_virtual_to_lpar(dev_info, bounce_buf);

	/* issue command */
	init_completion(&(dev_info->irq_done));
	ret64 = lv1_storage_send_device_command(dev_info->lv1_dev_info->device_id,
						LV1_STORAGE_SEND_ATAPI_COMMAND,
						p_to_lp(__pa(&atapi_cmnd)), /* no need special convert */
						sizeof(struct lv1_atapi_cmnd_block),
						atapi_cmnd.buffer,
						atapi_cmnd.arglen,
						&dev_info->lv1_dev_info->current_tag);
	if (ret64) {
		printk(KERN_ERR "%s: send_device failed lv1dev=%ld ret=%ld\n", __FUNCTION__,
		       dev_info->lv1_dev_info->device_id, ret64);
		dev_info->srb->result = DID_ERROR << 16; /* FIXME: is better other error code ? */
		if (dev_info->bounce_type == DYNAMIC_BOUNCE)
			kfree(bounce_buf);
		return -1;
	}

	/* wait interrupt */
	wait_for_completion(&(dev_info->irq_done));

	/* check error */
	if (!dev_info->lv1_status) {
		/* OK, completed */
		if (atapi_cmnd.in_out == DIR_READ)
			fill_from_dev_buffer(dev_info->srb, bounce_buf, atapi_cmnd.arglen);
		dev_info->srb->result = DID_OK << 16;
		if (dev_info->bounce_type == DYNAMIC_BOUNCE)
			kfree(bounce_buf);
		DPRINTK(KERN_ERR "normal end len=%d \n", len);
		return 0;
	}

	if (dev_info->bounce_type == DYNAMIC_BOUNCE) {
		kfree(bounce_buf);
		bounce_buf = NULL;
	}
	/* error */
	if (!auto_sense) {
		dev_info->srb->result = (DID_ERROR << 16) | (CHECK_CONDITION << 1);
		printk(KERN_ERR "%s: end error withtout autosense\n", __FUNCTION__);
		return 1;
	}

	/* followings are only for auto_sense*/
	if (!decode_lv1_status(dev_info->lv1_status,
			       &(keys[0]), &(keys[1]), &(keys[2]))) {
		/* lv1 may have issued autosense ... */
		dev_info->srb->sense_buffer[0]  = 0x70;
		dev_info->srb->sense_buffer[2]  = keys[0];
		dev_info->srb->sense_buffer[7]  = 16 - 6;
		dev_info->srb->sense_buffer[12] = keys[1];
		dev_info->srb->sense_buffer[13] = keys[2];
		dev_info->srb->result = SAM_STAT_CHECK_CONDITION;
	} else {
		/* do auto sense by our selves*/
		bounce_buf = kzalloc(18, GFP_NOIO);
		memset(&atapi_cmnd, 0, sizeof(struct lv1_atapi_cmnd_block));
		atapi_cmnd.pkt[0] = REQUEST_SENSE;
		atapi_cmnd.pkt[4] = 18;
		atapi_cmnd.pktlen = 12;
		atapi_cmnd.arglen = atapi_cmnd.blocks = atapi_cmnd.pkt[4];
		atapi_cmnd.block_size = 1;
		atapi_cmnd.proto = DMA_PROTO;
		atapi_cmnd.in_out = DIR_READ;
		atapi_cmnd.buffer = p_to_lp(__pa(bounce_buf));

		/* issue REQUEST_SENSE command */
		init_completion(&(dev_info->irq_done));
		ret64 = lv1_storage_send_device_command(dev_info->lv1_dev_info->device_id,
							LV1_STORAGE_SEND_ATAPI_COMMAND,
							p_to_lp(__pa(&atapi_cmnd)),
							sizeof(struct lv1_atapi_cmnd_block),
							atapi_cmnd.buffer,
							atapi_cmnd.arglen,
							&dev_info->lv1_dev_info->current_tag);
		if (ret64) {
			printk(KERN_ERR "%s: send_device for request sense failed lv1dev=%ld ret=%ld\n", __FUNCTION__,
			       dev_info->lv1_dev_info->device_id, ret64);
			dev_info->srb->result = DID_ERROR << 16; /* FIXME: is better other error code ? */
			kfree(bounce_buf);
			return -1;
		}

		/* wait interrupt */
		wait_for_completion(&(dev_info->irq_done));

		/* scsi spec says request sense should never get error */
		if (dev_info->lv1_status) {
			decode_lv1_status(dev_info->lv1_status,
					  &(keys[0]), &(keys[1]), &(keys[2]));
			printk(KERN_ERR "%s: auto REQUEST_SENSE error %#x %#x %#x\n", __FUNCTION__,
			       keys[0], keys[1], keys[2]);
		}

		memcpy(dev_info->srb->sense_buffer, bounce_buf, min((int)atapi_cmnd.pkt[4], SCSI_SENSE_BUFFERSIZE));
		kfree(bounce_buf);
		dev_info->srb->result = SAM_STAT_CHECK_CONDITION;
	}

	DPRINTK(KERN_ERR "error end \n");
	return 1;
}

/*
 * just send command with auto REQUEST_SENSE
 */
static int ps3pf_stor_atapi_handle_simple(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	int ret;

	ret = issue_atapi_by_srb(dev_info, 1);
	ps3pf_stor_srb_done(dev_info);
	return ret;
}

/*
 * just send command WITHOUT auto REQUEST_SENSE
 */
static int ps3pf_stor_atapi_handle_request_sense(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	int ret;

	ret = issue_atapi_by_srb(dev_info, 0);
	ps3pf_stor_srb_done(dev_info);
	return ret;
}

/******************************************************
 * handlers for HDD
 */

static int ps3pf_stor_hdd_handle_inquiry(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	unsigned char inquiry_reply[PS3PF_STOR_MAX_INQUIRY_DATA_SIZE];
	unsigned char *cmd = (unsigned char *)srb->cmnd;
	const char * msg;
	int alloc_len;
	int ret;

	FUNC_START;
	alloc_len = (cmd[3] << 8) + cmd[4];
	memset(inquiry_reply, 0, PS3PF_STOR_MAX_INQUIRY_DATA_SIZE);
	inquiry_reply[0] = dev_info->lv1_dev_info->device_type;
	inquiry_reply[1] = 0;  /* Removable flag */
	inquiry_reply[2] = 2;  /* ANSI version */
	inquiry_reply[3] = 2;  /* response_data_format==2 */
	inquiry_reply[4] = PS3PF_STOR_INQUIRY_DATA_SIZE - 5;

	sprintf(&inquiry_reply[8], "%-8s", "SCEI");
	if (dev_info->lv1_dev_info->device_type == TYPE_DISK) {
		switch (dev_info->lv1_dev_info->attached_port) {
		case 0:
			msg = "Pri:Master";
			break;
		case 1:
			msg = "Pri:Slave";
			break;
		case 2:
			msg = "Sec:Master";
			break;
		case 3:
			msg = "Sec:Slave";
			break;
		default:
			msg = "Unknown";
			break;

		}
	} else {
		msg = "Flash";
	}

	/* SCSI spec requires model name left aligned, spece padded */
	ret = sprintf(&inquiry_reply[16], "%s-%d",
		      msg,
		      dev_info->lv1_dev_info->region_info_array[srb->cmnd[1]>>5].region_index);
	if (ret < 16)
		memset(&(inquiry_reply[16 + ret]), ' ', 16 - ret);

	sprintf(&inquiry_reply[32], "%-4d", 4989);

	inquiry_reply[58] = 0x0; inquiry_reply[59] = 0x40; /* SAM-2 */
	inquiry_reply[60] = 0x3; inquiry_reply[61] = 0x0;  /* SPC-3 */
	inquiry_reply[62] = 0x1; inquiry_reply[63] = 0x80; /* SBC */

	ret = fill_from_dev_buffer(dev_info->srb, inquiry_reply, min(alloc_len, PS3PF_STOR_INQUIRY_DATA_SIZE));

	srb->result = DID_OK << 16;
	ps3pf_stor_srb_done(dev_info);
	FUNC_END;
	return ret;
}


static int ps3pf_stor_hdd_handle_request_sense(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	unsigned char sense_data[PS3PF_STOR_SENSE_LEN];
	int len = 18;

	memset(sense_data, 0, PS3PF_STOR_SENSE_LEN);

	if (dev_info->lv1_status) {
		if (!decode_lv1_status(dev_info->lv1_status,
				       &(sense_data[2]),
				       &(sense_data[12]),
				       &(sense_data[13]))) {
		} else {
			/* unknown error */
			printk(KERN_ERR "%s: FIXME issue real RS %#lx %#lx\n", __FUNCTION__,
			       dev_info->lv1_status, dev_info->lv1_retval);
			sense_data[2] = HARDWARE_ERROR;
			dev_info->srb->result = DID_OK << 16;
		}
		sense_data[0] = 0x70;
	} else {
		/* no sense */
		sense_data[0] = 0x70;
		dev_info->srb->result = DID_OK << 16;
	}

	fill_from_dev_buffer(dev_info->srb, sense_data, len);
	ps3pf_stor_srb_done(dev_info);
	return 0;
}

static int ps3pf_stor_hdd_handle_just_ok(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	dev_info->srb->result = DID_OK << 16;
	ps3pf_stor_srb_done(dev_info);
	return 0;
}

static int ps3pf_stor_hdd_handle_sync_cache(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	unsigned char keys[4];
	uint64_t ret64;

	/* issue command */
	init_completion(&(dev_info->irq_done));
	ret64 = lv1_storage_send_device_command(dev_info->lv1_dev_info->device_id,
						LV1_STORAGE_ATA_HDDOUT,
						0,
						0,
						0,
						0,
						&dev_info->lv1_dev_info->current_tag);
	if (ret64) {
		/* error */
		printk(KERN_ERR "%s: send_device failed. lv1dev=%ld ret=%ld\n", __FUNCTION__,
		       dev_info->lv1_dev_info->device_id, ret64);
		dev_info->srb->result = DID_ERROR << 16; /* FIXME: is better other error code? */
	} else {
		/* wait interrupt */
		wait_for_completion(&(dev_info->irq_done));

		/* check error */
		if (!dev_info->lv1_status) {
			dev_info->srb->result = DID_OK << 16;
		} else {
			decode_lv1_status(dev_info->lv1_status,
					  &(keys[0]), &(keys[1]), &(keys[2]));
			dev_info->srb->sense_buffer[0]  = 0x70;
			dev_info->srb->sense_buffer[2]  = keys[0];
			dev_info->srb->sense_buffer[7]  = 16 - 6;
			dev_info->srb->sense_buffer[12] = keys[1];
			dev_info->srb->sense_buffer[13] = keys[2];
			dev_info->srb->result = SAM_STAT_CHECK_CONDITION;
		}
	}

	ps3pf_stor_srb_done(dev_info);
	return 0;
}

static int ps3pf_stor_hdd_handle_read_capacity(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	unsigned char data[PS3PF_STOR_READCAP_DATA_SIZE];
	unsigned long len;
	int ret;

	FUNC_START;
	memset(data, 0, sizeof(data));
	len = (unsigned long)dev_info->lv1_dev_info->region_info_array[srb->cmnd[1] >> 5].region_size - 1;
	data[0] = (len >> 24) & 0xff;
	data[1] = (len >> 16) & 0xff;
	data[2] = (len >> 8)  & 0xff;
	data[3] =  len        & 0xff;

	len = (unsigned long)dev_info->lv1_dev_info->sector_size;
	data[4] = (len >> 24) & 0xff;
	data[5] = (len >> 16) & 0xff;
	data[6] = (len >> 8)  & 0xff;
	data[7] =  len        & 0xff;

	ret = fill_from_dev_buffer(dev_info->srb, data, PS3PF_STOR_READCAP_DATA_SIZE);
	dev_info->srb->result = DID_OK << 16;
	ps3pf_stor_srb_done(dev_info);
	FUNC_END;
	return ret;
}


static int copy_page_data(unsigned char * buf, const unsigned char * data,
			   int length, int changable)
{
	if (changable) {
		/* reports no parameters are changable */
		memcpy(buf, data, 2);
		memset(buf + 2, 0, length - 2);
	} else {
		memcpy(buf, data, length);
	}
	return length;
}

static int fill_mode_page(struct ps3pf_stor_dev_info * dev_info, unsigned char *buf, int page, int changable)
{
	int length;

	switch (page){
	case 8:
		/* TYPE_DISK; see sd_read_cache_type():sd.c */
		length = copy_page_data(buf, page_data_8, sizeof(page_data_8), changable);
		break;
	case 6:
		/* TYPE_RBC */
		length = copy_page_data(buf, page_data_6, sizeof(page_data_6), changable);
		break;
	case 0x3f: /* ALL PAGES, but sd.c checks only parameter header to see WriteProtect */
		length  = copy_page_data(buf, page_data_6, sizeof(page_data_6), changable);
		length += copy_page_data(buf + length, page_data_8, sizeof(page_data_8), changable);
		break;
	default:
		printk(KERN_ERR "%s: unknown page=%#x\n", __FUNCTION__, page);
		return 0;
	}

	return length;
}

/*
 * scsi disk driver asks only PAGE= 0x3f, 6(RBC), 8(SCSI disk)
 */
static int ps3pf_stor_hdd_handle_mode_sense(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	unsigned char sense_data[128];
	int offset = 0;

	/*
	 * NOTE: support MODE_SENSE_10 only
	 * see slave_cofigure()
	 */
	memset(sense_data, 0, sizeof(sense_data));
	/* parameter header */
	sense_data[2] = dev_info->lv1_dev_info->device_type;
	sense_data[3] = 0;      /* mid layer wants to see here     */
	/* bit 7=1 means WriteProtected    */
	offset = fill_mode_page(dev_info,
				&(sense_data[8]),
				dev_info->srb->cmnd[2] & 0x3f,
				dev_info->srb->cmnd[2] & 0xc0);
	sense_data[1] = offset + 8;        /* parameter length */
	sense_data[0] = (offset + 8) >> 8;

	fill_from_dev_buffer(dev_info->srb, sense_data, offset + 8);
	ps3pf_stor_srb_done(dev_info);
	return 0;
}


static int ps3pf_stor_common_handle_read(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	uint64_t ret64;
 	uint64_t lpar_addr, region_id;
	uint32_t sectors = 0;
	uint32_t start_sector = 0;
	unsigned char *cmnd = dev_info->srb->cmnd;
	int ret = 0;

	/* check transfer length */
	switch (cmnd[0]) {
	case READ_10:
		start_sector = (cmnd[2] << 24) +
			(cmnd[3] << 16) +
			(cmnd[4] <<  8) +
			cmnd[5];
		sectors = (cmnd[7] << 8) +
			cmnd[8];
		break;
	case READ_6:
		start_sector = (cmnd[1] << 16) +
			(cmnd[2] <<  8) +
			cmnd[3];
		sectors = cmnd[4];
		break;

	}

 	/* bounce buffer */
	if (dev_info->bounce_type == DYNAMIC_BOUNCE) {
		dev_info->bounce_buf = kmalloc(sectors * dev_info->sector_size, GFP_NOIO | __GFP_DMA);
		if (!dev_info->bounce_buf) {
			printk(KERN_ERR "%s: no memory \n", __FUNCTION__);
			dev_info->srb->result = DID_ERROR <<16;
			ps3pf_stor_srb_done(dev_info);
			return -1;
		}
	}

	/* issue read */
	read_lock(&dev_info->bounce_lock);
	lpar_addr = ps3pf_stor_virtual_to_lpar(dev_info, dev_info->bounce_buf);
	region_id = dev_info->lv1_dev_info->region_info_array[(cmnd[1] >> 5)].region_id;
	init_completion(&(dev_info->irq_done));
	ret64 = lv1_storage_read((uint64_t)dev_info->lv1_dev_info->device_id,
				 (uint64_t)region_id,
				 (uint64_t)start_sector,
				 (uint64_t)sectors,
				 (uint64_t)0, /* flags */
				 (uint64_t)lpar_addr,
				 &(dev_info->lv1_dev_info->current_tag));
	if (ret64) {
		/* error */
		printk(KERN_ERR "%s: error lv1dev =%ld ret=%ld\n", __FUNCTION__,
		       dev_info->lv1_dev_info->device_id, ret64);
		dev_info->srb->result = DID_ERROR << 16; /* FIXME: other error code? */
		ret = -1;
	} else {
		/* wait irq */
		wait_for_completion(&(dev_info->irq_done));
		if (dev_info->lv1_status) {
			/* error */
			memset(dev_info->srb->sense_buffer, 0, SCSI_SENSE_BUFFERSIZE);
			decode_lv1_status(dev_info->lv1_status,
					  &(dev_info->srb->sense_buffer[2]),
					  &(dev_info->srb->sense_buffer[12]),
					  &(dev_info->srb->sense_buffer[13]));
			dev_info->srb->sense_buffer[7] = 16 - 6;
			dev_info->srb->result = SAM_STAT_CHECK_CONDITION;
			ret =  1;
		} else {
			/* OK */
			fill_from_dev_buffer(dev_info->srb,
					     dev_info->bounce_buf,
					     sectors * dev_info->sector_size);

			dev_info->srb->result = DID_OK << 16;
			ret =  0;
		}
	}

	ps3pf_stor_srb_done(dev_info);
	if (dev_info->bounce_type == DYNAMIC_BOUNCE)
		kfree(dev_info->bounce_buf);
	read_unlock(&dev_info->bounce_lock);
	return ret;
}

static int ps3pf_stor_common_handle_write(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	int ret;
	uint64_t ret64;
	uint64_t lpar_addr, region_id;
	uint32_t start_sector = 0;
	uint32_t sectors = 0;
	unsigned char * cmnd = dev_info->srb->cmnd;

	/* check transfer length */
	switch (cmnd[0]) {
	case WRITE_10:
		start_sector = (cmnd[2] << 24) +
			(cmnd[3] << 16) +
			(cmnd[4] <<  8) +
			cmnd[5];
		sectors = (cmnd[7] << 8) +
			cmnd[8];
		break;
	case WRITE_6:
		start_sector = (cmnd[1] << 16) +
			(cmnd[2] <<  8) +
			cmnd[3];
		sectors = cmnd[4];
		break;
	}

	/* bounce buf */
	if (dev_info->bounce_type == DYNAMIC_BOUNCE) {
		dev_info->bounce_buf = kmalloc(sectors * dev_info->sector_size, GFP_NOIO | __GFP_DMA);
		if (!dev_info->bounce_buf) {
			printk(KERN_ERR "%s: no memory \n", __FUNCTION__);
			dev_info->srb->result = DID_ERROR <<16;
			ps3pf_stor_srb_done(dev_info);
			return -1;
		}
	}

	read_lock(&dev_info->bounce_lock);
	ret = fetch_to_dev_buffer(dev_info->srb,
				  dev_info->bounce_buf,
				  sectors * dev_info->sector_size);

 	lpar_addr = ps3pf_stor_virtual_to_lpar(dev_info, dev_info->bounce_buf);
	region_id = dev_info->lv1_dev_info->region_info_array[(cmnd[1] >> 5)].region_id;
	init_completion(&(dev_info->irq_done));
	ret64 = lv1_storage_write((uint64_t)dev_info->lv1_dev_info->device_id,
				 (uint64_t)region_id, /* region id */
				 (uint64_t)start_sector,
				 (uint64_t)sectors,
				 (uint64_t)0, /* flags */
				 (uint64_t)lpar_addr/*srb->request_buffer*/, /* assume non SG! */
				 &(dev_info->lv1_dev_info->current_tag));
	if (ret64) {
		/* error */
		printk(KERN_ERR "%s: error lv1dev =%ld ret=%ld\n", __FUNCTION__,
		       dev_info->lv1_dev_info->device_id, ret64);
		dev_info->srb->result = DID_ERROR << 16; /* FIXME: other error code? */
		ret = -1;
	} else {

		/* wait irq */
		wait_for_completion(&(dev_info->irq_done));

		if (dev_info->lv1_status) {
			/* error */
			memset(dev_info->srb->sense_buffer, 0, SCSI_SENSE_BUFFERSIZE);
			decode_lv1_status(dev_info->lv1_status,
					  &(dev_info->srb->sense_buffer[2]),
					  &(dev_info->srb->sense_buffer[12]),
					  &(dev_info->srb->sense_buffer[13]));
			dev_info->srb->sense_buffer[7] = 16 - 6;
			dev_info->srb->result = SAM_STAT_CHECK_CONDITION;
			ret = 1;
		} else {
			/* OK */
			dev_info->srb->result = DID_OK << 16;
			ret = 0;
		}

	}
	ps3pf_stor_srb_done(dev_info);
	if (dev_info->bounce_type == DYNAMIC_BOUNCE)
		kfree(dev_info->bounce_buf);
	read_unlock(&dev_info->bounce_lock);
	return ret;
}

static int is_aligned_flash(uint32_t sector, int sector_size)
{
	uint32_t flash_align_sector = FLASH_ALIGN / sector_size;

	return (sector % flash_align_sector)? 0 : 1;
}

static uint32_t floor_align_flash(uint32_t sector, int sector_size)
{
	uint32_t flash_align_sector = FLASH_ALIGN / sector_size;

	return sector & ~(flash_align_sector - 1);
}

static uint32_t ceil_align_flash(uint32_t sector, int sector_size)
{
	uint32_t flash_align_sector = FLASH_ALIGN / sector_size;

	return (sector + (flash_align_sector - 1)) & ~(flash_align_sector - 1);
}

/*
 * special handling for flash drive; do safer way to write in order to reduce
 * the risk of flash corruption by sudden power off.
 */
static int ps3pf_stor_handle_write_flash(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb)
{
	int ret = 0;
	uint64_t ret64, sector_size;
	uint64_t lpar_addr, region_id;
	uint64_t start_sector = 0;
	uint64_t start_sector_aligned = 0;
 	uint64_t sectors = 0;
 	uint64_t sectors_aligned = 0;
	uint64_t current_sector;
	uint64_t aligned_sector_count;
	unsigned char * cmnd = dev_info->srb->cmnd;
	void * current_buffer;
	struct ps3pf_stor_lv1_region_info * region_info;

	static int align_warned;

	DPRINTK(KERN_ERR "%s: start\n", __FUNCTION__);

	/* check transfer length */
	switch (cmnd[0]) {
	case WRITE_10:
		start_sector = (cmnd[2] << 24) +
			(cmnd[3] << 16) +
			(cmnd[4] <<  8) +
			cmnd[5];
		sectors = (cmnd[7] << 8) +
			cmnd[8];
		break;
	case WRITE_6:
		start_sector = (cmnd[1] << 16) +
			(cmnd[2] <<  8) +
			cmnd[3];
		sectors = cmnd[4];
		break;
	}


        /*
         *    start_sector_aligned
         *   /          start_sector
         *  /          /
         * +----------+--------------------+---+
         *            |<-    sectors     ->|   |
         *            |<-   sectors_aligned  ->|
         *
         * ^-----------------------------------^ 256K align
         */
	sector_size = dev_info->sector_size;
	aligned_sector_count = FLASH_ALIGN / sector_size;

	start_sector_aligned = floor_align_flash(start_sector, sector_size);
	sectors_aligned = ceil_align_flash(start_sector + sectors, sector_size) - start_sector;

	/* check aligned border exceed region */
	region_info = &dev_info->lv1_dev_info->region_info_array[cmnd[1] >> 5];
	if (!is_aligned_flash(region_info->region_start, sector_size) ||
	    (region_info->region_size < (start_sector_aligned + sectors_aligned))) {
		if (!align_warned) {
			printk(KERN_ERR "%s: region alignment is not 256k, continue to work with norman method\n",
			       __FUNCTION__);
			align_warned = 1;
		}
		return ps3pf_stor_common_handle_write(dev_info, srb);
	};

	/* bounce buf */
	if (dev_info->bounce_type == DYNAMIC_BOUNCE) {
		dev_info->bounce_buf = kmalloc(sectors_aligned * sector_size, GFP_NOIO | __GFP_DMA);
		if (!dev_info->bounce_buf) {
			printk(KERN_ERR "%s: no memory \n", __FUNCTION__);
			dev_info->srb->result = DID_ERROR <<16;
			ps3pf_stor_srb_done(dev_info);
			return -1;
		}
	}

	read_lock(&dev_info->bounce_lock);
	region_id = region_info->region_id;


	DPRINTK(KERN_ERR "%s: start=%#lx(%ld) start_a=%#lx(%ld) sec=%#lx(%ld) sec_a=%#lx(%ld)\n", __FUNCTION__,
		start_sector, start_sector,
		start_sector_aligned, start_sector_aligned,
		sectors, sectors,
		sectors_aligned, sectors_aligned);

	/*
	 * loop in the case that the requested write sectors across
	 * 245Kb alignment.  Since we have set max_sectors as 256kb,
	 * loop count is up to 2.
	 */
	for (current_sector = start_sector_aligned, ret = 0;
	     (current_sector < (start_sector + sectors_aligned)) && !ret;
	     current_sector += aligned_sector_count) {

		DPRINTK(KERN_ERR "%s: LOOP current=%#lx\n", __FUNCTION__, current_sector);

		current_buffer = dev_info->bounce_buf;

		/* read from (start_sector_aligned) to (start_sector) */
		if (current_sector < start_sector) {
			DPRINTK(KERN_ERR "%s: head read \n", __FUNCTION__);
			lpar_addr = ps3pf_stor_virtual_to_lpar(dev_info, current_buffer);
			init_completion(&(dev_info->irq_done));
			ret64 = lv1_storage_read((uint64_t)dev_info->lv1_dev_info->device_id,
						 (uint64_t)region_id,
						 (uint64_t)current_sector,
						 (uint64_t)(start_sector - current_sector),
						 (uint64_t)0,
						 (uint64_t)lpar_addr,
						 &(dev_info->lv1_dev_info->current_tag));
			DPRINTK(KERN_ERR "HEAD start=%#lx, len=%#lx\n",
				start_sector_aligned, (start_sector - start_sector_aligned));
			if (ret64) {
				/* error */
				printk(KERN_ERR "%s: error lv1dev =%ld ret=%ld\n", __FUNCTION__,
				       dev_info->lv1_dev_info->device_id, ret64);
				dev_info->srb->result = DID_ERROR << 16; /* FIXME: other error code? */
				ret = -1;
				goto done;
			} else {
				/* wait irq */
				wait_for_completion(&(dev_info->irq_done));
			}
			if (dev_info->lv1_status) {
				/* error */
				memset(dev_info->srb->sense_buffer, 0, SCSI_SENSE_BUFFERSIZE);
				decode_lv1_status(dev_info->lv1_status,
						  &(dev_info->srb->sense_buffer[2]),
						  &(dev_info->srb->sense_buffer[12]),
						  &(dev_info->srb->sense_buffer[13]));
				dev_info->srb->sense_buffer[7] = 16 - 6;
				dev_info->srb->result = SAM_STAT_CHECK_CONDITION;
				ret = 1;
				goto done;
			} else {
				/* OK */
				ret = 0;
				current_buffer += (start_sector - start_sector_aligned) * sector_size;
			}
		} /* head remainder */


		if ((start_sector + sectors) < (current_sector + aligned_sector_count)) {
			void * buf = dev_info->bounce_buf;
			DPRINTK(KERN_ERR "%s: tail read\n", __FUNCTION__);
			buf += (start_sector + sectors - current_sector) * sector_size;
			lpar_addr = ps3pf_stor_virtual_to_lpar(dev_info, buf);
			init_completion(&(dev_info->irq_done));
			ret64 = lv1_storage_read((uint64_t)dev_info->lv1_dev_info->device_id,
						 (uint64_t)region_id,
						 (uint64_t)(start_sector + sectors),
						 (uint64_t)(sectors_aligned - sectors),
						 (uint64_t)0,
						 (uint64_t)lpar_addr,
						 &(dev_info->lv1_dev_info->current_tag));
			DPRINTK(KERN_ERR "TAIL start=%#lx, len=%#lx\n",
				start_sector + sectors, sectors_aligned - sectors);
			if (ret64) {
				/* error */
				printk(KERN_ERR "%s: error lv1dev =%ld ret=%ld\n", __FUNCTION__,
				       dev_info->lv1_dev_info->device_id, ret64);
				dev_info->srb->result = DID_ERROR << 16; /* FIXME: other error code? */
				ret = -1;
				goto done;
			} else {
				/* wait irq */
				wait_for_completion(&(dev_info->irq_done));
			}
			if (dev_info->lv1_status) {
				/* error */
				memset(dev_info->srb->sense_buffer, 0, SCSI_SENSE_BUFFERSIZE);
				decode_lv1_status(dev_info->lv1_status,
						  &(dev_info->srb->sense_buffer[2]),
						  &(dev_info->srb->sense_buffer[12]),
						  &(dev_info->srb->sense_buffer[13]));
				dev_info->srb->sense_buffer[7] = 16 - 6;
				dev_info->srb->result = SAM_STAT_CHECK_CONDITION;
				ret = 1;
				goto done;
			} else {
				/* OK */
				ret = 0;
			}
		} /* tail remainder */

		{
			uint64_t copy_sectors_from, copy_sectors_to;

			/* start_sector is within this iteration */
			if ((current_sector < start_sector)  &&
			    (start_sector < (current_sector + aligned_sector_count))) {
				copy_sectors_from = start_sector;
			}
			else {
				copy_sectors_from = current_sector;
			}

			/* start_sector+sectors is within this iteration */
			if ((current_sector < (start_sector + sectors))  &&
			    ((start_sector + sectors) < (current_sector + aligned_sector_count))) {
				copy_sectors_to = start_sector + sectors;
			}
			else {
				copy_sectors_to = current_sector + aligned_sector_count;
			}

			DPRINTK(KERN_ERR "%s: copy to current=%p\n", __FUNCTION__, current_buffer);
			ret = fetch_to_dev_buffer_abs(dev_info->srb,
						      current_buffer,
						      (copy_sectors_from - start_sector) * sector_size,
						      (copy_sectors_to - start_sector) * sector_size);
		} /* write data */

		/* write 256K */
		DPRINTK(KERN_ERR "%s: WRITE sector=%#lx\n", __FUNCTION__, current_sector);
		lpar_addr = ps3pf_stor_virtual_to_lpar(dev_info, dev_info->bounce_buf);
		init_completion(&(dev_info->irq_done));
		ret64 = lv1_storage_write((uint64_t)dev_info->lv1_dev_info->device_id,
					  (uint64_t)region_id,
					  (uint64_t)current_sector,
					  (uint64_t)aligned_sector_count,
					  (uint64_t)0,
					  (uint64_t)lpar_addr,
					  &(dev_info->lv1_dev_info->current_tag));
		if (ret64) {
			/* error */
			printk(KERN_ERR "%s: error lv1dev =%ld ret=%ld\n", __FUNCTION__,
			       dev_info->lv1_dev_info->device_id, ret64);
			dev_info->srb->result = DID_ERROR << 16; /* FIXME: other error code? */
			ret = -1;
		} else {

			/* wait irq */
			wait_for_completion(&(dev_info->irq_done));

			if (dev_info->lv1_status) {
				/* error */
				memset(dev_info->srb->sense_buffer, 0, SCSI_SENSE_BUFFERSIZE);
				decode_lv1_status(dev_info->lv1_status,
						  &(dev_info->srb->sense_buffer[2]),
						  &(dev_info->srb->sense_buffer[12]),
						  &(dev_info->srb->sense_buffer[13]));
				dev_info->srb->sense_buffer[7] = 16 - 6;
				dev_info->srb->result = SAM_STAT_CHECK_CONDITION;
				ret = 1;
			} else {
				/* OK */
				dev_info->srb->result = DID_OK << 16;
				ret = 0;
			}

		}
	} /* for */
 done:
	ps3pf_stor_srb_done(dev_info);
	if (dev_info->bounce_type == DYNAMIC_BOUNCE)
		kfree(dev_info->bounce_buf);
	read_unlock(&dev_info->bounce_lock);
	DPRINTK(KERN_ERR "%s: end\n", __FUNCTION__);
	return ret;
}

/*
 * NOTE: If return 1, all buffers communicate with the device
 *       should be in dedicated buffer area.
 *       Currently common_handle_read, common_handle_write know this
 *       restriction.
 *       And should implement remap function in ps3pf_stor_set_max_sectors()
 */
static int need_dedicated_dma_region(struct ps3pf_stor_lv1_dev_info * lv1_dev_info)
{
	int ret = 0;
	switch (lv1_dev_info->device_type) {
	case TYPE_RBC:
		ret = 1; /* should be 1 */
		break;
	case TYPE_ROM:
		ret = 0;
		break;
	case TYPE_DISK:
		ret = 0;
		break;
	default:
		printk(KERN_ERR "%s: unknown type =%ld\n", __FUNCTION__, lv1_dev_info->device_type);
		ret =  0;
		break;
	}
	return ret;
}
/*
 * allocate static(dedicated) bounce buffer
 */
static int get_dedicated_buffer_type(struct ps3pf_stor_lv1_dev_info * lv1_dev_info)
{
	int ret = 0;
	switch (lv1_dev_info->device_type) {
	case TYPE_RBC:
		ret = DEDICATED_SPECIAL;
		break;
	case TYPE_ROM:
		ret = DEDICATED_KMALLOC;
		break;
	case TYPE_DISK:
		ret = DEDICATED_KMALLOC;
		break;
	default:
		printk(KERN_ERR "%s: unknown type =%ld\n", __FUNCTION__, lv1_dev_info->device_type);
		ret =  0;
		break;
	}
	return ret;
}

static int get_default_max_sector(struct ps3pf_stor_lv1_dev_info * lv1_dev_info)
{
	int ret = 0;
	switch (lv1_dev_info->device_type) {
	case TYPE_RBC:
		ret = FLASH_ALIGN / lv1_dev_info->sector_size;
		break;
	case TYPE_ROM:
		ret = 32;
		break;
	case TYPE_DISK:
		ret =  128;
		break;
	default:
		printk(KERN_ERR "%s: unknown type =%ld\n", __FUNCTION__, lv1_dev_info->device_type);
		ret =  0;
		break;
	}
	return ret;
}
/*
 * called from scsi mid layer when it want to probe a
 * device.
 * Prepare so that mid can issue SCSI commands later (slave_configure)
 */
static int ps3pf_stor_slave_alloc(struct scsi_device * scsi_dev)
{
        int error = 0;
        struct ps3pf_stor_host_info * host_info = NULL;
	struct ps3pf_stor_dev_info * dev_info = NULL;
        struct Scsi_Host *scsi_host;
	struct ps3pf_stor_lv1_bus_info * lv1_bus_info;
	struct ps3pf_stor_lv1_dev_info * lv1_dev_info = NULL;
	struct list_head * pos;
	uint64_t ret;
	int found;
	char thread_name[64];

	FUNC_START;

	scsi_host = scsi_dev->host;
	host_info = *(struct ps3pf_stor_host_info **)(scsi_host->hostdata);
	lv1_bus_info = host_info->lv1_bus_info;
	/*
	 * connect lv1_dev_info with scsi_device
	 * assume SCSI mid layer started scsi id with ZERO '0'
	 */
	found = 0;
	list_for_each(pos, &(lv1_bus_info->dev_list)) {
		lv1_dev_info = list_entry(pos, struct ps3pf_stor_lv1_dev_info, bus_dev_list);

		if ((lv1_dev_info->bus_device_index == scsi_dev->id) &&
		    (scsi_dev->lun < lv1_dev_info->accessible_regions)) {
			found = 1;
			break;
		}
	}

	if (!found) {
		error = -ENXIO;
		goto cleanup_0;
	}
	/*
	 * connect scsi_dev with dev_info
	 */
	found = 0;
	list_for_each(pos, &(host_info->dev_info_list)) {
		dev_info = list_entry(pos, struct ps3pf_stor_dev_info, dev_list);
		if (!dev_info->used) {
			dev_info->used = 1;
			dev_info->target = scsi_dev->id;
			dev_info->lv1_dev_info = lv1_dev_info;
			switch (lv1_dev_info->device_type)
			{
			case TYPE_DISK:
				dev_info->handler_info = scsi_cmnd_info_table_hdd;
				break;
			case TYPE_RBC:
				dev_info->handler_info = scsi_cmnd_info_table_flash;
				break;
			case TYPE_ROM:
				dev_info->handler_info = scsi_cmnd_info_table_atapi;
				break;
			}
			/* reverse link */
			lv1_dev_info->dev_info = dev_info;
			scsi_dev->hostdata = dev_info;
			/* copy sector length and capacity */
			dev_info->sector_size = lv1_dev_info->sector_size;
			found = 1;
			break;
		} else {
			if (dev_info->target == scsi_dev->id) {
				/* another lun ? */
				if (scsi_dev->lun < lv1_dev_info->accessible_regions) {
					/* ok, support this lun */
					scsi_dev->hostdata = dev_info;
					goto skip_per_device_configure;
				}
			}
		}
	}

	if (!found) {
		printk(KERN_ERR "%s: no empty dev_info for device id=%d lun=%d \n", __FUNCTION__,
		       scsi_dev->id, scsi_dev->lun);
		error = -ENODEV;
		goto cleanup_0;
	}
	FUNC_STEP_C("1");
	/* register irq */
	ret = ps3pf_connect_irq(&(lv1_dev_info->port_id),
			      &(lv1_dev_info->irq_plug_id),
			      &(lv1_dev_info->cpu_id),
			      ps3pf_stor_hdd_irq_handler,
			      0,
			      "PS3PF stor",
			      (void*)lv1_dev_info);
	if (ret) {
		error = -ENODEV;
		goto cleanup_0;
	}
	FUNC_STEP_C("2");

	/* open lv1 device */
	ret = lv1_open_device(lv1_dev_info->bus_id, lv1_dev_info->device_id, 0);
	if (ret) {
		printk(KERN_ERR "%s:open failed %ld\n", __FUNCTION__, ret);
		error = -ENODEV;
		goto cleanup_1;
	}

	if (ps3pf_get_device_interrupt_id(lv1_dev_info->bus_index, lv1_dev_info->device_index,
					  &lv1_dev_info->interrupt_id))
		lv1_dev_info->interrupt_id = PS3PF_STOR_DEFAULT_INTERRUPT_ID;

	ret = lv1_connect_interrupt_event_receive_port(lv1_dev_info->bus_id,
						       lv1_dev_info->device_id,
						       lv1_dev_info->port_id,
						       lv1_dev_info->interrupt_id);
	if (ret) {
		printk(KERN_ERR "%s:receive port failed %ld\n", __FUNCTION__, ret);
		error = -ENODEV;
		goto cleanup_2;
	}
	FUNC_STEP_C("3");

	/* prepare dma regions for the device */
	write_lock(&dev_info->bounce_lock);
	switch (get_dedicated_buffer_type(lv1_dev_info)) {
	case DEDICATED_KMALLOC:
		/*
		 * adjust max_sector count.
		 * mid layer already set default value from host template
		 */
		blk_queue_max_sectors(scsi_dev->request_queue, get_default_max_sector(lv1_dev_info));
		/* create its own static bouce buffer */
		dev_info->dedicated_bounce_size = get_default_max_sector(lv1_dev_info) * lv1_dev_info->sector_size;
		dev_info->bounce_buf = kmalloc(dev_info->dedicated_bounce_size, GFP_KERNEL | __GFP_DMA);
		write_unlock(&dev_info->bounce_lock);
		if (!dev_info->bounce_buf) {
			printk(KERN_ERR "%s:kmalloc for static bounce buffer failed %#x\n", __FUNCTION__,
			       dev_info->dedicated_bounce_size);
			error = -ENOMEM;
			goto cleanup_3;
		}
		dev_info->bounce_type = DEDICATED_KMALLOC;
		break;
	case DEDICATED_SPECIAL:
		blk_queue_max_sectors(scsi_dev->request_queue, get_default_max_sector(lv1_dev_info));
		/* use static buffer, kmalloc can not allocate 256K */
		dev_info->dedicated_bounce_size = FLASH_ALIGN;
		dev_info->bounce_buf = ps3pf_stor_alloc_separate_memory(FLASH_ALIGN,
									&dev_info->separate_bounce_lpar);
		if (!dev_info->bounce_buf) {
			error = -ENOMEM;
			goto cleanup_3;
		}
		write_unlock(&dev_info->bounce_lock);
		dev_info->bounce_type = DEDICATED_SPECIAL;
		break;
	case DYNAMIC_BOUNCE:
		write_unlock(&dev_info->bounce_lock);
		dev_info->bounce_type = DYNAMIC_BOUNCE;
		break;
	}
	/* allocate dma region */
	if (need_dedicated_dma_region(lv1_dev_info)) {
		ret = lv1_allocate_device_dma_region(lv1_dev_info->bus_id,
						     lv1_dev_info->device_id,
						     CEIL_ALIGN_4K(dev_info->dedicated_bounce_size),
						     12 /* 4K */,
						     0,
						     &lv1_dev_info->dma_region);
		if (ret || !lv1_dev_info->dma_region) {
			printk(KERN_ERR "%s:allocate dma region failed %ld\n", __FUNCTION__, ret);
			error = -ENOMEM;
			goto cleanup_3;
		}
		ret = lv1_map_device_dma_region(lv1_dev_info->bus_id,
						lv1_dev_info->device_id,
						ps3pf_stor_virtual_to_lpar(dev_info, dev_info->bounce_buf),
						lv1_dev_info->dma_region,
						CEIL_ALIGN_4K(dev_info->dedicated_bounce_size),
						0xf800000000000000UL);
		DPRINTK(KERN_ERR "%s:map bounce buffer %ld va=%p lp=%#lx pa=%#lx size=%#x dma=%#lx\n", __FUNCTION__,
		       ret,
		       dev_info->bounce_buf,
		       ps3pf_stor_virtual_to_lpar(dev_info, dev_info->bounce_buf),
		       __pa(dev_info->bounce_buf),
		       dev_info->dedicated_bounce_size,
		       lv1_dev_info->dma_region);
		if (ret) {
			lv1_free_device_dma_region(lv1_dev_info->bus_id,
						   lv1_dev_info->device_id,
						   lv1_dev_info->dma_region);

			error = -ENODEV;
			goto cleanup_3;
		}
		dev_info->dedicated_dma_region = 1;

	} else {
		lv1_dev_info->dma_region = ps3pf_allocate_dma_region(lv1_dev_info->bus_id,
								     lv1_dev_info->device_id,
								     0);
		if (!lv1_dev_info->dma_region) {
			printk(KERN_ERR "%s:create dma region failed %ld\n", __FUNCTION__, ret);
			error = -ENODEV;
			goto cleanup_3;
		}
	}
	FUNC_STEP_C("4");

	/* create receive thread */
	sprintf(thread_name, "ps3pfstor-%d-%d",
		scsi_host->host_no, scsi_dev->id);
	dev_info->thread_struct = kthread_create(ps3pf_stor_main_thread,
						 dev_info,
						 thread_name);
	if (IS_ERR(dev_info->thread_struct)) {
		error = -ENOMEM;
		dev_info->thread_struct = NULL;
		goto cleanup_3;
	}
	init_MUTEX_LOCKED(&(dev_info->thread_sema));
	wake_up_process(dev_info->thread_struct);

 skip_per_device_configure:
	FUNC_END;
        return 0;

 cleanup_3:
	FUNC_STEP_C("5");
	lv1_disconnect_interrupt_event_receive_port(lv1_dev_info->bus_id,
						    lv1_dev_info->device_id,
						    lv1_dev_info->port_id,
						    lv1_dev_info->interrupt_id);
 cleanup_2:
	FUNC_STEP_C("6");
	lv1_close_device(lv1_dev_info->bus_id, lv1_dev_info->device_id);
 cleanup_1:
	FUNC_STEP_C("7");
	ps3pf_free_irq(lv1_dev_info->port_id,
		     lv1_dev_info->irq_plug_id,
		     lv1_dev_info->cpu_id,
		     (void*)lv1_dev_info);
 cleanup_0:
	FUNC_END_C("error");
	return error;/* say failed to alloc */


}

static int ps3pf_stor_slave_configure(struct scsi_device * scsi_dev)
{

	if (scsi_dev->host->max_cmd_len != PS3PF_STOR_MAX_CMD_LEN)
		scsi_dev->host->max_cmd_len = PS3PF_STOR_MAX_CMD_LEN;

	if (scsi_dev->host->cmd_per_lun)
		scsi_adjust_queue_depth(scsi_dev, 0, scsi_dev->host->cmd_per_lun);
	/*
	 * ATAPI SFF8020 devices use MODE_SENSE_10,
	 * so we can prohibit MODE_SENSE_6
	 */
	scsi_dev->use_10_for_ms = 1;

	return 0;
}

static void ps3pf_stor_slave_destroy(struct scsi_device * scsi_dev)
{
	uint64_t ret64;
	struct ps3pf_stor_dev_info * dev_info =
				(struct ps3pf_stor_dev_info *)scsi_dev->hostdata;
	struct ps3pf_stor_lv1_dev_info * lv1_dev_info = dev_info->lv1_dev_info;

	/* only LUN=0 should do */
	if (scsi_dev->lun != 0) {
		printk(KERN_ERR "%s: id=%d lun=%d skipped\n", __FUNCTION__,
		       scsi_dev->id, scsi_dev->lun);
		return;
	}

	/* terminate main thread */
	dev_info->thread_wakeup_reason = THREAD_TERMINATE;
	init_completion(&(dev_info->thread_terminated));
	up(&(dev_info)->thread_sema);
	wait_for_completion(&(dev_info->thread_terminated));


	/* free resources */
	switch(dev_info->bounce_type) {
	case DEDICATED_SPECIAL:
		ps3pf_stor_release_separate_memory(dev_info->bounce_buf,
						   dev_info->separate_bounce_lpar);
		dev_info->bounce_buf = NULL;
		break;
	case DEDICATED_KMALLOC:
		kfree(dev_info->bounce_buf);
		dev_info->bounce_buf = NULL;
		break;
	}

	if (dev_info->dedicated_dma_region) {
		ret64 = lv1_unmap_device_dma_region(lv1_dev_info->bus_id,
						    lv1_dev_info->device_id,
						    lv1_dev_info->dma_region,
						    CEIL_ALIGN_4K(dev_info->dedicated_bounce_size));
		if (ret64) {
			printk(KERN_ERR "%s: unmap dma region failed %ld\n", __FUNCTION__, ret64);
		}
		ret64 = lv1_free_device_dma_region(lv1_dev_info->bus_id,
						   lv1_dev_info->device_id,
						   lv1_dev_info->dma_region);
		if (ret64) {
			printk(KERN_ERR "%s: unmap dma region failed %ld\n", __FUNCTION__, ret64);
		}
		dev_info->dedicated_dma_region = 0;
	} else {
		ret64 = ps3pf_free_dma_region(lv1_dev_info->bus_id,
					    lv1_dev_info->device_id,
					    lv1_dev_info->dma_region);
		if (ret64) {
			printk(KERN_ERR "%s: free dma region failed %ld\n", __FUNCTION__, ret64);
		}
	}
	ret64 = lv1_disconnect_interrupt_event_receive_port(lv1_dev_info->bus_id,
							    lv1_dev_info->device_id,
							    lv1_dev_info->port_id,
							    lv1_dev_info->interrupt_id);

	if (ret64) {
		printk(KERN_ERR "%s: disconnect interrupt event %ld\n", __FUNCTION__, ret64);
	}

	ret64 = lv1_close_device(lv1_dev_info->bus_id, lv1_dev_info->device_id);

	if (ret64) {
		printk(KERN_ERR "%s: close device %ld\n", __FUNCTION__, ret64);
	}

	ret64 = ps3pf_free_irq(lv1_dev_info->port_id,
			     lv1_dev_info->irq_plug_id,
			     lv1_dev_info->cpu_id,
			     lv1_dev_info);
	if (ret64) {
		printk(KERN_ERR "%s: free irq %ld\n", __FUNCTION__, ret64);
	}

	if (dev_info) {
		/* make this slot avaliable for re-use */
		dev_info->used = 0;
		scsi_dev->hostdata = NULL;
	}

}

static int ps3pf_stor_host_reset(struct scsi_cmnd * srb)
{
	return FAILED;
}

MODULE_AUTHOR("Sony Computer Entertainment Inc.");
MODULE_DESCRIPTION("PS3PF storage driver");
MODULE_LICENSE("GPL");
MODULE_VERSION(PS3PF_STOR_VERSION);

/*
 * return 1 specified region is accessible from linux
 */
static int ps3pf_stor_temporary_irq_handler(int irq, void * context, struct pt_regs * regs)
{
	struct ps3pf_stor_quirk_probe_info * info = context;

	(void) irq;
	(void) regs;

	info->lv1_retval = lv1_storage_get_async_status(info->device_id,
							&(info->lv1_ret_tag),
							&(info->lv1_status));
	complete(&(info->irq_done));

	return IRQ_HANDLED;
}

static int is_region_accessible(uint64_t bus_index, uint64_t bus_id,
				uint64_t device_index, uint64_t device_id,
				uint64_t device_type, uint64_t region_id)
{
	uint64_t ret64, port_id, interrupt_id;
	int accessible = 0;
	unsigned int irq_plug_id, cpu_id, dma_region;
	void * buf;
	struct ps3pf_stor_quirk_probe_info info;

	/*
	 * special case
	 * cd-rom is assumed always accessible
	 */
	if (device_type == TYPE_ROM)
		return 1;

	/*
	 * 1. open the device
	 * 2. register irq for the device
	 * 3. connect irq
	 * 4. map dma region
	 * 5. do read
	 * 6. umap dma region
	 * 7. disconnect irq
	 * 8. unregister irq
	 * 9. close the device
	 */
	memset(&info, 0, sizeof(info));

	ret64 = lv1_open_device(bus_id, device_id, 0);
	if (ret64)
		return 0;

	ret64 = ps3pf_connect_irq(&port_id,
				  &irq_plug_id,
				  &cpu_id,
				  ps3pf_stor_temporary_irq_handler,
				  0,
				  "PS3PF quirk",
				  (void*) &info);
	if (ret64)
		goto cleanup_0;

	if (ps3pf_get_device_interrupt_id(bus_index, device_index, &interrupt_id))
		interrupt_id = PS3PF_STOR_DEFAULT_INTERRUPT_ID;
	ret64 = lv1_connect_interrupt_event_receive_port(bus_id,
							 device_id,
							 port_id,
							 interrupt_id);
	if (ret64)
		goto cleanup_1;

	dma_region = ps3pf_allocate_dma_region(bus_id, device_id, 0);

	if (!dma_region)
		goto cleanup_2;

	/* 4k buffer is for fail safe of large sector devices */
	buf = kmalloc(4096, GFP_KERNEL);
	if (!buf) {
		printk(KERN_ERR "%s: no memory while probing dev=%lx",
		       __FUNCTION__, device_id);
		goto cleanup_3;
	};

	init_completion(&(info.irq_done));
	info.device_id = device_id;
	ret64 = lv1_storage_read(device_id,
				 region_id,
				 0, /* start sector */
				 1, /* sector count */
				 0, /* flags */
				 p_to_lp(__pa(buf)), /* no need special convert */
				 &info.lv1_tag);
	if (ret64)
		goto cleanup_4;

	wait_for_completion(&(info.irq_done));

	if ((info.lv1_retval == 0) && (info.lv1_status == 0)) {
		if (info.lv1_tag != info.lv1_ret_tag) {
			printk(KERN_ERR "%s: tag mismached dev=%lx\n", __FUNCTION__,
			       device_id);
		} else
			accessible = 1;
	}

 cleanup_4:
	kfree(buf);
 cleanup_3:
	ps3pf_free_dma_region(bus_id, device_id, dma_region);
 cleanup_2:
	lv1_disconnect_interrupt_event_receive_port(bus_id,
						    device_id,
						    port_id,
						    interrupt_id);
 cleanup_1:
	ps3pf_free_irq(port_id,
		       irq_plug_id,
		       cpu_id,
		       (void*) &info);
 cleanup_0:
	lv1_close_device(bus_id, device_id);

	return accessible;
}

/*
 * returns current number of found HDDs
 * and collect device info
 */
static int ps3pf_stor_enum_storage_drives(void)
{
	const uint64_t bus_index = 4;
	uint64_t num_of_dev, bus_id, bus_type;
	uint64_t device_index, device_id, device_type, pci_bus, pci_dev, pci_func;
	uint64_t int_type, int_number;
	uint64_t port;
	uint64_t blocks, blksize, regions;
	uint64_t region_size, region_id, region_start;

	int devices;
	struct ps3pf_stor_lv1_dev_info * lv1_dev_info;
	uint64_t i, accessible_regions;
	int j;

	num_of_dev = bus_id = bus_type = 0;
	device_id = device_type = pci_bus = pci_dev = pci_func = 0;
	int_type = int_number = 0;

	ps3pf_read_repository_bus_id(bus_index, &bus_id);
	ps3pf_read_repository_bus_type(bus_index, &bus_type);
	ps3pf_read_repository_num_of_dev(bus_index, &num_of_dev);
	devices = 0;

	/* is this storage ?*/
	if (bus_type != 5)
		return 0;
	for (device_index = 0; device_index < num_of_dev; device_index++) {
		if (ps3pf_read_repository_device_id(bus_index, device_index, &device_id))
			printk(KERN_ERR "no: ps3pf_read_repository_device_id\n");
		if (ps3pf_read_repository_device_type(bus_index, device_index, &device_type))
			printk(KERN_ERR "no: ps3pf_read_repository_device_type\n");

		if (device_type != TYPE_DISK && device_type != TYPE_ROM && device_type != TYPE_RBC) {
			ps3pf_read_repository_device_blksize(bus_index, device_index, &blksize);
			continue;
		}

		port = blocks = blksize = 0;

		if (ps3pf_read_repository_device_port(bus_index, device_index, &port))
			printk(KERN_ERR "%s: failed DEVICE_PORT\n", __FUNCTION__);

		if (ps3pf_read_repository_device_blksize(bus_index, device_index, &blksize))
			printk(KERN_ERR "%s: failed DEVICE_BLKSIZE\n", __FUNCTION__);

		if (ps3pf_read_repository_device_blocks(bus_index, device_index, &blocks))
			printk(KERN_ERR "%s: failed DEVICE_BLOCKS\n", __FUNCTION__);

		if (ps3pf_read_repository_device_regions(bus_index, device_index, &regions))
			printk(KERN_ERR "%s: failed to get regions \n", __FUNCTION__);
		if (sizeof(unsigned long) * 8 < regions) {
			printk(KERN_ERR "%s: region count exceeded (%ld).  the rest are ignored\n", __FUNCTION__, regions);
			regions = sizeof(unsigned long) * 8;
		}

		lv1_dev_info = &(ps3pf_stor_lv1_dev_info_array[ps3pf_stor_lv1_devnum]);
		INIT_LIST_HEAD(&(lv1_dev_info->bus_dev_list));

		lv1_dev_info->bus_index = bus_index;
		lv1_dev_info->bus_id = bus_id;
		lv1_dev_info->device_index = device_index;
		lv1_dev_info->device_id = device_id;
		lv1_dev_info->device_type = device_type;
		lv1_dev_info->sector_size = blksize;
		lv1_dev_info->attached_port = port;
		lv1_dev_info->device_index = device_index;
		lv1_dev_info->regions = regions;
		ps3pf_stor_lv1_devnum ++;
		/* check how many regions accessible */
		accessible_regions = 0;
		for (i = 0; i < regions; i++) {
			if (is_region_accessible(bus_index, bus_id, device_index, device_id, device_type, i)) {
				set_bit(i, &(lv1_dev_info->accessible_region_flag));
				accessible_regions ++;
			}
		}
		/* LUN limitaion */
		if (8 < accessible_regions) {
			printk(KERN_ERR "%s: accessible regions (=%ld) exceeded eight(8). The rest are ignored\n", __FUNCTION__,
			       accessible_regions);
			accessible_regions = 8;
		}

		lv1_dev_info->region_info_array = kzalloc(sizeof(struct ps3pf_stor_lv1_region_info) * accessible_regions,
							  GFP_KERNEL);
		if (!(lv1_dev_info->region_info_array)) {
			printk(KERN_ERR "%s: kzalloc failed for info array\n", __FUNCTION__);
			return 0;/* FIXME: is it better to panic here? */
		}

		lv1_dev_info->accessible_regions = accessible_regions;
		for (j = i = 0; i < regions; i++) {
			if (!test_bit(i,&(lv1_dev_info->accessible_region_flag)))
				continue;

			if (ps3pf_read_repository_device_region_id(bus_index,
								   device_index,
								   i,
								   &region_id))
				printk(KERN_ERR "%s: failed region id\n", __FUNCTION__);
			if (ps3pf_read_repository_device_region_size(bus_index,
								     device_index,
								     i,
								     &region_size))
				printk(KERN_ERR "%s: failed region size\n", __FUNCTION__);
			if (ps3pf_read_repository_device_region_start(bus_index,
								     device_index,
								     i,
								     &region_start))
				printk(KERN_ERR "%s: failed region start\n", __FUNCTION__);
			lv1_dev_info->region_info_array[j].region_index = i;
			lv1_dev_info->region_info_array[j].region_id = region_id;
			lv1_dev_info->region_info_array[j].region_size = region_size;
			lv1_dev_info->region_info_array[j].region_start = region_start;
			j++;
		}
		printk(KERN_INFO "ps3pf_stor: dev=%ld type=%lx port=%ld regions=%ld accessible=%ld\n",
		       device_id, device_type, port, regions, accessible_regions);
		devices ++;
	}
	return devices;
}

static int __init ps3pf_stor_init(void)
{
	int host_to_add, devices;
	int index;

	FUNC_START;

	/* register this driver thru devfs */
	platform_driver_register(&ps3pf_stor_platform_driver);

	/* scsi host template */
	ps3pf_stor_driver_template.proc_name = (char *)ps3pf_stor_proc_name;

	/* wait until expected numbers of devices become ready */
	devices = ps3pf_stor_wait_device_ready();

	/* init lv1_bus_info */
	for (index = 0; index < PS3PF_STORAGE_NUM_OF_BUS_TYPES; index++) {
		ps3pf_stor_lv1_bus_info_array[index].bus_type = index;
		INIT_LIST_HEAD(&(ps3pf_stor_lv1_bus_info_array[index].dev_list));
	}

	/* alloc lv1_dev_info for devices */
	ps3pf_stor_lv1_dev_info_array = kzalloc(sizeof(struct ps3pf_stor_lv1_dev_info) * devices, GFP_KERNEL);

	if (!ps3pf_stor_lv1_dev_info_array) {
		printk("init failed\n");
		goto clean;
	}
	for (index = 0; index < devices; index++) {
		INIT_LIST_HEAD(&(ps3pf_stor_lv1_dev_info_array[index].bus_dev_list));
	}

	/* calc how many HBA to add */
	ps3pf_stor_lv1_devnum = 0;
	devices = ps3pf_stor_enum_storage_drives();

	for (index = 0; index < devices; index++) {
		if (ps3pf_stor_lv1_dev_info_array[index].device_type == TYPE_DISK ||
		    ps3pf_stor_lv1_dev_info_array[index].device_type == TYPE_ROM) {
			if (ps3pf_stor_lv1_dev_info_array[index].attached_port & (1 << 1)) {
				ps3pf_stor_lv1_dev_info_array[index].bus_device_index =
					ps3pf_stor_lv1_bus_info_array[PS3PF_STORAGE_PATA_1].devices ++;
				list_add_tail(&(ps3pf_stor_lv1_dev_info_array[index].bus_dev_list),
					      &(ps3pf_stor_lv1_bus_info_array[PS3PF_STORAGE_PATA_1].dev_list));
			} else {
				ps3pf_stor_lv1_dev_info_array[index].bus_device_index =
					ps3pf_stor_lv1_bus_info_array[PS3PF_STORAGE_PATA_0].devices ++;
				list_add_tail(&(ps3pf_stor_lv1_dev_info_array[index].bus_dev_list),
					      &(ps3pf_stor_lv1_bus_info_array[PS3PF_STORAGE_PATA_0].dev_list));
			}

		}

		if (ps3pf_stor_lv1_dev_info_array[index].device_type == TYPE_RBC) {
			ps3pf_stor_lv1_dev_info_array[index].bus_device_index =
				ps3pf_stor_lv1_bus_info_array[PS3PF_STORAGE_FLASH].devices ++;
			list_add_tail(&(ps3pf_stor_lv1_dev_info_array[index].bus_dev_list),
				      &(ps3pf_stor_lv1_bus_info_array[PS3PF_STORAGE_FLASH].dev_list));
		}

	}

	host_to_add = 0;
	for (index = 0; index < PS3PF_STORAGE_NUM_OF_BUS_TYPES; index++) {
		if (ps3pf_stor_lv1_bus_info_array[index].devices)
			host_to_add ++;
	}


        /* add HBAs */
	ps3pf_stor_add_host = 0;
	for (index = 0; index < PS3PF_STORAGE_NUM_OF_BUS_TYPES; index++) {
		if (ps3pf_stor_lv1_bus_info_array[index].devices) {
			if (ps3pf_stor_add_adapter(&(ps3pf_stor_lv1_bus_info_array[index]))) {
				printk(KERN_ERR "ps3pf_stor_init: ps3pf_stor_add_adapter failed\n");
				break;
			} else
				host_to_add --;
		}
	}

	FUNC_END;
	return 0;

 clean:
	platform_driver_unregister(&ps3pf_stor_platform_driver);
	return -ENOMEM;
}

static void __exit ps3pf_stor_exit(void)
{
	int k;

	for (k = ps3pf_stor_add_host; k; k--)
		ps3pf_stor_remove_adapter();
	platform_driver_unregister(&ps3pf_stor_platform_driver);

	for (k = 0; k < ps3pf_stor_lv1_devnum; k++) {
		if (ps3pf_stor_lv1_dev_info_array[k].region_info_array) {
			kfree(ps3pf_stor_lv1_dev_info_array[k].region_info_array);
		}
	}
	kfree(ps3pf_stor_lv1_dev_info_array);
}

device_initcall(ps3pf_stor_init);
module_exit(ps3pf_stor_exit);


/*
  construct a host structure
  and associated structures for
  its devices.
  register the host thru device_register()
*/
static int ps3pf_stor_add_adapter(struct ps3pf_stor_lv1_bus_info * lv1_bus_info)
{
	int k;
        int error = 0;
        struct ps3pf_stor_host_info *host_info;
        struct ps3pf_stor_dev_info *dev_info;
        struct list_head *lh, *lh_sf;

        host_info = kzalloc(sizeof(struct ps3pf_stor_host_info), GFP_KERNEL);

        if (NULL == host_info) {
                printk(KERN_ERR "%s: out of memory \n", __FUNCTION__);
                return -ENOMEM;
        }
        INIT_LIST_HEAD(&host_info->dev_info_list);
	host_info->lv1_bus_info = lv1_bus_info;

	/* create structures for child devices of this adapter */
        for (k = 0; k < lv1_bus_info->devices; k++) {
                dev_info = kzalloc(sizeof(struct ps3pf_stor_dev_info), GFP_KERNEL);
                if (NULL == dev_info) {
                        printk(KERN_ERR "%s: out of memory \n", __FUNCTION__);
                        error = -ENOMEM;
			goto clean;
                }
                dev_info->host_info = host_info;
		INIT_LIST_HEAD(&dev_info->dev_list);
		spin_lock_init(&dev_info->srb_lock);
		rwlock_init(&dev_info->bounce_lock);
                list_add_tail(&dev_info->dev_list, &host_info->dev_info_list);
        }

        spin_lock(&ps3pf_stor_host_list_lock);
        list_add_tail(&host_info->host_list, &ps3pf_stor_host_list);
        spin_unlock(&ps3pf_stor_host_list_lock);

	/* copy struct platform_device */
        host_info->dev =  ps3pf_stor_platform_device;
        host_info->dev.id = ps3pf_stor_add_host;

        error = platform_device_register(&host_info->dev);

        if (error)
		goto clean;

	/* bump up registerd buses */
	++ps3pf_stor_add_host;

        return error;

clean:
	list_for_each_safe(lh, lh_sf, &host_info->dev_info_list) {
		dev_info = list_entry(lh, struct ps3pf_stor_dev_info, dev_list);
		list_del(&dev_info->dev_list);
		kfree(dev_info);
	}

	kfree(host_info);
        return error;
}

static void ps3pf_stor_remove_adapter(void)
{
        struct ps3pf_stor_host_info * host_info = NULL;

        spin_lock(&ps3pf_stor_host_list_lock);
        if (!list_empty(&ps3pf_stor_host_list)) {
                host_info = list_entry(ps3pf_stor_host_list.prev,
                                       struct ps3pf_stor_host_info, host_list);
		list_del(&host_info->host_list);
	}
        spin_unlock(&ps3pf_stor_host_list_lock);

	if (!host_info)
		return;

        platform_device_unregister(&host_info->dev);
	kfree(host_info);
        --ps3pf_stor_add_host;
}

static int ps3pf_stor_hdd_irq_handler(int irq, void * context, struct pt_regs * regs)
{
	struct ps3pf_stor_lv1_dev_info * lv1_dev_info = context;
	struct ps3pf_stor_dev_info * dev_info = lv1_dev_info->dev_info;
	int ret_val = IRQ_HANDLED;
	uint64_t tag;
	(void) irq;
	(void) regs;

	if (dev_info) {
		dev_info->lv1_retval = lv1_storage_get_async_status(lv1_dev_info->device_id,
								    &tag,
								    (uint64_t *)&dev_info->lv1_status);
		/*
		 * lv1_status = -1 may mean that ATAPI transport completed OK, but ATAPI command
		 * itself resulted CHECK CONDITION
		 * so, upper layer should issue REQUEST_SENSE to check the sense data
		 */
		if (tag != dev_info->lv1_dev_info->current_tag)
			printk("%s: tag=%#lx ctag=%#lx\n", __FUNCTION__,
			       tag, dev_info->lv1_dev_info->current_tag);
		if (dev_info->lv1_retval) {
			printk("%s: ret=%#lx status=%#lx\n", __FUNCTION__,
			       dev_info->lv1_retval, dev_info->lv1_status);
			//if (dev_info->lv1_retval == LV1_NO_ENTRY)
			//ret_val = IRQ_NONE;
		} else {
			complete(&(dev_info->irq_done));
		}
	}
	return ret_val;
}

static int ps3pf_stor_driver_probe(struct platform_device * dev)
{
        int error = 0;
        struct ps3pf_stor_host_info *host_info;
        struct ps3pf_stor_lv1_bus_info *lv1_bus_info;
        struct Scsi_Host *scsi_host;

	host_info = from_dev_to_ps3pf_stor_host(dev);
	lv1_bus_info = host_info->lv1_bus_info;

        scsi_host = scsi_host_alloc(&ps3pf_stor_driver_template, sizeof(struct ps3pf_stor_host_info*));
        if (NULL == scsi_host) {
                printk(KERN_ERR "%s: scsi_register failed\n", __FUNCTION__);
                error = -ENODEV;
		return error;
        }

        host_info->scsi_host = scsi_host;
	*((struct ps3pf_stor_host_info **)scsi_host->hostdata) = host_info;

	/*
	 * set maximum id as same as number of child devices
	 */
	scsi_host->max_id = lv1_bus_info->devices;
	scsi_host->max_lun = 8;

        error = scsi_add_host(scsi_host, &host_info->dev.dev);

        if (error) {
                printk(KERN_ERR "%s: scsi_add_host failed\n", __FUNCTION__);
                error = -ENODEV;
		scsi_host_put(scsi_host);
        } else {
		scsi_scan_host(scsi_host);
	}


        return error;
}

static void ps3pf_stor_driver_shutdown(struct platform_device * dev)
{
	ps3pf_stor_driver_remove(dev);
}

static int ps3pf_stor_driver_remove(struct platform_device * dev)
{
        struct list_head *lh, *lh_sf;
        struct ps3pf_stor_host_info *host_info;
        struct ps3pf_stor_dev_info *dev_info;

	host_info = from_dev_to_ps3pf_stor_host(dev);

	if (!host_info) {
		printk(KERN_ERR "%s: Unable to locate host info\n",
		       __FUNCTION__);
		return -ENODEV;
	}

        scsi_remove_host(host_info->scsi_host);

        list_for_each_safe(lh, lh_sf, &host_info->dev_info_list) {
                dev_info = list_entry(lh, struct ps3pf_stor_dev_info,
                                          dev_list);
                list_del(&dev_info->dev_list);
                kfree(dev_info);
        }

        scsi_host_put(host_info->scsi_host);

        return 0;
}

static void ps3pf_stor_device_release(struct device * device)
{
	FUNC_START;
	// place holder
	FUNC_END;
}

static ssize_t ps3pf_stor_get_max_sectors(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct scsi_device *scsi_dev = from_dev_to_scsi_device(dev);
	struct ps3pf_stor_dev_info * dev_info = (struct ps3pf_stor_dev_info *)scsi_dev->hostdata;
	ssize_t ret;

 	read_lock(&dev_info->bounce_lock);
	ret = sprintf(buf, "%u\n", scsi_dev->request_queue->max_sectors);
 	read_unlock(&dev_info->bounce_lock);
	return ret;
}

static ssize_t ps3pf_stor_set_max_sectors(struct device *dev,
					  struct device_attribute *attr, const char *buf, size_t count)
{
	struct scsi_device *scsi_dev = from_dev_to_scsi_device(dev);
	struct ps3pf_stor_dev_info * dev_info;
	struct ps3pf_stor_lv1_dev_info * lv1_dev_info;
	unsigned short max_sectors;
	void * bounce_buf;

	if (sscanf(buf, "%hu", &max_sectors) > 0 && max_sectors <= SCSI_DEFAULT_MAX_SECTORS) {
		dev_info = (struct ps3pf_stor_dev_info *)scsi_dev->hostdata;
		lv1_dev_info = dev_info->lv1_dev_info;
		/* if dedicated dma region, refuse to reset buffer */
		if (need_dedicated_dma_region(lv1_dev_info)) {
			/* FIXME: need remap dma region !!! */
			return -EINVAL;
		}
 		write_lock(&dev_info->bounce_lock);
		if (dev_info->bounce_type == DEDICATED_KMALLOC) {
			/* try to allocate new bounce buffer */
			bounce_buf = kmalloc(max_sectors * lv1_dev_info->sector_size, GFP_NOIO | __GFP_DMA | __GFP_NOWARN);
			if (!bounce_buf) {
				write_unlock(&dev_info->bounce_lock);
				return -ENOMEM;
			}
			kfree(dev_info->bounce_buf);
			dev_info->bounce_buf = bounce_buf;
			dev_info->dedicated_bounce_size = max_sectors * lv1_dev_info->sector_size;
		}
		blk_queue_max_sectors(scsi_dev->request_queue, max_sectors);
		write_unlock(&dev_info->bounce_lock);
		return strlen(buf);
	}
	return -EINVAL;
}

#define NOTIFICATION_DEVID ((uint64_t)(-1L))
struct device_probe_info {
	uint64_t device_id;
	uint64_t device_type;
	int      found;
	int      region_expected;
	int      region_ready;
};

static int ps3pf_stor_wait_device_ready(void)
{
	const uint64_t bus_index = 4;
	uint64_t num_of_dev, bus_id, bus_type;
	uint64_t ret64, tag, status;
	int retries;
	int ret = 0;
	int i;
	uint64_t * buf;
	int region_ready = 0;
	int region_expected = 0;
	struct device_probe_info * info_array;

	num_of_dev = bus_id = bus_type = 0;

	ps3pf_read_repository_bus_id(bus_index, &bus_id);
	ps3pf_read_repository_bus_type(bus_index, &bus_type);
	ps3pf_read_repository_num_of_dev(bus_index, &num_of_dev);

	/* is this storage ?*/
	if (bus_type != 5)
		return 0;

	/* 1) wait for expected devices becomes in repositry */
	retries = 0;
	while (retries++ < ps3pf_stor_wait_time)
	{
		if (ps3pf_read_repository_num_of_dev(bus_index, &num_of_dev)) {
			continue;
		}
		if (ps3pf_stor_wait_num_storages + 1 <= num_of_dev)
			break;
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(HZ);
		printk(".");
	}
	printk("\n");

	buf = kzalloc(512, GFP_KERNEL);
	if (!buf)
		return 0;

	info_array = kzalloc(sizeof(struct device_probe_info) * num_of_dev, GFP_KERNEL);
	if (!info_array) {
		ret = -1;
		goto cleanup_0;
	}

	/* 2) store the device info */
	for (i = 0; i < num_of_dev; i++) {
		if (ps3pf_read_repository_device_id(bus_index, i, &(info_array[i].device_id))) {
			BUG();
		}
		ps3pf_read_repository_device_type(bus_index, i, &(info_array[i].device_type));
		info_array[i].found = 1;

		switch(info_array[i].device_type) {
		case TYPE_DISK:
		case TYPE_RBC:
			info_array[i].region_expected = 1;
			region_expected ++;
			ret ++;
			break;
		case TYPE_ROM:
			ret ++;
		default:
			break;
		}
	} /* for */


	/* 2-1) open special event device */
	ret64 = lv1_open_device(bus_id, NOTIFICATION_DEVID, 0);
	if (ret64) {
		printk(KERN_ERR "%s: open failed notification dev %ld\n", __FUNCTION__, ret64);
		ret = 0;
		goto cleanup_1;
	}

	/* 2-2) write info to request notify */
	buf[0] = 0;
	buf[1] = (1 << 1); /* region update info only */
	ret64 = lv1_storage_write(NOTIFICATION_DEVID,
				  0, /* region */
				  0, /* lba */
				  1, /* sectors to write */
				  0, /* flags */
				  p_to_lp(__pa(buf)), /* no need special convert */
				  &tag);
	if (ret64) {
		printk(KERN_ERR "%s: notify request write failed %ld\n", __FUNCTION__, ret64);
		ret = 0;
		goto cleanup_2;
	}

	/* wait for completion in one sec */
	retries = 0;
	while ((ret64 = lv1_storage_check_async_status(NOTIFICATION_DEVID, tag, &status)) &&
	       (retries++ < 1000)) {
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1);
	}
	if (ret64) {
		/* write not completed */
		printk(KERN_ERR "%s: write not completed %ld\n", __FUNCTION__, ret64);
		ret = 0;
		goto cleanup_2;
	}

	/* 2-3) read to wait region notification for each device */
	while (region_ready < region_expected) {
		memset(buf, 0, 512);
		ret64 = lv1_storage_read(NOTIFICATION_DEVID,
					 0, /* region */
					 0, /* lba */
					 1, /* sectors to read */
					 0, /* flags */
					 p_to_lp(__pa(buf)), /* no need special convert */
					 &tag);
		retries = 0;
		while ((ret64 = lv1_storage_check_async_status(NOTIFICATION_DEVID, tag, &status)) &&
		       (retries++ < 1000)) {
			set_current_state(TASK_INTERRUPTIBLE);
			schedule_timeout(1);
		}
		if (ret64) {
			/* read not completed */
			printk(KERN_ERR "%s: read noy complated %d\n", __FUNCTION__, ret);
			break;
		}

		/* 2-4) verify the notification */
		if (buf[0] != 1) {
			/* other info notified */
			printk(KERN_ERR "%s: notification info %ld dev=%lx type=%lx\n", __FUNCTION__,
			       buf[0], buf[2], buf[3]);
		}

		for (i = 0; i < num_of_dev; i++) {
			if (info_array[i].found && info_array[i].device_id == buf[2]) {
				info_array[i].region_ready = 1;
				region_ready ++;
				break;
			}
		} /* for */
	} /* while */

 cleanup_2:
	lv1_close_device(bus_id, NOTIFICATION_DEVID);

 cleanup_1:
	kfree(info_array);
 cleanup_0:
	kfree(buf);
	return ret;
}

/*
 * allocate memory from lv1 and map it for bounce buffer.
 * return null if allocation failed
 * alloc_size: allocation size in 256KB unit
 */
void * ps3pf_stor_alloc_separate_memory(int alloc_size, uint64_t * lpar_addr)
{
	void * va;
	uint64_t ret64, lpar, muid;

	FUNC_START;
	if (!alloc_size || (alloc_size % FLASH_ALIGN))
		return NULL;

	ret64 = lv1_allocate_memory(alloc_size , /* 256Kb * n */
				    18, /* 256KB */
				    0,
				    (LV1_AM_TF_NO |
				     LV1_AM_DS_ANYTIME |
				     LV1_AM_FA_ALTERNATIVE |
				     LV1_AM_ADDR_ANY),
				    &lpar,
				    &muid);
	if (ret64) {
		printk(KERN_ERR "%s: alloc failed %ld\n", __FUNCTION__, ret64);
		return NULL;
	}
	if (lpar_addr)
		*lpar_addr = lpar;

	va = __ioremap(lpar, alloc_size, PAGE_KERNEL);
	if (!va) {
		printk(KERN_ERR "%s: remap failed for lpar=%016lx\n", __FUNCTION__, lpar);
		goto rewind;
	}

	FUNC_END;
	return va;

 rewind:
	lv1_release_memory(lpar);
	return NULL;
}

int ps3pf_stor_release_separate_memory(void * va, uint64_t lpar)
{
	uint64_t ret64;
	int ret;

	FUNC_START;
	iounmap(va);
	ret64 = lv1_release_memory(lpar);
	if (ret64) {
		printk(KERN_ERR "%s: ret=%ld\n", __FUNCTION__, ret64);
		ret = 1;
	} else
		ret = 0;
	FUNC_END;
	return ret;
}

/*
 * convert kernel virtal address to lpar address for storage IO
 * NOTE: va should be within allocated special buffer
 *       if DEDICATED_SPECIAL bounce type
 */
uint64_t ps3pf_stor_virtual_to_lpar(struct ps3pf_stor_dev_info * dev_info, void * va)
{
	if (unlikely(dev_info->bounce_type == DEDICATED_SPECIAL)) {
		return dev_info->separate_bounce_lpar + (va - dev_info->bounce_buf);
	} else {
		return p_to_lp(__pa(va));
	}
}
