/*
 * Copyright (C) 2006 Sony Computer Entertainment Inc.
 * storage support for PS3PF
 *
 * based on scsi_debug.h
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#ifndef _PS3PF_STOR_H
#define _PS3PF_STOR_H

#include <linux/types.h>

#define LV1_STORAGE_SEND_ATAPI_COMMAND (1)
#define LV1_STORAGE_ATA_HDDOUT         (0x23)

#define PS3PF_STOR_MAX_INQUIRY_DATA_SIZE  (128)
#define PS3PF_STOR_INQUIRY_DATA_SIZE      (86)
#define PS3PF_STOR_READCAP_DATA_SIZE      (8)
#define PS3PF_STOR_SENSE_LEN              (32)
#define PS3PF_STOR_VERSION                "1.00"
#define PS3PF_STOR_CANQUEUE               (1)
#define PS3PF_STOR_MAX_CMD_LEN            (16)

#define PS3PF_STOR_DEFAULT_INTERRUPT_ID (0)
#define PS3PF_MMIO_REGION_MAX           (4) /* see PS3PF_PCI_INT_MAX arch/powerpc/platforms/pci_res.h */

/*
 * lv1 return code
 */
enum {
	LV1_OK = 0,
	LV1_RESOURCE_SHORTAGE = -2,
	LV1_NO_ENTRY          = -6,
	LV1_NO_BUSY           = -9,
	LV1_NO_EMPTY          = -10
};

struct lv1_atapi_cmnd_block {
	uint8_t		pkt[32]; /* packet command block           */
	uint32_t	pktlen;  /* should be 12 for ATAPI 8020    */
	uint32_t	blocks;
	uint32_t	block_size;
	uint32_t	proto;   /* transfer mode                  */
	uint32_t	in_out;  /* transfer direction             */
	uint64_t	buffer;  /* parameter except command block */
	uint32_t	arglen;  /* length above                   */
};

enum lv1_atapi_proto {
	NON_DATA_PROTO     = 0,
	PIO_DATA_IN_PROTO  = 1,
	PIO_DATA_OUT_PROTO = 2,
	DMA_PROTO = 3
};

enum lv1_atapi_in_out {
	DIR_WRITE = 0,
	DIR_READ = 1
};

/*
 * describe protocol of an ATAPI command
 */
struct ps3pf_stor_dev_info;

struct scsi_command_handler_info {
	int buflen;
	int proto;
	int in_out;
	int (*cmnd_handler)(struct ps3pf_stor_dev_info *, struct scsi_cmnd *);
};

/*
 * srb position parameter
 */

enum {
	NOT_AVAIL          = -1,
	USE_SRB_10         = -2,
	USE_SRB_6          = -3,
	USE_CDDA_FRAME_RAW = -4
};
/*
 * for LV1 maintainance
 */
enum  {
	PS3PF_STORAGE_PATA_0, /* primary   PATA bus */
	PS3PF_STORAGE_PATA_1, /* secondary PATA bus */
	PS3PF_STORAGE_FLASH,
	PS3PF_STORAGE_NUM_OF_BUS_TYPES /* terminator */
};

/*
 * LV1 per physical bus info:
 * PATA0, PATA1, FLASH
 */
struct ps3pf_stor_lv1_bus_info {
	int bus_type;           /* PATA0, PATA1, FLASH */
	int devices;            /* number of devices on the bus */
	struct list_head dev_list;
};

/*
 * LV1 per region info
 */
struct ps3pf_stor_lv1_region_info {
	int region_index;       /* index of this region       */
	uint64_t region_id;     /* id of this region          */
	uint64_t region_size;   /* region size in sector      */
	uint64_t region_start;  /* start sector */
};

/*
 * LV1 per device info
 */
struct ps3pf_stor_lv1_dev_info {
	struct list_head bus_dev_list; /* device list of devices          */
				       /* which share same physical bus   */
	struct ps3pf_stor_dev_info * dev_info;
	/* repositry values */
 	uint64_t bus_index;     /* X of bus#X           */
 	uint64_t bus_id;        /* contents of above    */
 	uint64_t device_index;  /* Y of bus#X.dev#Y     */
 	uint64_t device_id;     /* contents of above    */
	uint64_t device_type;   /* bus#X.dev#Y.type     */
	uint64_t attached_port; /* bus#x.dev#Y.port     */
	uint64_t sector_size;   /* bus#X.dev#Y.blk_size */

	/* house keeping */
	int      bus_type;      /* PATA0,1 or FLASH */
	uint64_t port_id;       /* async event receive port */
	unsigned int irq_plug_id;
	uint64_t interrupt_id;
	unsigned int cpu_id;
	uint64_t dma_region;
	uint64_t current_tag;
	int      bus_device_index; /*
				    * device index of same lv1 phy bus.
				    * 0 for first device, 1 for second.
				    * should be same as SCSI id
				    */
	/* regions */
	uint64_t regions;           /* number of regions reported thru repository */
	unsigned long accessible_region_flag; /* flag of accessible regions */
	int      accessible_regions; /* number of accessible regions of this dev.
				      * currently, this includes region #0
				      * NOTE: maximum is 8, if exceed, the rest of
				      * regions are ignored
				      */
	struct ps3pf_stor_lv1_region_info * region_info_array;
};

enum read_or_write {
	SCSIDEBUG_READ,
	SCSIDEBUG_WRITE
};


enum thread_wakeup_reason {
	SRB_QUEUED,
	THREAD_TERMINATE
};

enum bounce_buffer_type {
	DYNAMIC_BOUNCE,
	DEDICATED_KMALLOC,
	DEDICATED_SPECIAL,
};

struct ps3pf_stor_dev_info {
	struct list_head dev_list;
	struct ps3pf_stor_lv1_dev_info * lv1_dev_info;
	struct ps3pf_stor_host_info *host_info;
	const struct scsi_command_handler_info * handler_info;
	unsigned int target;

	uint64_t sector_size;    /* copied from lv1 repository at initialize   */
	/* devices may change these value */
	rwlock_t bounce_lock;  /* protect the following members:
				* bounce_buf (pointer itself, not buffer),
				* dedicated_bounce_size
				* max_sectors in scsi_dev->request_queue
				*/
	int  dedicated_bounce;   /* set nonzero if the bounce buffer is dedicated */
	int  dedicated_bounce_size;
	int  dedicated_dma_region; /* set if partial dma region allocated */
	int  bounce_type;          /* bounce buffer type */
	void * bounce_buf;
	uint64_t separate_bounce_lpar; /* lpar address for separated buffer  */

	char used;

	/* main thread communication */
	struct task_struct * thread_struct;
	spinlock_t srb_lock;
	struct scsi_cmnd * srb;              /* queued srb; just one srb allowd             */
	struct semaphore thread_sema;        /* device main thread wakeup                   */
	struct completion thread_terminated; /* notify thread temination to slave_destory() */
	int thread_wakeup_reason;

	/* interrupt handler communication */
	struct completion irq_done;
	volatile uint64_t lv1_status;        /* result of get_async_status()       */
	volatile uint64_t lv1_retval;        /* return value of get_async_status() */

};

struct ps3pf_stor_host_info {
	struct list_head host_list;
	struct Scsi_Host *scsi_host;
	struct platform_device dev;
	struct list_head dev_info_list;
	struct ps3pf_stor_lv1_bus_info * lv1_bus_info;
};

#define from_dev_to_ps3pf_stor_host(p) \
 container_of(p, struct ps3pf_stor_host_info, dev)
#define from_dev_to_scsi_device(p) \
 container_of(p, struct scsi_device, sdev_gendev)

/*
 * extern
 */
extern unsigned long p_to_lp(long pa);
extern uint64_t ps3pf_allocate_dma_region(uint64_t, uint64_t, int);
extern uint64_t ps3pf_free_dma_region(uint64_t, uint64_t, uint64_t);

/*
 * local
 */
static int fill_from_dev_buffer(struct scsi_cmnd * srb, const unsigned char * buf, int buf_len);
static int fetch_to_dev_buffer(struct scsi_cmnd * srb, unsigned char * buf, int buf_len);
static void make_first_key(const char *name, uint64_t index, uint64_t *n);
static void make_key(const char *name, uint64_t index, uint64_t *n);
static int ps3pf_read_repository_bus_id(uint64_t bus_index, uint64_t *system_bus_id);
static int ps3pf_read_repository_bus_type(uint64_t bus_index, uint64_t *bus_type);
static int ps3pf_read_repository_num_of_dev(uint64_t bus_index, uint64_t *num_of_dev);
static int ps3pf_read_repository_device_id(uint64_t bus_index, uint64_t device_index,
					 uint64_t *device_id);
static int ps3pf_read_repository_device_type(uint64_t bus_index, uint64_t device_index,
					   uint64_t *device_type);
static int ps3pf_read_repository_device_port(uint64_t bus_index, uint64_t device_index,
					   uint64_t *port);
static int ps3pf_read_repository_device_blksize(uint64_t bus_index, uint64_t device_index,
					      uint64_t *blksize);
static int ps3pf_read_repository_device_blocks(uint64_t bus_index, uint64_t device_index,
					     uint64_t *blocks);

/*
 * local
 */
static int ps3pf_stor_driver_probe(struct platform_device *);
static int ps3pf_stor_driver_remove(struct platform_device *);
static void ps3pf_stor_driver_shutdown(struct platform_device * dev);
static int ps3pf_stor_main_thread(void * dev_info);
static void ps3pf_stor_process_srb(struct scsi_cmnd *);
static int ps3pf_stor_add_adapter(struct ps3pf_stor_lv1_bus_info *);
static void ps3pf_stor_remove_adapter(void);
static int ps3pf_stor_hdd_irq_handler(int irq, void * context, struct pt_regs * regs);
static int ps3pf_stor_slave_alloc(struct scsi_device * sdp);
static int ps3pf_stor_slave_configure(struct scsi_device * sdp);
static void ps3pf_stor_slave_destroy(struct scsi_device *);
static int ps3pf_stor_queuecommand(struct scsi_cmnd *, void (*done) (struct scsi_cmnd *));
static int ps3pf_stor_host_reset(struct scsi_cmnd *);
static void ps3pf_stor_device_release(struct device * dev);
/*
 * sysfs interface
 */
static ssize_t ps3pf_stor_set_max_sectors(struct device *dev,
					  struct device_attribute *attr, const char *buf, size_t count);
static ssize_t ps3pf_stor_get_max_sectors(struct device *dev, struct device_attribute *attr, char *buf);

/*
 * command handler for atapi
 */
static int ps3pf_stor_atapi_handle_simple(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
static int ps3pf_stor_atapi_handle_request_sense(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
/*
 * command handler for HDD
 */
static int ps3pf_stor_hdd_handle_inquiry(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
static int ps3pf_stor_hdd_handle_read_capacity(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
static int ps3pf_stor_hdd_handle_just_ok(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
static int ps3pf_stor_hdd_handle_sync_cache(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
static int ps3pf_stor_hdd_handle_mode_sense(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
static int ps3pf_stor_hdd_handle_request_sense(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);

/*
 * command handler for both
 */
static int ps3pf_stor_common_handle_read(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
static int ps3pf_stor_common_handle_write(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);
static int ps3pf_stor_handle_write_flash(struct ps3pf_stor_dev_info * dev_info, struct scsi_cmnd * srb);

struct ps3pf_stor_quirk_probe_info {
	struct completion irq_done;
	uint64_t device_id;
	uint64_t lv1_retval;
	uint64_t lv1_status;
	uint64_t lv1_tag;
	uint64_t lv1_ret_tag;
};
static int ps3pf_stor_temporary_irq_handler(int irq, void * context, struct pt_regs *regs);

static int ps3pf_stor_enum_storage_drives(void);
static int ps3pf_stor_wait_device_ready(void);
static void * ps3pf_stor_alloc_separate_memory(int alloc_size, uint64_t * lpar_addr);
static int ps3pf_stor_release_separate_memory(void * va, uint64_t lpar_addr);
static uint64_t ps3pf_stor_virtual_to_lpar(struct ps3pf_stor_dev_info * dev_info, void * va);
#endif
