/**********************************************************************
 * $pow example -- PLI application using VPI routines
 *
 * C source to calculate the result of a number to the power of an
 * exponent. The result is returned as a 32-bit integer.
 *
 * Usage: result = $pow(<base>,<exponent>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library  */
                       /*  TF library is used for aborting on error */

/* prototypes of PLI application routine names */
int PLIbook_PowSizetf(), PLIbook_PowCalltf(), PLIbook_PowCompiletf(),
    PLIbook_PowStartOfSim();

/**********************************************************************
 * $pow Registration Data
 * (add this function name to the vlog_startup_routines array)
 *********************************************************************/
void PLIbook_pow_register()
{
  s_vpi_systf_data tf_data;
  s_cb_data   cb_data_s;
  s_vpi_time  time_s;

  tf_data.type        = vpiSysFunc;
  tf_data.sysfunctype = vpiSysFuncSized;
  tf_data.tfname      = "$pow";
  tf_data.calltf      = PLIbook_PowCalltf;
  tf_data.compiletf   = PLIbook_PowCompiletf;
  tf_data.sizetf      = PLIbook_PowSizetf;
  vpi_register_systf(&tf_data);

  cb_data_s.reason    = cbStartOfSimulation;
  cb_data_s.cb_rtn    = PLIbook_PowStartOfSim;
  cb_data_s.obj       = NULL;
  cb_data_s.time      = NULL;
  cb_data_s.value     = NULL;
  cb_data_s.user_data = NULL;
  vpi_register_cb(&cb_data_s);
}

/**********************************************************************
 * Sizetf application
 *********************************************************************/
int PLIbook_PowSizetf(char *user_data)
{
  return(32);   /* $pow returns 32-bit values */
}

/**********************************************************************
 * compiletf application to verify valid systf args.
 *********************************************************************/
int PLIbook_PowCompiletf(char *user_data)
{
  vpiHandle systf_handle, arg_itr, arg_handle;
  int       tfarg_type;

  systf_handle = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_handle);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $pow requires 2 arguments\n");
    tf_dofinish();
    return(0);
  }
  arg_handle = vpi_scan(arg_itr);
  tfarg_type = vpi_get(vpiType, arg_handle);
  if ( (tfarg_type != vpiReg) &&
       (tfarg_type != vpiIntegerVar) &&
       (tfarg_type != vpiConstant)   ) {
    vpi_printf("ERROR: $pow arg1 must be number, variable or net\n");
    tf_dofinish();
    return(0);
  }
  arg_handle = vpi_scan(arg_itr);
  if (arg_handle == NULL) {
    vpi_printf("ERROR: $pow requires 2nd argument\n");
    tf_dofinish();
    return(0);
  }
  tfarg_type = vpi_get(vpiType, arg_handle);
  if ( (tfarg_type != vpiReg) &&
       (tfarg_type != vpiIntegerVar) &&
       (tfarg_type != vpiConstant)   ) {
    vpi_printf("ERROR: $pow arg2 must be number, variable or net\n");
    tf_dofinish();
    return(0);
  }
  if (vpi_scan(arg_itr) != NULL) {
    vpi_printf("ERROR: $pow requires only 2 arguments\n");
    vpi_free_object(arg_itr);
    tf_dofinish();
    return(0);
  }
}

/**********************************************************************
 * calltf to calculate base to power of exponent and return result.
 *********************************************************************/
#include <math.h>
int PLIbook_PowCalltf(char *user_data)
{
  s_vpi_value value_s;
  vpiHandle   systf_handle, arg_itr, arg_handle;
  int         base, exp, result;

  systf_handle = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_handle);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $pow failed to obtain systf arg handles\n");
    return(0);
  }

  /* read base from systf arg 1 (compiletf has already verified) */
  arg_handle = vpi_scan(arg_itr);
  value_s.format = vpiIntVal;
  vpi_get_value(arg_handle, &value_s);
  base = value_s.value.integer;

  /* read exponent from systf arg 2 (compiletf has already verified) */
  arg_handle = vpi_scan(arg_itr);
  vpi_free_object(arg_itr); /* not calling scan until returns null */
  vpi_get_value(arg_handle, &value_s);
  exp = value_s.value.integer;

  /* calculate result of base to power of exponent */
  result = (int)pow( (double)base, (double)exp );

  /* write result to simulation as return value $pow */
  value_s.value.integer = result;
  vpi_put_value(systf_handle, &value_s, NULL, vpiNoDelay);
  return(0);
}

/**********************************************************************
 * Start-of-simulation application
 *********************************************************************/
int PLIbook_PowStartOfSim()
{
  vpi_printf("\n$pow PLI application is being used.\n\n");
  return(0);
}
/*********************************************************************/

