"""
TODO:
    * TODOs are specified in the classes
    * being more verbose in 'container'-object's __repr__: add bugnumber to each object
    * standardize *.changed (output: frozenset/list, property)
    * dates: should all dates be date/time objects?
    * how to handle global attachment-related variables?
    * add revert()-function(s)
THIS IS STILL WORK IN PROGRESS!!
"""
import re
import os
import libxml2

import utils # TODO: split this modul
import bughelper_error as Error

from bugbase import Bug as BugBase
        

ATTACHMENT_PATH = "~/.bughelper/attachments-cache" #default should be None
CONTENT_TYPES = ["text/html"] #default should be an empty-list

DEBUG = True
DEBUG_URL = False
def debug(*arg,**args):
    if DEBUG:
        return utils.debug(*arg,**args)
        
#deactivate error messages from the validation [libxml2.htmlParseDoc]
def noerr(ctx, str):
    pass

libxml2.registerErrorHandler(noerr, None)
        

def _small_xpath(xml, expr):
    """ Returns the content of the first result of a xpath expression  """
    result = xml.xpathEval(expr)
    if not result:
        return False
    return result[0].content
        
def get_bug(id):
    return Bug._container_refs[id]


class user(str):
    def __new__(cls, lplogin, realname=None):
        obj = super(user, cls).__new__(user, lplogin)
        obj.__lplogin = lplogin
        obj.__realname = realname
        return obj
        
    @property
    def realname(self):
        return self.__realname or "unknown"
        
    def __repr__(self):
        return "<user %s (%s)>" %(self.__lplogin, self.realname)
        
    def __str__(self):
        return self.__lplogin or ""
        
    def __nonzero__(self):
        if self.__lplogin:
            return True
        else:
            return False


class product(str):
    def __new__(cls, lpname, longname=None, type=None, tracked_in=None):
        obj = super(product, cls).__new__(product, lpname)
        obj.__lpname = lpname
        obj.__longname = longname
        obj.__type = type
        obj.__tracked_in = tracked_in
        return obj
        
    @property
    def longname(self):
        return self.__longname or "unknown"
        
    @property
    def type(self):
        return self.__type
        
    @property
    def tracked_in(self):
        return self.__tracked_in
        
    def __str__(self):
        return self.__lpname or ""
        
    def __repr__(self):
        return "<product %s (%s, type=%s, tracked in=%s)>" %(self.__lpname,
                    self.longname, self.__type, self.__tracked_in)


class _change_obj(object):
    """ shows the status of a changed object  """
    def __init__(self, attr, action="changed"):
        """
        * attr: edited attribute
        * action: describes the way the attribute was edited
        """
        self.__attr = attr
        self.__action = action
        
    @property
    def component(self):
        return self.__attr
        
    def __str__(self):
        if self.__action == "changed":
            s = "(%s changed: %s)" %(repr(self.__attr), ", ".join([str(x) for x in getattr(self.__attr,"changed", [])]))
        else:
            s = "(%s %s)" %(repr(self.__attr), self.__action)
        return s
        
    def __repr__(self):
        return "<Changes to %s>" %self.__attr

def _blocked(func, error=None):
    def f(a, *args, **kwargs):
        try:
            x = a.infotable.current
        except AttributeError, e:
            error = "%s %s" %(f.error, e)
            raise AttributeError, error
        return func(a, *args, **kwargs)
    f.error = error or "Unable to get current InfoTable row."
    return f 

def _attr_ext(i, s):
    if i.startswith("__"):
        return "_%s%s" %(s.__class__.__name__,i)
    else:
        return i


def _gen_getter(attr):
    """ Returns a function to return the value of an attribute
    
    Example:
        get_example = _gen_getter("x.y")
    is like:
        def get_example(self):
            if not x.parsed:
                x.parse()
            return x.y
    
    """
    def func(s):
        attributes = attr.split(".")
        attributes = map(lambda a: _attr_ext(a, s), attributes)
        x = getattr(s, attributes[0])
        attributes.insert(0, s)
        if not x.parsed:
            x.parse()
        return reduce(getattr, attributes)
    return func

        
def _gen_setter(attr):
    """ Returns a function to set the value of an attribute
    
    Example:
        set_example = _gen_setter("x.y")
    is like:
        def set_example(self, value):
            if not x.parsed:
                x.parse()
            x.y = value
    
    """
    def func(s, value):
        attributes = attr.split(".")
        attributes = map(lambda a: _attr_ext(a, s), attributes)
        x = getattr(s, attributes[0])
        attributes.insert(0, s)
        if not x.parsed:
            x.parse()
        setattr(reduce(getattr, attributes[:-1]), attributes[-1], value)
    return func


        
class Info(object):
    """ The 'Info'-object represents one row of the 'InfoTable' of a bugreport
    
    * editable attributes:
        .sourcepackage: lp-name of a package/project
        .status: valid lp-status
        .importance: valid lp-importance (if the user is not permitted to
            change 'importance' an 'IOError' will be raised
        .assignee: lp-login of an user/group
        .milestone: value must be in '.valid_milestones'
        
    * read-only attributes:
        .affects, .target, .valid_milestones
        
    TODO: * rename 'Info' into 'Task'
    """
    def __init__(self, affects, status, importance, assignee, current,
                    editurl, type, milestone, available_milestone,
                    lock_importance, targeted_to, remote, editlock,
                    edit_fields, connection):
        self.__connection = connection
        self.__editlock = editlock
        self.__edit_fields = edit_fields
        self.__lock_importance = lock_importance
        self.__target = None
        self.__affects = affects
        temp = self.__affects.split(" ")
        if temp[0] == "Ubuntu":
            self.__sourcepackage = None
        else:
            self.__sourcepackage = temp[0]
        r = re.match(r'^.*\((.*)\)$', self.__affects.longname)
        if r:
            self.__target = r.group(1)
        self.__status = status
        self.__importance = importance
        self.__assignee = assignee
        self.__current = current
        self.__editurl = editurl
        self.__type = type
        self.__milestone = milestone
        self.__available_milestone = available_milestone
        self.__targeted_to = targeted_to
        self.__remote = remote
        self.__cache = {"sourcepackage" : self.__sourcepackage,
                "status": self.__status, "importance": self.__importance,
                "assignee": self.__assignee, "milestone": self.__milestone}
        
    
    def __str__(self):
        targeted_to = (self.__targeted_to and " (%s)" %self.__targeted_to) or ""
        remote = (self.__remote and " (remote)") or ""
        return "[%s%s%s: %s/%s]" % (self.affects.longname or self.affects,
                    targeted_to, remote, self.status, self.importance)
        
    def __repr__(self):
        targeted_to = (self.__targeted_to and " (%s)" %self.__targeted_to) or ""
        remote = (self.__remote and " (remote)") or ""
        number = (not self.__editlock and self.__editurl.split("/")[-2]) or self.__editurl.split("/")[-1]
        return "<Info of '%s%s%s (#%s)'>" %(self.affects.longname or self.affects,
                    targeted_to, remote, number)
                     
    @property
    def targeted_to(self):
        return self.__targeted_to
        
    @property
    def remote(self):
        return self.__remote
                     
    @property
    def affects(self):
        return self.__affects
        
    @property
    def target(self):
        return self.__target
        
    def get_sourcepackage(self):
        return self.__sourcepackage
        
    def set_sourcepackage(self, package):
        if self.__editlock:
            raise IOError, "The sourcepackage of this bug can't be edited, maybe because this bug is a duplicate of an other one"
        self.__sourcepackage = package
    sourcepackage = property(get_sourcepackage, set_sourcepackage, doc="sourcepackage of a bug")

    def get_assignee(self):
        return self.__assignee
        
    def set_assignee(self, lplogin):
        if self.__editlock:
            raise IOError, "The assignee of this bug can't be edited, maybe because this bug is a duplicate of an other one"
        if self.__remote:
            raise IOError, "This task is linked to a remote-bug system, please change the assignee there"
        self.__assignee = lplogin
    assignee = property(get_assignee, set_assignee, doc="assignee to a bugreport")
    
    def get_status(self):
        return self.__status
        
    def set_status(self, status):
        if self.__editlock:
            raise IOError, "The status of this bug can't be edited, maybe because this bug is a duplicate of an other one"
        if self.__remote:
            raise IOError, "This task is linked to a remote-bug system, please change the status there"
        if status not in Bug.STATUS.values():
            raise ValueError, "Unknown status '%s', status must be one of these: %s" %(status, Bug.STATUS.values())
        self.__status = status
    status = property(get_status, set_status, doc="status of a bugreport")
                              
        
    def get_importance(self):
        return self.__importance
        
    def set_importance(self, importance):
        if self.__editlock:
            raise IOError, "The importance of this bug can't be edited, maybe because this bug is a duplicate of an other one"
        if self.__remote:
            raise IOError, "This task is linked to a remote-bug system, please change the importance there"
        if self.__lock_importance:
            raise IOError, "'Importance' editable only by the maintainer or bug contact of the project/package"
        if importance not in Bug.IMPORTANCE.values():
            raise ValueError, "Unknown importance '%s', importance must be one of these: %s" %(importance, Bug.IMPORTANCE.values())
        self.__importance = importance
    importance = property(get_importance, set_importance, doc="importance of a bugreport")
    
    
    @property
    def valid_milestones(self):
        return self.__available_milestone
        
    def get_milestone(self):
        return self.__milestone
        
    def set_milestone(self, milestone):
        if self.__editlock:
            raise IOError, "The milestone of this bug can't be edited, maybe because this bug is a duplicate of an other one"
        if not self.__available_milestone:
            raise ValueError, "No milestones defined for this product"
        if milestone not in self.__available_milestone:
            raise ValueError, "Unknown milestone, milestone must be one of these: %s" %self.__available_milestone
        self.__milestone = milestone
    milestone = property(get_milestone, set_milestone, doc="milestone of a bugreport")
    
        
    def get_changed(self):
        changed = set()
        for k in self.__cache:
            if self.__cache[k] != getattr(self, k):
                changed.add(k)
        return frozenset(changed)
    changed = property(get_changed, doc="get a list of changed attributes")
    
    def commit(self, force_changes=False, ignore_lp_errors=True):
        """ Commits the local changes to launchpad.net
        
        * force_changes: general argument, has not effect in this case
        * ignore_lp_errors: if the user tries to commit invalid data to launchpad,
            launchpad returns an error-page. If 'ignore_lp_errors=False' Info.commit()
            will raise an 'ValueError' in this case, otherwise ignore this
            and leave the bugreport in launchpad unchanged (default=True)
        """
        changed = self.changed
        if changed:
            if self.__type:
                full_sourcepackage = self.__type.split(".")[0]
            else:
                full_sourcepackage = "%s_%s" %(self.targeted_to, str(self.__affects))
            s = self.sourcepackage
            if s == "ubuntu":
                s = ""
            args = { '%s.actions.save' %full_sourcepackage: '1',
                '%s.comment_on_change' %full_sourcepackage: ''#set_status_comment
                    }
            if self.__type:
                args[self.__type] = s
            if ".status" in self.__edit_fields:
                args['%s.status-empty-marker' %full_sourcepackage] = '1'
                args['%s.status' %full_sourcepackage] = self.status
            if ".importance" in self.__edit_fields:
                args['%s.importance' %full_sourcepackage] = self.importance
                args['%s.importance-empty-marker' %full_sourcepackage] = '1'
            if ".milestone" in self.__edit_fields:
                args['%s.mlestone' %full_sourcepackage] = ""
                args['%s.milestone-empty-marker' %full_sourcepackage] = '1'
            args['%s.assignee.option' %full_sourcepackage] = '%s.assignee.assign_to' %full_sourcepackage
            args['%s.assignee' %full_sourcepackage] = self.assignee or ""
                
            result = self.__connection.post(self.__editurl, args)
            if result.url == self.__editurl and not ignore_lp_errors:
                raise ValueError, "There is a problem with the information you entered. Please fix it and try again."


class InfoTable(object):
    """ The 'InfoTable'-object represents the tasks at the top of a bugreport
        
    * read-only attributes:
        .current: returns the highlighted Info-object of the bugreport
    
    TODO:  * rename 'InfoTable' into 'TaskTable'
           * allow adding of tasks (Also affects upstream/Also affects distribution)
           * does current/tracked work as expected?
           * remote: parse editable values
    """
    def __init__(self, connection, xml, url):
        self.__infolist = []
        self.__current = None 
        self.parsed = False
        self.__connection=connection
        self.__xml = xml
        self.__url = url

    def __getitem__(self, key):
        return self.__infolist[key]
        
    def __iter__(self):
        for i in self.__infolist:
            yield(i)

    def __repr__(self):
        return "<InfoTable>"
        
    def __str__(self):
        x = (len(self.__infolist) > 1 and "[%s]") or "%s"
        return x %",".join(str(i) for i in self.__infolist)

    def parse(self):
        """ Parsing the info-table
        
        * format:  'Affects'|'Status'|'Importance'|'Assigned To'
        
        TODO: * working on 'tracked in...' - currently there is only one 'tracked in'
                entry per bugreport supported
              * REMOTE BUG!!!
        """
        if self.parsed:
            return True
        rows = self.__xml[0].xpathEval('tbody/tr[not(@style="display: none") and not(@class="secondary")]')
        assert rows, "Wrong XPath-Expr in InfoTable.parse() 'rows' (%s)" %(DEBUG_URL or "unknown")
        highl_target = None
        #status of (remote)-bugs can be 'unknown''
        temp_status = Bug.STATUS.copy()
        temp_status["statusUNKNOWN"] = "Unknown"
        #importance of (remote)-bugs can be 'unknown''
        temp_importance = Bug.IMPORTANCE.copy()
        temp_importance["importanceUNKNOWN"] = "Unknown"
        
        tracked = False
        affects = product(None)
        for row in rows:
            edit_fields = set()
            tmp_affects = affects
            current = False
            remote = False
                
            if row.prop("class") == "highlight":
                current = True
            
            if not tracked:
                #parse affects
                m = row.xpathEval("td[1]")
                assert m, "Wrong XPath-Expr in InfoTable.parse() 'affects' (%s)" %(DEBUG_URL or "unknown")
                m_type = m[0].xpathEval("img")
                assert m_type, "Wrong XPath-Expr in InfoTable.parse() 'affects' (%s)" %(DEBUG_URL or "unknown")
                affects_type = m_type[0].prop("src").split("/")[-1]
                m_name = m[0].xpathEval("a")
                assert m_name, "Wrong XPath-Expr in InfoTable.parse() 'affects' (%s)" %(DEBUG_URL or "unknown")
                affects_lpname = m_name[0].prop("href").split("/")[-1]
                affects_longname = m_name[0].content
            affects = product(affects_lpname, affects_longname, affects_type)
                
            if row.xpathEval('td[2][@colspan]'):
                tracked = True
                if current:
                    highl_target = row.xpathEval('td[2]/span')[0].content.split("\n")[2].lstrip().rstrip()
                    current = False
                continue # ignore "tracked in ..." - rows
            tracked = False
                
            targeted_to = None
            if row.xpathEval('td[1]//img[@alt="Targeted to"]'):
                targeted_to = row.xpathEval('td[1]//a')[0].content
                affects = tmp_affects
                if highl_target:
                    if targeted_to.lower() == highl_target.lower():
                        current = True
                            
            xmledit = row.xpathEval('following-sibling::tr[@style="display: none"][1]')
            
            type = None
            milestone = None
            available_milestone = {}
            editurl = None
            editlock = False
            lock_importance = False
            
            if xmledit:
                xmledit = xmledit[0]
                editurl = xmledit.xpathEval('td/form')
                assert editurl, "Wrong XPath-Expr in InfoTable.parse() 'editurl' (%s)" %(DEBUG_URL or "unknown")
                editurl = utils.valid_lp_url(editurl[0].prop('action'), utils.BUG)
            
                if xmledit.xpathEval('descendant::select[contains(@id,".bugtracker")]'):
                    remote = True
                    
                if not remote:
                    for i in ["product", "sourcepackagename"]:
                        x = xmledit.xpathEval('td/form/div//table//input[contains(@id,".%s")]' %i)
                        if x:
                            type = x[0].prop("id")
                    if not type:
                        if not row.xpathEval('td[1]//img[contains(@src,"milestone")]'):
                            assert type, "Wrong XPath-Expr in InfoTable.parse() 'type' (%s)" %(DEBUG_URL or "unknown")
                
                    m = xmledit.xpathEval('descendant::select[contains(@id,".milestone")]//option')
                    if m:
                        for i in m:
                            available_milestone[i.prop("value")] = i.content
                            if i.prop("selected"):
                                milestone = i.content
                            
                    m = xmledit.xpathEval('descendant::td[@title="Editable only by the maintainer or bug contact of the project"]/span')
                    if len(m) == 2 and not milestone:
                        milestone = m[1].content
            
                if not xmledit.xpathEval('descendant::select[contains(@id,".importance")]'):
                    lock_importance = True
                m = set([".sourcepackagename", ".product", ".status", ".status-empty-marker",
                            ".importance", ".importance-empty-marker", ".milestone" ,
                            ".milestone-empty-marker"])
                for i in m:
                    x = xmledit.xpathEval('td/form//input[contains(@name,"%s")]' %i)
                    y = xmledit.xpathEval('td/form//select[contains(@name,"%s")]' %i)
                    if x or y:
                        edit_fields.add(i)
            else:
                editlock = True
                        
            s = row.xpathEval("td[2]")
            assert s, "Wrong XPath-Expr in InfoTable.parse() 'status' (%s)" %(DEBUG_URL or "unknown")
            assert s[0].prop("class") in temp_status, "unknown bugstatus '%s' in InfoTable.parse() (%s)" %(s[0].prop("class"), DEBUG_URL or "unknown")
            status = temp_status[s[0].prop("class")]
            
            s = row.xpathEval("td[3]")
            assert s, "Wrong XPath-Expr in InfoTable.parse() 'importance' (%s)" %(DEBUG_URL or "unknown")
            assert s[0].prop("class") in temp_importance, "unknown bugimportance '%s' in InfoTable.parse() (%s)" %(s[0].prop("class"), DEBUG_URL or "unknown")
            importance = temp_importance[s[0].prop("class")]
            
            assignee = row.xpathEval("td[4]//a")
            if assignee:
                if remote:
                    assignee = " ".join([i.lstrip().rstrip() for i in assignee[0].content.split("\n") if i.lstrip()])
                else:
                    assignee = user(assignee[0].prop("href").split('/')[1].lstrip('~'), assignee[0].content.lstrip("\n ").rstrip("\n "))
            else:
                assignee = user(None)
            
            if current:
                self.__current = len(self.__infolist)
            self.__infolist.append(Info(affects, status, importance,
                            assignee, current, editurl or self.__url, type, milestone,
                            available_milestone, lock_importance,
                            targeted_to, remote, editlock, edit_fields,
                            connection=self.__connection))

        self.parsed = True
        return True
        
    def get_current(self):
        if self.__current == None:
            raise AttributeError, "There is no row of the info-table linked to this bugreport (%s)" %self.__url
        assert isinstance(self.__current, int), "No 'highlight' in %s?" %self.__url
        return self.__infolist[self.__current]
    current = property(get_current, doc= "get the info-object for the current bug")
    
    def has_target(self, target):
        if not target:
            return None in [i.target for i in self.__infolist]
        target = target.lower()
        for i in self.__infolist:
            if i.target:
                if i.target.lower() == target:
                    return True
        return False
        
    @property
    def changed(self):
        return [_change_obj(i) for i in self.__infolist if i.changed]
        
    def commit(self, force_changes=False, ignore_lp_errors=True):
        """ delegates commit() to each changed element """
        for i in self.changed:
            i.component.commit(force_changes, ignore_lp_errors)

   
class BugReport(object):
    """ The 'BugReport'-object is the report itself
    
    * editable attributes:
        .description: any text
        .title/.summary: any text
        .tags: list, use list operations to add/remove tags
        .nickname
        
    * read-only attributes:
        .target: e.g. 'upstream'
        .sourcepackage: 'None' if not package specified
        .reporter, .date
    """
    def __init__(self, connection, xml, url):
        [self.__title, self.__description, self.__tags, self.__nickname,
        self.target, self.sourcepackage, self.reporter, self.date] = [None]*8
        self.__cache = {}
        self.parsed = False
        self.__connection=connection
        self.__xml = xml
        self.__url = url
        self.__description_raw = None
        
    def __repr__(self):
        return "<BugReport>"

    def parse(self):
        assert not self.parsed, "BugReport.parsed() should never excecuted twice" #TODO: add this to each parse()
        
        # getting text (description) of the bugreport
        description = self.__xml.xpathEval('//body//div[@class="report"]/\
div[@id="bug-description"]')
        assert description, "Wrong XPath-Expr in BugReport.parse() 'description' (%s)" %(DEBUG_URL or "unknown")
        
        # hack to change </p> into \n - any better idea? - NOT just eye-candy
        # this is also the format needed when committing changes
        p = description[0].xpathEval('p')
        description = ""
        for i in p[:-1]:
            description = "".join([description,i.content,"\n\n"])
        description = "".join([description,p[-1:].pop().content])
        self.__description = description
        
        # getting tile (summary) of the bugreport
        title = self.__xml.xpathEval('//title')
        assert title, "Wrong XPath-Expr in BugReport.parse() 'title' (%s)" %(DEBUG_URL or "unknown")
        titleFilter = 'Bug #[0-9]* in ([^:]*?): (.*)'
        title = re.findall(titleFilter, title[0].content)
        assert title, "Wrong RegEx in BugReport.parse() 'title' (%s)" %(DEBUG_URL or "unknown")
        self.__title = title[0][1].rstrip('\xe2\x80\x9d').lstrip('\xe2\x80\x9c')
        
        # getting target and sourcepackage
        target = title[0][0].split(" ")
        if len(target) == 2:
            self.target = target[1].lstrip("(").rstrip(")")
        self.sourcepackage = target[0]
        # fix sourcepackage
        if self.sourcepackage == 'Ubuntu':
            self.sourcepackage = None
        
        # getting tags of bugreport
        tags = self.__xml.xpathEval('//body//div[@class="report"]/\
div[@id="bug-tags"]//a/text()')
        self.__tags = [i.content for i in tags]

        m = self.__xml.xpathEval('//div[@id="maincontent"]//div[@style="float: left;"]')
        assert m, "Wrong XPath-Expr in BugReport.parse() 'info line' (%s)" %(DEBUG_URL or "unknown")
        r = re.search(r'\(([^\)]+)\)', m[0].content)
        self.__nickname = (r and r.group(1)) or None
        
        d = m[0].xpathEval('span')
        assert d, "Wrong XPath-Expr in BugReport.parse() 'date' (%s)" %(DEBUG_URL or "unknown")
        self.date = d[0].prop("title")
        
        d = m[0].xpathEval('a')
        assert d, "Wrong XPath-Expr in BugReport.parse() 'reporter' (%s)" %(DEBUG_URL or "unknown")
        self.reporter = user(d[0].prop('href').split('~').pop(), d[0].content.lstrip("\xc2\xa0"))
        
        self.__cache = {"title": self.__title, "description" : self.__description,
                            "tags" : self.__tags[:], "nickname" : self.__nickname}
        self.parsed = True
        return True
        
    def get_title(self):
        return self.__title
 
    def set_title(self, title):
        self.__title = title
    title = property(get_title, set_title, doc="title of a bugreport")
         
    def get_description(self):
        return self.__description
 
    def set_description(self, description):
        self.__description = description
    description = property(get_description, set_description,
                    doc="description of a bugreport")

    @property
    def tags(self):
        return self.__tags
         
    def get_nickname(self):
        return self.__nickname
 
    def set_nickname(self, nickname):
        self.__nickname = nickname
    nickname = property(get_nickname, set_nickname,
                    doc="nickname of a bugreport")
    
    @property
    def changed(self):
        changed = set()
        for k in self.__cache:
            if self.__cache[k] != getattr(self, k):
                changed.add(_change_obj(k))
        return frozenset(changed)
        
    @property
    def description_raw(self):
        if not self.__description_raw:
            url = "%s/+edit" %self.__url
            result = self.__connection.get(url)
            xmldoc = libxml2.htmlParseDoc(result.text, "UTF-8")
            x = xmldoc.xpathEval('//textarea[@name="field.description"]')
            assert x, "Wrong XPath-Expr in BugReport.commit() 'description' (%s)" %(DEBUG_URL or "unknown")
            self.__description_raw = x[0].content
        return self.__description_raw
        
    def commit(self, force_changes=False, ignore_lp_errors=True):
        """ Commits the local changes to launchpad.net
        
        * force_changes: if a user adds a tag which has not been used before
            and force_changes is True then commit() tries to create a new
            tag for this package; if this fails or force_changes=False
            commit will raise a 'ValueError'
        * ignore_lp_errors: if the user tries to commit invalid data to launchpad,
            launchpad returns an error-page. If 'ignore_lp_errors=False' Info.commit()
            will raise an 'ValueError' in this case, otherwise ignore this
            and leave the bugreport in launchpad unchanged (default=True)
        """
        if self.changed:
            if "description" not in [i.component for i in self.changed]:
                description = self.description_raw
            else:
                description = self.__description
            assert self.title and description, ".title and .description needed in BugReport.commit() (%s)" %(DEBUG_URL or "unknown")
            args = { 'field.actions.change': '1', 
                 'field.title': self.title, 
                 'field.description': description, 
                 'field.tags': ' '.join(self.tags),
                 'field.name': self.nickname or ""
                  }
            url = "%s/+edit" %self.__url
            result = self.__connection.post(url, args)
            
            if result.url == url and (not ignore_lp_errors or force_changes):
                x = libxml2.htmlParseDoc(result.text, "UTF-8")
                # informational message - 'new tag'
                if x.xpathEval('//p[@class="informational message"]/input[@name="field.actions.confirm_tag"]'):
                    if force_changes:
                        # commit new tag
                        args['field.actions.confirm_tag'] = '1'
                        url = "%s/+edit" %self.__url
                        result = self.__connection.post(url, args)
                    else:
                        raise ValueError, "Can't set 'tag', because it has not yet been used before."
                y = x.xpathEval('//p[@class="error message"]')
                if y:
                    raise ValueError, "launchpad.net error: %s" %y[0].content


class Attachment(object):
    """ Returns an 'Attachment'-object
    
    * editable attributes:
        .description: any text
        .contenttype: any text
        .is_patch: True/False
        
    * read-only attributes:
        .id: hash(local_filename) for local files,
            launchpadlibrarian-id for files uploaded to launchpadlibrarian.net
        .is_down: True if a file is downloaded to ATTACHMENT_PATH
        .is_up: True if file is uploaded to launchpadlibrarian.net
        ...
    TODO: work on docstring
    """
    def __init__(self, localfilename=None, localfileobject=None, description=None, is_patch=None, contenttype=None):
        assert not len([i for i in [localfilename,localfileobject] if i is not None]) == 2, "Attachment need localfilename or localfileobject, but not both"
        self.__description = description
        self.__is_patch = is_patch
        if self.__is_patch:
            self.__contenttype = "text/plain"
        else:
            self.__contenttype = contenttype
            
        self.__local_filename = localfilename
        self.__open_in_object = False
        if localfileobject:
            if isinstance(localfileobject, file):
                self.__local_fileobject = localfileobject
            else:
                raise TypeError, "Attachment: localfileobject needs to be a file-like object"
        else:
            if self.__local_filename:
                self.__local_fileobject = open(self.__local_filename, "r")
                self.__open_in_object = True
            else:
                self.__local_fileobject = None
                
        self.__url = None
        self.__id = None
        self.__text = None
        self._comment = None
        self.__cache = {"description": self.__description,
            "is_patch": self.__is_patch, "contenttype": self.__contenttype}

    def __repr__(self):
        s = set()
        if self.is_up:
            s.add("(up: #%s)" %self.lp_id)
        if self.is_down:
            s.add("(down: %s)" %self.__local_filename)
        return "<Attachment %s>" %", ".join(s)
        
    def __del__(self):
        if self.__open_in_object and self.__local_fileobject:
            self.__local_fileobject.close()
        
    def _set_comment(self, comment_id):
        self._comment = comment_id
           
    @property
    def id(self):
        if self.is_up:
            return int(self.__url.split("/")[-2])
        else:
            if self.__local_fileobject:
                return hash(self.__local_fileobject)
            elif self.__local_filename:
                return hash(self.__local_filename)
            raise ValueError, "unable to get hash of the given file"
            
        
    @property
    def is_down(self):
        """checks if
            * file is already in attachment cache => change .__local_filename to cache-location => returns True
            * file is in old cache location => move it to new location => change .__local_filename to cache-location => returns true
            * file is in any other location => return True
            * else return False
        """
        if self.is_up:
            filename_old = \
                os.path.expanduser(os.path.join(ATTACHMENT_PATH,
                    self.lp_id, self.lp_filename))
            cache_filename = \
                os.path.expanduser(os.path.join(ATTACHMENT_PATH,
                    self.sourcepackage, self.bugnumber,
                    self.lp_id, self.lp_filename))
            if os.path.exists(filename_old):
                os.renames(filename_old, cache_filename)
            if os.path.exists(cache_filename):
                self.__local_filename = cache_filename
                return True
        if self.__local_filename:
            return True
        else:
            return False
        
    @property
    def is_up(self):
        return bool(self.__url)
        
    @property
    def url(self):
        return self.__url or ""
    
    @property
    def lp_id(self):
        if self.is_up:
            return self.__url.split("/")[-2]
        
    @property
    def lp_filename(self):
        if self.is_up:
            return self.__url.split("/")[-1]
            
    @property
    def bugnumber(self):
        if self.is_up:
            return self.__edit.split("/")[-3]
                    
    @property
    def sourcepackage(self):
        if self.is_up:
            return self.__edit.split("/")[-5]
            
    @property
    def edit_url(self):
        if self.is_up:
            return utils.valid_lp_url(self.__edit, utils.BUG)
            
    @property
    def local_filename(self):
        return self.__local_filename
        
    @property
    def local_fileobject(self):
        return self.__local_fileobject
            
    def get_contenttype(self):
        return self.__contenttype
        
    def set_contenttype(self, type):
        if not self.__is_patch:
            self.__contenttype = type
    contenttype = property(get_contenttype, set_contenttype, doc="contenttype of an attachment")
    
    def get_description(self):
        return self.__description
        
    def set_description(self, text):
        self.__description = str(text)
    description = property(get_description, set_description, doc="description of an attachment")
    
    def get_ispatch(self):
        return self.__is_patch
        
    def set_ispatch(self, value):
        self.__is_patch = bool(value)
    is_patch = property(get_ispatch, set_ispatch, "type of an attachment")
            
    def set_attr(self, url, editurl, connection):
        assert connection, "Connection object needed"
        self.__connection = connection
        self.__url = url
        self.__edit = editurl                
        
    def get_changed(self):
        changed = set()
        for k in self.__cache:
            if self.__cache[k] != getattr(self, k):
                changed.add(k)
        return frozenset(changed)
    changed = property(get_changed, doc="get a list of changed attributes")
    
    
    @property
    def text(self):
        """get content of an attachment"""
        if not self.is_down:
            self.download()
        else:
            self.read_local()
        return self.__text
       
    def read_local(self):
        if self.is_down and self.__text == None:
            if not self.__local_fileobject:
                self.__local_fileobject = open(self.__local_filename, "r")
            self.__text = self.__local_fileobject.read()
       
    def download(self):
        """ save attachment in ATTACHMENT_PATH if not already done """
        if "text/plain" not in CONTENT_TYPES:
            CONTENT_TYPES.append("text/plain")
            
        self.__local_filename = \
            os.path.expanduser(os.path.join(ATTACHMENT_PATH,
                self.sourcepackage, self.bugnumber, self.lp_id, self.lp_filename))
                
        if self.is_up:
            attachment = self.__connection.get(self.__url)
            self.__text = attachment.text
            self.contenttype = attachment.contenttype
            utils.lazy_makedir(os.path.dirname(self.__local_filename))
            attachment_file = open(self.__local_filename, "w")
            attachment_file.write(self.__text)
            attachment_file.close()
        

class Attachments(object):
    def __init__(self, connection, xml, url):
        self.__attachments = {}
        self.parsed = False
        self.__cache = {}
        self.__connection=connection
        self.__xml = xml
        self.__url = url
        
    def __repr__(self):
        return "<Attachmentlist>"
        
    def __contains__(self, item):
        return self.__attachments.has_key(item)
        
    def __str__(self):
        return repr(self.__attachments.values()) #TODO: order
        
    def __iter__(self):
        """ allow direct iteration over instance of Attachments """
        return self.__attachments.itervalues()
            
    def __len__(self):
        return len(self.__attachments)
        
    def __getitem__(self, key):
        if self.__attachments.has_key(key):
            return self.__attachments[key]
        elif key in self.__attachments.values():
            return self.__attachments.values()[key]
        elif len(self.__attachments) > key:
            return self.__attachments[self.__attachments.keys()[key]]
        elif self.__cache.has_key(key) or key in self.__cache.values():
            raise ValueError, "This attachment has been removed"
        raise IndexError, "could not find '%s' in attachments ('%s')" %(key, self.__url) #list index out of range"
        
    def has_key(self, key):
        return self.__attachments.has_key(key)
        
    def filter(self, func):
        for a in self.__attachments.itervalues():
            if func(a):
                yield a
        
    def add(self, attachment):
        if isinstance(attachment, Attachment):
            # thats 'wrong' we have to add a comment-object to comments
            # (maybe use a fake object)`
            self.__attachments[attachment.id] = attachment
        else:
            raise IOError, "'attachment' must be an instance of 'Attachment'"
            
    def remove(self, key=None, func=None):
        assert len([i for i in (key,func) if not isinstance(i, type(None))]) == 1, "exactly one argument needed"
        if func:
            key = list(self.filter(func))
        def _isiterable(x):
            try:
                i = iter(x)
                return not isinstance(x, str) or False
            except TypeError:
                return False
        if _isiterable(key):
            for i in key:
                self.remove(key=i)
            return True
        if self.__attachments.has_key(key):
            self[key]._comment = None
            del self.__attachments[key]
            return True
        elif isinstance(key, Attachment):
            self[key.id]._comment = None
            del self.__attachments[key.id]
            return True 
        elif isinstance(key, int):
            try:
                x = self.__attachments.keys()[key]
            except IndexError:
                raise IndexError, "attachment '%s' not found" %key
            self[x]._comment = None
            del self.__attachments[x]
            return True
        raise TypeError, "unsupported type of 'key' in Attachment.remove()"
        
    def _ref_comment(self, attachments):
        (nr, att) = attachments
        for i in att:
            self[i]._comment = nr
        
    def parse(self):
        if self.parsed:
            return True
        if not self.__xml:
            self.parsed = True
            return True
        attachments = self.__xml[0].xpathEval('li[@class="download"]')
        for a in attachments:
            temp = a.xpathEval('a')[0]
            url = temp.prop("href")
            description = temp.content
            edit = a.xpathEval('small/a')[0].prop("href")
            attachment = Attachment(description=description)
            attachment.set_attr(url=url, editurl=edit, connection=self.__connection)
            self.add(attachment)
        self.__cache = self.__attachments.copy()
        self.parsed = True
        return True
        
    @property
    def deleted(self):
        deleted = set(self.__cache.keys()) - set(self.__attachments.keys())
        return frozenset(deleted)
    
    @property
    def added(self):
        added = set(self.__attachments.keys()) - set(self.__cache.keys())
        return frozenset(added)
    
    @property
    def changed(self):
        changed = []
        deleted = self.deleted
        added = self.added
        for i in deleted:
            changed.append(_change_obj(self.__cache[i], action="deleted"))
        for i in added:
            changed.append(_change_obj(self.__attachments[i], action="added"))
        for i in set(self.__attachments.keys()) - added - deleted:
            if self.__attachments[i].changed:
                changed.append(_change_obj(self.__attachments[i]))
        return changed
        
    def commit(self, force_changes=False, ignore_lp_errors=True):
        #nested functions:
        def _lp_edit(attachment):
            args = {'field.actions.change': '1', 'field.title': attachment.description,
                    'field.patch': attachment.is_patch and 'on' or 'off', 'field.contenttype': attachment.contenttype}
            self.__connection.post("%s/+edit" %attachment.edit_url, args)
        def _lp_delete(attachment):
            args = {'field.actions.delete': '1', 'field.title': attachment.description,
                    'field.patch': 'off', 'field.contenttype': 'text/plain'}
            self.__connection.post("%s/+edit" %attachment.edit_url, args)
            
        def _lp_add(attachment):
            """ delegated to comments """
            assert isinstance(attachment, Attachment), "<attachment> has to be an instance of 'Attachment'"
            if not attachment._comment:
                c = Comment(attachment=attachment)
                #delegate to Comments
                get_bug(id(self)).comments._lp_add_comment(c, force_changes, ignore_lp_errors)
            
        changed = set(self.changed)
        for i in changed:
            if i._change_obj__action == "added":
                _lp_add(i.component)
            elif i._change_obj__action == "deleted":
                _lp_delete(i.component)
            elif i._change_obj__action == "changed":
                _lp_edit(i.component)
            else:
                raise AttributeError, "Unknown action '%s' in Attachments.commit()" %i.component


class Comment(object):
    def __init__(self, subject=None, text=None, attachment=None):
        self.subject = subject
        self.text = text
        if not isinstance(attachment, (Attachment, type(None))):
            raise ValueError, "Unsupported attachment-type '%s'" %type(attachment)
        self.__nr = id(self) #  set unique number since comment is not attached to comments
        self._tmp_att = attachment and [attachment] # the attachment will be in an temp_list as long a comment is not added to a commentlist
        self._anker_list = None
        self.__user = None
        self.__date = None
        
    def set_attr(self, nr, user, date, attachments):
        self.__nr = nr
        self.__user = user
        self.__date = date
        if False in [isinstance(a, int) for a in attachments]:
            raise ValueError, "Each element in 'attachments' has to be an integer"
        get_bug(self._anker_list).attachments._ref_comment((self.__nr, attachments))
        
    def __repr__(self):
        if self.__nr and self.__user and self.__date:
            return "<Comment #%s by %s on %s>" %(self.__nr, self.__user, self.__date.split()[0])
        else:
            return "<Comment 'unknown'>"
        
    @property
    def number(self):
        return self.__nr
    
    @property
    def user(self):
        return self.__user
    
    @property
    def date(self):
        '''TODO: return Date-object ??'''
        return self.__date
        
        
    def get_attachments(self):
        if not self._anker_list:
            return self._tmp_att or []
        else:
            return [i for i in get_bug(self._anker_list).attachments if i._comment == self.__nr]
        
    def set_attachments(self, attachment):
        if isinstance(attachment, Attachment):
            if self._anker_list:
                #add to attachmentlist...
                assert not self._tmp_att, "Impossible to add more then one file to a comment"
                get_bug(self._anker_list).attachments.add(attachment)
                get_bug(self._anker_list).attachments._ref_comment((self.__nr, [attachment]))
            else:
                self._tmp_att = [attachment]
        else:
            raise TypeError, ""
    attachments = property(get_attachments, set_attachments, doc="attachment added to a comment")


class Comments(object):
    
    def __init__(self, connection, xml, url):
        self.__comments = []
        self.__cache = []
        self.__connection=connection
        self.__xml = xml
        self.__url = url
        self.parsed = False
        
    def __repr__(self):
        return "<Commentslist>"
        
    def __str__(self):
        return repr(self.__comments)
        
    def __iter__(self):
        """ allow direct iteration over instance of Attachments """
        for a in self.__comments:
            yield(a)
            
    def __len__(self):
        return len(self.__comments)
        
    def __getitem__(self, key):
        return self.__comments[key]
      
    def parse(self):
        for com in self.__xml:
            __com_user = com.xpathEval('div[@class="boardCommentDetails"]/a[1]')
            if not __com_user:
                return False
            __user = user(__com_user[0].prop('href').split('~').pop(), __com_user[0].content.lstrip("\xc2\xa0"))

            __com_url = com.xpathEval('div[@class="boardCommentDetails"]/a[2]')
            if not __com_url:
                return False
            __nr = __com_url[0].prop('href').split('/')[-1]

            __com_date = com.xpathEval('div[@class="boardCommentDetails"]/span')
            if not __com_date:
                return False
            __date = __com_date[0].prop('title')

            # Explicite parsing of comment-text is still NOT supported
            __com_text = com.xpathEval('div[@class="boardCommentBody"]/div')
            if not __com_text:
                return False
            __text = __com_text[0].content

            # Explicite parsing of comment-attachment is still NOT supported
            __attachments = set()
            __com_attach = com.xpathEval('div[@class="boardCommentBody"]/ul/li/a')
            for a in __com_attach:
                __attachments.add(int(a.prop('href').split('/')[-2]))
              
            c = Comment(subject=None, text=__text)
            c._anker_list = id(self)
            c.set_attr(__nr,__user,__date,__attachments)
            self.add(c)
        self.__cache = self.__comments[:]
        self.parsed = True
        return True
        
    def add(self, comment):
        if isinstance(comment, Comment):
            self.__comments.append(comment)
            if comment._tmp_att:
                for a in comment._tmp_att:
                    a._comment = comment.number
                    get_bug(id(self)).attachments.add(a)
                comment._tmp_att = None
            comment._anker_list = id(self)
        else:
            raise IOError, "'comment' must be an instance of 'Comment'"
    
    @property
    def changed(self):
        __changed = []
        __save = self.__comments[:]
        while True:
            if self.__cache == __save:
                return __changed
            else:
                x = __save.pop()
                __changed.insert(0,_change_obj(x, action="added"))
                
    def _lp_add_comment(self, comment, force_changes, ignore_lp_errors):
        assert isinstance(comment, Comment)
        #assert comment.subject
        #assert comment.text
        args = {
            'field.subject': comment.subject or "",
            'field.comment': comment.text or "", 
            'field.actions.save': '1',
            'field.filecontent.used': '',
            'field.email_me.used': ''
                }
        if comment.attachments:
            # attachment is always a list, currently only 1 attachment per new comment supported
            assert isinstance(comment.attachments, list), "comment.attachments has to be a list()"
            assert len(comment.attachments) == 1, "currently LP only supports uploading of one comment at a time"
            ca = comment.attachments[0]
            assert isinstance(ca, Attachment), "the file added to a comment has to be an instance of 'Attachment'"
            assert ca.description, "each attachment needs al east a description"
            args['field.patch.used'] = ''
            args['field.filecontent.used'] = ''
            args['field.filecontent'] = ca.local_fileobject
            args['field.attachment_description'] = ca.description or ""
            args['field.patch'] = ca.is_patch and 'on' or 'off'
         
        #print args #DEBUG
        __url = self.__url + '/+addcomment'
        __result = self.__connection.post(__url, args)
        #print __result.url #DEBUG
        if __result.url == __url and not ignore_lp_errors:# or force_changes):
            #print "check result" #DEBUG
            x = libxml2.htmlParseDoc(__result.text, "UTF-8")
            y = x.xpathEval('//p[@class="error message"]')
            if y:
                if force_changes:
                    z = x.xpathEval('//input[@name="field.subject" and @value=""]')
                    if z:
                        #print "has no 'subject' - add one!" #DEBUG
                        z = x.xpathEval('//div[@class="pageheading"]/div')
                        comment.subject = "Re: %s" %z[0].content.split("\n")[-2].lstrip().rstrip()
                        self._lp_add_comment(comment, False, ignore_lp_errors)
                else:
                    #print __result.text
                    raise ValueError, "launchpad.net error: %s" %y[0].content
        return __result
            
    def commit(self, force_changes=False, ignore_lp_errors=True):
        for i in self.changed:
            if i._change_obj__action == "added":
                self._lp_add_comment(i.component, force_changes, ignore_lp_errors)
            else:
                raise AttributeError, "Unknown action '%s' in Comments.commit()" %i.component


class Duplicates(object):
    def __init__(self, connection, xml, url):
        self.parsed = False
        self.__cache = None
        self.__connection=connection
        self.__xml = xml
        self.__url = url
        [self.__duplicate_of, self.__duplicates] = [None]*2
        
    def __repr__(self):
        return "<Duplicates>"
        
    def parse(self):
        """ bugnumber optional for debugging """
        if self.parsed:
            return True
        # Check if this bug is already marked as a duplicate
        nodes = self.__xml.xpathEval('//body//div[@id="mainarea"]//p[@class="informational message" and contains(.,"This report is a duplicate of")]/a')
        if len(nodes)>0:
            self.__duplicate_of = int(nodes[0].prop('href').split('/').pop())
        result = self.__xml.xpathEval('//body//div[@class="portlet"]/h2[contains(.,"Duplicates of this bug")]/../div[@class="portletBody"]/div/ul//li/a')
        self.__duplicates = set([int(i.prop("href").split('/')[-1]) for i in result])
        self.__cache = self.__duplicate_of
        self.parsed = True
        return True
        
    def get_duplicates(self):
        return self.__duplicates
    duplicates = property(get_duplicates, doc="get a list of duplicates")
    
    def get_duplicate_of(self):
        return self.__duplicate_of
        
    def set_duplicate_of(self, bugnumber):
        if bugnumber == None:
            self.__duplicate_of = None
        else:
            self.__duplicate_of = int(bugnumber)
    duplicate_of = property(get_duplicate_of, set_duplicate_of, doc="this bug report is duplicate of")
    
    @property
    def changed(self):
        __changed = set()
        if self.__cache != self.__duplicate_of:
            __changed.add("duplicate_of")
        return frozenset(__changed)
        
        
    def commit(self, force_changes=False, ignore_lp_errors=True):
        if self.changed:
            args = { 'field.actions.change': '1', 
                     'field.duplicateof': self.__duplicate_of or ""
                    }
            url = "%s/+duplicate" %self.__url
            result = self.__connection.post(url, args)
            
            if result.url == url and not ignore_lp_errors:# or force_changes):
                x = libxml2.htmlParseDoc(result.text, "UTF-8")
                # informational message - 'new tag'
                y = x.xpathEval('//p[@class="error message"]')
                if y:
                    raise ValueError, "launchpad.net error: %s" %y[0].content
            
    
class Secrecy(object):
    def __init__(self, connection, xml, url):
        self.parsed = False
        self.__cache = set()
        self.__connection=connection
        self.__xml = xml
        self.__url = url
        [self.__security, self.__private] = [False]*2
        
    def __repr__(self):
        return "<Secrecy>"
        
    def parse(self):
        if self.parsed:
            return True
        assert self.__xml, "Wrong XPath-Expr in Secrecy.parse() '__xml' (%s)" %(DEBUG_URL or "unknown")
        if self.__xml[0].xpathEval('img[@alt="(Security vulnerability)"]'):
            self.__security = True
        if self.__xml[0].xpathEval('img[@alt="(Private)"]'):
            self.__private = True
        self.__cache = {"security": self.__security, "private" : self.__private}
        self.parsed = True
        return True
            
    def _editlock(self):
        return bool(get_bug(id(self)).duplicate_of)
            
    def get_security(self):
        assert self.parsed, "parse first"
        return self.__security
    
    def set_security(self, security):
        if self._editlock():
            raise IOError, "'security' of this bug can't be edited, maybe because this bug is a duplicate of an other one"
        self.__security = bool(security)
    security = property(get_security, set_security, doc="security status")

    def get_private(self):
        assert self.parsed, "parse first"
        return self.__private
    
    def set_private(self, private):
        if self._editlock():
            raise IOError, "'private' of this bug can't be edited, maybe because this bug is a duplicate of an other one"
        self.__private = bool(private)
    private = property(get_private, set_private, doc="private status")

    def get_changed(self):
        __changed = set()
        for k in self.__cache:
            if self.__cache[k] != getattr(self, k):
                __changed.add(k)
        return frozenset(__changed)
    changed = property(get_changed, doc="get a list of changed attributes")
        
        
    def commit(self, force_changes=False, ignore_lp_errors=True):
        __url = "%s/+secrecy" %self.__url
        status = ["off", "on"]
        if self.changed:
            __args = { 'field.private': status[int(self.private)],
                 'field.security_related': status[int(self.security)],
                 'field.actions.change': 'Change'
               }
            __result = self.__connection.post(__url, __args)


class Subscribers(object):
    def __init__(self, connection, xml, url):
        self.parsed = False
        self.__cache = set()
        self.__connection=connection
        self.__xml = xml
        self.__url = url
        self.__subscribers = set()
        
    def __repr__(self):
        return "<Subscribers>"
        
    def parse(self):
        """TODO:
            * currently this only returns a set of people who are 'directly'
                subscribed to a bug, should we add people who are subscribed
                via dublicates?
        """
        if self.parsed:
            return True
        assert self.__xml, "Wrong XPath-Expr in Subscribers.parse() '__xml' (%s)" %(DEBUG_URL or "unknown")
        if self.__xml[0].prop("class") == "person":
            nodes = self.__xml[0].xpathEval("li/a")
            for i in nodes:
                self.__subscribers.add(user(i.prop("href").split("~").pop(),i.content))
        self.__cache = {"subscribers": self.__subscribers.copy()}
        self.parsed = True
        return True
        
    @property
    def subscribers(self):
        return self.__subscribers
        
    @property
    def changed(self):
        """get a list of changed attributes"""
        __changed = set()
        for k in self.__cache:
            if self.__cache[k] != getattr(self, k):
                __changed.add(k)
        return frozenset(__changed)
        
    def commit(self, force_changes=False, ignore_lp_errors=True):
        if self.changed:
            __remove = self.__cache["subscribers"] - self.__subscribers
            __add = self.__subscribers - self.__cache["subscribers"]
            for i in __remove:
                self._remove(self.__url, i)
            for i in __add:
                self._add(self.__url, i)
        
    def _add(self, url, lplogin):
        '''Add a subscriber to a bug.'''
        __url = "%s/+addsubscriber" %url
        __args = { 'field.person': lplogin, 
                 'UPDATE_SUBMIT': 'Add'
               }
        __result = self.__connection.post(__url, __args)
        if __result.url == __url:
            x = libxml2.htmlParseDoc(__result.text, "UTF-8")
            if x.xpathEval('//div[@class="message" and contains(.,"Invalid value")]'):
                raise ValueError, "The person's Launchpad ID or e-mail address. You can only subscribe someone who has a Launchpad account."
            else:
                raise ValueError, "Unknown rrror while subscribe %s to %s" %(lplogin, __url)
        
    def _remove(self, url, lplogin):
        '''Remove a subscriber from a bug.'''
        __url = "%s/" %url
        __args = { 'field.subscription': lplogin, 
                 'unsubscribe': 'Continue'
               }
        __result = self.__connection.post(__url, __args)
        

class ActivityWhat(str):
    def __new__(cls, what):
        obj = super(ActivityWhat, cls).__new__(ActivityWhat, what)
        obj.__task = None
        obj.__attribute = None
        x = what.split(":")
        assert len(x) < 3, "Wrong syntax in ActivityLog.ActivityWhat (%s)" %(DEBUG_URL or "unknown")
        if len(x) == 2:
            obj.__task = x[0]
            obj.__attribute = x[1].strip()
        else:
            obj.__attribute = x[0]
        return obj
        
    @property
    def task(self):
        return self.__task
        
    @property
    def attribute(self):
        return self.__attribute


     
class Activity(object):
    def __init__(self, date, user, what, old_value, new_value, message):
        self.__date = date
        self.__user = user
        self.__what = what
        self.__old_value = old_value
        self.__new_value = new_value
        self.__message = message
        
    def __repr__(self):
        return "<%s %s '%s'>" %(self.user, self.date, self.what)        
        
    @property
    def date(self):
        return self.__date
        
    @property
    def user(self):
        return self.__user
        
    @property
    def what(self):
        return self.__what
        
    @property
    def old_value(self):
        return self.__old_value
        
    @property
    def new_value(self):
        return self.__new_value
        
    @property
    def message(self):
        return self.__message
        
        
class ActivityLog(object):
    """
    TODO: there is nor clear relation between an entry in the activity log
        and a certain task, this is why the result of when(), completed(),
        assigned() and started_work() may differ from the grey infobox added
        to each task. Maybe we should also parse this box.
    """
    def __init__(self, connection, url):
        self.parsed = False
        self.__connection=connection
        page = self.__connection.get("%s/+activity" %url)
        
        r = re.compile(ur"\ufffd|\x0f|\x10|\x02")
        u = unicode(page.text, "UTF-8")
        self.__xmldoc = libxml2.htmlParseDoc(r.sub("??", u).encode("UTF-8"), "UTF-8")     
        self.__activity = []
        
    def __repr__(self):
        return "<activity log>"
        
    def __str__(self):
        return str(self.__activity)
        
    def __iter__(self):
        for i in self.activity:
            yield i
            
    def __getitem__(self, key):
        return self.activity[key]
        
    def __len__(self):
        return len(self.activity)
        
    @property
    def activity(self):
        if not self.parsed:
            self.parse()
        return self.__activity
        
    def _activity_rev(self):
        if not self.parsed:
            self.parse()
        return self.__activity_rev
        
    def parse(self):
        if self.parsed:
            return True
        table = self.__xmldoc.xpathEval('//body//table[@class="listing"][1]//tbody//tr')
        for row in table:
            r = row.xpathEval("td")
            assert len(r) == 6, "Wrong XPath-Expr in ActivityLog.parse() 'r('td')' (%s)" %(DEBUG_URL or "unknown")
            date = r[0].content
            x = r[1].xpathEval("a")
            lp_user = user(x[0].prop("href").split("~").pop(), x[0].content)
            what = r[2].content
            old_value = r[3].content
            new_value = r[4].content
            message = r[5].content
            
            self.__activity.append(Activity(date, lp_user, ActivityWhat(what),
                                        old_value, new_value, message))
        self.__activity_rev = self.__activity[::-1]
        self.parsed = True
        return True
        
    def assigned(self, task):
        for i in self._activity_rev():
            if i.what.task == task and i.what.attribute == "assignee":
                return i.date
                
    def completed(self, task):
        for i in self._activity_rev():
            if i.what.task == task and i.what.attribute == "status" and i.new_value in ["Invalid", "Fix Released"]:
                return i.date
                
    def when(self, task):
        for i in self._activity_rev():
            if i.what == "bug" and i.message.startswith("assigned to") and i.message.count(task):
                return i.date
                
    def started_work(self, task):
        for i in self._activity_rev():
            if i.what.task == task and i.what.attribute == "status" and i.new_value in ["In Progress", "Fix Committed"]:
                return i.date
            
                
                
        


class Bug(BugBase):
    _container_refs = {}

    def __init__(self, bug=None, url=None, connection=None):
            
        BugBase.__init__(self, bug, url, connection)

        if DEBUG == True:
            global DEBUG_URL
            DEBUG_URL = self.__url

        __bugpage = self.__connection.get(self.__url)

        #self.text contains the whole HTML-formated Bug-Page
        self.__text = __bugpage.text
        # Get the rewritten URL so that we have a valid one to attach comments
        # to
        self.__url = __bugpage.url
        
        r = re.compile(ur"\ufffd|\x0f|\x10|\x02")
        u = unicode(self.__text, "UTF-8")
        self.xmldoc = libxml2.htmlParseDoc(r.sub("??", u).encode("UTF-8"), "UTF-8")                
        
        self.__bugreport = BugReport(connection=self.__connection, xml=self.xmldoc, url=self.__url)
        self.__infotable = InfoTable(connection=self.__connection, xml=self.xmldoc.xpathEval('//body//table[@class="listing" or @class="duplicate listing"][1]'), url=self.__url)
        self.__attachments = Attachments(connection=self.__connection, xml=self.xmldoc.xpathEval('//body//div[@id="portlet-attachments"]/div/div/ul'), url=self.__url)
        self.__comments = Comments(connection=self.__connection, xml=self.xmldoc.xpathEval('//body//div[@class="boardComment"]'), url=self.__url)
        self.__duplicates = Duplicates(connection=self.__connection, xml=self.xmldoc, url=self.__url)
        self.__secrecy = Secrecy(connection=self.__connection, xml=self.xmldoc.xpathEval('//body//div[@id="bug-properties"]'), url=self.__url)
        self.__subscribers = Subscribers(connection=self.__connection, xml=self.xmldoc.xpathEval('//body//div[@id="portlet-subscribers"]/div/ul'), url=self.__url)
        self.__activity = ActivityLog(connection=self.__connection, url=self.__url)
        
        Bug._container_refs[id(self.__attachments)] = self
        Bug._container_refs[id(self.__comments)] = self
        Bug._container_refs[id(self.__secrecy)] = self

    def __del__(self):
        "run self.xmldoc.freeDoc() to clear memory"
        if hasattr(self, "xmldoc"):
            self.xmldoc.freeDoc()
        
    @property
    def changed(self):
        __result = []
        for i in filter(lambda a: a.startswith("_Bug__"),dir(self)):
            # just for Developing; later each object should have a 'changed' property
            try:
                a = getattr(self.__dict__[i], "changed")
            except AttributeError:
                continue
            if a:
                __result.append(_change_obj(self.__dict__[i]))
        return __result
        
    def commit(self, force_changes=False, ignore_lp_errors=True):
        for i in self.changed:
            result = i.component.commit(force_changes, ignore_lp_errors)
                        
    def revert(self):
        """ need a function to revert changes """
        pass
                
            
        
       
# Overwrite the abstract functions in bugbase.Bug        
        
    # read-only
    
    get_url = _gen_getter("__url")
    get_bugnumber = _gen_getter("__bugnumber")
    
    get_reporter = _gen_getter("__bugreport.reporter")
    get_date = _gen_getter("__bugreport.date")
    get_duplicates = _gen_getter("__duplicates.duplicates")
    get_description_raw = _gen_getter("__bugreport.description_raw")
    get_activity = _gen_getter("__activity")
        
    def get_text(self):
        return "%s\n%s" %(self.description,"\n".join([c.text for c in self.comments]))
    
    #...
    
    # +edit

    get_title = _gen_getter("__bugreport.title")
    get_description = _gen_getter("__bugreport.description")
    set_description = _gen_setter("__bugreport.description")
    get_tags = _gen_getter("__bugreport.tags")
    get_nickname = _gen_getter("__bugreport.nickname")
    
    #...

    # +editstatus

    get_infotable = _gen_getter("__infotable")
    get_info = _blocked(_gen_getter("__infotable.current"), "No 'current' available.")
    get_target = _blocked(_gen_getter("__infotable.current.target"), "Can't get 'target'.")
    get_importance = _blocked(_gen_getter("__infotable.current.importance"), "Can't get 'importance'.")
    set_importance = _blocked(_gen_setter("__infotable.current.importance"), "Can't set 'importance'.")
    get_status = _blocked(_gen_getter("__infotable.current.status"), "Can't get 'status'.")
    set_status = _blocked(_gen_setter("__infotable.current.status"), "Can't set 'status'.")
    get_assignee = _blocked(_gen_getter("__infotable.current.assignee"), "Can't get 'assignee'.")
    set_assignee = _blocked(_gen_setter("__infotable.current.assignee"), "Can't set 'assignee'.")
    get_milestone = _blocked(_gen_getter("__infotable.current.milestone"), "Can't get 'milestone'.")
    set_milestone = _blocked(_gen_setter("__infotable.current.milestone"), "Can't set 'milestone'.")
    get_sourcepackage = _blocked(_gen_getter("__infotable.current.sourcepackage"), "Can't get 'sourcepackage'.")
    set_sourcepackage = _blocked(_gen_setter("__infotable.current.sourcepackage"), "Can't set 'sourcepackage'.")
    get_affects = _blocked(_gen_getter("__infotable.current.affects"), "Can't get 'affects'.")
    
    def has_target(self, target):
        if not self.__infotable.parsed:
            self.__infotable.parse()
        return self.__infotable.has_target(target)
        
    # ...
    
    # +duplicate

    get_duplicates = _gen_getter("__duplicates.duplicates")
    get_duplicate = _gen_getter("__duplicates.duplicate_of")
    set_duplicate = _gen_setter("__duplicates.duplicate_of")
    
    #...
    
    # +secrecy
    
    get_security = _gen_getter("__secrecy.security")
    set_security = _gen_setter("__secrecy.security")
    get_private = _gen_getter("__secrecy.private")
    set_private = _gen_setter("__secrecy.private")
    
    #...
        
    # subscription
    
    get_subscriptions = _gen_getter("__subscribers.subscribers")
    get_comments = _gen_getter("__comments")
    get_attachments = _gen_getter("__attachments")    
    
    
def create_new_bugreport(product, summary, description, connection, tags=[], security_related=False):
    """ creates a new bugreport and returns its bug object
    
        product keys: "name", "target" (optional)
        tags: list of tags
    """
    
    args = {"field.title": summary,
            "field.comment": description,
            "field.actions.submit_bug": 1}
    if tags:
        args["field.tags"] = " ".join(tags)
    if security_related:
        args["field.security_related"] = "on"
    if product.has_key("target"):
        url = "https://bugs.launchpad.net/%s/+source/%s/+filebug-advanced" %(product["target"], product["name"])
        args["field.packagename"] = product["name"]
    else:
        url = "https://bugs.launchpad.net/%s/+filebug-advanced" %product["name"]
    
    result = connection.post(url, args)
    if not result.url.endswith("+filebug-advanced"):
        return Bug(url=result.url, connection=connection)
    else:
        raise ValueError, "There is a problem with the information you entered. \
 Please fix it and try again.\nurl: %s\nargs: %s" %(url, args)
