#!/usr/bin/python
# -*- coding: UTF-8 -*-

import os
import sys
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

from testbase import PyLPBugsTestSuite, PyLPBugsTestCase, ConfigTest, TestCaseIgnored

import libxml2
from launchpadbugs.lphelper import product as LPProduct, user as LPUser, unicode_for_libxml2


    
def _gen_product_test(product, result):
    def _testcase(self):
        x = LPProduct(*product)
        self.assert_(x.longname == result[0], "longname is '%s' but '%s' is expected" %(x.longname, result[0]))
        self.assert_(x.type == result[1], "type is '%s' but '%s' is expected" %(x.type, result[1]))
        self.assert_(x.tracked_in == result[2], "tracked_in is '%s' but '%s' is expected" %(x.tracked_in, result[2]))
        self.assert_(str(x) == result[3], "__str__() returns '%s' but '%s' is expected" %(x, result[3]))
        self.assert_(repr(x) == result[4], "__repr__() returns '%s' but '%s' is expected" %(repr(x), result[4]))
        self.assert_(bool(x) == result[5], "__nonzero__() returns '%s' but '%s' is expected" %(repr(x), result[5]))
    return _testcase
    
def _gen_user_test(user, result):
    def _testcase(self):
        x = LPUser(*user)
        self.assert_(x.lplogin == result[0], "lplogin is '%s' but '%s' is expected" %(x.lplogin, result[0]))
        self.assert_(x.realname == result[1], "realname is '%s' but '%s' is expected" %(x.realname, result[1]))
        self.assert_(x._attributes == result[2], "tracked_in is '%s' but '%s' is expected" %(x._attributes, result[2]))
        self.assert_(repr(x) == result[3], "__repr__() returns '%s' but '%s' is expected" %(repr(x), result[3]))
        self.assert_(str(x) == result[4], "__str__() returns '%s' but '%s' is expected" %(x, result[4]))
        self.assert_(bool(x) == result[5], "__nonzero__() returns '%s' but '%s' is expected" %(repr(x), result[5]))
    return _testcase
    
def _gen_parse_text_user_test(user, result):
    def _testcase(self):
        x = LPUser.parse_text_user(user)
        self.assert_(isinstance(x, LPUser), "parsing of user was not successfull")
        self.assert_(x.lplogin == result[0], "lplogin is '%s' but '%s' is expected" %(x.lplogin, result[0]))
        self.assert_(x.realname == result[1], "realname is '%s' but '%s' is expected" %(x.realname, result[1]))
    return _testcase
    
def _gen_parse_html_user_test(user, result):
    def _testcase(self):
        ctx = libxml2.htmlParseDoc(unicode_for_libxml2(user), "UTF-8")
        x = LPUser.parse_html_user(ctx.xpathEval("//a").pop())
        self.assert_(isinstance(x, LPUser), "parsing of user was not successfull")
        self.assert_(x.lplogin == result[0], "lplogin is '%s' but '%s' is expected" %(x.lplogin, result[0]))
        self.assert_(x.realname == result[1], "realname is '%s' but '%s' is expected" %(x.realname, result[1]))
    return _testcase
    
def _gen_parse_text_product_test(product, result):
    def _testcase(self):
        x, _ = LPProduct.parse_text_product(product)
        self.assert_(isinstance(x, LPProduct), "parsing of product was not successfull")
        self.assert_(str(x) == result[0], "__str__() returns '%s' but '%s' is expected" %(x, result[0]))
        self.assert_(x.longname == result[1], "longname is '%s' but '%s' is expected" %(x.longname, result[1]))
    return _testcase

class ProductTestCase(PyLPBugsTestCase):
    products = (
        ((None,), ("unknown", None, None, "", "<product None (unknown, type=None, tracked in=None)>", False)),
        (("foo",), ("foo", None, None, "foo", "<product foo (foo, type=None, tracked in=None)>", True)),
        (("foo", "foo-bar", "remote", "0.1"), ("foo-bar", "remote", "0.1", "foo", "<product foo (foo-bar, type=remote, tracked in=0.1)>", True)),
    )
    text_product = (
        ("bughelper", ("bughelper", "bughelper")),
        ("python-launchpad-bugs", ("python-launchpad-bugs","python-launchpad-bugs")),
        ("bughelper (Ubuntu)", ("bughelper","bughelper (Ubuntu)")),
        ("bughelper (Ubuntu Feisty)", ("bughelper", "bughelper (Ubuntu)")),
        ("Ubuntu Breezy", ("ubuntu", "Ubuntu")),
        ("gtk+2.0 (Ubuntu)", ("gtk+2.0", "gtk+2.0 (Ubuntu)")),
        ("mythbuntu/8.04", ("mythbuntu/8.04", "mythbuntu/8.04")),
    )
            
for i, (input, result) in enumerate(ProductTestCase.products):
    _testcase = _gen_product_test(input, result)
    _testcase.__doc__ = "create product #%s" %i
    setattr(ProductTestCase, "test_%.3d_create_product" %i, _testcase)
        
for j, (input, result) in enumerate(ProductTestCase.text_product):
    _testcase = _gen_parse_text_product_test(input, result)
    _testcase.__doc__ = "parse text-product '%s'" %input
    setattr(ProductTestCase, "test_%.3d_html_user" %(i+j+1), _testcase)
    

class UserTestCase(PyLPBugsTestCase):
    users = (
        ((None,), (None, "unknown" , set(), "<user None (unknown)>", "", False)),
        (("thekorn",), ("thekorn", "unknown", set(), "<user thekorn (unknown)>", "thekorn", True)),
        (("thekorn","Markus Korn"), ("thekorn", "Markus Korn", set(), "<user thekorn (Markus Korn)>", "thekorn", True)),
    )
    text_users = (
        ("Daniel Holbach (dholbach)", ("dholbach", "Daniel Holbach")),
        ("Markus Korn (thekorn)", ("thekorn", "Markus Korn")),
        ("Sebastian Dröge (slomo)", ("slomo", u"Sebastian Dröge")),
        ("foobar baz (bi-bi)", ("bi-bi", "foobar baz")),
        ("Alfred J.Tims (t1nt1n)", ("t1nt1n", "Alfred J.Tims")),
        ("Christer Edwards (christer.edwards)", ("christer.edwards", "Christer Edwards")),
        ('Evgeny "nekr0z" Kuznetsov (nekr0z)', ("nekr0z", 'Evgeny "nekr0z" Kuznetsov')),
        ("John McPherson (jrm+launchpadbugs)", ("jrm+launchpadbugs", "John McPherson")),
        ("jan-teichmann (teichmann-jan)", ("teichmann-jan", "jan-teichmann")),
        ("schmirrwurst@free.fr (schmirrwurst)", ("schmirrwurst", "schmirrwurst@free.fr")),
        ("Siegfried Gevatter (RainCT) (rainct)", ("rainct", "Siegfried Gevatter (RainCT)")),
        ("Paul O'Malley (ompaul)", ("ompaul", "Paul O'Malley")),
        ("Pyroman[FO] (pyroman)", ("pyroman", "Pyroman[FO]")),
        ("Model: Gateway MT3423 Notebook (mliberto0347)", ("mliberto0347", "Model: Gateway MT3423 Notebook")),
        ("OS/2-User (fzf7a2c02)", ("fzf7a2c02", "OS/2-User")),
        ("Ian! D. Allen (idallen)", ("idallen", "Ian! D. Allen")),
        ("^rooker (translator-duckburg)", ("translator-duckburg", "^rooker")),
        ("Michael R. Benfield, Jr. (leftfist)", ("leftfist", "Michael R. Benfield, Jr.")),
    )
    html_users = (
        ("""<a href="/~ogra">Oliver Grawert</a>""", ("ogra", "Oliver Grawert")),
        ("""<a href="/~macslow">Mirco Müller</a>""", ("macslow", u"Mirco Müller")),
        ("""<a href="https://blueprints.edge.launchpad.net/~ogra">
                <img alt="" width="14" height="14" src="/@@/person" />&nbsp;Oliver Grawert
            </a>""", ("ogra", "Oliver Grawert")),
        ("""<a href="https://blueprints.edge.launchpad.net/~macslow">
                <img alt="" width="14" height="14" src="/@@/person" />&nbsp;Mirco Müller
            </a>""", ("macslow", u"Mirco Müller")),
    )

for i, (input, result) in enumerate(UserTestCase.users):
    _testcase = _gen_user_test(input, result)
    _testcase.__doc__ = "create user #%s" %i
    setattr(UserTestCase, "test_%.3d_create_user" %i, _testcase)

for j, (input, result) in enumerate(UserTestCase.text_users):
    _testcase = _gen_parse_text_user_test(input, result)
    _testcase.__doc__ = "parse text-user '%s'" %input
    setattr(UserTestCase, "test_%.3d_text_user" %(i+j+1), _testcase)
        
for k, (input, result) in enumerate(UserTestCase.html_users):
    _testcase = _gen_parse_html_user_test(input, result)
    _testcase.__doc__ = "parse html-user #%s" %k
    setattr(UserTestCase, "test_%.3d_html_user" %(i+j+k+2), _testcase)
    
if __name__ == '__main__':
    import unittest
    
    ProductTestSuite = unittest.TestLoader().loadTestsFromTestCase(ProductTestCase)
    UserTestSuite = unittest.TestLoader().loadTestsFromTestCase(UserTestCase)
    
    alltests = PyLPBugsTestSuite((ProductTestSuite, UserTestSuite,))
    unittest.TextTestRunner(verbosity=2).run(alltests)
