/*  Screem:  screem-macro-manager.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <gdk/gdkkeysyms.h>

#include <glade/glade.h>

#include <string.h>

#include "screem-macro-manager.h"

#include "fileops.h"
#include "support.h"

#include "screem-application.h"
#include "screem-session.h"

/* quick hack for now, just access the editor directly */
#include "screem-window.h"
#include "screem-window-private.h"

struct ScreemMacroManagerPrivate {
	GtkAccelGroup *macros;
	GtkListStore *model;

	ScreemMacro *current;
};

enum {
	ARG_0
};

static void screem_macro_manager_class_init( ScreemMacroManagerClass *klass );
static void screem_macro_manager_init( ScreemMacroManager *macro_manager );
static void screem_macro_manager_finalize( GObject *object );
static void screem_macro_manager_set_prop( GObject *object, 
		guint property_id, const GValue *value, 
		GParamSpec *pspec );
static void screem_macro_manager_get_prop( GObject *object, 
		guint property_id, GValue *value, GParamSpec *pspec );


static gboolean screem_macro_manager_macro_activate( GtkAccelGroup *group,
		GObject *acceleratable, guint keyval,
		GdkModifierType modifier, gpointer data  );
static gboolean screem_macro_manager_add_macro( ScreemMacroManager *manager, 
		ScreemMacro *macro );
static void screem_macro_manager_load_macros( ScreemMacroManager *manager );
static void screem_macro_manager_save_macros( ScreemMacroManager *manager );

static gboolean screem_macro_manager_clear( GtkTreeModel *model,
		GtkTreePath *path, GtkTreeIter *iter, gpointer data );

/* G Object stuff */
#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_macro_manager_class_init( ScreemMacroManagerClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_macro_manager_finalize;
	parent_class = g_type_class_peek_parent( klass );

	object_class->get_property = screem_macro_manager_get_prop;
	object_class->set_property = screem_macro_manager_set_prop;
}

static void screem_macro_manager_init( ScreemMacroManager *manager )
{
	ScreemMacroManagerPrivate *priv;
	
	priv = manager->priv = g_new0( ScreemMacroManagerPrivate, 1 );

	priv->macros = gtk_accel_group_new();

	priv->model = gtk_list_store_new( SCREEM_MACRO_MODEL_NUM_COLS, 
			G_TYPE_STRING, G_TYPE_STRING,
			G_TYPE_POINTER, NULL );
	g_object_set_data( G_OBJECT( priv->model ), "manager",
			manager );


	screem_macro_manager_load_macros( manager );
}

static void screem_macro_manager_finalize( GObject *object )
{
	ScreemMacroManager *macro_manager;
	ScreemMacroManagerPrivate *priv;
	
	macro_manager = SCREEM_MACRO_MANAGER( object );
	priv = macro_manager->priv;

	g_object_unref( G_OBJECT( priv->macros ) );

	gtk_tree_model_foreach( GTK_TREE_MODEL( priv->model ),
			screem_macro_manager_clear, NULL );
	g_object_unref( G_OBJECT( priv->model ) );
	
	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static void screem_macro_manager_set_prop( GObject *object, guint property_id,
				    const GValue *value, GParamSpec *pspec )
{
	ScreemMacroManager *macro_manager;
	ScreemMacroManagerPrivate *priv;
	
	macro_manager = SCREEM_MACRO_MANAGER( object );
	priv = macro_manager->priv;
	
	switch( property_id ) {
		default:
			break;
	}
}

static void screem_macro_manager_get_prop( GObject *object, guint property_id,
				    GValue *value, GParamSpec *pspec)
{
	ScreemMacroManager *macro_manager;
	ScreemMacroManagerPrivate *priv;
	
	macro_manager = SCREEM_MACRO_MANAGER( object );
	priv = macro_manager->priv;
	
	switch( property_id ) {
		default:
			break;
	}
}

GType screem_macro_manager_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemMacroManagerClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_macro_manager_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemMacroManager ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_macro_manager_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemMacroManager",
					       &info, 0 );
	}

	return type;
}

/* Private API */

static void start_element( GMarkupParseContext *ctx,
		const gchar *name, 
		const gchar **attrs, const gchar **values, 
		gpointer data, GError **error )
{
	ScreemMacroManager *manager;
	ScreemMacroManagerPrivate *priv;
	ScreemMacro *macro;
	
	manager = SCREEM_MACRO_MANAGER( data );
	priv = manager->priv;

	if( attrs && values && ! strcmp( "macro", name ) ) {
		macro = priv->current = g_new0( ScreemMacro, 
				1 );
		while( *attrs && *values ) {
			if( ! strcmp( "accel", *attrs ) ) {
				gtk_accelerator_parse( *values, 
						&macro->key, 
						&macro->mod );
			}
			attrs ++;
			values ++;
		}
	} else {
		priv->current = NULL;
	}
}

static void text( GMarkupParseContext *ctx, const gchar *txt,
		gsize len, gpointer data, GError **error )
{
	ScreemMacroManager *manager;
	ScreemMacro *macro;

	manager = SCREEM_MACRO_MANAGER( data );

	macro = manager->priv->current;
	if( macro ) {
		macro->data = g_strndup( txt, len );
	}
}

static void end_element( GMarkupParseContext *ctx, const gchar *name,
		gpointer data, GError **error )
{
	ScreemMacroManager *manager;
	ScreemMacro *macro;
	
	manager = SCREEM_MACRO_MANAGER( data );

	macro = manager->priv->current;
	if( macro ) {	
		if( ! screem_macro_manager_add_macro( manager, macro ) ) {
			g_free( macro );
		}
		manager->priv->current = NULL;
	}
}

static const GMarkupParser parser = {
	start_element,
	end_element,
	text,
	NULL, /* passthrough */
	NULL, /* error */
};

static gboolean screem_macro_manager_macro_activate( GtkAccelGroup *group,
		GObject *acceleratable, guint keyval,
		GdkModifierType modifier, gpointer data  )
{
	ScreemMacro *macro;
	
	ScreemWindow *window;
	
	macro = (ScreemMacro *)data;
	window = SCREEM_WINDOW( acceleratable );
	
	switch( macro->type ) {
		case SCREEM_MACRO_INSERT_TEXT:
			screem_view_insert( SCREEM_VIEW( window->details->editor ), 
					-1, macro->data );
			break;
		default:
			break;
	};

	return TRUE;
}

static gboolean screem_macro_manager_add_macro( ScreemMacroManager *manager, 
		ScreemMacro *macro )
{
	ScreemMacroManagerPrivate *priv;
	GtkAccelFlags flags;
	gboolean valid;
	GClosure *closure;
	GtkTreeIter it;
	gchar *combo;
	
	priv = manager->priv;
	
	flags = GTK_ACCEL_VISIBLE;
	
	valid = gtk_accelerator_valid( macro->key, macro->mod );
	if( valid ) {		
		closure = g_cclosure_new( G_CALLBACK( screem_macro_manager_macro_activate ), macro, NULL );
		
		gtk_accel_group_connect( priv->macros, 
				macro->key, macro->mod, 
				flags, closure );

		combo = gtk_accelerator_get_label( macro->key,
				macro->mod );
	
		/* use huge pos to ensure an append */
		gtk_list_store_insert_with_values( priv->model, &it,
				G_MAXINT32,
				SCREEM_MACRO_MODEL_COMBO, combo,
				SCREEM_MACRO_MODEL_TEXT, macro->data,
				SCREEM_MACRO_MODEL_MACRO, macro,
				-1 );

		g_free( combo );
	} else {
		/* invalid accel in macros file */
	}

	return valid;
}

static void screem_macro_manager_load_macros( ScreemMacroManager *manager )
{
	ScreemMacroManagerPrivate *priv;
	gchar *tmp;
	gchar *macros;

	gchar *content;
	gsize len;
	GMarkupParseContext *ctx;
	
	priv = manager->priv;

	tmp = screem_get_dot_dir();
	macros = g_build_filename( tmp, "macros.xml", NULL );
	g_free( tmp );
	
	if( g_file_get_contents( macros, &content, &len, NULL ) ) {
		ctx = g_markup_parse_context_new( &parser,
				0, manager, NULL );
		g_markup_parse_context_parse( ctx, content, len, NULL );
		g_free( content );

		g_free( priv->current );
		priv->current = NULL;
		g_markup_parse_context_free( ctx );
	}

	g_free( macros );
}

static void screem_macro_manager_save_macros( ScreemMacroManager *manager )
{
	ScreemMacroManagerPrivate *priv;
	gchar *tmp;
	gchar *macros;
	GtkTreeModel *model;
	GtkTreeIter it;
	ScreemMacro *macro;
	GString *out;
	gchar *name;
	
	priv = manager->priv;

	tmp = screem_get_dot_dir();
	macros = g_build_filename( tmp, "macros.xml", NULL );
	g_free( tmp );
	
	model = GTK_TREE_MODEL( priv->model );
	if( gtk_tree_model_get_iter_first( model, &it ) ) {
		out = g_string_new( "<?xml version=\"1.0\"?><macros>" );
		do {
			macro = NULL;
			gtk_tree_model_get( model, &it,
					SCREEM_MACRO_MODEL_MACRO, &macro,
					-1 );
			if( macro ) {
				name = gtk_accelerator_name( macro->key,
						macro->mod );
				tmp = g_markup_escape_text( name,
						strlen( name ) );
				g_free( name );
				name = tmp;
				
				tmp = g_markup_escape_text( macro->data,
						strlen( macro->data ) );
				g_string_append_printf( out,
						"<macro accel=\"%s\">%s</macro>",
						name, tmp );
				g_free( tmp );
				g_free( name );	
			}
		} while( gtk_tree_model_iter_next( model, &it ) );
		g_string_append( out, "</macros>" );

		save_file( macros, out->str,
				( GNOME_VFS_PERM_USER_READ | 
				GNOME_VFS_PERM_USER_WRITE ), 
				FALSE, TRUE, NULL );

		g_string_free( out, TRUE );
	}
	
	g_free( macros );
}

static void screem_macro_manager_macro_select( GtkTreeSelection *sel,
		ScreemMacroManager *manager )
{
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkWidget *widget;
	GladeXML *xml;

    	view = gtk_tree_selection_get_tree_view( sel );

	xml = glade_get_widget_tree( GTK_WIDGET( view ) );

	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
		gchar *n;
		gchar *p;
		GtkWidget *widget;

		gtk_tree_model_get( model, &it, 
				SCREEM_MACRO_MODEL_COMBO, &n, 
				SCREEM_MACRO_MODEL_TEXT,  &p, -1 );
		
		widget = glade_xml_get_widget( xml, "macrocombo" );
		gtk_entry_set_text( GTK_ENTRY( widget ), n );

		widget = glade_xml_get_widget( xml, "macroaction" );
		widget = GTK_BIN( widget )->child;
		gtk_entry_set_text( GTK_ENTRY( widget ), p );

		g_free( n );
		g_free( p );

		widget = glade_xml_get_widget( xml, "remove" );
		gtk_widget_set_sensitive( widget, TRUE );
		widget = glade_xml_get_widget( xml, "update" );
		gtk_widget_set_sensitive( widget, TRUE );
	} else {
		widget = glade_xml_get_widget( xml, "macrocombo" );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		widget = glade_xml_get_widget( xml, "macroaction" );
		widget = GTK_BIN( widget )->child;
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "remove" );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "update" );
		gtk_widget_set_sensitive( widget, FALSE );
	}
}

/* also used for the update button as we perform a check to see
   if a helper already exists */
static void screem_macro_manager_insert_macro( GtkWidget *widget,
		ScreemMacroManager *manager )
{
	GladeXML *xml;
	const gchar *action;
	ScreemMacro *macro;

	xml = glade_get_widget_tree( widget );

	macro = g_new0( ScreemMacro, 1 );

	widget = glade_xml_get_widget( xml, "macrocombo" );
	macro->key = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), "keyval" ) );
	macro->mod = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), "state" ) );

	widget = glade_xml_get_widget( xml, "macroaction" );
	widget = GTK_BIN( widget )->child;
	action = gtk_entry_get_text( GTK_ENTRY( widget ) );

	macro->data = g_strdup( action );
	
	if( screem_macro_manager_add_macro( manager, macro ) ) {
		screem_macro_manager_save_macros( manager );
	} else {
		g_free( macro->data );
		g_free( macro );
	}
}

static void screem_macro_manager_remove_macro( GtkWidget *widget,
		ScreemMacroManager *manager )
{
	GladeXML *xml;
	GtkWidget *view;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeSelection *selection;

	const gchar *combo;

	xml = glade_get_widget_tree( widget );

	view = glade_xml_get_widget( xml, "macrolist" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {

		widget = glade_xml_get_widget( xml, "macrocombo" );
		combo = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		gtk_list_store_remove( GTK_LIST_STORE( model ), &it );

		screem_macro_manager_save_macros( manager );
	}
}


gboolean screem_macro_manager_grab_key( GtkWidget *widget, 
		GdkEventKey *event )
{
	gboolean ret;
	gchar *key;

	ret = FALSE;

	if( event->keyval != GDK_Tab &&
	    event->keyval != GDK_Up &&
	    event->keyval != GDK_Right &&
	    event->keyval != GDK_Down &&
	    event->keyval != GDK_Left ) {
		key = gtk_accelerator_get_label( event->keyval,
				event->state );
		if( key ) {
			gtk_entry_set_text (GTK_ENTRY(widget), key );
			g_free( key );
			g_object_set_data( G_OBJECT( widget ),
					"keyval",
					GUINT_TO_POINTER( event->keyval ) );
			g_object_set_data( G_OBJECT( widget ),
					"state",
					GUINT_TO_POINTER( event->state ) );
		} else {
			gtk_entry_set_text (GTK_ENTRY(widget), "" );
			g_object_set_data( G_OBJECT( widget ),
					"keyval", NULL );
			g_object_set_data( G_OBJECT( widget ),
					"state", NULL );
		}
		ret = TRUE;
	}

	return ret;
}

void screem_macro_manager_dialog_response( GtkWidget *dialog,
		gint button )
{
	GladeXML *xml;
	GtkWidget *widget;
	GtkTreeModel *model;
	ScreemMacroManager *manager;
	
	xml = glade_get_widget_tree( dialog );

	widget = glade_xml_get_widget( xml, "macrolist" );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	manager = g_object_get_data( G_OBJECT( model ), "manager" );

	switch( button ) {
		case 0: /* add */
			screem_macro_manager_insert_macro( widget,
					manager );
			break;
		case 1: /* remove */
			screem_macro_manager_remove_macro( widget,
					manager );
			break;
		case 2: /* refresh */
			screem_macro_manager_remove_macro( widget,
					manager );
			screem_macro_manager_insert_macro( widget,
					manager );
			break;
		default:
			gtk_widget_destroy( dialog );
			break;
	}
}

static gboolean screem_macro_manager_clear( GtkTreeModel *model,
		GtkTreePath *path, GtkTreeIter *iter, gpointer data )
{
	ScreemMacro *macro;
	
	macro = NULL;
	gtk_tree_model_get( model, iter,
			SCREEM_MACRO_MODEL_MACRO, &macro, -1 );
	if( macro ) {
		g_free( macro );
	}

	return FALSE;
}

/* Public API */

ScreemMacroManager *screem_macro_manager_new( void )
{
	ScreemMacroManager *manager;

	manager = SCREEM_MACRO_MANAGER( g_object_new( SCREEM_TYPE_MACRO_MANAGER, 
				NULL ) );

	return manager;
}

GtkAccelGroup *screem_macro_manager_get_accel_group( ScreemMacroManager *manager )
{
	g_return_val_if_fail( SCREEM_IS_MACRO_MANAGER( manager ),
			NULL );

	return manager->priv->macros;
}

GtkTreeModel *screem_macro_manager_get_model( ScreemMacroManager *manager )
{
	g_return_val_if_fail( SCREEM_IS_MACRO_MANAGER( manager ),
			NULL );

	return GTK_TREE_MODEL( manager->priv->model );
}


void screem_macro_manager_edit_macros( ScreemMacroManager *manager )
{
	GtkTreeModel *store;
	GtkWidget *view;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	static const gchar *titles[] = { "Key Combo", "Action/Text", NULL };
	gint i;
	GtkTreeSelection *selection;

	GladeXML *xml;
	GtkWidget *widget;

	/* FIXME: dialog isn't session managed */
	
	store = screem_macro_manager_get_model( manager );

	xml = glade_xml_new( GLADE_PATH"/screem.glade", "macros", NULL );
	glade_xml_signal_autoconnect( xml );	
	
	/* create tree view */
	view = glade_xml_get_widget( xml, "macrolist" );

	for( i = 0; titles[ i ]; ++ i ) {
		renderer = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, titles[ i ] );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );
		gtk_tree_view_column_set_attributes( col, renderer, 
						     "text", i, NULL );
	}
	
	gtk_tree_view_set_model( GTK_TREE_VIEW( view ), 
				 GTK_TREE_MODEL( store ) );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_macro_manager_macro_select ),
			  manager );

	widget = glade_xml_get_widget( xml, "macros" );
	
	gtk_widget_show_all( widget );
}

