/*  Screem:  screem-main.c
 *
 *  initialisation code, shouldn't do much as most of the init stuff
 *  belongs in screem-application.c
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gmodule.h>

#include <glib/gi18n.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomeui/gnome-authentication-manager.h>

#include <gtk/gtk.h>

#include <unistd.h>
#include <errno.h>

#include <glade/glade-init.h>
#include <glade/glade.h>

#include <gconf/gconf.h>

#include <libgnomevfs/gnome-vfs.h>

#include "screem-application.h"
#include "screem-session.h"

#include "fileops.h"
#include "support.h"

ScreemApplication *app;

static gchar *session_filename = SESSION_LAST;

/* passed parameters stuff */
static const struct poptOption options[] = {
	{ "load-session", 'l', POPT_ARG_STRING, &session_filename, 0,
	  N_("Load the given session file"),
	  N_("FILE") },
	{ NULL, '\0', 0, NULL, 0, NULL, NULL }
};

typedef enum {
	MISSING_GLADE,
	MISSING_SCHEMAS,
	MISSING_CONFIG_DIR
} MissingType;

static gboolean screem_main_start_app( const gchar **start_files );
static const gchar** parse_args( poptContext ctx, int argc, char *argv[] );

static void screem_main_missing( MissingType type );

int main( int argc, char *argv[] )
{
	GnomeProgram *program;
	poptContext ctx;
	const gchar** start_files;
	const gchar *icon;
	
	gchar *dotdir;
	GConfClient *client;
	
#ifdef ENABLE_NLS
	/* initialise i18n stuff */
        bindtextdomain( GETTEXT_PACKAGE, GNOMELOCALEDIR );
	bind_textdomain_codeset( GETTEXT_PACKAGE, "UTF-8" );
        textdomain( GETTEXT_PACKAGE );
#endif	
	program = gnome_program_init( PACKAGE, VERSION, 
			LIBGNOMEUI_MODULE,
			argc, argv,
			GNOME_PROGRAM_STANDARD_PROPERTIES,
			GNOME_PARAM_POPT_TABLE, options,
			GNOME_PARAM_APP_DATADIR, DATADIR,
			LIBGNOMEUI_PARAM_DEFAULT_ICON,
			DATADIR"/pixmaps/screem.png",
			GNOME_PARAM_HUMAN_READABLE_NAME,
			_( "Site CReation and Editing EnvironMent" ),
			NULL );

	gdk_threads_init();

	icon = DATADIR"/pixmaps/screem.png";
        gtk_window_set_default_icon_from_file( icon, NULL );

	/* parse arguments */
	g_object_get( G_OBJECT( program ), 
			GNOME_PARAM_POPT_CONTEXT, &ctx, NULL );
	start_files = parse_args( ctx, argc, argv );

        gnome_vfs_init();
	gnome_authentication_manager_init();

	glade_require( "gnome" );

	/* check schemas are installed, we can access ~/.screem,
	 * and we have a glade file */
	client = gconf_client_get_default();
	dotdir = screem_get_dot_dir();
	
	if( ! gconf_client_get_bool( client, "/apps/screem/installed",
				NULL ) ) {
		screem_main_missing( MISSING_SCHEMAS );
	} else if( ! uri_accessible( dotdir, GNOME_VFS_PERM_USER_ALL ) ) {
		screem_main_missing( MISSING_CONFIG_DIR );
	} else if( ! uri_exists( GLADE_PATH"/screem.glade", NULL ) ) {
		screem_main_missing( MISSING_GLADE );
	} else {
		app = screem_application_new();
		
		g_idle_add( (GSourceFunc)screem_main_start_app, 
				start_files );

		gdk_threads_enter();
		gtk_main();
		gdk_threads_leave();

		screem_application_close( app );
		g_object_unref( app );
	}

	g_free( dotdir );
	g_object_unref( client );

	gnome_vfs_shutdown();
	
	g_object_unref( program );

	xmlCleanupParser();
	
	return 0;
}

static gboolean screem_main_start_app( const gchar **start_files )
{
	screem_application_startup( app, session_filename, start_files );

	return FALSE;
}

static const gchar **parse_args( poptContext ctx, int argc, char *argv[] )
{
	const gchar** start_files;

        /* passed arguements */
        start_files = poptGetArgs( ctx );

	return start_files;
}

static void screem_main_missing( MissingType type )
{
	const gchar *primary;
	const gchar *secondary;

	primary =_( "Screem was unable to locate one of its files." ); 
	switch( type ) {
		case MISSING_GLADE:
			secondary = _( "The user interface file appears to be missing, this means that screem doesn't appear to have been installed correctly and can not continue." );
			break;
		case MISSING_SCHEMAS:
			secondary = _( "The default configuration could not be found, this means that screem doesn't appear to have been installed correctly and can not continue.  If you have just installed screem then try restarting gconfd." );
			break;
		case MISSING_CONFIG_DIR:
			secondary = _( "~/.screem could not be accessed, check that you, the owner, have read/write/execute permissions. If that directory doesn't exist then you need to make sure you have permissions to create .screem in your home directory." );
			break;
		default:
			secondary = "";
			break;
	}
	
	screem_hig_alert( GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
			primary, secondary, NULL );
}
