/*  Screem:  screem-preferences.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 */

#include <config.h>

#include <string.h>

#include <glade/glade.h>

#include <gconf/gconf-client.h>

#include <glib/gi18n.h>

#include <libgnomeui/gnome-icon-entry.h>

#include <gtk/gtk.h>

#include <gdk/gdkcolor.h>
#include <gdk/gdkkeysyms.h>

#include "screem-application.h"
#include "screem-preferences.h"
#include "screem-helper.h"
#include "screem-tagtree.h"
#include "screem-tagfile.h"

#include "fileops.h"
#include "support.h"

#include "screem-apps-model.h"

#include "screem-doctype-manager.h"

typedef enum {
	SCREEM_PREF_TOGGLE,
	SCREEM_PREF_RADIO,
	SCREEM_PREF_SPIN,
	SCREEM_PREF_SPIN_INT,
	SCREEM_PREF_COLOUR,
	SCREEM_PREF_STRING
} ScreemPrefNotifyType;

typedef struct {
	const gchar *widget;
	const gchar *key;
	ScreemPrefNotifyType type;
	gint value;
} ScreemPrefStruct;

static const ScreemPrefStruct misc_prefs[] = {
	{ "default_filename_entry",	"/apps/screem/default_filename",
		SCREEM_PREF_STRING,	0 },
	{ "default_charset_combo", 	"/apps/screem/editor/default_charset", 
		SCREEM_PREF_STRING,	0 },
	{ "mimetype",			"/apps/screem/default_mimetype",
		SCREEM_PREF_STRING,	0 },
	{ "create_backups",		"/apps/screem/general/create_backups", 
		SCREEM_PREF_TOGGLE,	0 },
	{ "auto_save",			"/apps/screem/general/auto_save", 
		SCREEM_PREF_TOGGLE,	0 },
	{ "auto_save_period",  	"/apps/screem/general/auto_save_interval",
		SCREEM_PREF_SPIN_INT,	0 },
	{ NULL, NULL, 0, 0 }
};

static const ScreemPrefStruct editor_colours[] = {
	{ "fontpicker",		"/apps/screem/editor/font",		SCREEM_PREF_STRING,	0 },
	{ "editortextpicker", 	"/apps/screem/editor/text", 		SCREEM_PREF_COLOUR,	0 },
	{ "editorbgpicker", 	"/apps/screem/editor/back", 		SCREEM_PREF_COLOUR,	0 },
	{ "themefont",		"/apps/screem/editor/themefont",	SCREEM_PREF_TOGGLE,	0 },
	{ "themecolours",	"/apps/screem/editor/themecolours",	SCREEM_PREF_TOGGLE,	0 },
	{ NULL, NULL, 0, 0 }
};

static const ScreemPrefStruct editor_features[] = {
	{ "highlight", 		  "/apps/screem/editor/highlight", 	 SCREEM_PREF_TOGGLE,	0 },
	{ "error_highlight", 	  "/apps/screem/editor/error_highlight", SCREEM_PREF_TOGGLE,	0 },
	{ "current_line",	  "/apps/screem/editor/highlight_current_line", 	SCREEM_PREF_TOGGLE,	0 },

	{ "tagging", 		  "/apps/screem/editor/inline_tagging",  SCREEM_PREF_TOGGLE,	0 },
	{ "indent", 		  "/apps/screem/editor/auto_indent", 	 SCREEM_PREF_TOGGLE,	0 },
	{ "tagclose", 		  "/apps/screem/editor/intelliclose",    SCREEM_PREF_TOGGLE,	0 },
	{ "wordwrap",		  "/apps/screem/editor/wordwrap",	 SCREEM_PREF_TOGGLE,	0 },
	{ "margin",		  "/apps/screem/editor/right_margin",	 SCREEM_PREF_TOGGLE,	0 },
	{ "margin_size",  "/apps/screem/editor/margin_column",	 SCREEM_PREF_SPIN,	0 },

	{ "autocomplete",	  "/apps/screem/editor/autocomplete",    SCREEM_PREF_TOGGLE,    0 },
	{ "tooltips",	  "/apps/screem/editor/tooltips",    SCREEM_PREF_TOGGLE,    0 },
	{ "spaces", 		  "/apps/screem/editor/spaces_instead_of_tabs", 	 SCREEM_PREF_TOGGLE,	0 },
	{ "tabwidth", 	   	  "/apps/screem/editor/tabwidth",	 SCREEM_PREF_SPIN,	0 },
	{ "autoenton",		  "/apps/screem/editor/entityinsertion", SCREEM_PREF_RADIO,	0 },
	{ "autoentoff",		  "/apps/screem/editor/entityinsertion", SCREEM_PREF_RADIO,	1 },
	{ "autoentbychar",	  "/apps/screem/editor/entityinsertion", SCREEM_PREF_RADIO,	2 },
	{ NULL, NULL, 0, 0 }
};

/*static const ScreemPrefStruct browser_prefs[] = {
	{ "browser1_static",  "/apps/screem/Browser1_static",  SCREEM_PREF_TOGGLE,	0 },
	{ "browser2_static",  "/apps/screem/Browser2_static",  SCREEM_PREF_TOGGLE,	0 },
	{ "browser3_static",  "/apps/screem/Browser3_static",  SCREEM_PREF_TOGGLE,	0 },
	{ "browser1_dynamic", "/apps/screem/Browser1_dynamic", SCREEM_PREF_TOGGLE,	0 },
	{ "browser2_dynamic", "/apps/screem/Browser2_dynamic", SCREEM_PREF_TOGGLE,	0 },
	{ "browser3_dynamic", "/apps/screem/Browser3_dynamic", SCREEM_PREF_TOGGLE,	0 },
	{ "browser1_path",    "/apps/screem/Browser1_path",    SCREEM_PREF_STRING,	0 },
	{ "browser2_path",    "/apps/screem/Browser2_path",    SCREEM_PREF_STRING,	0 },
	{ "browser3_path",    "/apps/screem/Browser3_path",    SCREEM_PREF_STRING,	0 },
	{ "browser1_icon",    "/apps/screem/Browser1_icon",    SCREEM_PREF_STRING,	0 },
	{ "browser2_icon",    "/apps/screem/Browser2_icon",    SCREEM_PREF_STRING,	0 },
	{ "browser3_icon",    "/apps/screem/Browser3_icon",    SCREEM_PREF_STRING,	0 },
	{ "browser1_name",    "/apps/screem/Browser1_name",    SCREEM_PREF_STRING,		0 },
	{ "browser2_name",    "/apps/screem/Browser2_name",    SCREEM_PREF_STRING,		0 },
	{ "browser3_name",    "/apps/screem/Browser3_name",    SCREEM_PREF_STRING,		0 },

	{ NULL, NULL, 0, 0 }
};*/

static void 
screem_preferences_syntax_highlighting( ScreemApplication *application,
				  GladeXML *xml );
/*static void 
screem_preferences_browsers( ScreemApplication *application,
		GladeXML *xml );*/

static void
screem_preferences_edit_tag_trees( ScreemApplication *application,
				GladeXML *xml );
static void screem_preferences_tag_tree_sel( GtkTreeSelection *sel,
					ScreemApplication *app );

static void screem_preferences_add_pref( GtkWidget *dialog,
				const gchar *key,
				ScreemPrefNotifyType type,
				gint value,
				GtkWidget *widget );


static void screem_preferences_add_page( GtkWidget *dialog,
					const ScreemPrefStruct *pref )
{
	guint i;
	GtkWidget *widget;
	GladeXML *xml;

	xml = glade_get_widget_tree( dialog );
	for( i = 0; pref[ i ].widget; ++ i ) {
		widget = glade_xml_get_widget( xml, pref[ i ].widget );
		screem_preferences_add_pref( dialog,
					pref[ i ].key,
					pref[ i ].type,
					pref[ i ].value,
					widget );
	}
}

void screem_preferences_edit( ScreemApplication *application )
{
	ScreemSession *session;
	GladeXML *xml;
	GtkWidget *widget;
	static GtkWidget *dialog = NULL;
	GtkTreeModel *list;
	
	if( dialog ) {
		gtk_widget_show( dialog );
		gdk_window_raise( GDK_WINDOW( dialog->window ) );
		return;
	}

	session = screem_application_get_session( application );

	/* create dialog */
	xml = glade_xml_new( GLADE_PATH"/screem.glade", 
			"preferences", NULL );
	dialog = glade_xml_get_widget( xml, "preferences" );
	g_object_set_data( G_OBJECT( dialog ), "session", session );
	glade_xml_signal_autoconnect( xml );
	
	widget = glade_xml_get_widget( xml, "default_filename_entry" );
	screem_gtk_add_history( widget );
	
	widget = glade_xml_get_widget( xml, "default_charset_combo" );
	list = screem_application_get_encodings( application );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), list ); 
	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
			0 );

	widget = glade_xml_get_widget( xml, "mimetype" );

	list = GTK_TREE_MODEL( screem_support_get_mime_types_store() );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), list ); 
	g_object_unref( list );
	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
			0 );

	screem_preferences_syntax_highlighting( application, xml );

/*	screem_preferences_browsers( application, xml );*/
	
	screem_doctype_manager_init( application, xml );

	screem_preferences_edit_tag_trees( application, xml );
	
	g_signal_connect_swapped( G_OBJECT( dialog ), "delete_event",
			G_CALLBACK( g_nullify_pointer ),
			&dialog );
	g_signal_connect_swapped( G_OBJECT( dialog ), "destroy",
			G_CALLBACK( g_nullify_pointer ),
			&dialog );

	screem_preferences_add_page( dialog, misc_prefs );
	screem_preferences_add_page( dialog, editor_colours );
	screem_preferences_add_page( dialog, editor_features );
/*	screem_preferences_add_page( dialog, browser_prefs );*/
	
	screem_session_restore_dialog( session, dialog );
	
	gtk_widget_show( dialog );
}

static void screem_preferences_toggle_notify( GConfClient *client,
					      guint cnxn_id,
					      GConfEntry *entry,
					      gpointer data )
{
	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		GtkWidget *toggle;
		gboolean wstate;
		gboolean state;
		
		toggle = GTK_WIDGET( data );

		wstate = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
		state = gconf_value_get_bool( entry->value );

		if( wstate != state ) {
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(toggle),
						      state );
		}
	}
}
static gboolean screem_preferences_toggle_changed( GtkWidget *toggle )
{
	GConfClient *client;
	gboolean state;
	const gchar *key;

	client = gconf_client_get_default();

	state = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) );
	
	key = g_object_get_data( G_OBJECT( toggle ), "key" );
	gconf_client_set_bool( client, key, state, NULL );

	g_object_unref( client );
	
	return TRUE;
}

static void screem_preferences_spin_notify( GConfClient *client,
					    guint cnxn_id,
					    GConfEntry *entry,
					    gpointer data )
{
	if( entry->value && entry->value->type == GCONF_VALUE_INT ) {
		GtkWidget *spin;
		gint value;
		gint nvalue;
		
		spin = GTK_WIDGET( data );
		value = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( spin ) );
		nvalue = gconf_value_get_int( entry->value );

		if( value != nvalue ) {
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( spin ),
						   nvalue );
		}
	} else if( entry->value && entry->value->type == GCONF_VALUE_FLOAT ) {
		GtkWidget *spin;
		gfloat value;
		gfloat nvalue;
		
		spin = GTK_WIDGET( data );
		value = gtk_spin_button_get_value( GTK_SPIN_BUTTON( spin ) );
		nvalue = gconf_value_get_float( entry->value );

		if( value != nvalue ) {
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( spin ),
						   nvalue );
		}
	}
}

static void screem_preferences_spin_changed( GtkWidget *spin )
{
	GConfClient *client;
	const gchar *key;
	gdouble value;
	ScreemPrefNotifyType ptype;

	client = gconf_client_get_default();
	key = g_object_get_data( G_OBJECT( spin ), "key" );
	ptype = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( spin ),
				"ptype" ) );
	
	value = gtk_spin_button_get_value( GTK_SPIN_BUTTON( spin ) );

	if( ptype == SCREEM_PREF_SPIN ) {
		gconf_client_set_float( client, key, value, NULL );
	} else if( ptype == SCREEM_PREF_SPIN_INT ) {
		gconf_client_set_int( client, key, (gint)value, NULL );
	}

	g_object_unref( client );
}

static void screem_preferences_colour_notify( GConfClient *client,
					      guint cnxn_id,
					      GConfEntry *entry,
					      gpointer data )
{
	if( entry->value && entry->value->type == GCONF_VALUE_STRING ) {
		const gchar *value; 
		GdkColor colour;
		GtkWidget *widget;
		GdkColor pcolour;
	
		value = gconf_value_get_string( entry->value );
		widget = GTK_WIDGET( data );
		gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ),
				&pcolour );

		if( gdk_color_parse( value, &colour ) && 
		    ! gdk_color_equal( &colour, &pcolour ) ) {
			gtk_color_button_set_color( GTK_COLOR_BUTTON(widget),
					&pcolour );
		}
	}
}

static void screem_preferences_radio_notify( GConfClient *client,
					    guint cnxn_id,
					    GConfEntry *entry,
					    gpointer data )

{
	if( entry->value && entry->value->type == GCONF_VALUE_INT ) {
		GtkWidget *toggle;
		const gchar *name;
		gboolean wstate;
		gint gval;
		gint val;
		
		toggle = GTK_WIDGET( data );
		name = gtk_widget_get_name( toggle );

		wstate= gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
		gval = gconf_value_get_int( entry->value );
		
		val = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( toggle ),
							  "val" ) );

		if( val != gval && wstate ) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle),
						     FALSE );
		} else if( val == gval && ! wstate ) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle),
						     TRUE );
		}
	}
}

static void screem_preferences_string_notify( GConfClient *client,
					      guint cnxn_id,
					      GConfEntry *entry,
					      gpointer data )
{
	const gchar *value;
	GtkWidget *widget;
	GtkWidget *gtkentry;
	const gchar *cvalue;

	GtkTreeModel *model;
	GtkTreeIter it;
	gint cols;
	gchar *colval;
	gboolean found;
	
	if( entry->value && entry->value->type != GCONF_VALUE_STRING ) {
		return;
	}

	value = gconf_value_get_string( entry->value );
	widget = GTK_WIDGET( data );
	gtkentry = NULL;
	colval = NULL;	
	
	if( GTK_IS_ENTRY( widget ) ) {
		gtkentry = widget;
	} else if( GTK_IS_COMBO_BOX_ENTRY( widget ) ) {
		gtkentry = GTK_BIN( widget )->child;

		/* combo boxes are special, they have display text
		 * and possibly value text as well, for our
		 * purposes they are up to 2 column GtkListStores,
		 * 1 column display == value, 2 column col 2 == value */
		model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
		cols = 0;
		if( model ) {
			cols = gtk_tree_model_get_n_columns( model );
		}

		if( cols == 2 ) {
			found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &it, 1, value );
			if( ! found) {
				found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &it, 0, value );
			}
			if( found ) {
				gtk_tree_model_get( model, &it,
					0, &colval, -1 );
				value = colval;
			}
		}
	} else if( GTK_IS_FILE_CHOOSER_BUTTON( widget ) ) {
		gchar *cvalue;

		cvalue = gtk_file_chooser_get_uri( GTK_FILE_CHOOSER( widget ) );
		if( ! cvalue || strcmp( cvalue, value ) ) {
			gtk_file_chooser_set_uri( GTK_FILE_CHOOSER( widget ),
					value );
		}
	} else if( GNOME_IS_ICON_ENTRY( widget ) ) {
		const gchar *cvalue;
		cvalue = gnome_icon_entry_get_filename( GNOME_ICON_ENTRY( widget ) );
		if( ! cvalue || strcmp( cvalue, value ) )
			gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( widget ),
						      value);
	} else if( GTK_IS_FONT_BUTTON( widget ) ) {
		const gchar *cvalue;
		cvalue = gtk_font_button_get_font_name( GTK_FONT_BUTTON( widget ) );
		if( ! cvalue || strcmp( cvalue, value ) )
			gtk_font_button_set_font_name( GTK_FONT_BUTTON( widget ), value );
		
	}
	if( gtkentry ) {
		cvalue = gtk_entry_get_text( GTK_ENTRY( gtkentry ) );
		if( value && strcmp( value, cvalue ) ) {
			gtk_entry_set_text( GTK_ENTRY( gtkentry ),
					    value );
		}
	}
	g_free( colval );
}

static gboolean screem_preferences_icon_changed( GtkWidget *widget )
{
	GConfClient *client;
	const gchar *key;
	gchar *value;

	client = gconf_client_get_default();
	key = g_object_get_data( G_OBJECT( widget ), "key" );
	
	value = gnome_icon_entry_get_filename( GNOME_ICON_ENTRY( widget ) );
	if( ! value )
		value = g_strdup( "" );

	gconf_client_set_string( client, key, value, NULL );

	g_free( value );

	g_object_unref( client );
	
	return TRUE;
}

static gboolean screem_preferences_string_set( GtkWidget *widget,
		GtkWidget *parent )
{
	GtkWidget *entry = NULL;
	const gchar *cvalue = NULL;
	const gchar *key;

	GtkTreeModel *model;
	GtkTreeIter it;
	gint cols;
	gchar *colval;
	gchar *disval;
	gboolean found;

	gchar *uri;
	
	GConfClient *client;

	client = gconf_client_get_default();
	
	key = g_object_get_data( G_OBJECT( widget ), "key" );
	colval = NULL;
	
	if( parent ) {
		widget = parent;
	}

	uri = NULL;

	if( GTK_IS_COMBO_BOX_ENTRY( widget ) ) {
		entry = GTK_BIN( widget )->child;
		cvalue = gtk_entry_get_text( GTK_ENTRY( entry ) );
		
		/* combo boxes are special, they have display text
		 * and possibly value text as well, for our
		 * purposes they are up to 2 column GtkListStores,
		 * 1 column display == value, 2 column col 2 == value */
		model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
		cols = 0;
		if( model ) {
			cols = gtk_tree_model_get_n_columns( model );
		}
		if( cols == 2 ) {
			found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &it, 1, cvalue );
			if( ! found ) {
				found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &it, 0, cvalue );
			}
			if( found ) {
				gtk_tree_model_get( model, &it,	
					0, &disval,
					1, &colval, -1 );
				cvalue = colval;

				if( strcmp( cvalue, disval ) ) {
					gtk_entry_set_text( GTK_ENTRY( entry ), disval );
				}
				g_free( disval );
			}
		}
	} else if( GTK_IS_FILE_CHOOSER_BUTTON( widget ) ) {
		uri = gtk_file_chooser_get_uri( GTK_FILE_CHOOSER( widget ) );
		cvalue = uri;
	} else if( GTK_IS_ENTRY( widget ) ) {
		cvalue = gtk_entry_get_text( GTK_ENTRY( widget ) );
	}
	
	if( cvalue ) {
		gconf_client_set_string( client, key, 
					 cvalue, NULL );
	}

	g_free( colval );
	g_free( uri );
	
	g_object_unref( client );
	
	return FALSE;
}


static void screem_preferences_font_set( GtkWidget *fp )
{
	GConfClient *client;
	const gchar *key;
	const gchar *font;

	client = gconf_client_get_default();
	key = g_object_get_data( G_OBJECT( fp ), "key" );
	font = gtk_font_button_get_font_name( GTK_FONT_BUTTON( fp ) );
	gconf_client_set_string( client, key, font, NULL );
	g_object_unref( client );
}

static void screem_preferences_colour_set( GtkColorButton *cb )
{
	GConfClient *client;
	const gchar *key;
	gchar *value;
	GdkColor color;

	client = gconf_client_get_default();
	key = g_object_get_data( G_OBJECT( cb ), "key" );
	gtk_color_button_get_color( cb, &color );
	value = g_strdup_printf( "#%.4x%.4x%.4x", 
			color.red, color.green, color.blue );
	gconf_client_set_string( client, key, value, NULL );
	g_free( value );
	g_object_unref( client );
}

static gboolean screem_preferences_radio_changed( GtkWidget *toggle )
{
	GConfClient *client;
	gboolean state;
	gint val;
	const gchar *key;

	client = gconf_client_get_default();
	val = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( toggle ),
						  "val" ) );
	state = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) );
	
	key = g_object_get_data( G_OBJECT( toggle ), "key" );
	
	if( state ) {
		gconf_client_set_int( client, key, val, NULL );
	}
	
	g_object_unref( client );
	
	return TRUE;
}

static void screem_preferences_language_changed_cb( GtkComboBox *box )
{
	GladeXML *xml;
	GtkTreeIter it;
	GtkTreeModel *model;
	GtkSourceLanguage *lang;
	GtkWidget *widget;
	GSList *tags;
	GSList *l;
	gchar *name;
	gchar *id;
	GtkTreePath *path;
	gint i;
	
	xml = glade_get_widget_tree( GTK_WIDGET( box ) );
	
	model = gtk_combo_box_get_model( box );
	
	if( gtk_combo_box_get_active_iter( box, &it ) ) {
		gtk_tree_model_get( model, &it, 1, &lang, -1 );

		/* FIXME: do we need to unref lang ? */
		
		widget = glade_xml_get_widget( xml, "tags_treeview" );
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
		gtk_list_store_clear( GTK_LIST_STORE( model ) );
		
		tags = gtk_source_language_get_tags( lang );
		
		for( i = 0, l = tags; l; l = l->next, ++ i ) {
			g_object_get( G_OBJECT( l->data ),
					"name", &name, "id", &id,
					NULL );
			gtk_list_store_insert_with_values( GTK_LIST_STORE( model ),
					&it, i,
					0, name, 1, id, -1 );
			
			g_object_unref( l->data );
		}
		g_slist_free( tags );

		/* trigger screem_preferences_styles_cb on first item */
		path = gtk_tree_path_new_first();
		gtk_tree_view_set_cursor( GTK_TREE_VIEW( widget ),
				path, NULL, FALSE );
		gtk_tree_path_free( path );
	}
}

static GtkSourceLanguage *screem_preferences_get_lang( GtkWidget *box )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkSourceLanguage *lang;
	
	model = gtk_combo_box_get_model( GTK_COMBO_BOX( box ) );
	lang = NULL;
	if( gtk_combo_box_get_active_iter( GTK_COMBO_BOX( box ), &it ) ) {
		gtk_tree_model_get( model, &it, 1, &lang, -1 );
	}

	return lang;
}

static GtkSourceTagStyle *screem_preferences_get_style( GtkWidget *treeview, GtkSourceLanguage **rlang, gchar **rid, GtkSourceTagStyle **rdef_style  )
{
	GladeXML *xml;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter it;
	gchar *id;
	GtkWidget *widget;
	GtkSourceLanguage *lang;
	GtkSourceTagStyle *style;
	GtkSourceTagStyle *def_style;
	
	if( rlang ) {
		*rlang = NULL;
	}
	if( rid ) {
		*rid = NULL;
	}
	if( rdef_style ) {
		*rdef_style = NULL;
	}
	
	xml = glade_get_widget_tree( treeview );
	
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( treeview ) );
	gtk_tree_view_get_cursor( GTK_TREE_VIEW( treeview ), &path,
				NULL );
	gtk_tree_model_get_iter( model, &it, path );
	gtk_tree_path_free( path );
	gtk_tree_model_get( model, &it, 1, &id, -1 );

	g_return_val_if_fail( id != NULL, NULL );
	
	widget = glade_xml_get_widget( xml, "hl_mode_combo" );
	lang = screem_preferences_get_lang( widget );

	style = NULL;
	if( lang ) {
		style = gtk_source_language_get_tag_style( lang, id );
		def_style = gtk_source_language_get_tag_default_style( lang, id );
		if( ! style ) {
			style = def_style;
		} else {
			style->is_default = TRUE;
			def_style->is_default = TRUE;

			style->is_default = ( memcmp( style,
						def_style,
						sizeof( GtkSourceTagStyle ) ) == 0 );
			if( ! rdef_style ) {
				gtk_source_tag_style_free( def_style );
			} else {
				*rdef_style = def_style;
			}
		}
		if( rlang ) {
			*rlang = lang;
		}
	}
	if( ! rid ) {
		g_free( id );
	} else {
		*rid = id;
	}

	return style;
}

static void 
screem_preferences_save_tag_style( const GtkSourceTagStyle *style,
		GtkSourceLanguage *lang, const gchar *id )
{
	GConfClient *client;
	gchar *path;
	gchar *value;
	gchar *fg;
	gchar *bg;

	client = gconf_client_get_default();
	path = g_strconcat( "/apps/screem/editor/syntax/",
			gtk_source_language_get_id( lang ),
			id, NULL );
	if( style ) {
		fg = screem_gdk_color_to_string( &style->foreground,
				FALSE );
		bg = screem_gdk_color_to_string( &style->background,
				FALSE );
		value = g_strdup_printf( "%d/%s/%s/%d/%d/%d/%d",
				style->mask, fg, bg,
				style->italic,
				style->bold,
				style->underline,
				style->strikethrough );
	
		gconf_client_set_string( client, path, value, NULL );

		g_free( fg );
		g_free( bg );
		g_free( value );
	} else {
		gconf_client_unset( client, path, NULL );
	}
	g_free( path );
	g_object_unref( client );
}

static void screem_preferences_style_changed( GtkWidget *widget )
{
	GladeXML *xml;
	GtkSourceTagStyle *style;
	GtkSourceTagStyle *new_style;
	GtkSourceTagStyle *def_style;
	GtkSourceLanguage *lang;
	gchar *id;
	guint mask;
	
	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );
	
	widget = glade_xml_get_widget( xml, "tags_treeview" );
	style = screem_preferences_get_style( widget, &lang, &id,
			NULL );
	if( style->is_default ) {
		style = gtk_source_tag_style_new();
	}

	widget = glade_xml_get_widget( xml, "hl_mode_combo" );
	lang = screem_preferences_get_lang( widget );
	
	new_style = gtk_source_tag_style_copy( style );
	widget = glade_xml_get_widget( xml, "bold_togglebutton" );
	new_style->bold = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "italic_togglebutton" );
	new_style->italic = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "underline_togglebutton" );
	new_style->underline = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "strikethrough_togglebutton" );
	new_style->strikethrough = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	widget = glade_xml_get_widget( xml, "foreground_checkbutton" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		new_style->mask |= GTK_SOURCE_TAG_STYLE_USE_FOREGROUND;
		widget = glade_xml_get_widget( xml, 
				"foreground_colorbutton" );
		gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ),
				&new_style->foreground );
		gtk_widget_set_sensitive( widget, TRUE );
	} else {
		new_style->mask &= ~GTK_SOURCE_TAG_STYLE_USE_FOREGROUND;
		widget = glade_xml_get_widget( xml, 
				"foreground_colorbutton" );
		gtk_widget_set_sensitive( widget, TRUE );
	}
	widget = glade_xml_get_widget( xml, "background_checkbutton" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		new_style->mask |= GTK_SOURCE_TAG_STYLE_USE_BACKGROUND;
		widget = glade_xml_get_widget( xml, 
				"background_colorbutton_2" );
		gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ),
				&new_style->background );
		gtk_widget_set_sensitive( widget, TRUE );
	} else {
		new_style->mask &= ~GTK_SOURCE_TAG_STYLE_USE_BACKGROUND;
		widget = glade_xml_get_widget( xml, 
				"background_colorbutton_2" );
		gtk_widget_set_sensitive( widget, TRUE );
	}

	if( memcmp( style, new_style, sizeof( GtkSourceTagStyle ) ) != 0 ) {
		def_style = gtk_source_language_get_tag_default_style( lang, id );
		mask = new_style->mask;
		if( ! ( mask & GTK_SOURCE_TAG_STYLE_USE_BACKGROUND ) ) {
			new_style->background = def_style->background;
		}
		if( ! ( mask & GTK_SOURCE_TAG_STYLE_USE_FOREGROUND ) ) {
			new_style->foreground = def_style->foreground;
		}
		
		widget = glade_xml_get_widget( xml, "reset_button" );
		gtk_widget_set_sensitive( widget,
				memcmp( new_style, def_style,
					sizeof( GtkSourceTagStyle ) ) != 0 );
		screem_preferences_save_tag_style( new_style, 
				lang, id );
				
		gtk_source_language_set_tag_style( lang, id, new_style );
		gtk_source_tag_style_free( def_style );
	}

	gtk_source_tag_style_free( style );
	gtk_source_tag_style_free( new_style );
	g_free( id );
}


static void screem_preferences_styles_cb( GtkWidget *treeview )
{
	GladeXML *xml;
	
	GtkWidget *widget;

	GtkSourceTagStyle *style;
	
	gboolean use;
	GdkColor color;
	
	xml = glade_get_widget_tree( treeview );

	/* get selected style */
	style = screem_preferences_get_style( treeview, NULL, NULL,
			NULL );

	g_return_if_fail( style != NULL );

	widget = glade_xml_get_widget( xml, "bold_togglebutton" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
			style->bold );
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "italic_togglebutton" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
			style->italic );
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "underline_togglebutton" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
			style->underline );
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "strikethrough_togglebutton" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
			style->strikethrough );

	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "foreground_checkbutton" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	use = ( style->mask & GTK_SOURCE_TAG_STYLE_USE_FOREGROUND );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), use );
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "foreground_colorbutton" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	if( use ) {
		gtk_color_button_set_color( GTK_COLOR_BUTTON( widget ),
				&style->foreground );
	} else {
		/* FIXME: use user defined text color if set */
		gdk_color_parse( "#000000000000", &color );
		gtk_color_button_set_color( GTK_COLOR_BUTTON( widget ),
				&color );
	}
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	gtk_widget_set_sensitive( widget, use );

	widget = glade_xml_get_widget( xml, "background_checkbutton" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	use = ( style->mask & GTK_SOURCE_TAG_STYLE_USE_BACKGROUND );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), use );
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "background_colorbutton_2" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	if( use ) {
		gtk_color_button_set_color( GTK_COLOR_BUTTON( widget ),
				&style->background );
	} else {
		/* FIXME: use user defined background color if set */
		gdk_color_parse( "#ffffffffffff", &color );
		gtk_color_button_set_color( GTK_COLOR_BUTTON( widget ),
				&color );
	}
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	gtk_widget_set_sensitive( widget, use );

	widget = glade_xml_get_widget( xml, "reset_button" );
	gtk_widget_set_sensitive( widget, ! style->is_default );

	gtk_source_tag_style_free( style );
}

static void screem_preferences_reset_style( GtkWidget *widget )
{
	GladeXML *xml;
	gchar *id;
	GtkSourceLanguage *lang;
	GtkSourceTagStyle *style;
	
	gtk_widget_set_sensitive( widget, FALSE );
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "tags_treeview" );
	style = screem_preferences_get_style( widget, &lang, &id, NULL );
	/* erase saved style */
	screem_preferences_save_tag_style( NULL, lang, id );
	
	gtk_source_language_set_tag_style( lang, id, NULL );
	
	gtk_source_tag_style_free( style );
	screem_preferences_styles_cb( widget );
}

static void 
screem_preferences_syntax_highlighting( ScreemApplication *application,
				  GladeXML *xml )
{
	GtkWidget *widget;
	GtkListStore *tags;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;

	GtkListStore *langs;
	GtkTreeIter it;
	GtkSourceLanguagesManager* lm;
	const GSList *languages;
	const GSList *l;
	GtkSourceLanguage *lang;
	gchar *name;
	gint i;
	
	/* setup tree view */
	tags = gtk_list_store_new( 2, G_TYPE_STRING, G_TYPE_STRING );

	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( tags ), 0, GTK_SORT_ASCENDING );
	widget = glade_xml_get_widget( xml, "tags_treeview" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
			GTK_TREE_MODEL( tags ) );
	g_object_unref( tags );
	
	g_signal_connect( G_OBJECT( widget ), "cursor-changed",
			G_CALLBACK( screem_preferences_styles_cb ),
			NULL );
	
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new_with_attributes( _( "Elements" ),
			rend, "text", 0, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );

	/* Add languages to combo box */
	widget = glade_xml_get_widget( xml, "hl_mode_combo" );
	langs = gtk_list_store_new( 2, G_TYPE_STRING, G_TYPE_OBJECT );
	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( langs ), 
			0, GTK_SORT_ASCENDING );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), 
			GTK_TREE_MODEL( langs ) );
	g_object_unref( langs );
	rend = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start( GTK_CELL_LAYOUT( widget ), rend,
			TRUE );
	gtk_cell_layout_set_attributes( GTK_CELL_LAYOUT( widget ),
			rend, "text", 0, NULL );
	
	lm = screem_application_load_syntax_tables();
	languages = gtk_source_languages_manager_get_available_languages( lm );
	for( i = 0, l = languages; l; l = l->next, ++ i ) {
		lang = GTK_SOURCE_LANGUAGE( l->data );
		name = gtk_source_language_get_name( lang );
		gtk_list_store_insert_with_values( langs, &it, i,
				0, name, 1, lang, -1 );
		g_free( name );
	}
	g_signal_connect( G_OBJECT( widget ), "changed",
			G_CALLBACK( screem_preferences_language_changed_cb ),
			NULL );
	gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );

	widget = glade_xml_get_widget( xml, "bold_togglebutton" );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "italic_togglebutton" );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "underline_togglebutton" );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "strikethrough_togglebutton" );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "foreground_checkbutton" );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "background_checkbutton" );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "foreground_colorbutton" );
	g_signal_connect( G_OBJECT( widget ), "color_set",
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "background_colorbutton_2" );
	g_signal_connect( G_OBJECT( widget ), "color_set",
			G_CALLBACK( screem_preferences_style_changed ),
			NULL );
	widget = glade_xml_get_widget( xml, "reset_button" );
	g_signal_connect( G_OBJECT( widget ), "clicked",
			G_CALLBACK( screem_preferences_reset_style ),
			NULL );
}

/*static void screem_preferences_browser_change( GtkComboBox *combo,
		GtkWidget *icon )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	GdkPixbuf *pixbuf;

	if( gtk_combo_box_get_active_iter( combo, &it ) ) {
		model = gtk_combo_box_get_model( combo );
		

	}
}

static void 
screem_preferences_browsers( ScreemApplication *application,
		GladeXML *xml )
{
	GtkTreeModel *model;
	GtkWidget *widget;
	GtkWidget *icon;

	model = screem_apps_model_get_default();
	
	widget = glade_xml_get_widget( xml, "browser1_path" );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), model );
	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
			AVAILABLE_APP_COLUMN_EXEC );
	screem_apps_model_setup_cell_layout( widget );

	icon = glade_xml_get_widget( xml, "browser1_icon" );
	g_signal_connect( G_OBJECT( widget ), "changed",
			G_CALLBACK( screem_preferences_browser_change ),
			icon );
	
	widget = glade_xml_get_widget( xml, "browser2_path" );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), model );
	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
			AVAILABLE_APP_COLUMN_EXEC );
	screem_apps_model_setup_cell_layout( widget );

	icon = glade_xml_get_widget( xml, "browser2_icon" );
	g_signal_connect( G_OBJECT( widget ), "changed",
			G_CALLBACK( screem_preferences_browser_change ),
			icon );

	widget = glade_xml_get_widget( xml, "browser3_path" );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), model );
	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
			AVAILABLE_APP_COLUMN_EXEC );
	screem_apps_model_setup_cell_layout( widget );

	icon = glade_xml_get_widget( xml, "browser3_icon" );
	g_signal_connect( G_OBJECT( widget ), "changed",
			G_CALLBACK( screem_preferences_browser_change ),
			icon );

	g_object_unref( model );
}
*/
static void
screem_preferences_tag_tree_toggle( GtkCellRendererToggle *toggle,
				const gchar *path,
				gpointer data )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	gboolean active;
	ScreemTagFile *tfile;
	
	model = GTK_TREE_MODEL( data );

	active = gtk_cell_renderer_toggle_get_active( toggle );
	
	if( gtk_tree_model_get_iter_from_string( model, &it, path ) ) {

		active = ! active;

		gtk_tree_model_get( model, &it,
				TAG_FILE_OBJ_COL, &tfile, -1 );

		g_object_set( G_OBJECT( tfile ), 
				"active", active, NULL );
	}
}

static void
screem_preferences_edit_tag_trees( ScreemApplication *application,
				GladeXML *xml )
{
	GtkListStore *model;
	GtkWidget *widget;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	GtkTreeSelection *sel;

	widget = glade_xml_get_widget( xml, "tagtreeview" );
	model = screem_tag_tree_get_file_model();

	rend = gtk_cell_renderer_toggle_new();
	g_signal_connect( G_OBJECT( rend ), "toggled",
			  G_CALLBACK( screem_preferences_tag_tree_toggle ),
			  model );
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, FALSE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
				"active", TAG_FILE_ACTIVE_COL, NULL );
	gtk_tree_view_column_set_title( col, _( "Active" ) );
	
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 
					     TAG_FILE_NAME_COL, NULL ); 
	gtk_tree_view_column_set_title( col, _( "Name" ) );

	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 
					     TAG_FILE_URI_COL, NULL ); 
	gtk_tree_view_column_set_title( col, "URI" );

	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( model ) );

	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	g_signal_connect( G_OBJECT( sel ), "changed",
			  G_CALLBACK( screem_preferences_tag_tree_sel ),
			  application );
}


static void screem_preferences_tag_tree_sel( GtkTreeSelection *sel,
					ScreemApplication *app )
{
	GtkWidget *widget;
	GladeXML *xml;
	GtkTreeModel *model;
	GtkTreeIter it;
	gboolean sens;
	ScreemTagFile *tfile;
	const gchar *uri;
	gchar *dot;
	
	widget = GTK_WIDGET( gtk_tree_selection_get_tree_view( sel ) );
	
	xml = glade_get_widget_tree( widget );
	sens = FALSE;
	uri = NULL;
	dot = screem_get_dot_dir();
	
	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
		sens = TRUE;

		gtk_tree_model_get( model, &it,
				TAG_FILE_OBJ_COL, &tfile, -1 );
		uri = screem_tag_file_get_uri( tfile );
	}

	widget = glade_xml_get_widget( xml, "refreshtagfile" );
	gtk_widget_set_sensitive( widget, sens );

	/* we don't want to allow removal of system tag trees,
	 * or tag trees from ~/.screem */
	if( uri &&
	    ( ! strncmp( DATADIR"/screem/tagtrees", uri,
		    strlen( DATADIR"/screem/tagtrees" ) ) ||
	      ! strncmp( dot, uri, strlen( dot ) ) ) )  {
		
		sens = FALSE;
	}
	
	widget = glade_xml_get_widget( xml, "removetagfile" );
	gtk_widget_set_sensitive( widget, sens );

	g_free( dot );
}

void screem_preferences_refresh_tag_file( GtkWidget *widget )
{
	GladeXML *xml;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	ScreemTagFile *tfile;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "tagtreeview" );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );

	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {

		gtk_tree_model_get( model, &it,
				TAG_FILE_OBJ_COL, &tfile, -1 );

		screem_tag_file_load( tfile, SCREEM_TAG_TREE_FORMAT );
	}
}

void screem_preferences_remove_tag_file( GtkWidget *widget )
{
	GladeXML *xml;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	ScreemTagFile *tfile;
	const gchar *uri;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "tagtreeview" );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );

	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {

		gtk_tree_model_get( model, &it,
				TAG_FILE_OBJ_COL, &tfile, -1 );
		uri = screem_tag_file_get_uri( tfile );
		
		screem_tag_tree_remove_uri( uri );
	}

}

void screem_preferences_add_tag_file( GtkWidget *widget )
{
	GladeXML *xml;
	GladeXML *dxml;
	const gchar *prompt;
	gchar *txt;
	gchar *title;
	const gchar *url;
	GtkTreeModel *model;
	GtkTreeIter it;
	gboolean found;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "tagtreeview" );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	
	dxml = glade_xml_new( GLADE_PATH"/screem.glade",
			"open_file_location", NULL );

	widget = glade_xml_get_widget( dxml, "prompt" );
	prompt = _( "Enter the URL of the tag tree to add" );
	txt = g_strconcat( "<b>", prompt, "</b>", NULL );
	gtk_label_set_markup( GTK_LABEL( widget ), txt );
	g_free( txt );
	
	widget = glade_xml_get_widget( dxml, "open_file_location" );
	title = g_strdup_printf( "%s - Screem",
			_( "Add Tag Tree from Location" ) );
	gtk_window_set_title( GTK_WINDOW( widget ),
				title );
	g_free( title );

	glade_xml_signal_autoconnect( dxml );
	
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( dxml, "url" );
		url = gtk_entry_get_text( GTK_ENTRY( widget ) );

		found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &it, TAG_FILE_URI_COL, url );

		if( ! found ) {
			/* release lock, as screem_tag_tree_add_uri()
			 * is called from a thread on startup it
			 * needs to claim the lock to do anything */
			gdk_threads_leave();
			screem_tag_tree_add_uri( url, 
					SCREEM_TAG_TREE_FORMAT );	
			gdk_threads_enter();
		} else {
			widget = gtk_message_dialog_new( NULL,
							 GTK_DIALOG_MODAL,
							 GTK_MESSAGE_WARNING,
							 GTK_BUTTONS_OK,
							 _( "A Tag Tree located at %s is already present.  Refusing to add it again." ),
							 url);
			gtk_dialog_run( GTK_DIALOG( widget ) );
			gtk_widget_destroy( widget );
		}
	}

	widget = glade_xml_get_widget( dxml, "open_file_location" );
	gtk_widget_destroy( widget );
	
	g_object_unref( dxml );
}

void screem_preferences_page_setup( void )
{
	GConfClient *client;
	GladeXML *xml;
	GtkWidget *widget;
	
	static const gchar *widgets[] = {
		"highlight", "/apps/screem/editor/print/highlight",
		"headers", "/apps/screem/editor/print/headers",
		"numbers", "/apps/screem/editor/print/lines",
		"wrap", "/apps/screem/editor/print/wrap_lines",
		"split", "/apps/screem/editor/print/split",
		NULL
	};
	guint i;
	const gchar *path;
	gboolean val;
	GSList *handles;
	guint handle;
	gdouble sval;
	
	xml = glade_xml_new( GLADE_PATH"/screem.glade",
			"page_setup", NULL );
	glade_xml_signal_autoconnect( xml );	

	client = gconf_client_get_default();
	
	for( handles = NULL, i = 0; widgets[ i ]; ++ i ) {
		
		widget = glade_xml_get_widget( xml, widgets[ i ] );
		path = widgets[ ++ i ];
		val = gconf_client_get_bool( client, path, NULL );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), val );
		g_object_set_data( G_OBJECT( widget ), "key", path );
		g_signal_connect( G_OBJECT( widget ), "toggled",
				G_CALLBACK( screem_preferences_toggle_changed ),
				NULL );
		handle = gconf_client_notify_add( client,
				path,
				screem_preferences_toggle_notify,
				widget, NULL, NULL );
		handles = g_slist_prepend( handles,
				GUINT_TO_POINTER( handle ) );
	}
	widget = glade_xml_get_widget( xml, "lines" ); 
	path = "/apps/screem/editor/print/nlines";
	sval = gconf_client_get_float( client, path, NULL );
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), sval );
	g_object_set_data( G_OBJECT( widget ), "key", path );
	g_signal_connect( G_OBJECT( widget ), "changed",
			G_CALLBACK( screem_preferences_spin_changed ),
			NULL );
	handle = gconf_client_notify_add( client,
			path,
			screem_preferences_spin_notify,
			widget, NULL, NULL );
	handles = g_slist_prepend( handles,
			GUINT_TO_POINTER( handle ) );
	
	widget = glade_xml_get_widget( xml, "page_setup" );
	g_object_set_data( G_OBJECT( widget ), "handles", handles );

	gtk_widget_show( widget );

	g_object_unref( G_OBJECT( client ) );
}

void screem_preferences_dialog_destroy( GtkWidget *widget )
{
	ScreemSession *session;
	GConfClient *client;
	GSList *handles;
	GSList *tmp;
	GladeXML *xml;

	session = g_object_get_data( G_OBJECT( widget ), "session" );
	screem_session_store_dialog( session, widget );
	
	handles = g_object_get_data( G_OBJECT( widget ), "handles" );
	if( handles ) {
		client = gconf_client_get_default();
		for( tmp = handles; tmp; tmp = tmp->next ) {
			gconf_client_notify_remove( client,
					GPOINTER_TO_UINT( tmp->data ) );
		}
		g_slist_free( handles );
		g_object_unref( client );
	}

	xml = glade_get_widget_tree( widget );
	if( xml ) {
		g_object_unref( G_OBJECT( xml ) );
	}
	
	gtk_widget_destroy( widget );
}

void screem_preferences_dialog_response_check( GtkWidget *widget, gint response )
{
	if( response < 0 ) {
		screem_preferences_dialog_destroy( widget );
	}
}

static void screem_preferences_add_pref( GtkWidget *dialog,
				const gchar *key,
				ScreemPrefNotifyType type,
				gint value,
				GtkWidget *widget )
{
	GConfClient *client;
	GConfClientNotifyFunc func;
	guint handle;
	GSList *handles;

	gboolean tval;
	gint ival;
	gdouble spval;
	gchar *sval;
	GdkColor colour;
	GtkWidget *gtkentry;
	
	g_return_if_fail( key != NULL );
	g_return_if_fail( widget != NULL );
	
	client = gconf_client_get_default();
	
	handles = g_object_get_data( G_OBJECT( dialog ), "handles" );
	
	g_object_set_data( G_OBJECT( widget ), "key", (gchar*)key );
	g_object_set_data( G_OBJECT( widget ), "ptype",
			GINT_TO_POINTER( type ) );
	
	switch( type ) {
		case SCREEM_PREF_TOGGLE:
			func = screem_preferences_toggle_notify;
			tval = gconf_client_get_bool( client, key, NULL );
			g_signal_connect( G_OBJECT( widget ), "toggled",
					  G_CALLBACK( screem_preferences_toggle_changed ),
					  NULL );
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
							tval );
			break;
		case SCREEM_PREF_RADIO:
			func = screem_preferences_radio_notify;
			g_object_set_data( G_OBJECT( widget ), "val", 
				   	   GINT_TO_POINTER( value ) );
			ival = gconf_client_get_int( client, key, NULL );
			if( value == ival ) {
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
								TRUE );
			}
			g_signal_connect( G_OBJECT( widget ), "toggled",
					  G_CALLBACK( screem_preferences_radio_changed ),
					  NULL );
			break;
		case SCREEM_PREF_SPIN:
			func = screem_preferences_spin_notify;
			spval = gconf_client_get_float( client, key, NULL );
			g_signal_connect( G_OBJECT( widget ), "value_changed",
					  G_CALLBACK( screem_preferences_spin_changed ),
					  NULL );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
						   spval );
			break;
		case SCREEM_PREF_SPIN_INT:
			func = screem_preferences_spin_notify;
			ival = gconf_client_get_int( client, key, NULL );
			g_signal_connect( G_OBJECT( widget ), "value_changed",
					  G_CALLBACK( screem_preferences_spin_changed ),
					  NULL );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
						   (gfloat)ival );
			break;
		case SCREEM_PREF_COLOUR:
			func = screem_preferences_colour_notify;
			sval = gconf_client_get_string( client, key, NULL );
			g_signal_connect( G_OBJECT( widget ), "color_set",
					  G_CALLBACK( screem_preferences_colour_set ),
					  NULL );
			if( gdk_color_parse( sval, &colour ) ) {
				gtk_color_button_set_color( GTK_COLOR_BUTTON( widget ),
						&colour );
			}
			break;
		case SCREEM_PREF_STRING:
			func = screem_preferences_string_notify;
			sval = gconf_client_get_string( client, key, NULL );
			gtkentry = NULL;
			if( GTK_IS_FILE_CHOOSER_BUTTON( widget ) ) {
				/* FIXME: connect to a signal
				 * that works!!!!
				 * http://bugzilla.gnome.org/show_bug.cgi?id=158428
				 * 
				 * */
				g_signal_connect( G_OBJECT( widget ),
						"focus_out_event",
						G_CALLBACK( screem_preferences_string_set ),
						NULL );
				if( sval ) {
					gtk_file_chooser_set_uri( GTK_FILE_CHOOSER( widget ), sval );
				}
			} else if( GNOME_IS_ICON_ENTRY( widget ) ) {
				g_signal_connect( G_OBJECT( widget ), "changed",
						  G_CALLBACK( screem_preferences_icon_changed ),
						  NULL );
				gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( widget ),
							       sval);
			} else if( GTK_IS_ENTRY( widget ) ) {
				g_signal_connect( G_OBJECT( widget ), "changed",
						  G_CALLBACK( screem_preferences_string_set ),
						  NULL );
				gtkentry = widget;
			} else if( GTK_IS_COMBO_BOX_ENTRY( widget ) ) {
				gtkentry = GTK_BIN( widget )->child;
				g_object_set_data( G_OBJECT( gtkentry ), "key", (gchar*)key );
				g_signal_connect( G_OBJECT( gtkentry ), "changed",
						  G_CALLBACK( screem_preferences_string_set ),
						  widget );
				/* FIXME: look for possible history */

			} else if( GTK_IS_FONT_BUTTON( widget ) ) {
				g_signal_connect( G_OBJECT( widget ), "font_set",
						  G_CALLBACK( screem_preferences_font_set ),
						  NULL );
				gtk_font_button_set_font_name( GTK_FONT_BUTTON( widget ), sval );
			}
			if( gtkentry && sval ) {
				gtk_entry_set_text( GTK_ENTRY( gtkentry ), sval );
			}
			g_free( sval );
			break;
		default:
			g_warning( "Invalid preference notify type\n" );
			return;
			break;
	}

	handle = gconf_client_notify_add( client,
					  key, func,
					  widget, NULL, NULL );
	handles = g_slist_prepend( handles,
			GUINT_TO_POINTER( handle ) );
	g_object_set_data( G_OBJECT( dialog ), "handles", handles );
	g_object_unref( client );
}

