/*  Screem:  screem-top-level-model.c
 *
 *  A GtkTreeModel which represents a list of top level nodes
 *  in another GtkTreeModel
 *
 *  Copyright (C) 2003  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 */

#include "screem-top-level-model.h"

/* iter format:
 * iter->stamp = stamp
 * iter->user_data = path
 */

struct ScreemTopLevelModelPrivate {
	GtkTreeModel *child;
	guint stamp;
};

static void screem_top_level_model_init( ScreemTopLevelModel *model );
static void screem_top_level_model_class_init( ScreemTopLevelModelClass *klass );
static void screem_top_level_model_model_init( GtkTreeModelIface *iface );
static void screem_top_level_model_finalize( GObject *object );

/* TreeModel signals */
static void row_changed( GtkTreeModel *model,
			GtkTreePath *path, GtkTreeIter *iter,
			gpointer data );
static void row_inserted( GtkTreeModel *model,
			GtkTreePath *path, GtkTreeIter *iter,
			gpointer data );
static void row_deleted( GtkTreeModel *model, GtkTreePath *path,
			gpointer data );
static void rows_reordered( GtkTreeModel *model,
			GtkTreePath *path, GtkTreeIter *iter,
			gint *new_order, gpointer data );

/* TreeModelIface impl */
static guint        screem_top_level_model_get_flags       (GtkTreeModel *model);
static gint         screem_top_level_model_get_n_columns   (GtkTreeModel *model);
static GType        screem_top_level_model_get_column_type (GtkTreeModel *model,
                                                          gint          index);
static gboolean     screem_top_level_model_get_iter        (GtkTreeModel *model,
                                                          GtkTreeIter  *iter,
					                  GtkTreePath  *path);
static GtkTreePath *screem_top_level_model_get_path        (GtkTreeModel *model,
                                                          GtkTreeIter  *iter);
static void         screem_top_level_model_get_value       (GtkTreeModel *model,
                                                          GtkTreeIter  *iter,
					                  gint          column,
					                  GValue       *value);
static gboolean     screem_top_level_model_iter_next       (GtkTreeModel *model,
                                                          GtkTreeIter  *iter);
static gboolean     screem_top_level_model_iter_children   (GtkTreeModel *model,
                                                          GtkTreeIter  *iter,
						          GtkTreeIter  *parent);
static gboolean     screem_top_level_model_iter_has_child  (GtkTreeModel *model,
                                                          GtkTreeIter  *iter);
static gint         screem_top_level_model_iter_n_children (GtkTreeModel *model,
                                                          GtkTreeIter  *iter);
static gboolean     screem_top_level_model_iter_nth_child  (GtkTreeModel *model,
                                                          GtkTreeIter  *iter,
						          GtkTreeIter  *parent,
						          gint          n);
static gboolean     screem_top_level_model_iter_parent     (GtkTreeModel *model,
                                                          GtkTreeIter  *iter,
						          GtkTreeIter  *child);
static void         screem_top_level_model_ref_node        (GtkTreeModel *model,
                                                          GtkTreeIter  *iter);
static void         screem_top_level_model_unref_node      (GtkTreeModel *model,
                                                          GtkTreeIter  *iter);


static GObjectClass *parent_class = NULL;

GType screem_top_level_model_get_type( void )
{
	static GType type = 0;

	if( ! type ) {

		static const GTypeInfo info = {
			sizeof( ScreemTopLevelModelClass ),
			NULL,
			NULL,
			(GClassInitFunc)screem_top_level_model_class_init,
			NULL,
			NULL,
			sizeof( ScreemTopLevelModel ),
			0,
			(GInstanceInitFunc)screem_top_level_model_init

		};

		static const GInterfaceInfo iinfo = {
			(GInterfaceInitFunc)screem_top_level_model_model_init,
			NULL,
			NULL
		};

		type = g_type_register_static( G_TYPE_OBJECT,
					"ScreemTopLevelModel",
					&info, 0 );
		g_type_add_interface_static( type,
					GTK_TYPE_TREE_MODEL,
					&iinfo );
		
	}

	return type;
}

static void screem_top_level_model_init( ScreemTopLevelModel *model )
{
	model->priv = g_new0( ScreemTopLevelModelPrivate, 1 );
	model->priv->stamp = g_random_int();
}

static void screem_top_level_model_class_init( ScreemTopLevelModelClass *klass )
{
	GObjectClass *obj_class;

	obj_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	obj_class->finalize = screem_top_level_model_finalize;
}

static void screem_top_level_model_model_init( GtkTreeModelIface *iface )
{
	iface->get_flags = screem_top_level_model_get_flags;
	iface->get_n_columns = screem_top_level_model_get_n_columns;
	iface->get_column_type = screem_top_level_model_get_column_type;
	iface->get_iter = screem_top_level_model_get_iter;
	iface->get_path = screem_top_level_model_get_path;
	iface->get_value = screem_top_level_model_get_value;
	iface->iter_next = screem_top_level_model_iter_next;
	iface->iter_children = screem_top_level_model_iter_children;
	iface->iter_has_child = screem_top_level_model_iter_has_child;
	iface->iter_n_children = screem_top_level_model_iter_n_children;
	iface->iter_nth_child = screem_top_level_model_iter_nth_child;
	iface->iter_parent = screem_top_level_model_iter_parent;
	iface->ref_node = screem_top_level_model_ref_node;
	iface->unref_node = screem_top_level_model_unref_node;
}


static void screem_top_level_model_finalize( GObject *obj )
{
	ScreemTopLevelModel *tmodel;

	tmodel = SCREEM_TOP_LEVEL_MODEL( obj );
	
	if( tmodel->priv->child ) {
		g_signal_handlers_disconnect_by_func( tmodel->priv->child,
				row_inserted, tmodel );
		g_signal_handlers_disconnect_by_func( tmodel->priv->child,
				row_deleted, tmodel );
		g_signal_handlers_disconnect_by_func( tmodel->priv->child,
				row_changed, tmodel );
		g_signal_handlers_disconnect_by_func( tmodel->priv->child,
				rows_reordered, tmodel );
	}
	
	parent_class->finalize( obj );
}

/* TreeModel signals */
static void row_changed( GtkTreeModel *model,
			GtkTreePath *path, GtkTreeIter *iter,
			gpointer data )
{
	GtkTreeIter it;

	if( gtk_tree_path_get_depth( path ) == 1 &&
	    gtk_tree_model_get_iter( GTK_TREE_MODEL( data ),
					&it, path ) ) {
		gtk_tree_model_row_changed( GTK_TREE_MODEL( data ),
					path, &it );
	}
}

static void row_inserted( GtkTreeModel *model,
			GtkTreePath *path, GtkTreeIter *iter,
			gpointer data )
{
	GtkTreeIter it;

	if( gtk_tree_path_get_depth( path ) == 1 &&
	    gtk_tree_model_get_iter( GTK_TREE_MODEL( data ),
					&it, path ) ) {
		gtk_tree_model_row_inserted( GTK_TREE_MODEL( data ),
					path, &it );
	}
}

static void row_deleted( GtkTreeModel *model, GtkTreePath *path,
			gpointer data )
{
	if( gtk_tree_path_get_depth( path ) == 1 ) {
		gtk_tree_model_row_deleted( GTK_TREE_MODEL( data ),
					path ) ;
	}
}

static void rows_reordered( GtkTreeModel *model,
			GtkTreePath *path, GtkTreeIter *iter,
			gint *new_order, gpointer data )
{
	GtkTreeIter it;

	if( gtk_tree_path_get_depth( path ) == 1 &&
	    gtk_tree_model_get_iter( GTK_TREE_MODEL( data ),
					&it, path ) ) {
		gtk_tree_model_rows_reordered( GTK_TREE_MODEL( data ),
					path, &it, new_order );
	}
}


/* TreeModelIface implementation */
static guint screem_top_level_model_get_flags( GtkTreeModel *model )
{
	ScreemTopLevelModel *tmodel;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ), 0 );
	
	tmodel = (ScreemTopLevelModel*)model;

	g_return_val_if_fail( tmodel->priv->child != NULL, 0 );

	return gtk_tree_model_get_flags( tmodel->priv->child );
}

static gint screem_top_level_model_get_n_columns( GtkTreeModel *model )
{
	ScreemTopLevelModel *tmodel;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ), 0 );
	
	tmodel = (ScreemTopLevelModel*)model;

	return gtk_tree_model_get_n_columns( tmodel->priv->child );
}

static GType
screem_top_level_model_get_column_type (GtkTreeModel *model,
                                      gint          index)
{
	ScreemTopLevelModel *tmodel;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ), 0 );
	
	tmodel = (ScreemTopLevelModel*)model;
	
	return gtk_tree_model_get_column_type( tmodel->priv->child,
						index );
}

static gboolean
screem_top_level_model_get_iter (GtkTreeModel *model,
                               GtkTreeIter  *iter,
			       GtkTreePath  *path)
{
	ScreemTopLevelModel *tmodel;
	GtkTreeIter c_iter;
	gint *indicies;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ), 
				FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );
	
	tmodel = (ScreemTopLevelModel*)model;

	/* we don't want a warning on the screen if this fails */
	if( ! tmodel->priv->child ) {
		iter->stamp = 0;
		return FALSE;
	}

	/* only allow top level nodes, any other path is invalid */
	if( gtk_tree_path_get_depth( path ) > 1 ) {
		iter->stamp = 0;
		return FALSE;
	}
	
	/* check if the iter actually exists in the child model */
	if( ! gtk_tree_model_get_iter( tmodel->priv->child, 
					&c_iter, path ) ) {
		iter->stamp = 0;
		return FALSE;
	}

	indicies = gtk_tree_path_get_indices( path );
	
	iter->stamp = tmodel->priv->stamp;
	iter->user_data = GINT_TO_POINTER( indicies[ 0 ] );
	iter->user_data2 = NULL;
	iter->user_data3 = NULL;

	return TRUE;
}

static GtkTreePath *
screem_top_level_model_get_path (GtkTreeModel *model,
                               GtkTreeIter  *iter)
{
	ScreemTopLevelModel *tmodel;
	GtkTreePath *path;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ), NULL );
	g_return_val_if_fail( iter != NULL, NULL );
	
	tmodel = (ScreemTopLevelModel*)model;

	g_return_val_if_fail( tmodel->priv->stamp == iter->stamp, NULL );
	g_return_val_if_fail( tmodel->priv->child != NULL, NULL );

	path = gtk_tree_path_new_from_indices( GPOINTER_TO_INT( iter->user_data ), -1 );

	return path;
}

static void
screem_top_level_model_get_value (GtkTreeModel *model,
                                GtkTreeIter  *iter,
				gint          column,
				GValue       *value)
{
	ScreemTopLevelModel *tmodel;
	GtkTreeIter child_iter;
	GtkTreePath *path;
	
	g_return_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ) );
	g_return_if_fail( iter != NULL );
	
	tmodel = (ScreemTopLevelModel*)model;

	g_return_if_fail( tmodel->priv->stamp == iter->stamp );
	g_return_if_fail( tmodel->priv->child != NULL );

	path = screem_top_level_model_get_path( model, iter );

	gtk_tree_model_get_iter( tmodel->priv->child, &child_iter, path );

	gtk_tree_path_free( path );
	
	gtk_tree_model_get_value( tmodel->priv->child, &child_iter, 
				  column, value );
}

static gboolean
screem_top_level_model_iter_next (GtkTreeModel *model,
                                GtkTreeIter  *iter)
{
	ScreemTopLevelModel *tmodel;
	GtkTreePath *path;
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ), 
				FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );
	
	tmodel = (ScreemTopLevelModel*)model;

	g_return_val_if_fail( tmodel->priv->stamp == iter->stamp, FALSE );
	g_return_val_if_fail( tmodel->priv->child != NULL, FALSE );

	path = screem_top_level_model_get_path( model, iter );
	gtk_tree_path_next( path );

	ret = screem_top_level_model_get_iter( model, iter, path );
	gtk_tree_path_free( path );

	return ret;
}

static gboolean
screem_top_level_model_iter_children (GtkTreeModel *model,
                                    GtkTreeIter  *iter,
				    GtkTreeIter  *parent)
{
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ),
				FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );

	ret = FALSE;
	
	if( ! parent ) {
		ret = gtk_tree_model_get_iter_first( model, iter );
	}

	/* as this model makes a list out of top level nodes only
	 * we do not allow children to be accessed */
	
	return ret;
}

static gboolean
screem_top_level_model_iter_has_child (GtkTreeModel *model,
                                     GtkTreeIter  *iter)
{
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ),
				FALSE );

	/* as this model makes a list out of top level nodes only
	 * we do not allow children to be accessed */
	
	return ( iter == NULL );
}
	
static gint
screem_top_level_model_iter_n_children (GtkTreeModel *model,
                                      GtkTreeIter  *iter)
{
	ScreemTopLevelModel *tmodel;
	gint ret;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ), 0 );

	tmodel = SCREEM_TOP_LEVEL_MODEL( model );
	
	ret = 0;
	
	if( ! iter ) {
		ret = gtk_tree_model_iter_n_children( tmodel->priv->child,
						NULL );
	} 
	
	/* as this model makes a list out of top level nodes only
	 * we do not allow children to be accessed */
	
	return ret;
}

static gboolean
screem_top_level_model_iter_nth_child (GtkTreeModel *model,
                                     GtkTreeIter  *iter,
				     GtkTreeIter  *parent,
				     gint          n)
{
	gboolean ret;
	GtkTreePath *path;
	
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ),
				FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );

	ret = FALSE;
	if( ! parent ) {
	      path = gtk_tree_path_new_from_indices( n, -1 );
	      ret = screem_top_level_model_get_iter( model, iter, path );
	      gtk_tree_path_free (path);
	}
	
	/* as this model makes a list out of top level nodes only
	 * we do not allow children to be accessed */
	
	return ret;
}

static gboolean
screem_top_level_model_iter_parent (GtkTreeModel *model,
                                  GtkTreeIter  *iter,
				  GtkTreeIter  *child)
{
	g_return_val_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( model ),
				FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );

	/* as this model makes a list out of top level nodes only
	 * we do not allow children to be accessed */
	
	return FALSE;
}

static void
screem_top_level_model_ref_node (GtkTreeModel *model,
                               GtkTreeIter  *iter)
{
}

static void
screem_top_level_model_unref_node (GtkTreeModel *model,
                                 GtkTreeIter  *iter)
{
}

/* Public */
GtkTreeModel *screem_top_level_model_new( void )
{
 	return g_object_new( SCREEM_TYPE_TOP_LEVEL_MODEL, NULL );
}

void screem_top_level_model_set_model( ScreemTopLevelModel *tmodel,
					GtkTreeModel *model )
{
	g_return_if_fail( SCREEM_IS_TOP_LEVEL_MODEL( tmodel ) );

	if( tmodel->priv->child ) {
		g_signal_handlers_disconnect_by_func( tmodel->priv->child,
				row_inserted, tmodel );
		g_signal_handlers_disconnect_by_func( tmodel->priv->child,
				row_deleted, tmodel );
		g_signal_handlers_disconnect_by_func( tmodel->priv->child,
				row_changed, tmodel );
		g_signal_handlers_disconnect_by_func( tmodel->priv->child,
				rows_reordered, tmodel );
	}
	
	if( model ) {
		tmodel->priv->child = model;
		g_signal_connect( model, "row_inserted",
				G_CALLBACK( row_inserted ), tmodel );
		g_signal_connect( model, "row_changed", 
				G_CALLBACK( row_changed ), tmodel );
		g_signal_connect( model, "row_deleted", 
				G_CALLBACK( row_deleted ), tmodel );
		g_signal_connect( model, "rows_reordered", 
				G_CALLBACK( rows_reordered ), tmodel );
 	
		gtk_tree_model_foreach( model, 
			(GtkTreeModelForeachFunc)row_inserted, tmodel );
	}
}

