/*  Screem:  screem-site-ui.c,
 *  this file contains GUI code that deals with the site
 * 
 *  Copyright (C) 1999 - 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <glib/gfileutils.h>

#include <gtk/gtktogglebutton.h>

#include <glib/gi18n.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>

#include <string.h>

#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>

#include <gtk/gtkinvisible.h>

#include <glade/glade.h>

#include <libxml/tree.h>

#include "fileops.h"
#include "pageUI.h"
#include "support.h"

#include "screem-application.h"
#include "screem-site.h"
#include "screem-site-ui.h"
#include "screem-page.h"

#include "screem-site-model.h"

void screem_site_settings_dialog_cb( GtkWidget *widget, gint button );
void screem_site_settings_change_url( GtkWidget *widget, gpointer data );

static void screem_site_gather_stats( ScreemSite *site, GladeXML *xml );
static void screem_site_setup_lists( ScreemSite *site, GladeXML *xml );

void screem_site_close_site( ScreemSite *site )
{
       	g_return_if_fail( site != NULL );

	/* if this is the fake site then it can't be closed */
	if( ! screem_site_get_fake_flag( site ) ) {
		screem_site_save( site );
	}
}

gboolean screem_site_open_with_filename( ScreemWindow *window,
					 ScreemApplication *application,
					 const gchar *filename )
{
	ScreemSite *site;
	gchar *path = NULL;
	GnomeVFSURI *uri;
	GnomeVFSFileInfo *info;
	gchar *msg;
	gchar *tmpname;

	g_return_val_if_fail( filename != NULL, FALSE );

	screem_application_set_cursor( application, GDK_WATCH );

	uri = gnome_vfs_uri_new( filename );

	tmpname = gnome_vfs_uri_extract_short_name( uri );

	path = gnome_vfs_make_uri_from_input( filename );

	gnome_vfs_uri_unref( uri );

	/* ensure we have a dir */
	info = gnome_vfs_file_info_new();

	if( gnome_vfs_get_file_info( path, info, 
				 GNOME_VFS_FILE_INFO_FOLLOW_LINKS ) != 
			GNOME_VFS_OK ) {
		g_free( path );
		screem_application_set_cursor( application, 
				GDK_LAST_CURSOR );
		return FALSE;
	}

	if( info->type != GNOME_VFS_FILE_TYPE_DIRECTORY )  {
		gchar *temp = g_path_get_dirname( path );
		g_free( path );
		path = temp;
	}

	gnome_vfs_file_info_unref( info );

	/* if we already have the site loaded, then don't do so again */
	site = screem_application_get_site( application, path );
	if( site ) {
		g_free( path );
		if( window ) {
			screem_window_set_current( window, site );
		}
		screem_application_set_cursor( application, 
				GDK_LAST_CURSOR );
		g_free( tmpname );
		return TRUE;
	}

	site = screem_site_new( G_OBJECT( application ) );
	screem_site_set_pathname( site, path );
	screem_site_set_name( site, tmpname );
	g_free( tmpname );

	if( window ) {
		msg = g_strdup_printf( _( "Loading site: %s" ), path );
		screem_window_show_message( window, msg, FALSE );
		g_free( msg );
		gdk_threads_leave(); 
		while( g_main_context_iteration(NULL,FALSE) ) { }
		gdk_threads_enter();
	}
	
	if( ! screem_site_load( site ) ) {
		/* we failed to load the site */
		g_object_unref( G_OBJECT( site ) );
		if( window ) {
			msg = g_strdup_printf( _( "Failed to load site: %s" ), path );
			screem_window_show_message( window, msg, FALSE );
			g_free( msg );
			g_free( path );
		}
		screem_application_set_cursor( application, 
				GDK_LAST_CURSOR );
		return FALSE;
	}
		
	g_free( path );
		
	/* add this site to the list of loaded sites */
	screem_application_add_site( application, site );
	screem_application_set_cursor( application, GDK_LAST_CURSOR );

	if( window ) {
		screem_window_set_current( window, site );
		
		msg = g_strdup_printf( _( "Loaded site: %s" ), screem_site_get_name( site ) );
		screem_window_show_message( window, msg, FALSE );
		g_free( msg );
		gdk_threads_leave(); 
		while( g_main_context_iteration(NULL,FALSE) ) { }
		gdk_threads_enter();
		
		/* if we are importing a site open the site settings
		 * dialog */
		if( screem_site_get_is_import( site ) ) {
			screem_site_settings_dialog( site, window );
			screem_site_set_is_import( site, FALSE );
		}
	}

	/* site loaded, check for cvs update,
	   only do this if the site auto update flag is false,
	   and the user hasn't disabled displaying of the prompting dialog */

	if( screem_site_get_cvs_root( site ) && 
	    screem_site_get_auto_update( site ) ) {
		GladeXML *xml;
		GtkWidget *widget;
		gint button;
		GtkWidget *toggle;
		gboolean ask;
	
		xml = glade_xml_new( GLADE_PATH"/screem.glade",
				     "auto_cvs_update", NULL );
		widget = glade_xml_get_widget( xml, "auto_cvs_update" );
		toggle = glade_xml_get_widget( xml, "dont_ask" );

		gtk_window_set_wmclass( GTK_WINDOW( widget ), 
					"Screem", "auto_cvs_update" );
		gtk_window_set_transient_for( GTK_WINDOW( widget ),
					      GTK_WINDOW( window ) );
		if( screem_site_get_auto_update_ask( site ) ) {
			button = gtk_dialog_run( GTK_DIALOG( widget ) );
			ask = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
			screem_site_set_auto_update_ask( site, ask );
		} else {
			ask = TRUE;
			button = 0;
		}

		gtk_widget_destroy( widget );

		/* FIXME can't update from session load as window == NULL */
		if( button == 0 && window ) {
			/* do a cvs update */
			GtkAction *action;
			
			action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
								"UpdateSite" );
			gtk_action_activate( action );
		} else if( ! ask ) {
			/* don't ask was selected and the dialog was
			   just closed */
			screem_site_set_auto_update( site, FALSE );
		}
	}
	
	return TRUE;
}

/**
 * screem_site_settings_dialog:
 * opens the site settings dialog
 *
 * return values: none
 */
void screem_site_settings_dialog( ScreemSite *site, 
		ScreemWindow *window )
{
	ScreemApplication *app;
	ScreemSession *session;
	GtkWidget *dialog1;
     	GtkWidget *entry;
	GtkWidget *toggle;
	ScreemSitePerms permissions;
	ScreemSiteSymlinks symlinks;
	const gchar *text;
	GladeXML *xml;

	gint button;
		
	g_return_if_fail( site != NULL );
	g_return_if_fail( ! screem_site_get_fake_flag( site ) );

	app = SCREEM_APPLICATION( window->application );
	session = screem_application_get_session( app );
	
	xml = glade_xml_new( GLADE_PATH"/screem.glade",
			"site_settings", NULL );

	glade_xml_signal_autoconnect( xml );

	screem_site_setup_lists( site, xml );
		
	dialog1 = glade_xml_get_widget( xml, "site_settings" );
	
	/* fill in the details */
	/* name */
	entry = glade_xml_get_widget( xml, "site_name_entry" );
	screem_gtk_add_history( entry );
	entry = GTK_BIN( entry )->child;
	
	text = screem_site_get_name( site );
	if( ! text ) {
		text = _( "Imported Site ");
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* path */
	entry = glade_xml_get_widget( xml, "site_path_entry" );
	text = screem_site_get_pathname( site );
	if( ! text ) {
		text = "";
	}
	gtk_label_set_text( GTK_LABEL( entry ), text );
	
	/* cvs root */
	entry = glade_xml_get_widget( xml, "cvs_file_entry" );
	screem_gtk_add_history( entry );
	text = screem_site_get_cvs_root( site );
	toggle = glade_xml_get_widget( xml, "cvs_check" );
	if( text ) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
					     TRUE );
		entry = GTK_BIN( entry )->child;
		gtk_entry_set_text( GTK_ENTRY( entry ), text );
		gtk_widget_set_sensitive( entry, TRUE );
	}
	
	/* template path */
	entry = glade_xml_get_widget( xml, "template_file_entry" );
	if( ( text = screem_site_get_pathname( site ) ) ) {
		gtk_file_chooser_set_current_folder_uri( GTK_FILE_CHOOSER( entry ), text );
	}
	text = screem_site_get_template_path( site );
	if( text ) {
		toggle = glade_xml_get_widget( xml, "template_check" );
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
					     TRUE );
		gtk_file_chooser_set_uri( GTK_FILE_CHOOSER( entry ),
				text );
		/* required as file chooser is custom */
		gtk_widget_set_sensitive( entry, TRUE );
	} else {
		/* required as file chooser is custom */
		gtk_widget_set_sensitive( entry, FALSE );
	}
	
	/* auto cvs update */
	toggle = glade_xml_get_widget( xml, "auto_cvs_update" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
				      screem_site_get_auto_update( site ) );

	/* ctags */
	toggle = glade_xml_get_widget( xml, "ctags" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
				      screem_site_get_use_ctags( site ) );
	
	/* internal preview settings */
	toggle = glade_xml_get_widget( xml, "preview_static" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
				      screem_site_get_preview_static( site ) );

	toggle = glade_xml_get_widget( xml, "preview_dynamic" );
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
				     screem_site_get_preview_dynamic( site ) );

	/* remote url */
	entry = glade_xml_get_widget( xml, "remote_url_entry" );
	screem_gtk_add_history( entry );
	entry = GTK_BIN( entry )->child;
	
	text = screem_site_get_remote_url( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* http url */
	entry = glade_xml_get_widget( xml, "http_url_entry" );
	screem_gtk_add_history( entry );
	entry = GTK_BIN( entry )->child;
	text = screem_site_get_http_url( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* remote method */
	entry = glade_xml_get_widget( xml, "remote_method" );
	gtk_combo_box_set_active( GTK_COMBO_BOX( entry ), 
			screem_site_get_remote_method( site ) );
	
	/* no file delete */
	if( screem_site_get_no_delete( site ) ) {
		toggle = glade_xml_get_widget( xml, "nodelete" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}
	
	/* check for moved files */
	if( screem_site_get_check_moved( site ) ) {
		toggle = glade_xml_get_widget( xml, "checkmoved" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}
	
	/* no file overwrite */
	if( screem_site_get_no_overwrite( site ) ) {
		toggle = glade_xml_get_widget( xml, "nooverwrite" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}
	
	/* permissions handling */
	permissions = screem_site_get_permissions( site );
	toggle = glade_xml_get_widget( xml, permission_strings[permissions] );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	
	/* symbolic link handling */
	symlinks = screem_site_get_symlinks( site );
	toggle = glade_xml_get_widget( xml, symlink_strings[ symlinks ] );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	
	/* remote path */
	entry = glade_xml_get_widget( xml, "remote_path_entry" );
	screem_gtk_add_history( entry );
	entry = GTK_BIN( entry )->child;
	text = screem_site_get_remote_path( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* remote user */
	entry = glade_xml_get_widget( xml, "remote_user_entry" );
	screem_gtk_add_history( entry );
	entry = GTK_BIN( entry )->child;
	text = screem_site_get_remote_user( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* remote password */
	entry = glade_xml_get_widget( xml, "remote_password_entry" );
	text = screem_site_get_remote_pass( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* attach the site to the dialog */
	g_object_set_data( G_OBJECT( dialog1 ), "site", site );
	
	gtk_window_set_transient_for( GTK_WINDOW( dialog1 ),
				      GTK_WINDOW( window ) );

	screem_session_restore_dialog( session, dialog1 );
	
	button = gtk_dialog_run( GTK_DIALOG( dialog1 ) );
	
	if( button == GTK_RESPONSE_CLOSE ) {
		screem_site_settings_dialog_cb( dialog1, button );
	}
	
	screem_session_store_dialog( session, dialog1 );
	
	gtk_widget_destroy( dialog1 );
	g_object_unref( xml );
}

void screem_site_settings_change_url( GtkWidget *widget, gpointer data )
{

}

void screem_site_settings_dialog_cb( GtkWidget *widget, gint button ) 
{
	GtkWidget *entry;
	GtkWidget *toggle;
	ScreemSite *site;
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site") );
	
	/* name */
	entry = glade_xml_get_widget( xml, "site_name_entry" );
	entry = GTK_BIN( entry )->child;
	screem_site_set_name( site, 
			      gtk_entry_get_text( GTK_ENTRY(entry) ) );

	/* cvs root */
	toggle = glade_xml_get_widget( xml, "cvs_check" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) ) {
		const gchar *uri;
		
		entry = glade_xml_get_widget(xml,"cvs_file_entry");
		entry = GTK_BIN( entry )->child;
		uri = gtk_entry_get_text( GTK_ENTRY( entry ) );
		
		screem_site_set_cvs_root( site, uri );
	} else {
		screem_site_set_cvs_root( site, NULL );
	}
	
	/* template */
	toggle = glade_xml_get_widget( xml, "template_check" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) ) {
		gchar *path;

		entry = glade_xml_get_widget( xml, 
					      "template_file_entry" );
		path = gtk_file_chooser_get_uri( GTK_FILE_CHOOSER( entry ) );
		screem_site_set_template_path( site, path ); 
		g_free( path );
	} else {
		screem_site_set_template_path( site, NULL );
	}

	/* cvs update */
	toggle = glade_xml_get_widget( xml, "auto_cvs_update" );
	screem_site_set_auto_update( site,
				     gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) );

	/* ctags */
	toggle = glade_xml_get_widget( xml, "ctags" );
	screem_site_set_use_ctags( site,
		     gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) );

	/* preview */
	toggle = glade_xml_get_widget( xml, "preview_static" );
	screem_site_set_preview_static( site,
					gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) );

	toggle = glade_xml_get_widget( xml, "preview_dynamic" );
	screem_site_set_preview_dynamic( site,
					 gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) );
	
	/* remote url */
	entry = glade_xml_get_widget( xml, "remote_url_entry" );
	entry = GTK_BIN( entry )->child;
	screem_site_set_remote_url( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	/* http url */
	entry = glade_xml_get_widget( xml, "http_url_entry" );
	entry = GTK_BIN( entry )->child;
	screem_site_set_http_url( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	
	/* remote method */
	entry = glade_xml_get_widget( xml, "remote_method" );
	screem_site_set_remote_method( site,
			gtk_combo_box_get_active( GTK_COMBO_BOX( entry ) ) );
	
	/* no file delete */
	toggle = glade_xml_get_widget( xml, "nodelete" );
	screem_site_set_no_delete( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));
	
	/* check for moved files */
	toggle = glade_xml_get_widget( xml, "checkmoved" );
	screem_site_set_check_moved( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));
	
	/* no file overwrite */
	toggle = glade_xml_get_widget( xml, "nooverwrite" );
	screem_site_set_no_overwrite( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));
	
	/* permissions handling */
	toggle = glade_xml_get_widget( xml, "p_ignore" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_permissions( site, PERMS_IGNORE );
	}
	toggle = glade_xml_get_widget( xml, "p_exec" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_permissions( site, PERMS_EXEC );
	}
	toggle = glade_xml_get_widget( xml, "p_all" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_permissions( site, PERMS_ALL );
	}
	
	/* symbolic link handling */
	toggle = glade_xml_get_widget( xml, "s_ignore" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_symlinks( site, SYM_IGNORE );
	}
	toggle = glade_xml_get_widget( xml, "s_follow" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_symlinks( site, SYM_FOLLOW );
	}
	toggle = glade_xml_get_widget( xml, "s_maintain" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_symlinks( site, SYM_MAINTAIN );
	}
	
	/* remote path */
	entry = glade_xml_get_widget( xml, "remote_path_entry" );
	entry = GTK_BIN( entry )->child;
	screem_site_set_remote_path( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	
	/* remote user */
	entry = glade_xml_get_widget( xml, "remote_user_entry" );
	entry = GTK_BIN( entry )->child;
	screem_site_set_remote_user( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	
	/* remote password */
	entry = glade_xml_get_widget( xml, 
				      "remote_password_entry");
	screem_site_set_remote_pass( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

	screem_site_save( site );
}

static void screem_site_gather_stats( ScreemSite *site, GladeXML *xml )
{
	GtkWidget *last_upload;
	GtkWidget *files_list;
	GtkWidget *total_size;

	GtkCellRenderer *prenderer;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	GtkTreeModel *model;

	gchar *num;

	gint total;
	gchar *sitecopy_file;
	const gchar *name;

	struct stat st;

	last_upload = glade_xml_get_widget( xml, "last_upload" );
	files_list = glade_xml_get_widget( xml, "files_list" );
	total_size = glade_xml_get_widget( xml, "total_size" );

	name = screem_site_get_name( site );

	sitecopy_file = g_strconcat( g_get_home_dir(), G_DIR_SEPARATOR_S,
				     ".sitecopy", G_DIR_SEPARATOR_S, name,
				     NULL );
	if( ! stat( sitecopy_file, &st ) ) {
		gchar *tmp;

		tmp = ctime( &st.st_mtime );
		tmp[ strlen( tmp ) - 1 ] = '\0';
		gtk_label_set_text( GTK_LABEL( last_upload ), tmp );
	} else {
		gtk_label_set_text( GTK_LABEL( last_upload ), _("Unknown") );
	}
	g_free( sitecopy_file );

	model = screem_site_get_statistics( site );
	total = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( model ), 
						    "total" ) );
	num = gnome_vfs_format_file_size_for_display( total );
	gtk_label_set_text( GTK_LABEL( total_size ), num );
	g_free( num );

	prenderer = gtk_cell_renderer_pixbuf_new();
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();

	gtk_tree_view_column_set_title( column, _( "Type" ) );
	gtk_tree_view_column_pack_start( column, prenderer, FALSE );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_column_set_resizable( column, TRUE );
	gtk_tree_view_column_set_sort_column_id( column, 0 );
	
	gtk_tree_view_append_column( GTK_TREE_VIEW( files_list ), column );
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 0,
					     NULL );
	gtk_tree_view_column_set_attributes( column, prenderer,
					     "pixbuf", 1,
					     NULL );

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_set_title( column, _( "Number" ) );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( files_list ), column );
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 2, NULL );
	gtk_tree_view_column_set_sort_column_id( column, 2 );

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_set_title( column, _( "Total Size" ) );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( files_list ), column );
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 3, NULL );
	gtk_tree_view_column_set_sort_column_id( column, 5 );

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_set_title( column, _( "Average Size" ) );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( files_list ), column );
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 4, NULL );
	gtk_tree_view_column_set_sort_column_id( column, 6 );

	gtk_tree_view_set_headers_clickable( GTK_TREE_VIEW( files_list ),
			TRUE );
	
	gtk_tree_view_set_model( GTK_TREE_VIEW( files_list ), model );
	g_object_unref( model );
}

static void screem_site_lists_toggle_remove( GtkTreeSelection *selection,
					     gpointer data )
{
	gint count;

	count = gtk_tree_selection_count_selected_rows( selection );
	gtk_widget_set_sensitive( GTK_WIDGET( data ), ( count > 0 ) );
}

static gboolean screem_site_manage_pattern_add_idle( GtkWidget *view )
{
	GtkTreeViewColumn *col;
	GtkTreePath *path;

	gdk_threads_enter();
	
	path = g_object_get_data( G_OBJECT( view ), "path" );
	g_object_set_data( G_OBJECT( view ), "path", NULL );	
	
	col = gtk_tree_view_get_column( GTK_TREE_VIEW( view ), 0 );
	
	gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( view ),
				      path, col, FALSE, 0, 0 );
	
	gtk_tree_view_set_cursor( GTK_TREE_VIEW( view ), path, col, TRUE );

	gtk_tree_path_free( path );
	
	gdk_threads_leave();
	
	return FALSE;
}

void screem_site_manage_pattern_add( GtkWidget *view, gpointer button )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreePath *path;
	
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );

	/* use large pos to ensure appending */
	gtk_list_store_insert_with_values( GTK_LIST_STORE( model ), &it,
			G_MAXINT32,
			0, _( "Enter Pattern" ),
			1, TRUE,
			-1 );

	path = gtk_tree_model_get_path( model, &it );
	g_object_set_data( G_OBJECT( view ), "path", path );	

	g_idle_add_full( G_PRIORITY_LOW, 
			(GSourceFunc)screem_site_manage_pattern_add_idle,
			view, NULL );
}

void screem_site_manage_pattern_remove( GtkWidget *view, gpointer button )
{
	GList *rows;
	GList *tmp;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreePath *path;
	GtkTreeIter it;
	GtkTreeRowReference *ref;
	ScreemSite *site;

	site = g_object_get_data( G_OBJECT( view ), "site" );
	g_assert( SCREEM_IS_SITE( site ) );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	
	model = NULL;
	
	rows = gtk_tree_selection_get_selected_rows( selection, &model );
	g_assert( model );
	for( tmp = rows; tmp; tmp = tmp->next ) {
		path = (GtkTreePath*)tmp->data;
		tmp->data = gtk_tree_row_reference_new( model, path );
		gtk_tree_path_free( path );
	}
	for( tmp = rows; tmp; tmp = tmp->next ) {
		ref = (GtkTreeRowReference*)( tmp->data );
		path = gtk_tree_row_reference_get_path( ref );
		
		if( gtk_tree_model_get_iter( model, &it, path ) ) {
			gtk_list_store_remove( GTK_LIST_STORE( model ), &it );
			screem_site_model_update( site );
			screem_site_save( site );
		}	
		gtk_tree_path_free( path );
		gtk_tree_row_reference_free( ref );
	}
	g_list_free( rows );
}

static void screem_site_lists_edited( GtkCellRendererText *renderer,
					const gchar *path,
					const gchar *new_text,
					ScreemSite *site )
{
	GtkListStore *model;
	GtkTreePath *tpath;
	GtkTreeIter it;
	
	model = GTK_LIST_STORE( g_object_get_data( G_OBJECT( renderer ),
							"model" ) );
	
	tpath = gtk_tree_path_new_from_string( path );
	
	if( gtk_tree_model_get_iter( GTK_TREE_MODEL( model ), &it, tpath ) ) {
		gtk_list_store_set( model, &it, 0, new_text, -1 );
		screem_site_model_update( site );
		screem_site_save( site );
	}

	gtk_tree_path_free( tpath );
		
}

static void screem_site_setup_lists( ScreemSite *site, GladeXML *xml )
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkWidget *widget;
	GtkTreeSelection *selection;


	/* setup our exclude/ignore lists */
	widget = glade_xml_get_widget( xml, "exclude_list" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( site->priv->excludes ) );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "Pattern" ) );
	gtk_tree_view_column_pack_start( col, renderer,
					TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ),
				     col );
	gtk_tree_view_column_set_attributes( col, renderer,
					    "text", 0,
					    "editable", 1,
					    NULL );
	g_object_set_data( G_OBJECT( widget ), "site", site );
	g_signal_connect( G_OBJECT( renderer ), "edited",
			  G_CALLBACK( screem_site_lists_edited ),
			  site );
	g_object_set_data( G_OBJECT( renderer ), "model", 
			   site->priv->excludes );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_MULTIPLE );
	widget = glade_xml_get_widget( xml, "exclude_remove" );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_site_lists_toggle_remove ),
			  widget );
	
	widget = glade_xml_get_widget( xml, "ignore_list" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( site->priv->ignores ) );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "Pattern" ) );
	gtk_tree_view_column_pack_start( col, renderer,
					TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ),
				     col );
	gtk_tree_view_column_set_attributes( col, renderer,
					    "text", 0,
					    "editable", 1,
					    NULL );
	g_object_set_data( G_OBJECT( widget ), "site", site );
	g_signal_connect( G_OBJECT( renderer ), "edited",
			  G_CALLBACK( screem_site_lists_edited ),
			  site );
	g_object_set_data( G_OBJECT( renderer ), "model", 
			   site->priv->ignores );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_MULTIPLE );
	widget = glade_xml_get_widget( xml, "ignore_remove" );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_site_lists_toggle_remove ),
			  widget );
}

gboolean screem_site_save_confirmation( ScreemSite *site,
		ScreemWindow *window )
{
	ScreemApplication *app;
	ScreemPage *page;
	gint button;
	GList *list;
	GList *tmp;
	GSList *save;
	gboolean ret;
	
	ret = TRUE;
	button = GTK_RESPONSE_YES;

	save = NULL;
	
	for( tmp = list = screem_site_get_pages( site, TRUE, TRUE ); 
		list && button != GTK_RESPONSE_CANCEL ;
		list = list->next ) {

		page = SCREEM_PAGE( list->data );
		if( screem_page_is_loaded( page ) &&
			screem_page_get_changed( page ) ) {
			save = g_slist_prepend( save, page );
		}
	}
	g_list_free( tmp );
	
	if( save ) {
		g_object_get( G_OBJECT( window ), "app", &app, NULL );
		button = screem_page_save_pages_confirmation( save,
				window, app );
				
		if( button == GTK_RESPONSE_CANCEL ||
		    button == GTK_RESPONSE_NONE ||
		    button == GTK_RESPONSE_DELETE_EVENT ) {
			ret = FALSE;
		}
		g_object_unref( app );
		g_slist_free( save );
	}

	return ret;
}

void screem_site_stats_dialog( ScreemSite *site, ScreemWindow *window )
{
	ScreemApplication *app;
	ScreemSession *session;
	GtkWidget *dialog1;
	GladeXML *xml;

	gint button;
		
	g_return_if_fail( site != NULL );
	g_return_if_fail( ! screem_site_get_fake_flag( site ) );

	app = SCREEM_APPLICATION( window->application );
	session = screem_application_get_session( app );
	
	xml = glade_xml_new( GLADE_PATH"/screem.glade",
			"site_stats", NULL );

	glade_xml_signal_autoconnect( xml );
		
	screem_site_gather_stats( site, xml );
	
	dialog1 = glade_xml_get_widget( xml, "site_stats" );
		
	/* attach the site to the dialog */
	g_object_set_data( G_OBJECT( dialog1 ), "site", site );
	
	gtk_window_set_transient_for( GTK_WINDOW( dialog1 ),
				      GTK_WINDOW( window ) );

	screem_session_restore_dialog( session, dialog1 );
	
	button = gtk_dialog_run( GTK_DIALOG( dialog1 ) );
		
	screem_session_store_dialog( session, dialog1 );
	
	gtk_widget_destroy( dialog1 );
	g_object_unref( xml );
}
