/*  Screem:  screem-site_view.c
 *
 *  Copyright (C) 2005  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <glib/gi18n.h>

#include <glade/glade.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <sys/types.h>

#include <grp.h>
#include <pwd.h>
#include <string.h>
#include <time.h>

#include "screem-site-view.h"
#include "screem-site.h"
#include "screem-site-model.h"
#include "screem-file-browser.h"

#include "fileops.h"
#include "support.h"

#include "pageUI.h" /* needed for screem_page_open_with_filename() */

#include "screemmarshal.h"

enum {
	PROP_0,
	PROP_WINDOW,
	PROP_SITE
};

static void screem_site_view_class_init( ScreemSiteViewClass *klass );
static void screem_site_view_init( ScreemSiteView *site_view );
static void screem_site_view_finalize( GObject *site_view );
static void screem_site_view_set_prop( GObject *object, guint property_id,
				  const GValue *value, 
				  GParamSpec *pspec );
static void screem_site_view_get_prop( GObject *object, guint property_id,
				  GValue *value, GParamSpec *pspec );
static void screem_site_view_size_request( GtkWidget *widget,
                           		GtkRequisition *requisition );
static void screem_site_view_size_allocate( GtkWidget *widget,
                              		GtkAllocation *allocation );

static void screem_site_view_set_columns( GtkWidget *widget );
static void screem_site_view_window_set( ScreemSiteView *view );
static void screem_site_view_site_set( ScreemSiteView *view );

static gboolean screem_site_view_motion( GtkWidget *widget, 
		GdkDragContext *context, gint x, gint y, guint time, 
		gpointer data );
static void screem_site_view_set_dnd_data( GtkWidget *widget,
		GdkDragContext *context, GtkSelectionData *data,
		guint info, guint time, gpointer udata );
static void screem_site_view_dnd_delete( GtkWidget *widget,
		GdkDragContext *context, gpointer data );
static void screem_site_view_drop( GtkWidget *widget, 
		GdkDragContext *context, gint x, gint y, 
		GtkSelectionData *selectionData, guint info, 
		guint time, gpointer data );
static gboolean screem_site_view_drop_uris( ScreemSite *site,
		gchar **uris, const gchar *target_uri,
		gboolean moving );
static void screem_site_view_fm_expanded( GtkTreeView *view,
		GtkTreeIter *iter, GtkTreePath *path, gpointer data );
static void screem_site_view_fm_collapsed( GtkTreeView *view,
		GtkTreeIter *iter, GtkTreePath *path, gpointer data );
static void screem_site_view_fm_activated( GtkTreeView *view,
		GtkTreePath *path, GtkTreeViewColumn *column, 
		gpointer data );
static gboolean screem_site_view_file_view_press( GtkWidget *widget,
		GdkEventButton *event, ScreemSiteView *view );

static void site_view_menu_new_dir_callback( GtkAction *action, 
		gpointer user_data );
static void site_view_menu_delete_file_callback( GtkAction *action, 
		gpointer user_data );
static void site_view_menu_file_props_callback( GtkAction *action, 
		gpointer user_data );

static void screem_site_view_show_properties( ScreemSite *site, 
		const gchar *uri, GtkWindow *window );

static gboolean screem_site_view_filter_folders( GtkTreeModel *model,
		GtkTreeIter *iter, ScreemSiteView *view );
static gboolean screem_site_view_filter_files( GtkTreeModel *model,
		GtkTreeIter *iter, ScreemSiteView *view );
static void screem_site_view_folder_sel( GtkTreeSelection *sel,
		ScreemSiteView *view );

static gboolean screem_site_view_file_view_release( GtkWidget *widget,
		GdkEventButton *event, ScreemSiteView *view );

static void site_view_menu_show_hidden( GtkAction *action,
		gpointer data );
static void site_view_menu_show_backup( GtkAction *action,
		gpointer data );

static void site_view_menu_exclude( GtkAction *action, gpointer data );
static void site_view_menu_ignore( GtkAction *action, gpointer data );


struct ScreemSiteViewPrivate {
	ScreemWindow *window;
	ScreemSite *site;

	GtkWidget *folders;
	GtkWidget *files;

	GtkWidget *menu_view;

	GSList *expand;
	GSList *collapse;
};

enum {
	TARGET_URI_LIST
};

static const GtkTargetEntry drag_types[] = {
	{ "text/uri-list", 0, TARGET_URI_LIST }
};
static const int num_drag_types = sizeof(drag_types) / sizeof(drag_types[ 0 ]);


static const GtkTargetEntry drop_types[] = {
	{ "text/uri-list", 0, TARGET_URI_LIST }
};
static const int num_drop_types = sizeof(drop_types) / sizeof(drop_types[ 0 ]);


/* DnD menu */
static void dnd_cb( GtkAction *action, gpointer data );

static const gchar *dnd_menu = "\
<ui>\
<popup action=\"site_model_dnd_menu\">\
<menuitem action=\"move\"/>\
<menuitem action=\"copy\"/>\
<separator name=\"site_model_dnd_sep1\"/>\
<menuitem action=\"cancel drag\"/>\
</popup>\
</ui>";
static GtkActionEntry dnd_actions[] = {
	{ "site_model_dnd_menu", "", NULL, NULL, NULL, NULL },

	{ "move", NULL, N_( "_Move here" ), NULL, N_( "Move file" ),
	  G_CALLBACK( dnd_cb ) },
	  
	{ "copy", NULL, N_( "_Copy here" ), NULL,  N_( "Copy file" ),
	  G_CALLBACK( dnd_cb ) },

	  { "cancel drag", NULL, N_( "Cancel drag" ), NULL,
	  N_( "Cancel the drag and drop operation" ),
	  G_CALLBACK( dnd_cb ) }
};
static guint dnd_actions_n = G_N_ELEMENTS( dnd_actions );

static GtkActionEntry view_actions[] = {
	{ "siteviewmenu", NULL, "", NULL, NULL, NULL },
	
	{ "New Directory", GTK_STOCK_NEW, N_("New Directory"),
	  "", NULL, 
	  G_CALLBACK( site_view_menu_new_dir_callback ) },

	{ "Delete File", GTK_STOCK_DELETE, N_("Delete"),
	  "", NULL, 
	  G_CALLBACK( site_view_menu_delete_file_callback ) },

	{ "File Properties", GTK_STOCK_PROPERTIES, N_("Properties"),
	  "", NULL, 
	  G_CALLBACK( site_view_menu_file_props_callback ) }
};
static guint view_actions_n = G_N_ELEMENTS( view_actions );

static GtkToggleActionEntry view_toggle_actions[] = {
	{ "ShowHidden", NULL, N_("Show Hidden Files"),
	  "", N_("Show/Hide hidden files/folders"),
	  G_CALLBACK( site_view_menu_show_hidden ) },
	  
	{ "ShowBackup", NULL, N_("Show Backup Files"),
	  "", N_("Show/Hide backup files"),
	  G_CALLBACK( site_view_menu_show_backup ) },

	{ "ExcludeFlag", NULL, N_("Exclude"),
	  NULL, NULL, 
	  G_CALLBACK( site_view_menu_exclude ) },

	{ "IgnoreFlag", NULL, N_("Ignore"),
	  NULL, NULL,
	  G_CALLBACK( site_view_menu_ignore ) }
};
static guint view_toggle_actions_n = G_N_ELEMENTS( view_toggle_actions );

/* G Object stuff */
#define PARENT_TYPE GTK_TYPE_BIN

static gpointer parent_class;

static void screem_site_view_class_init( ScreemSiteViewClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	GParamSpec *pspec;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;
	parent_class = g_type_class_peek_parent( klass );


	object_class->finalize = screem_site_view_finalize;
	object_class->get_property = screem_site_view_get_prop;
	object_class->set_property = screem_site_view_set_prop;

	widget_class->size_request = screem_site_view_size_request;
	widget_class->size_allocate = screem_site_view_size_allocate;
	
	pspec = g_param_spec_pointer( "window", "window",
				      "window",
				      G_PARAM_READABLE |
				      G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_WINDOW,
					 pspec );
	pspec = g_param_spec_object( "site", "site",
				     "site",
				     SCREEM_TYPE_SITE,
				     G_PARAM_READABLE |
				     G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_SITE,
					 pspec );
}

static void screem_site_view_init( ScreemSiteView *view )
{
	ScreemSiteViewPrivate *priv;
	
	GtkWidget *sw;
	GtkWidget *pane;
	GtkWidget *widget;

	GtkTreeSelection *selection;

	priv = view->priv = g_new0( ScreemSiteViewPrivate, 1 );

	pane = gtk_vpaned_new();
	gtk_widget_show( pane );

	/* folder view */	
	sw = gtk_scrolled_window_new( NULL, NULL );	
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	priv->folders = widget = gtk_tree_view_new();

	gtk_tree_view_set_rules_hint( GTK_TREE_VIEW( widget ), TRUE );
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( widget ),
					   TRUE );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	gtk_tree_selection_set_mode( GTK_TREE_SELECTION( selection ),
			GTK_SELECTION_SINGLE );
	g_signal_connect( G_OBJECT( selection ), "changed",
			G_CALLBACK( screem_site_view_folder_sel ),
			view );
	screem_site_view_set_columns( widget );

	gtk_container_add( GTK_CONTAINER( sw ), widget );
	gtk_paned_pack1( GTK_PANED( pane ), sw, TRUE, TRUE );


	gtk_drag_source_set( widget, 
			GDK_BUTTON1_MASK | GDK_BUTTON2_MASK,
			drag_types, num_drag_types,
			GDK_ACTION_COPY |GDK_ACTION_MOVE | GDK_ACTION_ASK);
	gtk_drag_dest_set( widget,
			GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP |
			GTK_DEST_DEFAULT_MOTION,
			drop_types, num_drop_types,
			GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_ASK );
	g_signal_connect( G_OBJECT( widget ), "drag_motion",
			  G_CALLBACK( screem_site_view_motion ), NULL );
	g_signal_connect( G_OBJECT( widget ), "drag_data_get",
			  G_CALLBACK( screem_site_view_set_dnd_data ), 
			  view );
	g_signal_connect( G_OBJECT( widget ), "drag_data_delete",
			  G_CALLBACK( screem_site_view_dnd_delete ),
			  view );
	g_signal_connect( G_OBJECT( widget ), "drag_data_received",
			  G_CALLBACK( screem_site_view_drop ), 
			  view );

	g_signal_connect( G_OBJECT( widget ), "row_activated",
			  G_CALLBACK( screem_site_view_fm_activated ),
			  view );
	g_signal_connect( G_OBJECT( widget ), "row_expanded",
			  G_CALLBACK( screem_site_view_fm_expanded ),
			  view );
	g_signal_connect( G_OBJECT( widget ), "row_collapsed",
			  G_CALLBACK( screem_site_view_fm_collapsed ),
			  view );
	g_signal_connect( G_OBJECT( widget ), "button_press_event",
			G_CALLBACK( screem_site_view_file_view_press ),
			view );
	g_signal_connect( G_OBJECT( widget ), "button_release_event",
			G_CALLBACK( screem_site_view_file_view_release ),
			view );
		
	/* file view */
	sw = gtk_scrolled_window_new( NULL, NULL );	
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	priv->files = widget = gtk_tree_view_new();
	gtk_tree_view_set_rules_hint( GTK_TREE_VIEW( widget ), TRUE );
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( widget ),
					   FALSE );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	gtk_tree_selection_set_mode( GTK_TREE_SELECTION( selection ),
			GTK_SELECTION_SINGLE );
	screem_site_view_set_columns( widget );

	gtk_container_add( GTK_CONTAINER( sw ), widget );
	gtk_paned_pack2( GTK_PANED( pane ), sw, TRUE, TRUE );

	gtk_drag_source_set( widget, 
			GDK_BUTTON1_MASK | GDK_BUTTON2_MASK,
			drag_types, num_drag_types,
			GDK_ACTION_COPY |GDK_ACTION_MOVE | GDK_ACTION_ASK);
	gtk_drag_dest_set( widget,
			GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP |
			GTK_DEST_DEFAULT_MOTION,
			drop_types, num_drop_types,
			GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_ASK );
	g_signal_connect( G_OBJECT( widget ), "drag_motion",
			  G_CALLBACK( screem_site_view_motion ), NULL );
	g_signal_connect( G_OBJECT( widget ), "drag_data_get",
			  G_CALLBACK( screem_site_view_set_dnd_data ), 
			  view );
	g_signal_connect( G_OBJECT( widget ), "drag_data_delete",
			  G_CALLBACK( screem_site_view_dnd_delete ),
			  view );
	g_signal_connect( G_OBJECT( widget ), "drag_data_received",
			  G_CALLBACK( screem_site_view_drop ), view );

	g_signal_connect( G_OBJECT( widget ), "row_activated",
			  G_CALLBACK( screem_site_view_fm_activated ),
			  view );
	g_signal_connect( G_OBJECT( widget ), "button_press_event",
			G_CALLBACK( screem_site_view_file_view_press ),
			view );
	
	gtk_widget_show_all( pane );
	gtk_container_add( GTK_CONTAINER( view ), pane );
}

static void screem_site_view_finalize( GObject *view )
{
	ScreemSiteView *sview;
	ScreemSiteViewPrivate *priv;
	
	sview = SCREEM_SITE_VIEW( view );
	priv = sview->priv;
	
	g_free( priv );

	G_OBJECT_CLASS( parent_class )->finalize( view );
}

static void screem_site_view_set_prop( GObject *object, 
		guint property_id, const GValue *value, 
		GParamSpec *pspec )
{
	ScreemSiteView *view;
	ScreemSiteViewPrivate *priv;
	
	view = SCREEM_SITE_VIEW( object );
	priv = view->priv;
	
	switch( property_id ) {
		case PROP_WINDOW:
			priv->window = SCREEM_WINDOW( g_value_get_pointer( value ) );
			screem_site_view_window_set( view );
			break;
		case PROP_SITE:
			priv->site = g_value_get_object( value );
			screem_site_view_site_set( view );
			break;
	}
}

static void screem_site_view_get_prop( GObject *object, 
		guint property_id, GValue *value, GParamSpec *pspec )
{
	ScreemSiteView *view;
	ScreemSiteViewPrivate *priv;

	view = SCREEM_SITE_VIEW( object );
	priv = view->priv;

	switch( property_id ) {
		case PROP_WINDOW:
			g_value_set_pointer( value, priv->window );
			break;
		case PROP_SITE:
			g_value_set_object( value, priv->site );
			break;
	}
}

static void screem_site_view_size_request( GtkWidget *widget,
                           		GtkRequisition *requisition )
{
	GtkBin *bin;

	bin = GTK_BIN (widget);

	requisition->width = GTK_CONTAINER( widget )->border_width * 2;
	requisition->height = GTK_CONTAINER( widget )->border_width * 2;

	if( bin->child && GTK_WIDGET_VISIBLE( bin->child ) ) {
		GtkRequisition child_requisition;

		gtk_widget_size_request( bin->child, &child_requisition );
		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void screem_site_view_size_allocate( GtkWidget *widget,
                            		GtkAllocation *allocation )
{
	GtkBin *bin;
	GtkAllocation child_allocation;

	bin = GTK_BIN( widget );
	widget->allocation = *allocation;
	
	if( bin->child ) {
		child_allocation.x = allocation->x + 
				GTK_CONTAINER( widget )->border_width; 
		child_allocation.y = allocation->y + 
				GTK_CONTAINER (widget)->border_width;
		child_allocation.width = MAX( allocation->width - 
				      GTK_CONTAINER( widget )->border_width * 2,
					0);
		child_allocation.height = MAX( allocation->height - 
				       GTK_CONTAINER (widget)->border_width * 2,
					0);
		gtk_widget_size_allocate( bin->child, &child_allocation );
	}
}

GType screem_site_view_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSiteViewClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_site_view_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSiteView ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_site_view_init
		};
		
		type = g_type_register_static( PARENT_TYPE,
					       "ScreemSiteView",
					       &info, 0 );
	}

	return type;
}

/* static stuff */

static void screem_site_view_set_columns( GtkWidget *widget )
{
	GtkCellRenderer *prenderer;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTooltips *tip;

	/* column 1 */
	prenderer = gtk_cell_renderer_pixbuf_new();
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_pack_start( column, prenderer, FALSE );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_column_set_resizable( column, TRUE );

	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), column );
	
	gtk_tree_view_column_set_attributes( column, renderer,
			"text", FILE_BROWSER_NAME_COL, NULL );
	gtk_tree_view_column_set_attributes( column, prenderer,
			"pixbuf", FILE_BROWSER_ICON_COL, NULL );
	
	gtk_tree_view_column_set_sort_column_id( column,
			FILE_BROWSER_NAME_COL );

	tip = gtk_tooltips_new();
	gtk_tooltips_set_tip( tip, column->button, _( "Site name" ),
			"" );
	g_object_set_data_full( G_OBJECT( column->button ), "tipobj",
			tip, (GDestroyNotify)gtk_object_destroy );
	gtk_tree_view_set_expander_column( GTK_TREE_VIEW( widget ), 
			column );

	/* column 2 */
	
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_set_title( column, "" );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_column_set_sizing( column, GTK_TREE_VIEW_COLUMN_FIXED );
	gtk_tree_view_column_set_fixed_width( column, 32 );

	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), column );
	
	gtk_tree_view_column_set_attributes( column, renderer,
			"text", FILE_BROWSER_USER_COL, NULL );
	
	tip = gtk_tooltips_new();
	gtk_tooltips_set_tip( tip, column->button,
			_( "Site Operation Flags" ), "" );
	g_object_set_data_full( G_OBJECT( column->button ), "tipobj",
			tip, (GDestroyNotify)gtk_object_destroy );
	
	gtk_tree_view_columns_autosize( GTK_TREE_VIEW( widget ) );
}

static void screem_site_view_window_set( ScreemSiteView *view )
{
	ScreemSiteViewPrivate *priv;
	ScreemWindow *window;
	GtkActionGroup *group;
	GError *error;
	GtkUIManager *merge;
	GtkAction *action;
	
	priv = view->priv;
	
	window = priv->window;
	group = window->action_group;
	merge = GTK_UI_MANAGER( window->merge );

	/* view popup */
	gtk_action_group_add_actions( group, view_actions, 
			view_actions_n, view );

	/* toggle actions in view popup */
	gtk_action_group_add_toggle_actions( group,
			view_toggle_actions, view_toggle_actions_n,
			view );

	/* dnd popup */
	gtk_action_group_add_actions( GTK_ACTION_GROUP( group ),
			dnd_actions, dnd_actions_n, view );

	error = NULL;
	if( ! gtk_ui_manager_add_ui_from_string( GTK_UI_MANAGER( merge ),
				dnd_menu, strlen( dnd_menu ),
				&error ) ) {
		g_message( "Site View dnd menu string error = %s", 
				error->message );
		g_error_free( error );
	}

	/* init properties item to unsensitive  */
	action = gtk_action_group_get_action( group, 
			"File Properties" );
	gtk_action_set_sensitive( GTK_ACTION( action ), FALSE );
}

static gboolean set_expanded( GtkTreeModel *model, GtkTreePath *path,
		GtkTreeIter *iter, ScreemSiteView *view )
{
	ScreemSiteViewPrivate *priv;
	ScreemSite *site;
	ScreemApplication *app;
	ScreemSession *session;
	const gchar *name;
	gboolean status;
	gchar *uri;

	const gchar *root;

	priv = view->priv;
	site = priv->site;
	
	root = screem_site_get_pathname( site );
	if( ! root ) {
		root = g_get_home_dir();
	}
	
	gtk_tree_model_get( model, iter, FILE_BROWSER_URI_COL,
			&uri, -1 );

	if( uri && strcmp( uri, root ) ) {
		
		g_object_get( G_OBJECT( site ), "app", &app, NULL );
		session = screem_application_get_session( app );
		g_object_unref( app );
	
		name = screem_site_get_name( site );
			
		status = screem_session_site_get_expanded( session, 
				name, uri );
		
		if( status ) {

			priv->expand = g_slist_prepend( priv->expand,
					gtk_tree_path_copy( path ) );
		} else {
			priv->collapse = g_slist_prepend( priv->collapse,
					gtk_tree_path_copy( path ) );
		}
	}
	g_free( uri );

	return FALSE;
}

static void screem_site_view_site_set( ScreemSiteView *view )
{
	ScreemSiteViewPrivate *priv;
	ScreemSite *site;
	gchar *ename;	
	GtkTreeViewColumn *col;
	GtkTreePath *path;

	GtkTreeModel *model;
	
	GtkTreeModel *filter;
	GSList *tmp;
	
	priv = view->priv;
	site = priv->site;

	model = screem_site_get_model( site, FALSE );

	gtk_tree_view_set_model( GTK_TREE_VIEW( priv->files ), NULL );
	gtk_tree_view_set_model( GTK_TREE_VIEW( priv->folders ), NULL );

	filter = gtk_tree_model_filter_new( model, NULL );
	gtk_tree_model_filter_set_visible_func( GTK_TREE_MODEL_FILTER( filter ),
			(GtkTreeModelFilterVisibleFunc)screem_site_view_filter_folders, view, NULL );
		
	gtk_tree_view_set_model( GTK_TREE_VIEW( priv->folders ),
			filter );
	
	ename = screem_escape_underlines( screem_site_get_name( site ) );
	col = gtk_tree_view_get_column( GTK_TREE_VIEW( priv->folders ),
			0 );
	gtk_tree_view_column_set_title( col, ename );

	col = gtk_tree_view_get_column( GTK_TREE_VIEW( priv->files ),
			1 );
	gtk_tree_view_column_set_visible( col, 
			! screem_site_get_fake_flag( site ) );

	/* hide header for individual files as per HIG guidelines */
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( priv->folders  ),
			! screem_site_get_fake_flag( site ) ); 

	
	col = gtk_tree_view_get_column( GTK_TREE_VIEW( priv->files ),
			0 );
	gtk_tree_view_column_set_title( col, ename );

	col = gtk_tree_view_get_column( GTK_TREE_VIEW( priv->files ),
			1 );
	gtk_tree_view_column_set_visible( col, 
			! screem_site_get_fake_flag( site ) );

	g_free( ename );

	
	g_object_set_data( G_OBJECT( priv->folders ), "block",
			GUINT_TO_POINTER( 1 ) );

	/* expand root node */
	path = gtk_tree_path_new_first();
	gtk_tree_view_expand_row( GTK_TREE_VIEW( priv->folders ),
				  path, FALSE );
	gtk_tree_path_free( path );
		
	if( ! screem_site_get_fake_flag( site ) ) {
		gtk_tree_model_foreach( GTK_TREE_MODEL( filter ),
				(GtkTreeModelForeachFunc)set_expanded, 
				view );

		priv->collapse = g_slist_reverse( priv->collapse );
		for( tmp = priv->collapse; tmp; tmp = tmp->next ) {
			path = (GtkTreePath*)tmp->data;

			gtk_tree_view_collapse_row( GTK_TREE_VIEW( priv->folders ),
					path );
		
			gtk_tree_path_free( path );
		}
		g_slist_free( priv->collapse );
		priv->collapse = NULL;
	
		priv->expand = g_slist_reverse( priv->expand );
		for( tmp = priv->expand; tmp; tmp = tmp->next ) {
			path = (GtkTreePath*)tmp->data;

			gtk_tree_view_expand_to_path( GTK_TREE_VIEW( priv->folders ), 
					path );
			gtk_tree_view_expand_row( GTK_TREE_VIEW( priv->folders ),
					path, FALSE );

			gtk_tree_path_free( path );
		}
		g_slist_free( priv->expand );
		priv->expand = NULL;
	}
	g_object_set_data( G_OBJECT( priv->folders ), "block",
			GINT_TO_POINTER( 0 ) );
        	
	g_object_unref( filter );
	
}

static void dnd_cb( GtkAction *action, gpointer data )
{
	gchar *name;
	gint i;
	gint item;

	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	item = -1;
	for( i = 1; i < dnd_actions_n; ++ i ) {
		if( ! strcmp( name, dnd_actions[ i ].name ) ) {
			item = i - 1;
			break;
		}
	}
	g_object_set_data( G_OBJECT( data ), "dnd_popup_item",
			GINT_TO_POINTER( item ) );
}

static gboolean screem_site_view_motion( GtkWidget *widget, 
		GdkDragContext *context, gint x, gint y, guint time, 
		gpointer data )
{
	GdkDragAction action;
       	GdkModifierType modifiers;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	gint cx;
	gint cy;
	
        gdk_window_get_pointer (NULL, NULL, NULL, &modifiers);
	
	gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW( widget ),
				       x, y, &path, &column,
				       &cx, &cy );
      
        if( context->suggested_action != GDK_ACTION_ASK ) {
                action = GDK_ACTION_COPY;
	} else {
                action = GDK_ACTION_ASK;
	}

	if( ( modifiers & GDK_MOD1_MASK ) != 0 ) {
                action = GDK_ACTION_ASK;
	}
        
        gdk_drag_status( context, action, time );

        return TRUE;
}

static void screem_site_view_set_dnd_data( GtkWidget *widget,
		GdkDragContext *context, GtkSelectionData *data,
		guint info, guint time, gpointer udata )
{
	ScreemSiteView *view;
	ScreemSiteViewPrivate *priv;
	
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	GValue value = {0,};
	const gchar *uri;
	ScreemSite *site;

	view = SCREEM_SITE_VIEW( udata );
	priv = view->priv;
	
	site = priv->site;

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	
	gtk_tree_selection_get_selected( selection, &model, &it );

	gtk_tree_model_get_value( model, &it, 
			FILE_BROWSER_URI_COL, &value );
	uri = g_value_get_string( &value );

	if( uri ) {
		gtk_selection_data_set( data, data->target,
				8, (guchar*)uri, strlen( uri ) );
	} else {
		gtk_selection_data_set( data, data->target, 8, NULL, 0 );
	}
}

static void screem_site_view_dnd_delete( GtkWidget *widget,
		GdkDragContext *context, gpointer data )
{
	ScreemSiteView *view;
	ScreemSiteViewPrivate *priv;
	ScreemSite *site;

	view = SCREEM_SITE_VIEW( data );
	priv = view->priv;
	
	site = priv->site;

	g_signal_stop_emission_by_name( G_OBJECT( widget ), 
			"drag_data_delete" );
}

static void screem_site_view_drop( GtkWidget *widget, 
		GdkDragContext *context, gint x, gint y, 
		GtkSelectionData *selectionData, guint info, 
		guint time, gpointer data )
{
	ScreemSiteView *view;
	ScreemSiteViewPrivate *priv;
	gboolean moving;
	gchar **uris;
	GtkTreeModel *filter;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter it;
	GtkTreeIter cit;
	GtkTreeSelection *selection;
	GValue value = {0};
	const gchar *target_uri;
	ScreemSite *site;
	GtkUIManager *merge;

	view = SCREEM_SITE_VIEW( data );
	priv = view->priv;
	
	site = priv->site;

	g_signal_stop_emission_by_name( G_OBJECT( widget ),
			"drag_data_received" );

	merge = GTK_UI_MANAGER( priv->window->merge );
	
	if( context->action == GDK_ACTION_ASK ) {
		GtkWidget *popup;
		gint item;
		popup = gtk_ui_manager_get_widget( merge, 
				"/site_model_dnd_menu" );
		g_object_set_data( G_OBJECT( view ), 
				"dnd_popup_item", GINT_TO_POINTER( -1 ) );
		priv->menu_view = widget;
                screem_popup_menu_do_popup_modal( popup, 0, 0, 0, 0, 
				GTK_WIDGET( view ) );
		item = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( view ), "dnd_popup_item" ) );
		switch( item ) {
		case 0:
			context->action = GDK_ACTION_MOVE;
			break;
		case 1:
			context->action = GDK_ACTION_COPY;
			break;
		default:
			gtk_drag_finish( context, FALSE, FALSE, time );
			return;
			break;
		}
	}
	
	moving = ( context->action == GDK_ACTION_MOVE );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( priv->folders ) );

	path = NULL;
	/* get selected path of priv->folders if the drop is on
	 * priv->files */
	if( widget == priv->files ) {
		if( gtk_tree_selection_get_selected( selection, &model,
					&it ) ) {
			path = gtk_tree_model_get_path( model, &it );
		}
	} else {
		gtk_tree_view_get_dest_row_at_pos( GTK_TREE_VIEW( widget ),
						   x, y, &path, NULL );
	}
	if( ! path ) {
		gtk_drag_finish( context, FALSE, FALSE, time );
		return;
	}
	
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
		
	if( ! gtk_tree_model_get_iter( model, &it, path ) ) {
		gtk_tree_path_free( path );
		return;
	}

	gtk_tree_path_free( path );

	/* we want to work with the full model, not the folder or files
	 * one */
	gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &cit, &it );
	model = screem_site_get_model( site, FALSE );

	gtk_tree_model_get_value( model, &cit, FILE_BROWSER_URI_COL, 
			&value );
	target_uri = g_value_get_string( &value );

	if( ! target_uri ) {
		gtk_drag_finish( context, FALSE, FALSE, time );
		g_value_unset( &value );
		return;
	}

	/* what type of drop do we have? */
	switch( info ) {
	case TARGET_URI_LIST:
		uris = g_strsplit( (gchar*)selectionData->data, "\r\n", 0 );

		/* drops onto iters that aren't directories require
		   us to get the parent node, which should be a directory */
		if( ! screem_uri_is_dir( target_uri ) ) {
			GtkTreeIter pit;

			g_value_unset( &value );
			gtk_tree_model_iter_parent( model, &pit, &cit );
			gtk_tree_model_get_value( model, &pit, 
						  FILE_BROWSER_URI_COL,
						  &value );
			target_uri = g_value_get_string( &value );
			if( ! target_uri ) {
				gtk_drag_finish( context, FALSE, FALSE, time );
				g_value_unset( &value );
				return;
			}
			memcpy( &cit, &pit, sizeof( GtkTreeIter ) );
		}
		filter = gtk_tree_view_get_model( GTK_TREE_VIEW( priv->folders ) );
		gtk_tree_model_filter_convert_child_iter_to_iter( GTK_TREE_MODEL_FILTER( filter ),
				&it, &cit );
		path = gtk_tree_model_get_path( filter, &it );
		
		screem_site_view_drop_uris( site, uris, 
				target_uri, moving );
		
		g_strfreev( uris );

		/* select the item we are dropping onto */
	
		gtk_tree_selection_select_path( GTK_TREE_SELECTION( selection ), path );
		gtk_tree_path_free( path );

		break;
	}
	
	g_value_unset( &value );

	gtk_drag_finish( context, TRUE, FALSE, time );
}

static gboolean screem_site_view_drop_uris( ScreemSite *site,
		gchar **uris, const gchar *target_uri,
		gboolean moving )
{
	gint i;
	GnomeVFSURI *source;
	GnomeVFSURI *target;
	GnomeVFSURI *realtarget;
	gchar *basename;
	GnomeVFSFileInfo *vfsinfo;
	GnomeVFSFileInfoOptions options;
	gboolean res;
	gchar *tgt;

	/* processed all uris, they have been
	   dropped onto target_uri */
	for( i = 0; uris[ i ] && *uris[ i ] != '\0'; ++ i ) {
		/* make sure we aren't dropping onto the
		   same uri, first we need to append the
		   basename of source onto target */
		source = gnome_vfs_uri_new( uris[ i ] );
		
		basename = gnome_vfs_uri_extract_short_name( source );
		
		target = gnome_vfs_uri_new( target_uri );
		realtarget = gnome_vfs_uri_append_file_name(target,
							    basename );
				
		if( gnome_vfs_uri_equal( source, target ) ) {
			/* copy to same location, don't do it. */
			gnome_vfs_uri_unref( realtarget );

		} else if( ! gnome_vfs_uri_equal( source, realtarget ) ) {
			gnome_vfs_uri_unref( target );
			target = realtarget;

			tgt = gnome_vfs_uri_to_string(target,
						      GNOME_VFS_URI_HIDE_NONE);
			
			vfsinfo = gnome_vfs_file_info_new();
			options = GNOME_VFS_FILE_INFO_FOLLOW_LINKS;
			gnome_vfs_get_file_info( uris[ i ], vfsinfo, 
						 options );
			if( vfsinfo->type == GNOME_VFS_FILE_TYPE_DIRECTORY ) {
				res = copy_dir( uris[ i ], tgt, 
						FALSE,
						screem_site_model_file_op,
						site->priv->model );
			} else {
				res = copy_file( uris[ i ], tgt, 
						GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
						screem_site_model_file_op,
						site->priv->model );
			}
			gnome_vfs_file_info_unref( vfsinfo );
			
			if( res ) {
				/* we need to copy the buffer if
				   we are copying a file which is
				   already loaded */
				ScreemPage *page;
				ScreemPage *npage;
				page = screem_site_locate_page( site, 
								uris[ i ] );
				npage = screem_site_locate_page( site, tgt );
				if( page && npage ) {
					gchar *data;
					data = screem_page_get_data( page );
					if( screem_page_is_loaded(page))
						screem_page_set_data( npage,
								      data );
					g_free( data );
				}
				if( moving ) {
					screem_site_file_change( site, 
								 uris[i],tgt );
					/* will call delete_file()
					   if needed */
					delete_dir( uris[ i ], 
						screem_site_model_file_op,
						site->priv->model );
				}
			}
			g_free( tgt );
			g_free( basename );
		}
		gnome_vfs_uri_unref( target );
		gnome_vfs_uri_unref( source );
	}
		
	return TRUE;
}	

static void screem_site_view_fm_expanded( GtkTreeView *view,
		GtkTreeIter *iter, GtkTreePath *path, gpointer data )
{
	ScreemSiteView *sview;
	ScreemSiteViewPrivate *priv;
	ScreemSite *site;
	GtkTreeModel *model;
	GtkTreeIter it;

	const gchar *name;
	gchar *uri;

	ScreemApplication *app;
	ScreemSession *session;
	
	sview = SCREEM_SITE_VIEW( data );
	priv = sview->priv;
	
	site = priv->site;

	model = gtk_tree_view_get_model( view );

	if( ! g_object_get_data( G_OBJECT( view ), "block" ) ) {
		gtk_tree_model_get( model, iter,
				FILE_BROWSER_URI_COL, &uri, -1 );
		
		/* fake site needs to scan the directory */
		if(  screem_site_get_fake_flag( site ) ) {
			ScreemFileBrowser *browser; 
		
			browser = site->priv->model->browser;

			/* filter model to sort model */
			gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &it, iter );
		
			screem_file_browser_scan_iter( browser, &it, -1 );
		
	                g_object_set_data( G_OBJECT( view ), "block",
        	                           GUINT_TO_POINTER( 1 ) );
	                gtk_tree_view_expand_row( view, path, FALSE );
        	        g_object_set_data( G_OBJECT( view ), "block", NULL );
	        }
		name = screem_site_get_name( site );
		g_object_get( G_OBJECT( site ), "app", &app, NULL );
		session = screem_application_get_session( app );
		g_object_unref( app );
		screem_session_site_set_expanded( session, name, 
				uri, TRUE );
		
		g_free( uri );
	}
}

static void screem_site_view_fm_collapsed( GtkTreeView *view,
		GtkTreeIter *iter, GtkTreePath *path, gpointer data )
{
	ScreemSiteView *sview;
	ScreemSiteViewPrivate *priv;
	ScreemSite *site;
	GtkTreeModel *model;

	const gchar *name;
	gchar *uri;

	ScreemApplication *app;
	ScreemSession *session;
	
	sview = SCREEM_SITE_VIEW( data );
	priv = sview->priv;
	
	site = priv->site;

	model = gtk_tree_view_get_model( view );

	if( ! g_object_get_data( G_OBJECT( view ), "block" ) ) {
		gtk_tree_model_get( model, iter,
				FILE_BROWSER_URI_COL, &uri, -1 );
		name = screem_site_get_name( site );
		g_object_get( G_OBJECT( site ), "app", &app, NULL );
		session = screem_application_get_session( app );
		g_object_unref( app );
		screem_session_site_set_expanded( session, name, uri, 
				FALSE );
		g_free( uri );
	}
}

static void screem_site_view_fm_activated( GtkTreeView *view,
		GtkTreePath *path, GtkTreeViewColumn *column, 
		gpointer data )
{
	ScreemSiteView *sview;
	ScreemSiteViewPrivate *priv;
	ScreemFileBrowser *browser;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeIter cit;
	ScreemSite *site;
	ScreemWindow *window;
	ScreemApplication *app;

	GList *list;
	GList *ret;
	ScreemPage *page;

	sview = SCREEM_SITE_VIEW( data );
	priv = sview->priv;
	
	site = priv->site;
	window = priv->window;

	browser = site->priv->model->browser;

	model = gtk_tree_view_get_model( view );
	
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );

	ret = NULL;
        if( gtk_tree_model_get_iter( model, &it, path ) ) {
		gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &cit, &it );
	
		list = g_list_append( NULL, &cit );
		ret = screem_file_browser_get_pathnames( browser,
				list, FALSE );
		g_list_free( list );
	}
	if( ret ) {
		const gchar *current = ret->data;
		g_list_free( ret );
	
		if( current && screem_page_is_file_page( current ) ) {
			page = screem_site_locate_page( site, current );
			if( page ) {
				screem_window_set_document( window, 
						page );
			} else if( screem_site_get_fake_flag( site ) ) {
				screem_page_open_with_filename( site, 
						window,	current );
			} else {
				/* this can occur as we do fast mime
				 * type checking when initially scanning
				 * the site, it is a valid page, so
				 * add it */
				g_object_get( G_OBJECT( window ),
						"app", &app, NULL );
				page = screem_page_new( G_OBJECT( app ) );
				g_object_unref( app );
				screem_page_set_pathname( page,
						current );
				screem_site_add_page( site, page );
				g_object_unref( page );
				screem_window_set_document( window, 
						page );
				/* FIXME: update site view mime type and
				 * icon */
			}
		} else if( current && ! screem_uri_is_dir( current ) ) {
			/* non page, open with default gnome app */
			if( ! screem_execute_default_app( current ) ) {
				/* failed */

			}
		}
	}
}

static gboolean screem_site_view_file_view_press( GtkWidget *widget,
		GdkEventButton *event, ScreemSiteView *view )
{
	ScreemSiteViewPrivate *priv;
	ScreemWindow *window;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	gint cx;
	gint cy;

	GtkTreeModel *model;
	GtkTreeIter it;

	ScreemSiteViewNodeInfo *info;
	GString *pathname;
	ScreemSite *site;
	GtkWidget *menu;
	GtkAction *action;
	gboolean status;
		
	GtkTreeSelection *selection;

	priv = view->priv;
	window = priv->window;
	site = priv->site;
	
	/* right mouse clicked */
	gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW( widget ), 
			event->x, event->y, &path, &column, &cx, &cy );

	action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ), "File Properties" );
	gtk_action_set_sensitive( GTK_ACTION( action ), path != NULL );

	/* need to check properties even if it isn't the right
	 * mouse button clicked, so the main file menu properties
	 * item is set sensitive correctly */
	
	if( event->button != 3 ) {
		return FALSE;
	}

	g_signal_stop_emission_by_name( G_OBJECT( widget ),
			"button_press_event" );

	pathname = NULL;
	info = NULL;
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	if( path && gtk_tree_model_get_iter( GTK_TREE_MODEL( model ), 
				&it, path ) ) {
		
		pathname = NULL;
		if( ! screem_site_get_fake_flag( site ) ) {
			gtk_tree_model_get( model, &it, 
					FILE_BROWSER_DATA_COL, &info,
					-1 );
			if( info ) {
				pathname = g_string_new( info->fullname );
			}
		}
		if( ! pathname ) {
			gchar *tmp;
			
			gtk_tree_model_get( model, &it, 
					FILE_BROWSER_URI_COL, &tmp,
					-1 );
			pathname = g_string_new( tmp );
			info = NULL;
		}

		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
		gtk_tree_selection_select_iter( GTK_TREE_SELECTION( selection ),
						&it );
	}
	/* set upload status toggle buttons */
	menu = gtk_ui_manager_get_widget( GTK_UI_MANAGER( window->merge ), "/siteviewmenu" );

	priv->menu_view = NULL;
	
	/* disable delete file  / properties 
	 * if we haven't clicked on a file/folder */
	action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ), "Delete File" );
	gtk_action_set_sensitive( GTK_ACTION( action ), path != NULL );

	/* set exclude / ignore status / sensitivity */	
	status = FALSE;
	if( pathname ) {
		status = screem_site_is_excluded( site, pathname->str );
	}
	action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ), "ExcludeFlag" );
	gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( action ),
			status );
	gtk_action_set_sensitive( GTK_ACTION( action ), info != NULL );

	status = FALSE;
	if( pathname ) {
		status = screem_site_is_ignored( site, pathname->str );
	}
	action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ), "IgnoreFlag" );
	gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( action ),
					status );
	gtk_action_set_sensitive( GTK_ACTION( action ), info != NULL );

	
	priv->menu_view = widget;
	screem_popup_menu_do_popup_modal( menu, 0, 0, event, 0, widget );

	if( pathname ) {
		g_string_free( pathname, TRUE );
	}
	if( path ) {
		gtk_tree_path_free( path );
	}
	
	return ( path != NULL );
}

static void site_view_menu_new_dir_callback( GtkAction *action, 
		gpointer user_data )
{
	ScreemSiteView *view;
	ScreemSiteViewPrivate *priv;
	ScreemWindow *window;
	GtkWidget *widget;
	GladeXML *xml;
	ScreemSite *site;
	const gchar *string;
	
	view = SCREEM_SITE_VIEW( user_data );
	priv = view->priv;

	g_return_if_fail( priv->menu_view != NULL );
	
	window = priv->window;
	
	xml = glade_xml_new( GLADE_PATH"/screem.glade",
			"create_dir", NULL );
	
	widget = glade_xml_get_widget( xml, "create_dir" );		
	gtk_window_set_transient_for( GTK_WINDOW( widget ),
			GTK_WINDOW( window ) );
		
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		site = screem_window_get_current( window );
		widget = glade_xml_get_widget( xml, "dirname" );
		string = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		widget = priv->menu_view;

		if( widget == priv->files ) {
			widget = priv->folders;
		}
		
		if( ( ! string ) || *string == '\0' ) {
	                screem_window_show_message( window,
					_( "No directory name specified." ), 
					FALSE );
		} else if( ! screem_site_model_create_dir( site, 
					widget, string ) ) {
	                screem_window_show_message( window,
					_( "Directory could not be created." ), 
					FALSE );
        	}
	}
	widget = glade_xml_get_widget( xml, "create_dir" );
	gtk_widget_destroy( widget );
	g_object_unref( G_OBJECT( xml ) );
}

static void site_view_menu_delete_file_callback( GtkAction *action, 
		gpointer user_data )
{
	ScreemSiteView *sview;
	ScreemSiteViewPrivate *priv;
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemSite *site;
	GtkWidget *view;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter it;
	gchar *uri;
	GtkWidget *confirm;
	GtkWidget *label;
	gchar *txt;
	gchar *base;
	gchar *temp;
	GladeXML *xml;
	
	sview = SCREEM_SITE_VIEW( user_data );
	priv = sview->priv;
	
	g_return_if_fail( priv->menu_view != NULL );
	
	window = priv->window;
	app = window->application;
	
	site = screem_window_get_current( window );
	view = priv->menu_view;
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	
	gtk_tree_selection_get_selected( selection, &model, &it );
	
	gtk_tree_model_get( model, &it, 
			FILE_BROWSER_URI_COL, &uri, -1 );
	if( uri ) {		
		xml = glade_xml_new( GLADE_PATH"/screem.glade",
				"delete_file", NULL );
	
		confirm = glade_xml_get_widget( xml, "delete_file" );
		label = glade_xml_get_widget( xml, "delete_label" );
		base = g_path_get_basename( uri );
		temp = gnome_vfs_unescape_string_for_display( base );
		txt = g_strdup_printf( _( "Are you sure you want to permanently delete \"%s\"?" ),
					temp );
		g_free( temp );
		g_free( base );
		gtk_label_set_text( GTK_LABEL( label ), txt );
		g_free( txt );
		
		gtk_window_set_wmclass( GTK_WINDOW( confirm ),
					"Screem",
					"delete_confirm_prompt" );
		gtk_window_set_transient_for( GTK_WINDOW( confirm ),
					      GTK_WINDOW( window ) );
		gtk_window_set_type_hint( GTK_WINDOW( confirm ), 
					  GDK_WINDOW_TYPE_HINT_DIALOG );
		if(gtk_dialog_run( GTK_DIALOG( confirm ) ) == GTK_RESPONSE_OK){
			delete_dir( uri, screem_application_file_op, app );
		}
		gtk_widget_destroy( confirm );
		g_object_unref( G_OBJECT( xml ) );
		g_free( uri );
	}
}

static void site_view_menu_file_props_callback( GtkAction *action, 
		gpointer user_data )
{
	ScreemSiteView *sview;
	ScreemSiteViewPrivate *priv;
	ScreemWindow *window;
	ScreemWindowDetails *details;
	ScreemSite *site;
	GtkTreeModel *model;
	GtkWidget *view;
	GtkTreeSelection *selection;
	GtkTreeIter it;
	gchar *uri;

	sview = SCREEM_SITE_VIEW( user_data );
	priv = sview->priv;

	if( ! priv->menu_view ) {
		view = priv->files;
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
		if( ! gtk_tree_selection_get_selected( selection, &model, &it ) ) {
			view = priv->folders;
		}
		
	} else {
		view = priv->menu_view;
	}
	
	window = priv->window;
	details = window->details;

	site = screem_window_get_current( window );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		gtk_tree_model_get( model, &it, 
			FILE_BROWSER_URI_COL, &uri, -1 );
	
		if( uri ) {
			/* display info on info fullname */
			screem_site_view_show_properties( site, uri,
					GTK_WINDOW( window ) );
			g_free( uri );
		}
	}
}

static void screem_site_view_show_properties( ScreemSite *site, 
		const gchar *uri, GtkWindow *window )
{
	GladeXML *xml;
	GtkWidget *dialog;
	GtkWidget *widget;

	GnomeVFSFileInfoOptions options;
	GnomeVFSFileInfo *vfsinfo;
	GtkWidget *label;

	GdkPixbuf *pixbuf;
	GtkWidget *image;

	const gchar *mime_type;
	
	struct passwd *userinfo;
	struct group *fgroup;
	struct group *group;
	GtkListStore *groups;
	GtkTreeIter it;
	GtkCellRenderer *rend;
	gint grouppos;
	gint i;
	gchar *title;

	gchar *name;

	ScreemFileBrowser *browser;

	ScreemApplication *app;
	ScreemSession *session;
	ScreemIconCache *cache;
		
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	session = screem_application_get_session( app );
	cache = screem_application_get_icon_cache( app );
	g_object_unref( app );
	
	browser = site->priv->model->browser;
	
	xml = glade_xml_new( GLADE_PATH"/screem.glade", 
			"file_properties", NULL );

	if( screem_site_get_fake_flag( site ) ) {
		widget = glade_xml_get_widget( xml, "exclude" );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "ignore" );
		gtk_widget_set_sensitive( widget, FALSE );
	}
	
	dialog = glade_xml_get_widget( xml, "file_properties" );

	vfsinfo = gnome_vfs_file_info_new();
	options = GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
		GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS;

	gnome_vfs_get_file_info( uri, vfsinfo, options );

	title = g_strconcat( vfsinfo->name, " - Properties", NULL );
	gtk_window_set_title( GTK_WINDOW( dialog ), title );
	g_free( title );

	label = glade_xml_get_widget( xml, "basename" );
	gtk_entry_set_text( GTK_ENTRY( label ), vfsinfo->name );

	label = glade_xml_get_widget( xml, "size" );
	if( vfsinfo->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE ) {
		gchar *txt;

		txt = gnome_vfs_format_file_size_for_display( vfsinfo->size );
		gtk_label_set_text( GTK_LABEL( label ), txt );
		g_free( txt );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
	}

	label = glade_xml_get_widget( xml, "location" );
	{
		gchar *dirname;

		dirname = g_path_get_dirname( uri );	
		gtk_label_set_text( GTK_LABEL( label ), dirname );
		g_free( dirname );
	}

	label = glade_xml_get_widget( xml, "mime_type" );
	image = glade_xml_get_widget( xml, "file_image" );
	
	mime_type = NULL;
	if( vfsinfo->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE ) {
		mime_type = vfsinfo->mime_type;

		gtk_label_set_text( GTK_LABEL( label ), mime_type );
		label = glade_xml_get_widget( xml, "type" );
		gtk_label_set_text( GTK_LABEL( label ),
				    gnome_vfs_mime_get_description( mime_type ) );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
		label = glade_xml_get_widget( xml, "type" );
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
	}
	pixbuf = screem_icon_cache_get_pixbuf( cache, uri, mime_type,
			48 );
	if( pixbuf ) {
		gtk_image_set_from_pixbuf( GTK_IMAGE( image ), pixbuf );
		g_object_unref( pixbuf );
	}
	
	label = glade_xml_get_widget( xml, "modified" );
	if( vfsinfo->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME ) {
		gchar *t = ctime( &vfsinfo->mtime );
		t[ strlen( t ) - 1 ] = '\0';
		gtk_label_set_text( GTK_LABEL( label ), t );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
	}


	label = glade_xml_get_widget( xml, "accessed" );
	if( vfsinfo->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ATIME ) {
		gchar *t = ctime( &vfsinfo->atime );
		t[ strlen( t ) - 1 ] = '\0';
		gtk_label_set_text( GTK_LABEL( label ), t );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
	}

	userinfo = getpwuid( vfsinfo->uid );
	if( userinfo ) {
		gchar *t;
		gchar *tmp;

		label = glade_xml_get_widget( xml, "user" );
		t = g_strconcat( userinfo->pw_name, " - ",
				 userinfo->pw_gecos, NULL );
		tmp = strchr( t, ',' );
		if( tmp ) {
			*tmp = '\0';
		}
		gtk_label_set_text( GTK_LABEL( label ), t );
		g_free( t );
	}
	
	userinfo = getpwuid( geteuid() );

	fgroup = getgrgid( vfsinfo->gid );
	grouppos = -1;
	i = 0;
	groups = gtk_list_store_new( 2, G_TYPE_STRING, G_TYPE_INT );
	while( ( group = getgrent() ) ) {
		if( ! strcmp( group->gr_name, fgroup->gr_name ) ) {
			gtk_list_store_append( groups, &it );
			gtk_list_store_set( groups, &it,
					0, group->gr_name,
					1, group->gr_gid,
					-1 );
			grouppos = i ++;
		} else {
			gint ni;

			for( ni = 0; group->gr_mem[ ni ]; ++ ni ) {
				if( ! strcmp( group->gr_mem[ ni ],
					      userinfo->pw_name ) ) {
					gtk_list_store_append( groups, &it );
					gtk_list_store_set( groups, &it,
							0, 
							group->gr_name,
							1, 
							group->gr_gid,
							-1 );
					i ++;
					break;
				}
			}
		}
	}
	endgrent();
	if( grouppos == -1 ) {
		gtk_list_store_append( groups, &it );
		gtk_list_store_set( groups, &it,
				0, fgroup->gr_name,
				1, fgroup->gr_gid,
				-1 );		
		grouppos = i;
	}

	label = glade_xml_get_widget( xml, "group" );
	rend = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start( GTK_CELL_LAYOUT( label ), rend,
			TRUE );
	gtk_cell_layout_set_attributes( GTK_CELL_LAYOUT( label ),
			rend, "text", 0, NULL );
	gtk_combo_box_set_model( GTK_COMBO_BOX( label ), 
			GTK_TREE_MODEL( groups ) );
	g_object_unref( groups );
	gtk_combo_box_set_active( GTK_COMBO_BOX( label ), grouppos );
	
	label = glade_xml_get_widget( xml, "owner_read" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_USER_READ ) );	
	label = glade_xml_get_widget( xml, "owner_write" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_USER_WRITE ) );
	label = glade_xml_get_widget( xml, "owner_exec" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_USER_EXEC ) );
	label = glade_xml_get_widget( xml, "group_read" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_GROUP_READ ) );	
	label = glade_xml_get_widget( xml, "group_write" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_GROUP_WRITE ) );
	label = glade_xml_get_widget( xml, "group_exec" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_GROUP_EXEC ) );
	label = glade_xml_get_widget( xml, "all_read" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_OTHER_READ ) );	
	label = glade_xml_get_widget( xml, "all_write" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_OTHER_WRITE ) );
	label = glade_xml_get_widget( xml, "all_exec" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_OTHER_EXEC ) );

	label = glade_xml_get_widget( xml, "exclude" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      screem_site_is_excluded(site,
							      uri));
	label = glade_xml_get_widget( xml, "ignore" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      screem_site_is_ignored(site,
							     uri));


	gnome_vfs_file_info_unref( vfsinfo );

	g_object_set_data( G_OBJECT( dialog ), "site", site );
	g_object_set_data( G_OBJECT( dialog ), "fullname",
			   g_strdup( uri ) );

	glade_xml_signal_autoconnect( xml );

	gtk_window_set_transient_for( GTK_WINDOW( dialog ), window );
	screem_session_restore_dialog( session, dialog );
	gtk_dialog_run( GTK_DIALOG( dialog ) );
	screem_session_store_dialog( session, dialog );

	name = g_object_get_data( G_OBJECT( dialog ), "fullname" );
	if( name ) {
		g_free( name );
	}

	g_object_set_data( G_OBJECT( dialog ), "fullname", NULL );
	
	g_object_unref( cache );
	
	gtk_widget_destroy( dialog );
	g_object_unref( xml );
}

gboolean screem_site_view_properties_rename( GtkWidget *widget )
{
	GladeXML *xml;
	const gchar *name;
	ScreemSite *site;
	const gchar *fullname;
	gchar *base;
	gchar *dirname;
	gchar *newpath;
	ScreemPage *page;
	gboolean done;

	name = gtk_entry_get_text( GTK_ENTRY( widget ) );

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );

	base = NULL;
	if( fullname ) {
		base = g_path_get_basename( fullname );
	}
	if( base && strcmp( name, base ) ) {
		/* file has been renamed */
		page = screem_site_locate_page( site, fullname );
		
		dirname = g_path_get_dirname( fullname );
		newpath = g_build_filename( dirname, name, NULL );

		if( page ) {
			screem_page_set_pathname( page, newpath );
		}
		if( screem_uri_is_dir( fullname ) ) {
	       		done = copy_dir( fullname, newpath, FALSE,
					screem_site_model_file_op,
					site->priv->model );
		} else {
			done = copy_file( fullname, newpath, 
					GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
					screem_site_model_file_op,
					site->priv->model );
		}
		if( done ) {
			screem_site_file_change( site,
						 fullname,
						 newpath );
			
			delete_dir( fullname, screem_site_model_file_op,
					site->priv->model );
			
			g_object_set_data( G_OBJECT( widget ), "fullname",
					   newpath );
			g_free( (gchar*)fullname );
		} else if( page ) {
			screem_page_set_pathname( page, fullname );
			g_free( newpath );
		}
		g_free( dirname );
	}
	g_free( base );

	return FALSE;
}

void screem_site_view_properties_chown( GtkWidget *widget )
{
	GladeXML *xml;
	GtkTreeModel *model;
	GtkTreeIter it;
	GnomeVFSFileInfo *vfsinfo;
	guint gid;
	const gchar *fullname;

	xml = glade_get_widget_tree( widget );

	model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
	gtk_combo_box_get_active_iter( GTK_COMBO_BOX( widget ), &it );
	gtk_tree_model_get( model, &it, 1, &gid, -1 );	

	widget = glade_xml_get_widget( xml, "file_properties" );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );
	if( fullname ) {
		vfsinfo = gnome_vfs_file_info_new();
		
		gnome_vfs_get_file_info( fullname, vfsinfo,
					 GNOME_VFS_FILE_INFO_FOLLOW_LINKS );
		
		vfsinfo->gid = gid;
		
		gnome_vfs_set_file_info( fullname, vfsinfo,
					 GNOME_VFS_SET_FILE_INFO_OWNER );
		
		gnome_vfs_file_info_unref( vfsinfo );
	}
}

void screem_site_view_properties_chmod( GtkWidget *widget )
{
	GladeXML *xml;
	GnomeVFSFileInfo *vfsinfo;
	GnomeVFSFilePermissions perms;
	const gchar *fullname;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );

	perms = 0;
	widget = glade_xml_get_widget( xml, "owner_read" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_USER_READ;
	}
	widget = glade_xml_get_widget( xml, "owner_write" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_USER_WRITE;
	}
	widget = glade_xml_get_widget( xml, "owner_exec" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_USER_EXEC;
	}
	widget = glade_xml_get_widget( xml, "group_read" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_GROUP_READ;
	}
	widget = glade_xml_get_widget( xml, "group_write" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_GROUP_WRITE;
	}
	widget = glade_xml_get_widget( xml, "group_exec" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_GROUP_EXEC;
	}
	widget = glade_xml_get_widget( xml, "all_read" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_OTHER_READ;
	}
	widget = glade_xml_get_widget( xml, "all_write" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_OTHER_WRITE;
	}
	widget = glade_xml_get_widget( xml, "all_exec" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_OTHER_EXEC;
	}

	if( fullname ) {
		vfsinfo = gnome_vfs_file_info_new();
		
		vfsinfo->permissions = perms;
		
		gnome_vfs_set_file_info( fullname, vfsinfo,
					 GNOME_VFS_SET_FILE_INFO_PERMISSIONS );
		
		gnome_vfs_file_info_unref( vfsinfo );
	}
}

void screem_site_view_properties_exclude( GtkWidget *widget )
{
	GladeXML *xml;
	gboolean exclude;
	const gchar *fullname;
	ScreemSite *site;

	exclude = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );
	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ),
					       "site" ) );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );

	if( fullname ) {
		if( exclude ) {
			screem_site_add_exclude( site, fullname );
		} else {
			screem_site_remove_exclude( site, fullname );
		}
	}
}

void screem_site_view_properties_ignore( GtkWidget *widget )
{
	GladeXML *xml;
	gboolean ignore;
	const gchar *fullname;
	ScreemSite *site;

	ignore = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );
	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ),
					       "site" ) );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );
	
	if( fullname ) {
		if( ignore ) {
			screem_site_add_ignore( site, fullname );
		} else {
			screem_site_remove_ignore( site, fullname );
		}
	}
}

static gboolean screem_site_view_filter_folders( GtkTreeModel *model,
		GtkTreeIter *iter, ScreemSiteView *view )
{
	gboolean ret;
	const gchar *mime;
	
	mime = NULL;
	gtk_tree_model_get( model, iter, FILE_BROWSER_MIME_COL, &mime,
			-1 );
	ret = TRUE;
	if( mime ) {
		ret = ( ! strcmp( "x-directory/normal", mime ) );
	}
	return ret;
}

static gboolean screem_site_view_filter_files( GtkTreeModel *model,
		GtkTreeIter *iter, ScreemSiteView *view )
{
	gboolean ret;
	const gchar *mime;
	
	ret = TRUE;

	mime = NULL;
	gtk_tree_model_get( model, iter, FILE_BROWSER_MIME_COL, &mime,
			-1 );
	if( mime ) {
		ret = ( strcmp( "x-directory/normal", mime ) );
	}
	
	return ret;
}


static void screem_site_view_folder_sel( GtkTreeSelection *sel,
		ScreemSiteView *view )
{
	ScreemSiteViewPrivate *priv;
	ScreemSite *site;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeIter cit;
	GtkTreePath *path;
	GtkTreeModel *filter;
	ScreemFileBrowser *browser; 
		
	priv = view->priv;
	site = priv->site;

	gtk_tree_view_set_model( GTK_TREE_VIEW( priv->files ), NULL );
	
	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
		/* filter model to sort model */
		gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &cit, &it );

		model = screem_site_get_model( site, FALSE );
		path = gtk_tree_model_get_path( model, &cit );

		if(  screem_site_get_fake_flag( site ) &&
				! gtk_tree_view_row_expanded( GTK_TREE_VIEW( priv->folders ), path ) ) {
			browser = site->priv->model->browser;
			screem_file_browser_scan_iter( browser, &cit, -1 );
		}
				
		filter = gtk_tree_model_filter_new( model, path );
		gtk_tree_model_filter_set_visible_func( GTK_TREE_MODEL_FILTER( filter ),
			(GtkTreeModelFilterVisibleFunc)screem_site_view_filter_files, view, NULL );
		
		gtk_tree_view_set_model( GTK_TREE_VIEW( priv->files ),
			filter );

		gtk_tree_path_free( path );	

		g_object_unref( filter );
	} else {
		GtkAction *action;

		/* set properties item to unsensitive  */
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "File Properties" );
		gtk_action_set_sensitive( GTK_ACTION( action ), 
				FALSE );
	}
}

static gboolean screem_site_view_file_view_release( GtkWidget *widget,
		GdkEventButton *event, ScreemSiteView *view )
{
	ScreemSiteViewPrivate *priv;
	GtkTreeViewColumn *col;
	gint width;
	
	priv = view->priv;

	col = gtk_tree_view_get_column( GTK_TREE_VIEW( widget ), 0 );
	width = gtk_tree_view_column_get_width( col );

	col = gtk_tree_view_get_column( GTK_TREE_VIEW( priv->files ), 0 );
	gtk_tree_view_column_set_min_width( col, width );
	gtk_tree_view_column_set_max_width( col, width );
	
	return FALSE;
}

static void site_view_menu_show_hidden( GtkAction *action,
		gpointer data )
{
	ScreemSiteView *view;
	ScreemSite *site;
	ScreemFileBrowser *browser;
	gboolean active;

	view = SCREEM_SITE_VIEW( data );
	site = view->priv->site;
	
	g_return_if_fail( site != NULL );
		
	browser = site->priv->model->browser;
	
	active = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );

	screem_file_browser_set_show_hidden( browser, active );
}

static void site_view_menu_show_backup( GtkAction *action,
		gpointer data )
{
	ScreemSiteView *view;
	ScreemSite *site;
	ScreemFileBrowser *browser;
	gboolean active;

	view = SCREEM_SITE_VIEW( data );
	site = view->priv->site;

	g_return_if_fail( site != NULL );
	
	browser = site->priv->model->browser;
	
	active = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );

	screem_file_browser_set_show_backups( browser, active );
}

static void site_view_menu_exclude( GtkAction *action, gpointer data )
{
	ScreemSiteView *view;
	ScreemSiteViewPrivate *priv;
	ScreemSite *site;
	GtkTreeView *tree;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	gchar *uri;
	gboolean active;
	
	view = SCREEM_SITE_VIEW( data );
	priv = view->priv;

	if( ! priv->menu_view ) {
		return;
	}
	
	site = priv->site;
	tree = GTK_TREE_VIEW( priv->menu_view );
	sel = gtk_tree_view_get_selection( tree );
	gtk_tree_selection_get_selected( sel, &model, &it );

	gtk_tree_model_get( model, &it,
			FILE_BROWSER_URI_COL, &uri, -1 );
	
	if( uri ) {
		active = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );
		if( active ) {
			screem_site_add_exclude( site, uri );
		} else {
			screem_site_remove_exclude( site, uri );
		}
		g_free( uri );
	}
}

static void site_view_menu_ignore( GtkAction *action, gpointer data )
{
	ScreemSiteView *view;
	ScreemSiteViewPrivate *priv;
	ScreemSite *site;
	GtkTreeView *tree;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	gchar *uri;
	gboolean active;
	
	view = SCREEM_SITE_VIEW( data );
	priv = view->priv;

	if( ! priv->menu_view ) {
		return;
	}
	
	site = priv->site;
	tree = GTK_TREE_VIEW( priv->menu_view );
	sel = gtk_tree_view_get_selection( tree );
	gtk_tree_selection_get_selected( sel, &model, &it );

	gtk_tree_model_get( model, &it,
			FILE_BROWSER_URI_COL, &uri, -1 );
	
	if( uri ) {
		active = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );
		if( active ) {
			screem_site_add_ignore( site, uri );
		} else {
			screem_site_remove_ignore( site, uri );
		}
		g_free( uri );
	}
}

/* public stuff */

ScreemSiteView *screem_site_view_new( void )
{
	ScreemSiteView *view;
	GType type;

	type = screem_site_view_get_type();

	view = SCREEM_SITE_VIEW( g_object_new( type, NULL ) );

	return view;
}

ScreemWindow *screem_site_view_get_window( ScreemSiteView *view )
{
	g_return_val_if_fail( SCREEM_IS_SITE_VIEW( view ), NULL );

	return view->priv->window;
}

void screem_site_view_set_window( ScreemSiteView *view,
		ScreemWindow *window )
{
	g_return_if_fail( SCREEM_IS_SITE_VIEW( view ) );
	g_return_if_fail( SCREEM_IS_WINDOW( window ) );
	
	g_object_set( G_OBJECT( view ), "window", window, NULL );
}

ScreemSite *screem_site_view_get_site( ScreemSiteView *view )
{
	g_return_val_if_fail( SCREEM_IS_SITE_VIEW( view ), NULL );

	return view->priv->site;
}

void screem_site_view_set_site( ScreemSiteView *view, ScreemSite *site )
{
	g_return_if_fail( SCREEM_IS_SITE_VIEW( view ) );
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	
	g_object_set( G_OBJECT( view ), "site", site, NULL );
}

