/*  Screem:  screem-spell.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <ctype.h>

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <glade/glade.h>

#include <gtk/gtk.h>

#include <gconf/gconf-client.h>

#ifdef HAVE_ENCHANT
#include <enchant/enchant.h>
#endif

#include "screem-spell.h"
#include "screem-page.h"

#include "support.h"

#include "screemmarshal.h"

static void screem_spell_class_init( ScreemSpellClass *klass );
static void screem_spell_init( ScreemSpell *spell );
static void screem_spell_finalize( GObject *object );

typedef struct ScreemSpellWord {
	gchar *word;
	gint pos;
} ScreemSpellWord;


struct ScreemSpellPrivate {
	GList *words;

	GtkListStore *store;
#ifdef HAVE_ENCHANT
	EnchantBroker *broker;
	EnchantDict *dict;
	EnchantDict *pwl;
#else
	/* pid of ispell process */
	gint pid;

	gint inpipe[ 2 ];
	gint outpipe[ 2 ];
#endif

	GConfClient *client;
	gboolean ignore_tags;
	gboolean ignore_uppercase;
	gboolean ignore_numbers;
	GSList *notifies;
};

static const gchar *ispell_command = "ispell";
static gchar *const ispell_command_args[ 16 ] = {
	"screem-spellchecker",
	"-a",
	NULL
};

enum { ACCEPT = 0, REPLACE, INSERT, INSERTCASE, SKIP };


enum {
	HIGHLIGHT,
	LAST_SIGNAL
};
static guint screem_spell_signals[LAST_SIGNAL] = { 0 };


static ScreemSpellWord* screem_spell_word_new( const gchar *text, gint pos );
static void screem_spell_word_destroy( ScreemSpellWord *word );

static void screem_spell_split_text( ScreemSpell *spell, const gchar *text );
static gboolean screem_spell_check( ScreemSpell *spell,
				    const gchar *word,
				    gchar **buffer );


ScreemSpell* screem_spell_new()
{
	ScreemSpell *spell;
	GType type;
	
	type = screem_spell_get_type();

	spell = SCREEM_SPELL( g_object_new( type, NULL ) );

	return spell;
}


gboolean screem_spell_check_word( ScreemSpell *spell, const gchar *word )
{
	return screem_spell_check( spell, word, NULL );
}


static void dict_describe_cb( const char * const lang_tag,
		const char * const provider_name,
		const char * const provider_desc,
		const char * const provider_file,
		void * user_data )
{
	GtkListStore *store;
	GtkTreeIter tit;

	store = GTK_LIST_STORE( user_data );

	gtk_list_store_insert_with_values( store, &tit, G_MAXINT,
			0, lang_tag, -1 );
}

static void screem_spell_ignore_tag_toggle( GtkToggleButton *button,
		ScreemSpell *spell )
{
	ScreemSpellPrivate *priv;
	gboolean flag;

	priv = spell->priv;
	flag = gtk_toggle_button_get_active( button );
	
	gconf_client_set_bool( priv->client,
			"/apps/screem/spell/ignore_tags", flag,
			NULL );
}

static void screem_spell_ignore_uppercase_toggle( GtkToggleButton *button,
		ScreemSpell *spell )
{
	ScreemSpellPrivate *priv;
	gboolean flag;

	priv = spell->priv;
	flag = gtk_toggle_button_get_active( button );
	
	gconf_client_set_bool( priv->client,
			"/apps/screem/spell/ignore_uppercase", flag,
			NULL );
}

static void screem_spell_ignore_numbers_toggle( GtkToggleButton *button,
		ScreemSpell *spell )
{
	ScreemSpellPrivate *priv;
	gboolean flag;

	priv = spell->priv;
	flag = gtk_toggle_button_get_active( button );
	
	gconf_client_set_bool( priv->client,
			"/apps/screem/spell/ignore_numbers", flag,
			NULL );
}

void screem_spell_check_interactive( ScreemSpell *spell )
{
	ScreemSpellPrivate *priv;
	GladeXML *xml;
	GtkWidget *widget;
	gpointer data;
	GList *list;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	ScreemPage *page;
	gchar *text;
	GtkTextBuffer *tbuffer;
	GtkTextIter it;
	GtkTextIter eit;
	GtkWidget *dialog;
	gint offset;

	GtkListStore *store;
	
	g_return_if_fail( SCREEM_IS_SPELL( spell ) );
	
	data = g_object_get_data( G_OBJECT( spell ), "page" );
	
	if( ! data )
		return;
	
	priv = spell->priv;
	
	page = SCREEM_PAGE( data );
	tbuffer = GTK_TEXT_BUFFER( page );
	text = screem_page_get_data( page );
	screem_spell_split_text( spell, text );
	g_free( text );

	xml = glade_xml_new( GLADE_PATH"/screem.glade",
			"spellcheck_dialog", NULL );
	
	widget = glade_xml_get_widget( xml, "spellaltlist" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( spell->priv->store ) );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Alternatives" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 0, NULL );

	/* config tab */

	/* ignore */
	widget = glade_xml_get_widget( xml, "ignore_markup" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
			priv->ignore_tags );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_spell_ignore_tag_toggle ),
			spell );
	
	widget = glade_xml_get_widget( xml, "ignore_upper" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
			priv->ignore_uppercase );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_spell_ignore_uppercase_toggle ),
			spell );
	
	widget = glade_xml_get_widget( xml, "ignore_numbers" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
			priv->ignore_numbers );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			G_CALLBACK( screem_spell_ignore_numbers_toggle ),
			spell );

	/* dictionary */
	widget = glade_xml_get_widget( xml, "default_dict" );
	gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
	store = GTK_LIST_STORE( gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) ) );
#ifdef HAVE_ENCHANT
	enchant_broker_list_dicts( priv->broker,
			dict_describe_cb, store );
#endif
	dialog = glade_xml_get_widget( xml, "spellcheck_dialog" );
	
	glade_xml_signal_autoconnect( xml );

	offset = 0;
	list = spell->priv->words;
	while( dialog ) {
		gchar *buffer;
		GSList *alts = NULL;
		GSList *alist;
		gchar c;
		gchar *curword;
		gboolean replace = FALSE;
		ScreemSpellWord *word;

		if( list ) {
			word = (ScreemSpellWord*)list->data;
			list = list->next;
		} else {
			GtkWidget *notebook;
			notebook = glade_xml_get_widget( xml, "check_table" );
			gtk_widget_set_sensitive( notebook, FALSE );
			word = NULL;
			gtk_dialog_run( GTK_DIALOG( dialog ) );
			gtk_widget_destroy( dialog );
			dialog = NULL;
		}
	
		if( ! word ) {
			continue;
		}
		
		if( ! screem_spell_check( spell, word->word, &buffer ) ) {
#ifdef HAVE_ENCHANT
			char **suggestions;
			size_t num;
			
			suggestions = enchant_dict_suggest( priv->dict,
					word->word, 
					strlen( word->word ),
					&num );
			while( num > 0 ) {
				num --;
				alts = g_slist_prepend( alts, suggestions[ num ] );
			}
#else
			/* buffer format:
			   <symbol> <word> <number> <word number>: <word>, ...
			*/
			if( buffer ) {
				gchar tmp[ BUFSIZ ];
				gchar *temp;
				gint num;
				
				sscanf( buffer, "%c %s %d", &c, tmp, &num );
				temp = strchr( buffer, ':' );
				temp += 2;
				while( num ) {
					curword = temp;
					temp = strchr( curword, ',' );
					if( temp ) {
						*temp = '\0';
						temp += 2;
					} else {
						temp = strpbrk( curword, "\r\n" );
						if( temp )
							*temp = '\0';
					}
					alts = g_slist_append( alts, curword );
					num --;
				}
			}
#endif
			/* fill in entries in dialog */
			widget = glade_xml_get_widget( xml, "spellword" );
			gtk_entry_set_text( GTK_ENTRY( widget ), word->word );
			gtk_list_store_clear( spell->priv->store );
			for( alist = alts; alist; alist = alist->next ) {
				GtkTreeIter it;
				gtk_list_store_append( spell->priv->store,
						       &it );
				gtk_list_store_set( spell->priv->store,
						    &it,
						    0, alist->data,
						    1, spell,
						    -1 );
			}
#ifdef HAVE_ENCHANT
			g_slist_foreach( alts, (GFunc)g_free, NULL );
			g_free( suggestions );				
#endif
			g_slist_free( alts );
			/* highlight the word */
			g_signal_emit( G_OBJECT( spell ),
				       screem_spell_signals[ HIGHLIGHT ], 0,
				       word->pos + offset, 
				       word->pos + offset + 
				       strlen( word->word ) );
			/* run dialog */
			c = '\0';
			switch( gtk_dialog_run( GTK_DIALOG( dialog ) ) ) {
			case ACCEPT:
				c = '@';
				break;
			case REPLACE:
				replace = TRUE;
				break;
			case INSERT:
				c = '+';
				replace = TRUE;
				break;
			case INSERTCASE:
				c = '&';
				replace = TRUE;
				break;
			case SKIP:
				break;
			default:
				gtk_widget_destroy( dialog );
				dialog = NULL;
				break;
			}
			if( ! dialog )
				break;
			curword = (gchar*)
				gtk_entry_get_text( GTK_ENTRY( widget ) );

			/* add to dict */
			if( c != '\0' ) {
#ifdef HAVE_ENCHANT
				if( c == '@' ) {
					enchant_dict_add_to_session( priv->dict, curword, strlen( curword ) );
				} else {
					enchant_dict_add_to_pwl( priv->pwl, curword, strlen( curword ) );
				}
#else
				/* FIXME: remove command characters from
				   curword */
				gchar *temp = g_strdup_printf( "%c%s\n",
							       c, curword );
				write( spell->priv->outpipe[ 1 ],
				       temp, strlen( temp ) );
				g_free( temp );
#endif
			}
			
			if( replace ) {
				gint len = strlen( word->word );

				gtk_text_buffer_get_iter_at_offset( tbuffer,
								    &it,
								    word->pos +
								    offset );
				gtk_text_buffer_get_iter_at_offset( tbuffer,
								    &eit,
								    word->pos +
								    offset +
								    len );

				gtk_text_buffer_delete( tbuffer, &it, &eit );
				gtk_text_buffer_insert( tbuffer, &it,
							curword, 
							strlen( curword ) );
				offset += strlen( curword ) - len;
			}
		}
	}

	g_object_unref( xml );
}

void screem_spell_alt_selected( GtkTreeView *view, GtkTreePath *path,
				GtkTreeViewColumn *column, gpointer data )
{
	GladeXML *xml;
	GtkWidget *widget;
	GtkTreeIter it;
	GtkTreeModel *model;
	GValue value = {0};
	const gchar *word;

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
	if( gtk_tree_model_get_iter( GTK_TREE_MODEL( model ), &it, path ) ) {
		gtk_tree_model_get_value( GTK_TREE_MODEL( model ), &it, 0,
					  &value );
		word = g_value_get_string( &value );
		
		xml = glade_get_widget_tree( GTK_WIDGET( view ) );
		widget = glade_xml_get_widget( xml, "spellword" );
		gtk_entry_set_text( GTK_ENTRY( widget ), word );
		
		g_value_unset( &value );
	}
}

void screem_spell_accept( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), ACCEPT );
}
void screem_spell_replace( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), REPLACE );
}
void screem_spell_insert( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), INSERT );
}
void screem_spell_insertcase( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), INSERTCASE );
}
void screem_spell_skip( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), SKIP );
}

/* static stuff */
static ScreemSpellWord* screem_spell_word_new( const gchar *text, gint pos )
{
	ScreemSpellWord *word;

	word = g_new0( ScreemSpellWord, 1 );
	word->word = g_strdup( text );
	word->pos = pos;

	return word;
}

static void screem_spell_word_destroy( ScreemSpellWord *word )
{
	g_free( word->word );
	g_free( word );
}


/* splits text up into its individual words, ignoring html tags,
   FIXME: should really handle any text within tag attributes such as alt
   in an <img> tag */
static void screem_spell_split_text( ScreemSpell *spell, const gchar *text )
{
	ScreemSpellPrivate *priv;
	GList *list;
	gint len;
	const gchar *start;
	ScreemSpellWord *w;
	gchar *word;
	gint wordlen;
	gunichar c;
	const gchar *tstart;
	gboolean containsnum;
	gboolean allupper;
	gboolean newword;
	
	g_return_if_fail( SCREEM_IS_SPELL( spell ) );
	
	priv = spell->priv;
	
	g_list_foreach( spell->priv->words,
			(GFunc)screem_spell_word_destroy,
			NULL );
	g_list_free( spell->priv->words );
	spell->priv->words = NULL;

	list = NULL;
	len = strlen( text );

	c = ' ';
	tstart = text;
	containsnum = FALSE;
	allupper = FALSE;
	for( start = text; c != '\0'; text = g_utf8_next_char( text ) ) {
		c = g_utf8_get_char( text );

		newword = ( g_unichar_isspace( c ) ||
			( g_unichar_ispunct( c ) && c != '\'' ) );
	
		if( start == text ) {
			allupper = g_unichar_isupper( c );
		} else if( ! newword ) {
			allupper &= g_unichar_isupper( c );
		}
		if( g_unichar_isdigit( c ) ) {
			containsnum = TRUE;
		}

		if( newword ) {
			word = NULL;
			if( start != text ) {
				wordlen = text - start;
				word = g_strndup( start, wordlen );
			}
			if( word && priv->ignore_numbers && containsnum ) {
				g_free( word );
				word = NULL;
			}
			if( word && priv->ignore_uppercase && allupper ) {
				g_free( word );
				word = NULL;
			}
			if( word ) {
				w = screem_spell_word_new( word, 
						start - tstart );
				list = g_list_prepend( list, w );
				g_free( word );
			}
			start = g_utf8_next_char( text );
			containsnum = FALSE;
		}
		if( priv->ignore_tags ) {
			if( c == '<' ) {
				while( c != '>' && c != '\0' ) {
					text = g_utf8_next_char( text );
					c = g_utf8_get_char( text );
				}
				start = g_utf8_next_char( text );
				containsnum = FALSE;
			} else if( c == '&' ) {
				/* we stopped at a char encoding element */
				while( c != '>' && c != '\0' ) {
					text = g_utf8_next_char( text );
					c = g_utf8_get_char( text );
				}
				start = g_utf8_next_char( text );
				containsnum = FALSE;
			}
		}
	}
	spell->priv->words = g_list_reverse( list );
}

static gboolean screem_spell_check( ScreemSpell *spell,
				    const gchar *word,
				    gchar **buffer )
{
	ScreemSpellPrivate *priv;
	gboolean success = FALSE;
#ifndef HAVE_ENCHANT
	gchar *temp;
	gint size;
	gchar buf[ BUFSIZ ];
#endif

	g_return_val_if_fail( SCREEM_IS_SPELL( spell ), TRUE );
	
	priv = spell->priv;
#ifdef HAVE_ENCHANT

	success = ! enchant_dict_check( priv->dict, word, strlen( word ) );
	if( ! success ) {
		success = ! enchant_dict_check( priv->pwl, word,
				strlen( word ) );
	}
#else
	if( buffer )
		*buffer = NULL;

	temp = g_strconcat( "^", word, "\n", NULL );

	if( write( spell->priv->outpipe[ 1 ], temp, strlen( temp ) ) < 0 ) {
		g_free( temp );
		return success;
	}

	g_free( temp );

	if( ( size = read( spell->priv->inpipe[ 0 ], buf, BUFSIZ ) ) < 0 ) {
		return success;
	}

	buf[ size ] = '\0';

	switch( buf[ 0 ] ) {
	case '*':
		/* word is ok */
		success = TRUE;
		break;
	case '+':
		/* word found */
		success = TRUE;
		break;
	case '-':
		/* found as compound */
		success = TRUE;
		break;
	case '\n':
		/* ignore the word */
		success = TRUE;
		break;
	case '#':
		/* not found, no idea what it is */
		break;
	case '?':
		/* not found, but we have guesses */
	case '&':
		/* not found, but here is some near misses */
		if( buffer )
			*buffer = g_strdup( buf );
		break;
	default:
		/* probably a number */
		success = TRUE;
		break;
	}
#endif

	return success;
}

static void screem_spell_ignore_tags_notify( GConfClient *client,
		guint cnxn_id, GConfEntry *entry, gpointer data )
{
	ScreemSpell *spell;
	ScreemSpellPrivate *priv;
	gboolean value;
	
	spell = SCREEM_SPELL( data );
	priv = spell->priv;

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		value = gconf_value_get_bool( entry->value );
		priv->ignore_tags = value;
	}
}

static void screem_spell_ignore_uppercase_notify( GConfClient *client,
		guint cnxn_id, GConfEntry *entry, gpointer data )
{
	ScreemSpell *spell;
	ScreemSpellPrivate *priv;
	gboolean value;
	
	spell = SCREEM_SPELL( data );
	priv = spell->priv;

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		value = gconf_value_get_bool( entry->value );
		priv->ignore_uppercase = value;
	}
}

static void screem_spell_ignore_numbers_notify( GConfClient *client,
		guint cnxn_id, GConfEntry *entry, gpointer data )
{
	ScreemSpell *spell;
	ScreemSpellPrivate *priv;
	gboolean value;
	
	spell = SCREEM_SPELL( data );
	priv = spell->priv;

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		value = gconf_value_get_bool( entry->value );
		priv->ignore_numbers = value;
	}
}



/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_spell_class_init( ScreemSpellClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_spell_finalize;


	screem_spell_signals[HIGHLIGHT] = 
		g_signal_new( "highlight",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemSpellClass, highlight ),
			      NULL, NULL,
			      screem_marshal_VOID__INT_INT,
			      G_TYPE_NONE, 2,
			      G_TYPE_INT,
			      G_TYPE_INT );
}

static void screem_spell_init( ScreemSpell *spell )
{
	ScreemSpellPrivate *priv;
	const gchar *lang;
	const gchar *const *langs;
	GConfClient *client;
	gchar *glang;
	guint handle;
	
	priv = spell->priv = g_new0( ScreemSpellPrivate, 1 );

	priv->store = gtk_list_store_new( 2,
			G_TYPE_STRING, G_TYPE_POINTER, NULL );
	priv->client = client = gconf_client_get_default();
	
#ifdef HAVE_ENCHANT
	priv->broker = enchant_broker_init();
	
	langs = g_get_language_names();
	if( ! langs ) {
		lang = "en";
	} else {
		for( lang = *langs; lang; lang = *(langs++) ) {
			if( enchant_broker_dict_exists( priv->broker,
						lang ) ) {
				break;
			}
		}
		if( ! lang ) {
			lang = "en";
		}
	}
	glang = gconf_client_get_string( client,
			"/apps/screem/spell/dictionary",
			NULL );
	if( glang && *glang != '\0' &&
			enchant_broker_dict_exists( priv->broker,
						glang ) ) {
		lang = glang;
	}
	
	priv->dict = enchant_broker_request_dict( priv->broker, lang );
	priv->pwl = enchant_broker_request_pwl_dict( priv->broker,
			lang );
	g_free( glang );
#else
	/* spawn an ispell process */
	pipe( spell->priv->inpipe );
	pipe( spell->priv->outpipe );

	spell->priv->pid = fork();

	switch( spell->priv->pid ) {
	case -1:
		/* error */
		break;
	case 0:
		/* child process */
		close( 0 );
		dup( spell->priv->outpipe[ 0 ] );
		close( spell->priv->outpipe[ 1 ] );
		close( 1 );
		dup( spell->priv->inpipe[ 1 ] );
		close( spell->priv->inpipe[ 0 ] );
		execvp( ispell_command, ispell_command_args );
		_exit( 1 );
		break;
	default:
		/* parent process */
		close( spell->priv->outpipe[ 0 ] );
		close( spell->priv->inpipe[ 1 ] );

		/* read in the ispell banner */
		{
			gchar buffer[ BUFSIZ + 1 ];
			read( spell->priv->inpipe[ 0 ], buffer, BUFSIZ );
		}
		break;
	}
#endif
	priv->ignore_tags = gconf_client_get_bool( client,
			"/apps/screem/spell/ignore_tags",
			NULL );	
	handle = gconf_client_notify_add( client,
			"/apps/screem/spell/ignore_tags",
			screem_spell_ignore_tags_notify,
			spell, NULL, NULL );
	priv->notifies = g_slist_prepend( priv->notifies,
			GUINT_TO_POINTER( handle ) );

	priv->ignore_uppercase = gconf_client_get_bool( client,
			"/apps/screem/spell/ignore_uppercase",
			NULL );
	handle = gconf_client_notify_add( client,
			"/apps/screem/spell/ignore_uppercase",
			screem_spell_ignore_uppercase_notify,
			spell, NULL, NULL );
	priv->notifies = g_slist_prepend( priv->notifies,
			GUINT_TO_POINTER( handle ) );
			
	priv->ignore_numbers = gconf_client_get_bool( client,
			"/apps/screem/spell/ignore_numbers",
			NULL );
	handle = gconf_client_notify_add( client,
			"/apps/screem/spell/ignore_numbers",
			screem_spell_ignore_numbers_notify,
			spell, NULL, NULL );
	priv->notifies = g_slist_prepend( priv->notifies,
			GUINT_TO_POINTER( handle ) );
}

static void screem_spell_finalize( GObject *object )
{
	ScreemSpell *spell;
	ScreemSpellPrivate *priv;
	gint status;
	GSList *tmp;
	
	spell = SCREEM_SPELL( object );
	priv = spell->priv;
	
#ifdef HAVE_ENCHANT
	enchant_broker_free_dict( priv->broker, priv->dict );
	enchant_broker_free( priv->broker );
#else
	kill( priv->pid, SIGTERM );
	waitpid( priv->pid, &status, 0 );
	close( priv->inpipe[ 0 ] );
	close( priv->outpipe[ 1 ] );
#endif

	for( tmp = priv->notifies; tmp; tmp = tmp->next ) {
		gconf_client_notify_remove( priv->client,
				GPOINTER_TO_UINT( tmp->data ) );
	}
	g_slist_free( priv->notifies );
	
	if( priv->client ) {
		g_object_unref( G_OBJECT( priv->client ) );
	}
	
	if( spell->priv->words ) {
		g_list_foreach( spell->priv->words,
				(GFunc)screem_spell_word_destroy,
				NULL );
		g_list_free( spell->priv->words );
	}

	g_object_unref( spell->priv->store );

	g_free( spell->priv );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

GType screem_spell_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSpellClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_spell_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSpell ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_spell_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemSpell",
					       &info, 0 );
	}

	return type;
}
