/* This file is part of Strigi Desktop Search
 *
 * Copyright (C) 2006 Jos van den Oever <jos@vandenoever.info>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include <kdebug.h>
#include <kinstance.h>
#include <klocale.h>
#include <kurl.h>
#include <kmimetype.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <strigi/archivereader.h>
#include <vector>

#include "jstream.h"
#include <strigi/fileinputstream.h>

// http://www.heise.de/ct/english/01/05/242/

using namespace KIO;
using namespace std;
using namespace Strigi;

class QFileStreamOpener : public StreamOpener {
public:
    ~QFileStreamOpener() {}
    StreamBase<char>* openStream(const string& url);
    int stat(const string& url, EntryInfo& e);
};
StreamBase<char>*
QFileStreamOpener::openStream(const string& url) {
    StreamBase<char>* stream = new FileInputStream(QString(url.c_str()));
    if (stream->status() != Ok) {
        delete stream;
        stream = 0;
    }
    return stream;
}
int
QFileStreamOpener::stat(const string& url, EntryInfo& e) {
    QFileInfo f(url.c_str());
    if (!f.exists()) {
        return -1;
    }
    e.type = EntryInfo::Unknown;
    if (f.isFile()) {
        e.type = EntryInfo::File;
    }
    if (f.isDir()) {
        e.type = EntryInfo::Dir;
    }
    e.size = f.size();
    e.mtime = f.lastModified().toTime_t();
    QByteArray filename = f.fileName().utf8();
    e.filename.assign((const char*)filename, filename.size());

    return 0;
}

kio_jstreamProtocol::kio_jstreamProtocol(const QCString &pool_socket,
        const QCString &app_socket)
    : SlaveBase("kio_jstream", pool_socket, app_socket) {
    kdDebug(7101) << "kio_jstreamProtocol::kio_jstreamProtocol()" << endl;
    reader = new ArchiveReader();
    opener = new QFileStreamOpener();
    reader->addStreamOpener(opener);
}
kio_jstreamProtocol::~kio_jstreamProtocol() {
    kdDebug(7101) << "kio_jstreamProtocol::~kio_jstreamProtocol()" << endl;
    delete reader;
    delete opener;
}
void
kio_jstreamProtocol::get(const KURL& url ) {
    kdDebug(7101) << "kio_jstream::get(const KURL& url)" << endl ;
    string path = (const char*)url.path().utf8();
    if (path.size() && path[path.size()-1] == '/') path.resize(path.size()-1);

    Strigi::StreamBase<char>* in = reader->openStream(path);
    if (in == 0) {
        error( KIO::ERR_CANNOT_OPEN_FOR_READING, path );
        return;
    }

    const char* buf;
    int32_t nread = in->read(buf, 1024, 0);
    QByteArray d;
    if (nread > 0) {
        mimeType(KMimeType::findByContent(d)->name());
        d.setRawData(buf, nread);
        data(d);
        d.resetRawData(buf, nread);
        nread = in->read(buf, 1, 0);
    }
    while (nread > 0) {
        d.setRawData(buf, nread);
        data(d);
        d.resetRawData(buf, nread);
        nread = in->read(buf, 1, 0);
    }
    data(QByteArray()); // empty array means we're done sending the data
    finished();
}
UDSEntry
QFileInfoToUDSEntry(const QFileInfo& fi) {
    UDSEntry entry;
    UDSAtom atom;
    atom.m_uds = KIO::UDS_NAME;
    atom.m_str = fi.fileName();
    entry.append(atom);
    atom.m_uds = KIO::UDS_FILE_TYPE;
    atom.m_long = 0;
    if (fi.isDir()) {
        atom.m_long |= S_IFDIR;
    } else if (fi.isFile()) {
        atom.m_long |= S_IFREG;
    }
    entry.append(atom);
    atom.m_uds = KIO::UDS_SIZE;
    atom.m_long = fi.size();
    entry.append(atom);
    return entry;
}
UDSEntry
makeDirEntry(const QString& n, int size) {
    UDSEntry entry;
    UDSAtom atom;
    atom.m_uds = KIO::UDS_NAME;
    atom.m_str = n;
    entry.append(atom);
    atom.m_uds = KIO::UDS_FILE_TYPE;
    atom.m_long = S_IFDIR;
    entry.append(atom);
    atom.m_uds = KIO::UDS_SIZE;
    atom.m_long = size;
    entry.append(atom);
    return entry;
}
UDSEntry
kio_jstreamProtocol::statQFileInfo(const QFileInfo& f) const {
    UDSEntry entry;
    if (f.isFile()) {
        string path = (const char*)f.filePath().utf8();
        if (path.size() && path[path.size() - 1] == '/') {
            path.resize(path.size() - 1);
        }
        DirLister dl = reader->dirEntries(path);
        EntryInfo e;
        if (dl.nextEntry(e)) {
            entry = makeDirEntry(f.fileName(), f.size());
            return entry;
        }
    }
    entry = QFileInfoToUDSEntry(f);
    return entry;
}
UDSEntry
entryInfoToUDSEntry(const EntryInfo& e) {
    UDSEntry entry;
    UDSAtom atom;
    atom.m_uds = KIO::UDS_NAME;
    atom.m_str = e.filename;
    entry.append(atom);
    atom.m_uds = KIO::UDS_FILE_TYPE;
    atom.m_long = 0;
    if (e.type & EntryInfo::Dir) {
        atom.m_long |= S_IFDIR;
    } else if (e.type & EntryInfo::File) {
        atom.m_long |= S_IFREG;
    }
    entry.append(atom);
    atom.m_uds = KIO::UDS_SIZE;
    atom.m_long = e.size;
    entry.append(atom);
    return entry;
}
void
kio_jstreamProtocol::stat(const KURL& url)
{
    UDSEntry entry;
    kdDebug(7101) << "kio_jstream::stat(const KURL& url)" << endl ;
    QFileInfo f(url.path());
    if (f.exists()) {
        entry = statQFileInfo(f);
        statEntry(entry);
        finished();
        return;
    }
    string path = (const char*)url.path().utf8();
    if (path.size() && path[path.size()-1] == '/') path.resize(path.size()-1);
    EntryInfo e;
    if (reader->stat(path, e) != 0) {
        error( KIO::ERR_CANNOT_OPEN_FOR_READING, path );
        return;
    }
        
    entry = entryInfoToUDSEntry(e);
    statEntry(entry);
    finished();
}
void
kio_jstreamProtocol::mimetype(const KURL & /*url*/)
{
    kdDebug(7101) << "kio_jstream::mimetype(const KURL& url)" << endl ;
    finished();
}
void
kio_jstreamProtocol::listDir(const KURL& url) {
    UDSEntry entry;
    QFileInfo f(url.path());
    if (f.isDir()) {
        QDir d(url.path());
        d.setFilter(QDir::Dirs | QDir::Files | QDir::Hidden | QDir::NoSymLinks);
        QFileInfoListIterator it(*d.entryInfoList());
        QFileInfo* fi;
        while ((fi = it.current()) != 0) {
            entry = statQFileInfo(*fi);
            listEntry(entry, false);
            ++it;
        }
        listEntry(entry, true);
        finished();
        return;
    }

    kdDebug(7101) << "kio_jstream::listDir(const KURL& url)" << endl ;
    string path = (const char*)url.path().utf8();
    if (path.size() && path[path.size() - 1] == '/') {
        path.resize(path.size() - 1);
    }
    DirLister dl = reader->dirEntries(path);
    EntryInfo e;
    while (dl.nextEntry(e)) {
        entry = entryInfoToUDSEntry(e);
        listEntry(entry, false);
    }
    listEntry(entry, true);
    finished();
}

extern "C"
{
    int kdemain(int argc, char **argv)
    {
        KInstance instance( "kio_jstream" );
        
        kdDebug(7101) << "*** Starting kio_jstream " << endl;
        
        if (argc != 4) {
            kdDebug(7101) << "Usage: kio_jstream  protocol domain-socket1 domain-socket2" << endl;
            exit(-1);
        }
        
        kio_jstreamProtocol slave(argv[2], argv[3]);
        slave.dispatchLoop();
        
        kdDebug(7101) << "*** kio_jstream Done" << endl;
        return 0;
    }
} 
