#!/usr/bin/env python

from translate.storage import po
from translate.misc import wStringIO

class TestPO:
    def poparse(self, posource):
        """helper that parses po source without requiring files"""
        dummyfile = wStringIO.StringIO(posource)
        pofile = po.pofile(dummyfile)
        return pofile

    def poregen(self, posource):
        """helper that converts po source to pofile object and back"""
        return str(self.poparse(posource))

    def test_simpleentry(self):
        """checks that a simple po entry is parsed correctly"""
        posource = '#: test.c\nmsgid "test"\nmsgstr "rest"\n'
        pofile = self.poparse(posource)
        assert len(pofile.poelements) == 1
        thepo = pofile.poelements[0]
        assert thepo.getsources() == ["test.c"]
        assert po.unquotefrompo(thepo.msgid) == "test"
        assert po.unquotefrompo(thepo.msgstr) == "rest"

    def test_combine_msgidcomments(self):
        """checks that we don't get duplicate msgid comments"""
        posource = 'msgid "test me"\nmsgstr ""'
        pofile = self.poparse(posource)
        thepo = pofile.poelements[0]
        thepo.msgidcomments.append('"_: first comment\\n"')
        thepo.msgidcomments.append('"_: second comment\\n"')
        regenposource = str(pofile)
        assert regenposource.count("_:") == 1

    def test_merge_duplicates(self):
        """checks that merging duplicates works"""
        posource = '#: source1\nmsgid "test me"\nmsgstr ""\n\n#: source2\nmsgid "test me"\nmsgstr ""\n'
        pofile = self.poparse(posource)
        assert len(pofile.poelements) == 2
        pofile.removeduplicates("merge")
        assert len(pofile.poelements) == 1
        assert pofile.poelements[0].getsources() == ["source1", "source2"]

    def test_merge_blanks(self):
        """checks that merging adds msgid_comments to blanks"""
        posource = '#: source1\nmsgid ""\nmsgstr ""\n\n#: source2\nmsgid ""\nmsgstr ""\n'
        pofile = self.poparse(posource)
        assert len(pofile.poelements) == 2
        pofile.removeduplicates("merge")
        assert len(pofile.poelements) == 2
        assert po.getunquotedstr(pofile.poelements[0].msgidcomments) == "_: source1\\n"
        assert po.getunquotedstr(pofile.poelements[1].msgidcomments) == "_: source2\\n"

    def test_keep_blanks(self):
        """checks that keeping keeps blanks and doesn't add msgid_comments"""
        posource = '#: source1\nmsgid ""\nmsgstr ""\n\n#: source2\nmsgid ""\nmsgstr ""\n'
        pofile = self.poparse(posource)
        assert len(pofile.poelements) == 2
        pofile.removeduplicates("keep")
        assert len(pofile.poelements) == 2
        # check we don't add msgidcomments
        assert po.getunquotedstr(pofile.poelements[0].msgidcomments) == ""
        assert po.getunquotedstr(pofile.poelements[1].msgidcomments) == ""

    def test_getunquotedstr(self):
        """checks that getunquotedstr works as advertised"""
        assert po.getunquotedstr(['"First line\nSecond line"'], includeescapes=False) == "First line\nSecond line"

    def test_parse_source_string(self):
        """parse a string"""
        posource = '#: test.c\nmsgid "test"\nmsgstr "rest"\n'
        pofile = po.pofile(posource)
        assert len(pofile.poelements) == 1

    def test_parse_file(self):
        """test parsing a real file"""
        posource = '#: test.c\nmsgid "test"\nmsgstr "rest"\n'
        pofile = self.poparse(posource)
        assert len(pofile.poelements) == 1

    def test_output_str_unicode(self):
        """checks that we can str(element) which is in unicode"""
        posource = u'''#: nb
msgid "Norwegian Bokm\xe5l"
msgstr ""
'''
        pofile = po.pofile(wStringIO.StringIO(posource.encode("UTF-8")), encoding="UTF-8")
        assert len(pofile.poelements) == 1
        print str(pofile)
        thepo = pofile.poelements[0]
        assert str(thepo) == posource.encode("UTF-8")
        # extra test: what if we set the msgid to a unicode? this happens in prop2po etc
        thepo.msgid = po.quoteforpo(u"Norwegian Bokm\xe5l")
        assert str(thepo) == posource.encode("UTF-8")
        # Now if we set the msgstr to Unicode
        # this is an escaped half character (1/2)
        halfstr = "\xbd ...".decode("latin-1")
        halfmsgstr = 'msgstr "%s"' % halfstr
        thepo.msgstr = po.quoteforpo(halfstr)
        assert halfmsgstr in str(thepo).decode("UTF-8")
        thepo.msgstr = po.quoteforpo(halfstr.encode("UTF-8"))
        assert halfmsgstr.encode("UTF-8") in thepo.getsource()

