/*
 * Copyright © 2014 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Thomas Voß <thomas.voss@canonical.com>
 */

#ifndef CORE_TRUST_DBUS_AGENT_H_
#define CORE_TRUST_DBUS_AGENT_H_

#include <core/trust/agent.h>
#include <core/trust/dbus_agent.h>

#include <core/trust/dbus/codec.h>
#include <core/trust/remote/helpers.h>

#include <core/dbus/macros.h>
#include <core/dbus/object.h>

namespace core
{
namespace trust
{
namespace dbus
{
struct Agent
{
    static const std::string& name()
    {
        static const std::string s
        {
            "core.trust.dbus.Agent"
        };
        return s;
    }

    struct Errors
    {
        struct CouldNotDetermineConclusiveAnswer : public std::runtime_error
        {
            static constexpr const char* name
            {
                "core.trust.dbus.Agent.Errors.CouldNotDetermineConclusiveAnswer"
            };

            CouldNotDetermineConclusiveAnswer()
                : std::runtime_error("Could not determine conclusive answer to trust request.")
            {
            }
        };
    };

    // Just for scoping purposes
    struct Methods
    {
        // And thus not constructible
        Methods() = delete;

        DBUS_CPP_METHOD_WITH_TIMEOUT_DEF(AuthenticateRequestWithParameters, Agent, 120000)
    };

    class Stub : public core::trust::Agent
    {
    public:
        Stub(const core::dbus::Object::Ptr& object)
            : object{object}
        {
        }

        Request::Answer authenticate_request_with_parameters(const RequestParameters& parameters) override
        {
            auto result = object->transact_method
                    <
                        Methods::AuthenticateRequestWithParameters,
                        core::trust::Request::Answer
                    >(parameters);

            if (result.is_error()) throw std::runtime_error
            {
                result.error().print()
            };

            return result.value();
        }

    private:
        core::dbus::Object::Ptr object;
    };

    class Skeleton : public core::trust::Agent
    {
    public:
        // All creation time parameters go here.
        struct Configuration
        {
            // The object that should implement core.trust.Agent
            core::dbus::Object::Ptr object;
            // The underlying bus instance
            core::dbus::Bus::Ptr bus;
            // The agent implementation
            std::function<core::trust::Request::Answer(const core::trust::Agent::RequestParameters&)> agent;
        };

        Skeleton(const Configuration& config)
            : configuration(config)
        {
            configuration.object->install_method_handler<Methods::AuthenticateRequestWithParameters>([this](const core::dbus::Message::Ptr& in)
            {
                core::trust::Agent::RequestParameters params; in->reader() >> params;

                core::dbus::Message::Ptr reply;
                try
                {
                    reply = core::dbus::Message::make_method_return(in);
                    reply->writer() << authenticate_request_with_parameters(params);
                } catch(const std::exception& e)
                {
                    // TODO(tvoss): we should report the error locally here.
                    reply = core::dbus::Message::make_error(in, Errors::CouldNotDetermineConclusiveAnswer::name, "");
                } catch(...)
                {
                    reply = core::dbus::Message::make_error(in, Errors::CouldNotDetermineConclusiveAnswer::name, "");
                }

                configuration.bus->send(reply);
            });
        }

        Request::Answer authenticate_request_with_parameters(const RequestParameters& request) override
        {
            return configuration.agent(request);
        }

    private:
        // We just store all creation time parameters.
        Configuration configuration;
    };
};
}
}
}

/**
 * @brief create_per_user_agent_for_bus_connection creates a trust::Agent implementation communicating with a remote agent
 * implementation living in the same user session.
 * @param connection An existing DBus connection.
 * @param service The name of the service we are operating for.
 * @throws std::runtime_error in case of issues.
 */
std::shared_ptr<core::trust::Agent> core::trust::dbus::create_per_user_agent_for_bus_connection(
        const std::shared_ptr<core::dbus::Bus>& connection,
        const std::string& service_name)
{
    auto object =
            core::dbus::Service::use_service(connection,"core.trust.dbus.Agent." + service_name)
            ->object_for_path(core::dbus::types::ObjectPath{"/core/trust/dbus/Agent"});

    return std::shared_ptr<core::trust::dbus::Agent::Stub>
    {
        new core::trust::dbus::Agent::Stub{object}
    };
}

#endif // CORE_TRUST_DBUS_AGENT_H_
