# ubuntuone.syncdaemon.config - SyncDaemon config utilities
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""SyncDaemon config"""

from __future__ import with_statement

import os
import functools
import logging

import ConfigParser
from configglue import TypedConfigParser
from xdg.BaseDirectory import (
    load_config_paths,
    save_config_path,
    xdg_data_home,
    xdg_cache_home,
)


CONFIG_FILE = 'syncdaemon.conf'

# sections
THROTTLING = 'bandwidth_throttling'

# get (and  possibly create if don't exists) the user config file
_user_config_path = os.path.join(save_config_path('ubuntuone'), CONFIG_FILE)

# module private config instance.
# this object is the shared config
_user_config = None


def home_dir_parser(value):
    """ Parser for the root_dir and shares_dir options.
    returns the path using user home + value.
    """
    return os.path.expanduser(value)


def xdg_cache_dir_parser(value):
    """ Parser for the data_dir option.
    returns the path using xdg_cache_home + value.
    """
    return os.path.join(xdg_cache_home, value)


def xdg_data_dir_parser(value):
    """ Parser for the data_dir option.
    returns the path using xdg_data_home + value.
    """
    return os.path.join(xdg_data_home, value)


def log_level_parser(value):
    """perser for "logging" module log levels."""
    level = getattr(logging, value, None)
    if level is None:
        # if level don't exists in our custom levels, fallback to DEBUG
        level = getattr(logging, 'DEBUG')
    return level


def throttling_limit_parser(value):
    """parser for throttling limit values, if value < 0 returns None"""
    value = int(value)
    if value < 0:
        return None
    else:
        return value


def get_parsers():
    """returns a list of tuples: (name, parser)"""
    return [('home_dir', home_dir_parser),
            ('xdg_cache', xdg_cache_dir_parser),
            ('xdg_data', xdg_data_dir_parser),
            ('log_level', log_level_parser),
            ('throttling_limit', throttling_limit_parser)]


def get_config_files():
    """ return the path to the config files or and empty list.
    The search path is based on the paths returned by load_config_paths
    but it's returned in reverse order (e.g: /etc/xdg first).
    """
    config_files = []
    for xdg_config_dir in load_config_paths('ubuntuone'):
        config_file = os.path.join(xdg_config_dir, CONFIG_FILE)
        if os.path.exists(config_file):
            config_files.append(config_file)

    # reverse the list as load_config_paths returns the user dir first
    config_files.reverse()
    # if we are running from a branch, get the config file from it too
    config_file = os.path.join('data', CONFIG_FILE)
    if os.path.exists(config_file):
        config_files.append(config_file)
    return config_files


def get_user_config(config_file=_user_config_path):
    """return the shared _Config instance"""
    global _user_config
    if _user_config is None:
        _user_config = _Config(config_file)
    return _user_config


def requires_section(section):
    """decorator to enforce the existence of a section in the config."""
    def wrapper(meth):
        """the wrapper"""
        def wrapped(self, *args, **kwargs):
            """the real thing, wrap the method and do the job"""
            if not self.has_section(section):
                self.add_section(section)
            return meth(self, *args, **kwargs)
        functools.update_wrapper(wrapped, meth)
        return wrapped
    return wrapper


class _Config(TypedConfigParser):
    """Minimal config object to read/write config values
    to the user config file.

    Only supports bandwidth throttling options.

    Ideally TypedConfigParser should implement a write method that converts
    from configglue.attributed.ValueWithAttrs back to str in order to take
    advantage of all the nice tricks of configglue.
    """

    def __init__(self, config_file=_user_config_path):
        """Create the instance, add our custom parsers and
        read the config file
        """
        super(_Config, self).__init__()
        for name, parser in get_parsers():
            self.add_parser(name, parser)
        self.config_file = config_file
        self.read(config_file)
        self.default = self._load_defaults(get_config_files())

    @staticmethod
    def _load_defaults(config_files):
        """load typed defaults from config_files"""
        cp = TypedConfigParser()
        for name, parser in get_parsers():
            cp.add_parser(name, parser)
        cp.read(config_files)
        cp.parse_all()
        return cp

    def save(self):
        """save the config object to disk"""
        with open(self.config_file+'.new', 'w') as fp:
            self.write(fp)
        if os.path.exists(self.config_file):
            os.rename(self.config_file, self.config_file+'.old')
        os.rename(self.config_file+'.new', self.config_file)

    def get_parsed(self, section, option):
        """custom get that fallback to our custom defaults"""
        try:
            value = super(_Config, self).get(section, option)
            # get the parser from the default config
            default = self.default.get(section, option)
            return default.parser(value)
        except ConfigParser.NoOptionError, e:
            return self.default.get(section, option).value

    # throttling section get/set
    @requires_section(THROTTLING)
    def set_throttling(self, enabled):
        self.set(THROTTLING, 'on', str(enabled))

    @requires_section(THROTTLING)
    def set_throttling_read_limit(self, bytes):
        self.set(THROTTLING, 'read_limit', bytes)

    @requires_section(THROTTLING)
    def set_throttling_write_limit(self, bytes):
        self.set(THROTTLING, 'write_limit', bytes)

    @requires_section(THROTTLING)
    def get_throttling(self):
        return self.get_parsed(THROTTLING, 'on')

    @requires_section(THROTTLING)
    def get_throttling_read_limit(self):
        return self.get_parsed(THROTTLING, 'read_limit')

    @requires_section(THROTTLING)
    def get_throttling_write_limit(self):
        return self.get_parsed(THROTTLING, 'write_limit')

