#
# Author: Facundo Batista <facundo@canonical.com>
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Event Queue."""

import logging
import unittest

from ubuntuone.syncdaemon import (
    event_queue,
    filesystem_manager,
)
from contrib.testing import testcase
from twisted.internet import defer


class BaseEQTestCase(testcase.BaseTwistedTestCase):
    """ Setup an EQ for test. """

    def setUp(self):
        """Setup the test."""
        testcase.BaseTwistedTestCase.setUp(self)
        self.fsmdir = self.mktemp('fsmdir')
        self.partials_dir = self.mktemp('partials_dir')
        self.root_dir = self.mktemp('root_dir')
        self.home_dir = self.mktemp('home_dir')
        self.vm = testcase.FakeVolumeManager(self.root_dir)
        self.fs = filesystem_manager.FileSystemManager(self.fsmdir,
                                                       self.partials_dir,
                                                       self.vm)
        self.fs.create(path=self.root_dir,
                       share_id='', is_dir=True)
        self.fs.set_by_path(path=self.root_dir,
                              local_hash=None, server_hash=None)
        self.eq = event_queue.EventQueue(self.fs)
        self.fs.register_eq(self.eq)

        # add a Memento handler to the logger
        self.log_handler = testcase.MementoHandler()
        self.log_handler.setLevel(logging.DEBUG)
        self.eq.log.addHandler(self.log_handler)

    def tearDown(self):
        """Clean up the tests."""
        for listener in self.eq._listeners:
            self.eq.unsubscribe(listener)
        self.eq.shutdown()
        self.rmtree(self.tmpdir)
        testcase.BaseTwistedTestCase.tearDown(self)


class SubscriptionTests(BaseEQTestCase):
    """Test to subscribe and unsubscribe to the EQ."""

    def test_subscription(self):
        """Test that subscription works."""
        # pylint: disable-msg=W0212
        # simple one
        o = object()
        self.eq.subscribe(o)
        self.assertTrue(o in self.eq._listeners)

        # not duplicates
        o = object()
        self.eq.subscribe(o)
        self.eq.subscribe(o)
        self.assertTrue(self.eq._listeners.count(o) == 1)

    def test_unsubscription(self):
        """Test that unsubscription works."""
        # pylint: disable-msg=W0212
        # simple one
        o = object()
        self.eq.subscribe(o)
        self.eq.unsubscribe(o)
        self.assertFalse(o in self.eq._listeners)

        # duplicate ok (two suscs)
        o = object()
        self.eq.subscribe(o)
        self.eq.subscribe(o)
        self.eq.unsubscribe(o)
        self.assertFalse(o in self.eq._listeners)

        # duplicate error (two unsuscs)
        o = object()
        self.eq.subscribe(o)
        self.eq.subscribe(o)
        self.eq.unsubscribe(o)
        self.assertRaises(ValueError, self.eq.unsubscribe, o)

        # indecisive
        o = object()
        self.eq.subscribe(o)
        self.eq.unsubscribe(o)
        self.eq.subscribe(o)
        self.eq.unsubscribe(o)
        self.assertFalse(o in self.eq._listeners)


class PushTests(BaseEQTestCase):
    """Test the event distribution machinery."""

    def test_push_simple(self):
        """Test that events can be pushed (not listening yet)."""
        # not even an event
        self.assertRaises(TypeError, self.eq.push)

        # bad event
        self.assertRaises(event_queue.InvalidEventError,
                          self.eq.push, "no-such-event")

        # not enough or incorrect args for that event
        self.assertRaises(TypeError, self.eq.push, "FS_FILE_MOVE")
        self.assertRaises(TypeError, self.eq.push, "FS_FILE_MOVE", 1, 2, 3)
        self.assertRaises(TypeError,
                      self.eq.push, "FS_FILE_MOVE", 1, 2, path_to=3)
        self.assertRaises(TypeError,
                      self.eq.push, "FS_FILE_MOVE", 1, path_from=2, path_to=3)

        # all ok... args, kwargs, and mixed
        self.eq.push("FS_FILE_MOVE", 1, 2)
        self.eq.push("FS_FILE_MOVE", path_from=1, path_to=2)
        self.eq.push("FS_FILE_MOVE", 1, path_to=2)

    def test_events_kwargs(self):
        """Test that all events are defined correctly with tuples or lists.

        This is to avoid a typical mistake of making it a "(param)", not
        a "(param,)".
        """
        for name, params in event_queue.EVENTS.iteritems():
            self.assertTrue(isinstance(params, (tuple, list)),
                            "%s event has params bad defined!" % name)

    def test_listened_pushs(self):
        """Push events and listem them."""

        # helper class, pylint: disable-msg=C0111
        class Create(object):
            def __init__(self):
                self.a = None
            def handle_FS_FILE_CREATE(self, a):
                self.a = a

        # it get passed!
        c = Create()
        self.eq.subscribe(c)
        self.eq.push("FS_FILE_CREATE", 1)
        self.assertEqual(c.a, 1)
        self.eq.unsubscribe(c)

        # don't get what don't listen
        c = Create()
        self.eq.subscribe(c)
        self.eq.push("FS_FILE_DELETE", 1)
        self.assertEqual(c.a, None)
        self.eq.unsubscribe(c)

    def test_signatures(self):
        """Check that the handle signatures are forced when passing."""

        # helper class, pylint: disable-msg=C0111
        class Create(object):
            def handle_FS_FILE_CREATE(self, a, b): # it should be only 'a' here
                pass

        # it get passed!
        c = Create()
        self.eq.subscribe(c)

        # caught by EQ
        self.assertRaises(TypeError, self.eq.push, "FS_FILE_CREATE")
        self.assertRaises(TypeError, self.eq.push, "FS_FILE_CREATE", 1, 2)

        # approved by EQ, but the listener has a wrong signature
        # this is logged as an error/exception
        self.eq.push("FS_FILE_CREATE", 1)
        self.assertTrue(self.log_handler.check_error('FS_FILE_CREATE',
                                                     'Create object'))

        self.eq.unsubscribe(c)

    def test_log_pushing_data(self):
        """Pushed event and info should be logged."""
        self.eq.push("AQ_QUERY_ERROR", 'item', error='err')
        self.assertTrue(self.log_handler.check_debug(
            "push_event: AQ_QUERY_ERROR, args:('item',), kw:{'error': 'err'}"))

    def test_log_delete_in_info(self):
        """Pushed any deletion event should be logged in info."""
        self.eq.push("FS_DIR_DELETE", 'path')
        self.assertTrue(self.log_handler.check_info(
                        "push_event: FS_DIR_DELETE"))

    def test_log_pushing_private_data(self):
        """SYS_USER_CONNECT event info must not be logged."""
        self.eq.push("SYS_USER_CONNECT", access_token='foo')
        self.assertTrue(self.log_handler.check_debug(
            "push_event: SYS_USER_CONNECT, args:*, kw:*"))


class PushTestsWithCallback(BaseEQTestCase):
    """Test the error handling in the event distribution machinery."""

    def test_keep_going(self):
        """ Check that if a listener raises an Exception or have a
        wrong signature, the next listeners are called.
        """
        d = defer.Deferred()
        # helper class, pylint: disable-msg=C0111
        class BadListener(object):
            def handle_FS_FILE_CREATE(self, a, b): # it should be only 'a' here
                d.callback(False)

        class GoodListener(object):
            def handle_FS_FILE_CREATE(self, a):
                d.callback(a)

        bl = BadListener()
        gl = GoodListener()
        self.eq.subscribe(bl)
        self.eq.subscribe(gl)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(bl)
            self.eq.unsubscribe(gl)
        self.addCleanup(cleanup)

        # caught by EQ
        self.assertRaises(TypeError, self.eq.push, "FS_FILE_CREATE")
        self.assertRaises(TypeError, self.eq.push, "FS_FILE_CREATE", 1, 2)

        # approved by EQ, but one listener has a wrong signature
        self.eq.push("FS_FILE_CREATE", 1)
        def callback(result):
            """ asserts that GoodListener was called. """
            self.assertTrue(result)
            self.assertEquals(1, result)

        d.addCallback(callback)
        return d

    def test_default_handler(self):
        """ Check that handler_default is called. """
        d = defer.Deferred()
        # helper class, pylint: disable-msg=C0111
        class Listener(object):
            def handle_default(self, *args, **kwargs):
                d.callback(args)

        l = Listener()
        self.eq.subscribe(l)

        def cleanup():
            """ unsubscribe the listeners """
            self.eq.unsubscribe(l)
        self.addCleanup(cleanup)

        # push some event and expect it'll be handled by handle_default
        self.eq.push("FS_FILE_CREATE", 1)
        def callback(result):
            """ asserts that GoodListener was called. """
            self.assertEquals(2, len(result))
            self.assertEquals('FS_FILE_CREATE', result[0])
            self.assertEquals(1, result[1])

        d.addCallback(callback)
        return d

    def test_ordered_dispatch(self):
        """ Check that the events are pushed to all listeners in order. """
        d = defer.Deferred()
        # helper class, pylint: disable-msg=C0111
        class Listener(object):
            def __init__(self, eq):
                self.eq = eq
                self.events = []

            def handle_FS_FILE_CREATE(self, a):
                self.events.append('FS_FILE_CREATE')
                self.eq.push('FS_FILE_MOVE', a, 2)

            def handle_FS_FILE_DELETE(self, a):
                self.events.append('FS_FILE_DELETE')

            def handle_FS_FILE_MOVE(self, *args):
                self.events.append('FS_FILE_MOVE')
                d.callback(True)

        # create 10 listeners in order to create an event madness
        listeners = []
        # pylint: disable-msg=W0612
        for i in xrange(0, 10):
            l = Listener(self.eq)
            listeners.append(l)
            self.eq.subscribe(l)

        def cleanup():
            """ unsubscribe the listeners """
            for l in listeners:
                self.eq.unsubscribe(l)
        self.addCleanup(cleanup)

        # push some events to unleash the event madness
        self.eq.push("FS_FILE_CREATE", 1)
        self.eq.push('FS_FILE_DELETE', 2)

        def callback(result):
            """ asserts that Listener was called in the right order"""
            listeners_events = [listener.events for listener in listeners]
            for l_events in listeners_events:
                for other_l_events in listeners_events:
                    self.assertEqual(l_events, other_l_events)

        d.addCallback(callback)
        return d


class MuteFilterTests(unittest.TestCase):
    """Tests the MuteFilter class."""

    def setUp(self):
        self.mf = event_queue.MuteFilter()

    def test_empty(self):
        """Nothing there."""
        self.assertFalse(self.mf._cnt)

    def test_add_one(self):
        """Adds one element."""
        self.mf.add("foo")
        self.assertEqual(self.mf._cnt, dict(foo=1))

    def test_add_two_different(self):
        """Adds two different elements."""
        self.mf.add("foo")
        self.mf.add("bar")
        self.assertEqual(self.mf._cnt, dict(foo=1, bar=1))

    def test_add_two_equal(self):
        """Adds one element twice."""
        self.mf.add("foo")
        self.mf.add("foo")
        self.assertEqual(self.mf._cnt, dict(foo=2))

    def test_add_two_equal_and_third(self):
        """Adds one element."""
        self.mf.add("foo")
        self.mf.add("bar")
        self.mf.add("bar")
        self.assertEqual(self.mf._cnt, dict(foo=1, bar=2))

    def test_pop_simple(self):
        """Pops one element."""
        self.mf.add("foo")
        self.assertFalse(self.mf.pop("bar"))
        self.assertEqual(self.mf._cnt, dict(foo=1))
        self.assertTrue(self.mf.pop("foo"))
        self.assertFalse(self.mf._cnt)

    def test_pop_complex(self):
        """Pops several elements."""
        # add several
        self.mf.add("foo")
        self.mf.add("bar")
        self.mf.add("bar")
        self.assertEqual(self.mf._cnt, dict(foo=1, bar=2))

        # clean bar
        self.assertTrue(self.mf.pop("bar"))
        self.assertEqual(self.mf._cnt, dict(foo=1, bar=1))
        self.assertTrue(self.mf.pop("bar"))
        self.assertEqual(self.mf._cnt, dict(foo=1))
        self.assertFalse(self.mf.pop("bar"))
        self.assertEqual(self.mf._cnt, dict(foo=1))

        # clean foo
        self.assertTrue(self.mf.pop("foo"))
        self.assertFalse(self.mf._cnt)
        self.assertFalse(self.mf.pop("foo"))
        self.assertFalse(self.mf._cnt)


class IgnoreFileTests(unittest.TestCase):
    """Tests the ignore files behaviour."""

    def test_filter_none(self):
        """Still works ok even if not receiving a regex to ignore."""
        p = event_queue._GeneralINotifyProcessor(None)
        self.assertFalse(p.is_ignored("froo.pyc"))

    def test_filter_one(self):
        """Filters stuff that matches (or not) this one regex."""
        p = event_queue._GeneralINotifyProcessor(None, ['\A.*\\.pyc\Z'])
        self.assertTrue(p.is_ignored("froo.pyc"))
        self.assertFalse(p.is_ignored("froo.pyc.real"))
        self.assertFalse(p.is_ignored("otherstuff"))

    def test_filter_two_simple(self):
        """Filters stuff that matches (or not) these simple regexes."""
        p = event_queue._GeneralINotifyProcessor(None, ['\A.*foo\Z',
                                                        '\A.*bar\Z'])
        self.assertTrue(p.is_ignored("blah_foo"))
        self.assertTrue(p.is_ignored("blah_bar"))
        self.assertFalse(p.is_ignored("bar_xxx"))
        self.assertFalse(p.is_ignored("--foo--"))
        self.assertFalse(p.is_ignored("otherstuff"))

    def test_filter_two_complex(self):
        """Filters stuff that matches (or not) these complex regexes."""
        p = event_queue._GeneralINotifyProcessor(None, ['\A.*foo\Z|\Afoo.*\Z',
                                                        '\A.*bar\Z'])
        self.assertTrue(p.is_ignored("blah_foo"))
        self.assertTrue(p.is_ignored("blah_bar"))
        self.assertTrue(p.is_ignored("foo_xxx"))
        self.assertFalse(p.is_ignored("--foo--"))
        self.assertFalse(p.is_ignored("otherstuff"))


def test_suite():
    # pylint: disable-msg=C0111
    return unittest.TestLoader().loadTestsFromName(__name__)

if __name__ == "__main__":
    unittest.main()
