/*
 * Authors: Alejandro J. Cura <alecu@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "highlight.h"

#include <glib.h>

#define ASSERT_HIGHLIGHT(a,b,c) \
	ret = highlight_result (a, b); \
	g_assert_cmpstr (ret, ==, c); \
	g_free (ret);

static void
test_highlight_matches (void)
{
	gchar *ret = NULL;
	/* match at beginning */
	ASSERT_HIGHLIGHT ("john", "john q. public", "<b>john</b> q. public");

	/* match in the middle */
	ASSERT_HIGHLIGHT ("john", "andrew john q. public", "andrew <b>john</b> q. public");

	/* no match */
	ASSERT_HIGHLIGHT ("john", "andrew q. public", "andrew q. public");

	/* double match */
	ASSERT_HIGHLIGHT ("jo sm", "john smith", "<b>jo</b>hn <b>sm</b>ith");
	ASSERT_HIGHLIGHT ("john", "john johnson", "<b>john</b> <b>john</b>son");

	/* empty search */
	ASSERT_HIGHLIGHT ("", "john q. public", "john q. public");

	/* empty search with entities */
	ASSERT_HIGHLIGHT ("", "john & public", "john &amp; public");

	/* empty search term */
	ASSERT_HIGHLIGHT (" ", "john q. public", "john q. public");

	/* empty result */
	ASSERT_HIGHLIGHT ("john", "", "");

	/* case insensitive match */
	ASSERT_HIGHLIGHT ("john", "John Smith", "<b>John</b> Smith");
	ASSERT_HIGHLIGHT ("john", "Brian JOHNSON", "Brian <b>JOHN</b>SON");
	ASSERT_HIGHLIGHT ("br jo", "Brian JOHNSON", "<b>Br</b>ian <b>JO</b>HNSON");

	/* nested match */
	ASSERT_HIGHLIGHT ("edward wa", "edward wall", "<b>ed<b>wa</b>rd</b> <b>wa</b>ll");

	/* utf-8 strings */
	ASSERT_HIGHLIGHT ("ñandÚ", "Ñandú", "<b>Ñandú</b>");

	/* xml entities in the search */
	ASSERT_HIGHLIGHT ("john &", "john sons", "<b>john</b> sons");

	/* xml entities in the result */
	ASSERT_HIGHLIGHT ("john sons", "john & sons", "<b>john</b> &amp; <b>sons</b>");

	/* xml entities everywhere */
	ASSERT_HIGHLIGHT ("john & sons", "john & sons", "<b>john</b> <b>&amp;</b> <b>sons</b>");

	/* make sure the name of the entities is not highlighted */
	ASSERT_HIGHLIGHT ("john amp sons", "john & sons", "<b>john</b> &amp; <b>sons</b>");
}

int
main (int argc, gchar *argv[])
{
	g_test_init (&argc, &argv, NULL);

	g_test_add_func ("/testcontacts/TestHighlightMatches", test_highlight_matches);

	return g_test_run ();
}
