# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The file sync status widget."""

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel import backend, cache
from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import (
    LOADING,
    PLEASE_WAIT,
)
from ubuntuone.controlpanel.gui.qt import (
    FILE_SYNC_STATUS,
    pixmap_from_name,
)
from ubuntuone.controlpanel.gui.qt.ui import filesyncstatus_ui


logger = setup_logging('qt.filesyncstatus')


class FileSyncStatus(cache.Cache, QtGui.QWidget):
    """The FileSyncStatus widget"""

    def __init__(self, *args, **kwargs):
        """Initialize the UI of the widget."""
        super(FileSyncStatus, self).__init__(*args, **kwargs)
        self.ui = filesyncstatus_ui.Ui_Form()
        self.ui.setupUi(self)

        self._backend_method = None

    # Invalid name "showEvent"
    # pylint: disable=C0103

    def showEvent(self, event):
        """Load info."""
        super(FileSyncStatus, self).showEvent(event)
        self.load()

    # pylint: enable=C0103

    @defer.inlineCallbacks
    def load(self):
        """Load specific tab info."""
        info = yield self.backend.file_sync_status()
        self.process_info(info)
        self.backend.add_status_changed_handler(self.process_info)

    @log_call(logger.debug)
    def process_info(self, status):
        """Match status with signals."""
        if status is None:
            return
        try:
            status_key = status[backend.STATUS_KEY]
            data = FILE_SYNC_STATUS[status_key]
        except (KeyError, TypeError):
            logger.exception('process_info: received unknown status dict %r',
                             status)
            return

        msg = data['msg']
        if status_key in (backend.FILE_SYNC_ERROR, backend.FILE_SYNC_UNKNOWN):
            reason = status[backend.MSG_KEY]
            if reason:
                msg += ' (' + reason + ')'
        self.ui.sync_status_label.setText(msg)

        icon = data.get('icon')
        if icon is not None:
            pixmap = pixmap_from_name(icon)
            self.ui.sync_status_icon.setPixmap(pixmap)

        action = data.get('action')
        if action is not None:
            self.ui.sync_status_button.setText(action)
            self.ui.sync_status_button.setEnabled(True)
            self.ui.sync_status_button.show()
            default = (status_key == backend.FILE_SYNC_DISCONNECTED)
            self.ui.sync_status_button.setDefault(default)
        else:
            self.ui.sync_status_button.hide()

        tooltip = data.get('tooltip')
        if tooltip is not None:
            self.ui.sync_status_button.setToolTip(tooltip)

        self._backend_method = getattr(self.backend, data['backend_method'])

    @QtCore.pyqtSlot()
    def on_sync_status_button_clicked(self, *a, **kw):
        """Button was clicked, act accordingly to the label."""
        self.ui.sync_status_button.setEnabled(False)
        if self._backend_method is not None:
            self.ui.sync_status_label.setText(LOADING)
            self.ui.sync_status_button.setText(PLEASE_WAIT)
            pixmap = pixmap_from_name('sync_status_loading')
            self.ui.sync_status_icon.setPixmap(pixmap)
            self.ui.sync_status_button.setDefault(False)
            self._backend_method()
        else:
            logger.error('on_sync_status_button_clicked: backend method is '
                         'None!')
