# -*- coding: utf-8 -*-
#
# Copyright 2012-2013 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for darwin-specific parts of control panel's Qt main."""

from PyQt4 import QtGui

import subprocess
import sys

from mock import DEFAULT, Mock, patch

from twisted.internet.defer import inlineCallbacks
from twisted.internet.error import ReactorNotRunning

from ubuntuone.controlpanel import backend
from ubuntuone.controlpanel.gui.qt import main
from ubuntuone.controlpanel.tests import TestCase
from ubuntuone.controlpanel.gui.tests import FakeSignal


class FakeApplication(object):
    """Fake Application with quit signal."""
    def __init__(self):
        """Set up fake signal."""
        # pylint: disable=C0103
        self.aboutToQuit = FakeSignal()
        self.aboutToQuit.connect(self.record_call)
        self.signal_called = None

    def record_call(self, *args):
        """record call"""
        self.signal_called = args


class FakeBackend(object):
    """Mock backend"""

    def __init__(self):
        """init"""

        super(FakeBackend, self).__init__()
        self.handlers = []

    def add_status_changed_handler(self, h):
        """add handler"""
        self.handlers.append(h)

    def remove_status_changed_handler(self, h):
        """remove handler """
        self.handlers.remove(h)

    def clear(self):
        """reset handlers"""
        self.handlers = []


class ExternalGetDirectoryTestCase(TestCase):
    """Test calling applescript to show file picker."""

    @inlineCallbacks
    def setUp(self):
        """Simplify applescript for test."""
        yield super(ExternalGetDirectoryTestCase, self).setUp()
        self.patch(main.darwin.DarwinFileDialog,
                   "FOLDER_CHOOSER_SCRIPT",
                   "testscript: %s")
        self.patch(main.darwin.subprocess, 'check_output',
                   self._fake_check_output)

        self.args = []
        self.fake_raises = False

    def _fake_check_output(self, args):
        """mock for subprocess.check_output"""
        self.args += args
        if self.fake_raises:
            raise subprocess.CalledProcessError(-1, 'fakecmd')
        return "testpath"

    def test_path_ok(self):
        """Test returning the path."""

        rv = main.darwin.DarwinFileDialog.getExistingDirectory(
            directory="HOME")
        self.assertEqual(rv, "testpath")
        self.assertEqual(self.args, ["osascript", "-e", "testscript: HOME"])

    def test_path_raises(self):
        """Test returning the path."""
        self.fake_raises = True
        rv = main.darwin.DarwinFileDialog.getExistingDirectory(
            directory="HOME")
        self.assertEqual(rv, "")
        self.assertEqual(self.args, ["osascript", "-e", "testscript: HOME"])


@patch.multiple(main.darwin, SyncDaemonTool=DEFAULT, twisted_main=DEFAULT)
class QuitTestCase(TestCase):
    """Test quit event handling."""

    @inlineCallbacks
    def setUp(self):
        """Set up common fakes."""
        yield super(QuitTestCase, self).setUp()
        self.fake_app = FakeApplication()

    @inlineCallbacks
    def test_cmd_q_func_quits_sd(self, **mocks):
        """Test that we call syncdaemontool.quit when asked"""

        yield main.darwin.handle_cmd_q(self.fake_app, None,
                                       should_quit_sd=True)
        self.assertTrue(mocks['SyncDaemonTool'].called)

    @inlineCallbacks
    def test_cmd_q_func_doesnt_quit_sd(self, **mocks):
        """Test that we don't call syncdaemontool.quit by default"""

        yield main.darwin.handle_cmd_q(self.fake_app, None)
        self.assertFalse(mocks['SyncDaemonTool'].called)

    @inlineCallbacks
    def test_cmd_q_func_ignores_exception(self, **mocks):
        """Test that we keep going when SDT raises."""
        mocks['SyncDaemonTool'].quit.side_effect = Exception()

        yield main.darwin.handle_cmd_q(self.fake_app, None)
        mocks['twisted_main'].main_quit.assert_called_once_with(self.fake_app)

        self.assertEqual(self.fake_app.signal_called, ())

    @inlineCallbacks
    def test_cmd_q_func_calls_main_quit(self, **mocks):
        """Test that we call twisted_main.main_quit"""

        yield main.darwin.handle_cmd_q(self.fake_app, None,
                                       should_quit_sd=True)
        mocks['twisted_main'].main_quit.assert_called_once_with(self.fake_app)

    @inlineCallbacks
    def test_cmd_q_func_ignores_main_quit_exception(self, **mocks):
        """Test that we ignore exceptions in twisted_main.main_quit."""

        mocks['twisted_main'].main_quit.side_effect = ReactorNotRunning()

        yield main.darwin.handle_cmd_q(self.fake_app, None,
                                       should_quit_sd=True)
        mocks['twisted_main'].main_quit.assert_called_once_with(self.fake_app)

    @inlineCallbacks
    def test_reactor_iterate_called(self, **mocks):
        """Test that we poke the reactor to quit immediately."""
        # A bit of gymnastics to mock the internal import:
        mock_twisted = Mock()
        modules = {'twisted': mock_twisted,
                   'twisted.internet': mock_twisted.internet,
                   'twisted.internet.reactor': mock_twisted.internet.reactor}
        with patch.dict('sys.modules', modules):
            yield main.darwin.handle_cmd_q(self.fake_app, None,
                                           should_quit_sd=True)
            mock_twisted.internet.reactor.iterate.assert_called_once_with()


class InstallEventHandlersTestCase(TestCase):
    """Test creating Qt menu items."""

    def test_install_handlers_creates_kill_action(self):
        """Test creating menu item that will kill sd"""
        self.patch(main.darwin, 'handle_cmd_q',
                   self._set_called)
        app = QtGui.QApplication.instance()
        k = dict(quit_kills_sd=True)
        menubar = main.darwin.install_platform_event_handlers(app, **k)
        quit_action = menubar.findChild(QtGui.QAction, "darwin-cmd-q")
        quit_action.trigger()
        self.assertEqual(self._called, ((app, False, True), {}))

    def test_install_handlers_creates_nonkill_action(self):
        """Test creating menu item that will not kill sd"""
        self.patch(main.darwin, 'handle_cmd_q',
                   self._set_called)
        app = QtGui.QApplication.instance()
        menubar = main.darwin.install_platform_event_handlers(app)
        quit_action = menubar.findChild(QtGui.QAction, "darwin-cmd-q")
        quit_action.trigger()
        self.assertEqual(self._called, ((app, False, False), {}))


class LauncherTestCase(TestCase):
    """Test pyobjc menu launcher."""

    @inlineCallbacks
    def setUp(self):
        """Set up launcher and patches"""
        yield super(LauncherTestCase, self).setUp()

        self.patch(backend, 'ControlBackend', FakeBackend)
        self.patch(main.darwin, 'get_program_path',
                   lambda a, **kw: 'testpath')
        self.addCleanup(lambda: self.launcher.backend.clear())

        self.launcher = main.darwin.MenubarIconLauncher()

    def test_add_on_init(self):
        """Test adding to handler."""
        self.assertEqual(self.launcher.backend.handlers,
                         [self.launcher.handle_status_update])

    def test_handle_status_update_error(self):
        """Test doing nothing on error in handle_status_update"""
        self.patch(self.launcher, 'start_menu_process', self._set_called)
        self.launcher.handle_status_update({backend.STATUS_KEY:
                                            backend.FILE_SYNC_ERROR})
        self.assertEqual(self._called, False)

    def test_handle_status_update(self):
        """Test calling start_menu_process and removing handler"""
        self.patch(self.launcher, 'start_menu_process', self._set_called)
        self.launcher.handle_status_update({backend.STATUS_KEY: 'copacetic'})
        self.assertEqual(self._called, ((), {}))
        self.assertEqual(self.launcher.backend.handlers, [])

    def test_start_menu_process_nonfrozen(self):
        """Start does nothing when not frozen."""
        sys.frozen = None
        self.addCleanup(delattr, sys, 'frozen')

        self.patch(subprocess, 'Popen', self._set_called)
        self.launcher.start_menu_process()
        self.assertEqual(self._called, False)

    def test_start_menu_process_frozen(self):
        """Start launches menu when frozen."""
        sys.frozen = 'macosx'
        self.addCleanup(delattr, sys, 'frozen')

        self.patch(subprocess, 'Popen', self._set_called)
        self.launcher.start_menu_process()
        self.assertEqual(self._called, (('testpath',), {}))
