# -*- coding: utf-8 -*-
#
# Copyright 2012-2013 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the control panel's Qt main."""

import os
import sys

from PyQt4 import QtCore
from twisted.internet.defer import inlineCallbacks

from ubuntuone.controlpanel.gui.qt import main
from ubuntuone.controlpanel.tests import TestCase
from ubuntuone.controlpanel.gui.tests import FakeSignal


class FakeTranslator(object):

    """A fake QTranslator."""

    args = None

    def load(self, *args, **kwargs):
        """fake load."""
        self.args = (args, kwargs)


class FakeApplication(object):

    """A fake application."""

    def __init__(self):
        self.args = None
        self.style = None
        self.translator = None
        self.new_instance = FakeSignal()
        self.raster = False

    def __call__(self, argv, *args, **kwargs):
        """Fake arg filtering function."""
        if 'raster' in argv and '-graphicssystem' in argv:
            self.raster = True
        for arg in ('-title', '-graphicssystem', 'raster'):
            try:
                argv.remove(arg)
            except ValueError:
                pass
        self.args = (argv, args, kwargs)
        return self

    # pylint: disable=C0103
    def setStyleSheet(self, *args, **kwargs):
        """Fake setStyleSheet."""
        self.style = (args, kwargs)

    def installTranslator(self, translator):
        """Fake installTranslator."""
        self.translator = translator

    def isRightToLeft(self):
        """Fake isRightToLeft."""
        return False
    # pylint: enable=C0103

    def arguments(self):
        """Fake arguments function."""
        return self.args[0]

    def exec_(self):
        """Fake event loop."""
        pass


class FakeMainWindow(object):

    """A fake MainWindow."""

    tabname = None
    urgent = None
    raised = None

    def switch_to(self, tabname):
        """Fake switch_to."""
        self.tabname = tabname

    def set_urgent(self, value):
        """Fake set_urgent."""
        self.urgent = value

    def raise_(self):
        """Fake raise."""
        self.raised = True


class FakeStart(object):

    """Fake start function."""

    def __init__(self):
        self.args = None
        self.window = None

    def __call__(self, *args, **kwargs):
        if kwargs.get('minimized', False):
            self.window = None
        else:
            self.window = FakeMainWindow()
        self.args = (args, kwargs)
        return None, self.window


class MainTestCase(TestCase):

    """Test the argument parsing and passing in main."""

    @inlineCallbacks
    def setUp(self):
        yield super(MainTestCase, self).setUp()
        self.app = FakeApplication()
        self.translator = FakeTranslator()
        self.start = FakeStart()
        self.patch(main, "UniqueApplication", self.app)
        self.patch(main, "start", self.start)
        self.patch(main.source, "main_start", lambda app: None)
        self.patch(QtCore, "QTranslator", lambda: self.translator)
        self.patch(main, 'install_platform_event_handlers',
                   lambda app, **kwargs: None)

        self.qt4reactor_installed = False

        def fake_install_qt4reactor():
            """Record the install without importing."""
            self.qt4reactor_installed = True

        self.patch(main, "install_qt4reactor",
                   fake_install_qt4reactor)

    def test_wm_class(self):
        """Test that we set the 1st argument, used for WM_CLASS, correctly."""
        main.main([sys.argv[0]])
        self.assertEqual(self.app.args,
            (['ubuntuone-installer', sys.argv[0]],
            ('ubuntuone-control-panel',), {}))

    def _test_gtk_module_cleanup(self, modules, expected):
        """Check if the module cleanup works."""
        old_modules = os.environ.get('GTK_MODULES', '')

        def clean_env():
            """Reset the environment variable."""
            os.environ['GTK_MODULES'] = old_modules

        self.addCleanup(clean_env)
        os.environ['GTK_MODULES'] = modules
        main.main([sys.argv[0]])
        self.assertEqual(os.environ['GTK_MODULES'], expected)

    def test_gtk_module_cleanup_1(self):
        """Test that we deactivate the overlay scrollbar GTK module."""
        self._test_gtk_module_cleanup('aaa:overlay-scrollbar:bbb', 'aaa:bbb')

    def test_gtk_module_cleanup_2(self):
        """Test that we deactivate the overlay scrollbar GTK module."""
        self._test_gtk_module_cleanup('overlay-scrollbar', '')

    def test_gtk_module_cleanup_3(self):
        """Test that we deactivate the overlay scrollbar GTK module."""
        self._test_gtk_module_cleanup('overlay-scrollbar:overlay-scrollbars',
            'overlay-scrollbars')

    def test_title_not_fail(self):
        """Ensure -title is removed before it gets to argparse."""
        main.main([sys.argv[0], "-title"])
        # Did not crash!

    def test_truncate_argv(self):
        """Ensure the binary name is not given to argparse."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main(["foo", "bar", sys.argv[0], "--minimized"])
        self.assertEqual(self.start.args[1],
            {'minimized': True, 'with_icon': False, 'installer': False})

    def test_darwin_defult_is_with_icon(self):
        """On Darwin, the default is not to show the icon."""
        self.patch(sys, 'platform', 'darwin')
        main.main([sys.argv[0]])
        self.assertEqual(self.start.args[1],
            {'minimized': False, 'with_icon': False, 'installer': False})

    def test_not_darwin_defult_is_with_icon(self):
        """On Not-darwin, the default is to not show the icon."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main([sys.argv[0]])
        self.assertEqual(self.start.args[1],
            {'minimized': False, 'with_icon': False, 'installer': False})

    def test_minimized_option(self):
        """Ensure the --minimized option is parsed and passed correctly."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main([sys.argv[0], "--minimized"])
        self.assertEqual(self.start.args[1],
            {'minimized': True, 'with_icon': False, 'installer': False})

    def test_with_icon_option(self):
        """Ensure the --with-icon option is parsed and passed correctly."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main([sys.argv[0], "--with-icon"])
        self.assertEqual(self.start.args[1],
            {'minimized': False, 'with_icon': True, 'installer': False})

    def test_all_styles_load(self):
        """Ensure the platform style is loaded."""
        main.main([sys.argv[0]])
        data = []
        for qss_name in (":/ubuntuone.qss", main.source.PLATFORM_QSS):
            qss = QtCore.QResource(qss_name)
            data.append(unicode(qss.data()))
        self.assertEqual((('\n'.join(data),), {}), self.app.style)

    def test_switch_to_option(self):
        """Ensure the --switch-to option is parsed and passed correctly."""
        main.main([sys.argv[0], "--switch-to", "folders"])
        self.assertEqual(self.start.window.tabname, "folders")

    def test_installer_option(self):
        """Ensure the --installer option is parsed and passed correctly."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main([sys.argv[0], "--installer"])
        self.assertEqual(self.start.args[1],
            {'minimized': False, 'with_icon': False, 'installer': True})

    def test_minimized_with_icon_options(self):
        """Ensure you can be minimized and with icon at the same time."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main([sys.argv[0], "--minimized", "--with-icon"])
        self.assertEqual(self.start.args[1],
            {'minimized': True, 'with_icon': True, 'installer': False})

    def test_nondarwin_is_not_raster(self):
        """Ensure the raster argument is not injected in non-darwin."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main([sys.argv[0]])
        self.assertFalse(self.app.raster)

    def test_darwin_is_raster(self):
        """Ensure the raster argument is injected in darwin."""
        self.patch(sys, 'platform', 'darwin')
        main.main([sys.argv[0]])
        self.assertTrue(self.app.raster)

    def test_translator(self):
        """Ensure the Qt translator is loaded."""
        main.main([sys.argv[0]])
        locale = unicode(QtCore.QLocale.system().name())
        self.assertEqual(self.app.translator, self.translator)
        self.assertEqual(self.translator.args, (("qt_" + locale,
            QtCore.QLibraryInfo.location(
                QtCore.QLibraryInfo.TranslationsPath)), {}))

    def test_new_instance(self):
        """Ensure the new_instance signal is connected."""
        main.main([sys.argv[0]])
        self.assertEqual(self.app.new_instance.target,
            [self.start.window.raise_])

    def test_darwin_installs_qt4reactor(self):
        """Ensure the qt4 reactor is installed when requested."""
        self.patch(sys, 'platform', 'darwin')
        main.main([sys.argv[0]], install_reactor_darwin=True)
        self.assertEqual(self.qt4reactor_installed, True)

    def test_darwin_doesnt_install_qt4reactor(self):
        """Ensure the qt4 reactor isn't installed by default."""
        self.patch(sys, 'platform', 'darwin')
        main.main([sys.argv[0]])
        self.assertEqual(self.qt4reactor_installed, False)

    def test_nondarwin_ignores_true_install_flag(self):
        """Ensure the qt4 reactor isn't installed on non-darwin."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main([sys.argv[0]], install_reactor_darwin=True)
        self.assertEqual(self.qt4reactor_installed, False)

    def test_nondarwin_ignores_false_install_flag(self):
        """Ensure the qt4 reactor isn't installed on non-darwin."""
        self.patch(sys, 'platform', 'not-darwin')
        main.main([sys.argv[0]], install_reactor_darwin=False)
        self.assertEqual(self.qt4reactor_installed, False)

    def test_install_config(self):
        """Test that install_config_and_daemons is called."""
        self.patch(main, 'install_config_and_daemons',
                   self._set_called)

        main.main([sys.argv[0]], install_reactor_darwin=False)
        self.assertEqual(self._called, ((), {}))

    def test_install_event_handlers_darwin(self):
        """Test that install_platform_event_handlers is called on darwin."""
        self.patch(sys, 'platform', 'darwin')
        self.patch(main, 'install_platform_event_handlers',
                   self._set_called)

        main.main([sys.argv[0]], install_reactor_darwin=False)
        self.assertEqual(self._called, ((self.app,), {'quit_kills_sd': False}))

    def test_install_event_handlers_non_darwin(self):
        """Test install_platform_event_handlers is called on not-darwin."""
        self.patch(sys, 'platform', 'not-darwin')
        self.patch(main, 'install_platform_event_handlers',
                   self._set_called)

        main.main([sys.argv[0], '--with-icon'], install_reactor_darwin=False)
        self.assertEqual(self._called, ((self.app,), {'quit_kills_sd': False}))

    def test_fix_turkish_locale_called(self):
        """The fix_turkish_locale function is called, always."""
        self.patch(main, 'fix_turkish_locale', self._set_called)
        main.main([sys.argv[0]])
        self.assertEqual(self._called, ((), {}))
