# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for each device widget."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import device as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    BaseTestCase,
    CrashyBackend,
    CrashyBackendException,
    FakedDialog,
    SAMPLE_COMPUTER_INFO,
    SAMPLE_PHONE_INFO,
)

# Access to a protected member
# Instance of 'ControlBackend' has no '_called' member
# pylint: disable=W0212, E1103


class DeviceWidgetTestCase(BaseTestCase):
    """Test the DeviceWidget class."""

    innerclass_ui = gui.device_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.DeviceWidget
    device_id = 'zaraza'
    kwargs = {'device_id': device_id}
    logger = gui.logger

    def test_has_id(self):
        """The device as an id accepted as creation param."""
        self.assertEqual(self.ui.id, self.device_id)

    def test_id_can_be_none(self):
        """The device id is None by default."""
        ui = self.class_ui()
        self.assertEqual(ui.id, None)

    def test_setting_id_to_none_disables_remove_button(self):
        """If the id is set to None, the remove button is disabled."""
        self.ui.id = None
        self.assertFalse(self.ui.ui.remove_device_button.isEnabled())

    def test_setting_id_to_not_none_enables_remove_button(self):
        """If the id is set to not None, the remove button is enabled."""
        self.ui.id = 'not None'
        self.assertTrue(self.ui.ui.remove_device_button.isEnabled())

    def test_remove_button(self):
        """The remove button is visible."""
        self.assertTrue(self.ui.ui.remove_device_button.isVisible())

    def test_update_device_info(self):
        """The widget is updated with the info."""
        info = SAMPLE_COMPUTER_INFO
        expected_name = info["name"]
        self.ui.update_device_info(info)
        self.assertEqual(unicode(self.ui.ui.device_name_label.text()),
            expected_name)

    # pylint: disable=C0103
    def assertIconMatchesType(self, device_type, base_icon_name):
        """A given device type yields the right icon."""
        self.assertEqual(gui.icon_name_from_type(device_type), base_icon_name)
    # pylint: enable=C0103

    def test_icon_name_from_type(self):
        """Get the right icon name for a device type."""
        self.assertIconMatchesType(gui.DEVICE_TYPE_COMPUTER, gui.COMPUTER_ICON)
        self.assertIconMatchesType(gui.DEVICE_TYPE_PHONE, gui.PHONE_ICON)
        self.assertIconMatchesType("other random type", gui.COMPUTER_ICON)

    def assert_update_device_info_sets_right_icon(self, info):
        """The widget is updated with the right icon."""
        self.ui.update_device_info(info)
        pixmap_name = gui.icon_name_from_type(info["type"])
        expected_pixmap = gui.pixmap_from_name(pixmap_name)
        label_pixmap = self.ui.ui.device_icon_label.pixmap()
        self.assertEqualPixmaps(label_pixmap, expected_pixmap)

    def test_update_device_info_sets_computer_icon(self):
        """The computer icon is set."""
        self.assert_update_device_info_sets_right_icon(SAMPLE_COMPUTER_INFO)

    def test_update_device_info_sets_phone_icon(self):
        """The phone icon is set."""
        self.assert_update_device_info_sets_right_icon(SAMPLE_PHONE_INFO)


class RemoveDeviceTestCase(DeviceWidgetTestCase):
    """The test suite for the device deletion."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(RemoveDeviceTestCase, self).setUp()
        FakedDialog.response = gui.YES

    def test_remove_device_opens_confirmation_dialog(self):
        """A confirmation dialog is opened when user clicks 'delete device'."""
        self.ui.ui.remove_device_button.click()

        msg = gui.DEVICE_CONFIRM_REMOVE
        buttons = gui.YES | gui.NO
        self.assertEqual(FakedDialog.args,
                         (self.ui, '', msg, buttons, gui.NO))
        self.assertEqual(FakedDialog.kwargs, {})

    def test_remove_device_does_not_remove_if_answer_is_no(self):
        """The device is not removed is answer is No."""
        FakedDialog.response = gui.NO
        self.ui.removed.connect(self._set_called)
        self.ui.ui.remove_device_button.click()

        self.assertNotIn('remove_device', self.ui.backend._called)
        self.assertEqual(self._called, False)

    def test_remove_device_does_remove_if_answer_is_yes(self):
        """The device is removed is answer is Yes."""
        FakedDialog.response = gui.YES
        self.ui.ui.remove_device_button.click()

        self.assert_backend_called('remove_device', device_id=self.device_id)

    @defer.inlineCallbacks
    def test_remove_device_emits_signal_when_removed(self):
        """The signal 'removed' is emitted when removed."""
        d = defer.Deferred()

        def check(device_id):
            """Fire deferred when the device was removed."""
            d.callback(device_id)

        FakedDialog.response = gui.YES
        self.ui.removed.connect(self._set_called)
        self.patch(self.ui.backend, 'remove_device', check)
        self.ui.ui.remove_device_button.click()

        yield d
        self.assertEqual(self._called, ((), {}))

    def test_remove_device_emits_signal_when_not_removed(self):
        """The signal 'removeCanceled' is emitted when user cancels removal."""
        FakedDialog.response = gui.NO
        self.ui.removeCanceled.connect(self._set_called)
        self.ui.ui.remove_device_button.click()

        self.assertEqual(self._called, ((), {}))

    @defer.inlineCallbacks
    def test_backend_error_is_handled(self):
        """Any error from the backend is properly handled."""
        self.patch(self.ui, 'backend', CrashyBackend())
        yield self.ui.ui.remove_device_button.click()

        self.assertTrue(self.memento.check_exception(CrashyBackendException))


class RemoteDeviceWidgetTestCase(DeviceWidgetTestCase):
    """Test the RemoteDeviceWidget class."""

    class_ui = gui.RemoteDeviceWidget

    def test_remove_button(self):
        """The remove button is hidden."""
        self.assertFalse(self.ui.ui.remove_device_button.isVisible())
