# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Qt UI."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import gui
from ubuntuone.controlpanel.gui.qt.tests import BaseTestCase, FakedDialog


class FakeEntry(object):

    """A fake unity launcher entry."""

    called = None

    def set_property(self, *args, **kwargs):
        """Fake set_property."""
        self.called = (args, kwargs)


class MainWindowTestCase(BaseTestCase):
    """Test the qt main window."""

    innerclass_ui = gui.mainwindow_ui
    innerclass_name = "Ui_MainWindow"
    class_ui = gui.MainWindow

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        yield super(MainWindowTestCase, self).setUp()

    def test_close_event_calls_custom_close_callback(self):
        """When closing the window, close_callback is called."""
        self.ui.close_callback = self._set_called
        self.ui.closeEvent(event=gui.QtGui.QCloseEvent())
        self.assertEqual(self._called,
                         ((), {}), 'close_callback called.')

    def test_close_callback_can_be_none(self):
        """The close_callback can be None."""
        self.ui.close_callback = None
        self.ui.closeEvent(event=gui.QtGui.QCloseEvent())
        # world did not explode

    def test_on_signin_canceled(self):
        """On SigninPanel's signinCanceled, close."""
        self.patch(self.ui, 'closeEvent', self._set_called)
        self.ui.ui.control_panel.finished.emit()
        self.assertEqual(len(self._called[0]), 1)
        self.assertIsInstance(self._called[0][0], gui.QtGui.QCloseEvent)

    def test_quit_action(self):
        """Check that the app can quit using the keyboard."""
        self.assertEqual(self.ui.quit_action.shortcuts(),
            ["Ctrl+q", "Ctrl+w"])
        self.ui.close_callback = self._set_called
        self.ui.quit_action.trigger()
        self.assertFalse(self.ui.isVisible())
        self.assertEqual(self._called,
                         ((), {}), 'close_callback called.')

    def test_switch_to(self):
        """Check that switch_to changes the current tab"""
        self.assertEqual(
            self.ui.ui.control_panel.ui.tab_widget.currentIndex(),
            self.ui.ui.control_panel.ui.tab_widget.indexOf(
                self.ui.ui.control_panel.ui.folders_tab))
        self.ui.switch_to("foobar")
        self.assertEqual(
            self.ui.ui.control_panel.ui.tab_widget.currentIndex(),
            self.ui.ui.control_panel.ui.tab_widget.indexOf(
                self.ui.ui.control_panel.ui.folders_tab))
        self.ui.switch_to("share_links")
        self.assertEqual(
            self.ui.ui.control_panel.ui.tab_widget.currentIndex(),
            self.ui.ui.control_panel.ui.tab_widget.indexOf(
                self.ui.ui.control_panel.ui.share_links_tab))
        self.ui.switch_to("devices")
        self.assertEqual(
            self.ui.ui.control_panel.ui.tab_widget.currentIndex(),
            self.ui.ui.control_panel.ui.tab_widget.indexOf(
                self.ui.ui.control_panel.ui.devices_tab))
        self.ui.switch_to("settings")
        self.assertEqual(
            self.ui.ui.control_panel.ui.tab_widget.currentIndex(),
            self.ui.ui.control_panel.ui.tab_widget.indexOf(
                self.ui.ui.control_panel.ui.preferences_tab))
        self.ui.switch_to("account")
        self.assertEqual(
            self.ui.ui.control_panel.ui.tab_widget.currentIndex(),
            self.ui.ui.control_panel.ui.tab_widget.indexOf(
                self.ui.ui.control_panel.ui.account_tab))
        self.ui.switch_to("folders")
        self.assertEqual(
            self.ui.ui.control_panel.ui.tab_widget.currentIndex(),
            self.ui.ui.control_panel.ui.tab_widget.indexOf(
                self.ui.ui.control_panel.ui.folders_tab))

    def test_focus_in(self):
        """Test that focusing removes urgent bit from launcher entry."""
        entry = FakeEntry()
        self.patch(self.ui, "entry", entry)
        self.ui.focusInEvent(None)
        self.assertEqual(entry.called, (('urgent', False), {}))

    def test_set_urgent(self):
        """Test that set_urgent calls with the right arguments."""
        entry = FakeEntry()
        self.patch(self.ui, "entry", entry)
        self.ui.set_urgent("foo")
        self.assertEqual(entry.called, (('urgent', "foo"), {}))


class CheckUpdatesNoUpdatesTestCase(MainWindowTestCase):
    """Test the check_updates method when there are no updates."""

    updates_available = False

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        yield super(CheckUpdatesNoUpdatesTestCase, self).setUp()
        self.patch(gui.utils, 'are_updates_present',
                   lambda: defer.succeed(self.updates_available))
        self.patch(gui.utils, 'perform_update',
                   lambda: defer.fail(ValueError()))

    @defer.inlineCallbacks
    def test_update_confirmation_dialog(self):
        """The confirmation dialog is not shown."""
        yield self.ui.check_updates()
        self.assertEqual(None, FakedDialog.args)
        self.assertEqual(None, FakedDialog.kwargs)

    @defer.inlineCallbacks
    def test_perform_update(self):
        """The perform_update method is not called."""
        yield self.ui.check_updates()
        self.assertFalse(self._called)


class CheckUpdatesUserSaysNoTestCase(CheckUpdatesNoUpdatesTestCase):
    """Test check_updates when there are updates and user rejects them."""

    updates_available = True
    user_response = gui.QtGui.QMessageBox.No

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        yield super(CheckUpdatesUserSaysNoTestCase, self).setUp()
        self.patch(FakedDialog, 'response', self.user_response)

    @defer.inlineCallbacks
    def test_update_confirmation_dialog(self):
        """The confirmation dialog is shown with correct params."""
        yield self.ui.check_updates()
        args = (self.ui, gui.UPDATE_TITLE, gui.UPDATE_SOFTWARE,
                gui.QtGui.QMessageBox.Yes | gui.QtGui.QMessageBox.No,
                gui.QtGui.QMessageBox.Yes)
        self.assertEqual(args, FakedDialog.args)
        self.assertEqual({}, FakedDialog.kwargs)


class CheckUpdatesUserSaysYesTestCase(CheckUpdatesUserSaysNoTestCase):
    """Test check_updates when there are updates and user accepts them."""

    user_response = gui.QtGui.QMessageBox.Yes

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        self.updated = defer.Deferred()
        yield super(CheckUpdatesUserSaysYesTestCase, self).setUp()
        self.patch(gui.utils, 'perform_update',
                   lambda *a, **kw: self.updated.callback((a, kw)))

    @defer.inlineCallbacks
    def test_perform_update(self):
        """The perform_update method is called."""
        yield self.ui.check_updates()
        result = yield self.updated  # perform_update will fire this deferred

        self.assertEqual(result, ((), {}))


class AutoStartTestCase(MainWindowTestCase):
    """Test the add_to_autostart call."""

    @defer.inlineCallbacks
    def setUp(self):
        # Be sure to patch add_to_autostart *before* class_ui creation occurs.
        self.patch(gui.utils, "add_to_autostart", self._set_called)
        yield super(AutoStartTestCase, self).setUp()

    def test_add_to_autostart(self):
        """Test that the add_to_autostart function is called when CP opens."""
        self.assertTrue(self._called)
