# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Miscelaneous functions and constants for darwin."""

import os
import shutil
import sys

from Cocoa import (NSURL, NSUserDefaults)
from LaunchServices import (
    kLSSharedFileListFavoriteItems,
    kLSSharedFileListItemBeforeFirst,
    LSSharedFileListCreate,
    LSSharedFileListInsertItemURL,
    LSSharedFileListItemRef)

from ctypes import (
    byref,
    CDLL,
    POINTER,
    Structure,
    c_bool,
    c_char_p,
    c_double,
    c_int32,
    c_long,
    c_uint32,
    c_void_p)

from ctypes.util import find_library

from twisted.internet import defer

from dirspec.basedir import save_config_path
from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.utils.common import call_updater
from ubuntuone.platform import expand_user

logger = setup_logging('utils.darwin')

ADDED_TO_FINDER_SIDEBAR_ONCE = "ADDED_TO_FINDER_SIDEBAR_ONCE"

LOGO_ICON = "ubuntuone_logo.png"

AUTOUPDATE_BIN_NAME = 'autoupdate-darwin'
UNINSTALL_BIN_NAME = 'uninstall-darwin'

FSEVENTSD_JOB_LABEL = "com.ubuntu.one.fsevents"

# pylint: disable=C0103
CFPath = find_library("CoreFoundation")
CF = CDLL(CFPath)

CFRelease = CF.CFRelease
CFRelease.restype = None
CFRelease.argtypes = [c_void_p]

kCFStringEncodingUTF8 = 0x08000100


class CFRange(Structure):
    """CFRange Struct"""
    _fields_ = [("location", c_long),
                ("length", c_long)]

CFShow = CF.CFShow
CFShow.argtypes = [c_void_p]
CFShow.restype = None

CFStringCreateWithCString = CF.CFStringCreateWithCString
CFStringCreateWithCString.restype = c_void_p
CFStringCreateWithCString.argtypes = [c_void_p, c_void_p, c_uint32]

kCFAllocatorDefault = c_void_p()

CFErrorCopyDescription = CF.CFErrorCopyDescription
CFErrorCopyDescription.restype = c_void_p
CFErrorCopyDescription.argtypes = [c_void_p]

CFDictionaryGetValue = CF.CFDictionaryGetValue
CFDictionaryGetValue.restype = c_void_p
CFDictionaryGetValue.argtypes = [c_void_p, c_void_p]

CFArrayGetValueAtIndex = CF.CFArrayGetValueAtIndex
CFArrayGetValueAtIndex.restype = c_void_p
CFArrayGetValueAtIndex.argtypes = [c_void_p, c_uint32]

CFURLCreateWithFileSystemPath = CF.CFURLCreateWithFileSystemPath
CFURLCreateWithFileSystemPath.restype = c_void_p
CFURLCreateWithFileSystemPath.argtypes = [c_void_p, c_void_p, c_uint32, c_bool]

CFBundleCopyInfoDictionaryForURL = CF.CFBundleCopyInfoDictionaryForURL
CFBundleCopyInfoDictionaryForURL.restype = c_void_p
CFBundleCopyInfoDictionaryForURL.argtypes = [c_void_p]

CFStringGetDoubleValue = CF.CFStringGetDoubleValue
CFStringGetDoubleValue.restype = c_double
CFStringGetDoubleValue.argtypes = [c_void_p]

SecurityPath = find_library("Security")
Security = CDLL(SecurityPath)

AuthorizationCreate = Security.AuthorizationCreate
AuthorizationCreate.restype = c_int32
AuthorizationCreate.argtypes = [c_void_p, c_void_p, c_int32, c_void_p]

AuthorizationFree = Security.AuthorizationFree
AuthorizationFree.restype = c_uint32
AuthorizationFree.argtypes = [c_void_p, c_uint32]

kAuthorizationFlagDefaults = 0
kAuthorizationFlagInteractionAllowed = 1 << 0
kAuthorizationFlagExtendRights = 1 << 1
kAuthorizationFlagDestroyRights = 1 << 3
kAuthorizationFlagPreAuthorize = 1 << 4

kAuthorizationEmptyEnvironment = None

errAuthorizationSuccess = 0
errAuthorizationDenied = -60005
errAuthorizationCanceled = -60006
errAuthorizationInteractionNotAllowed = -60007

ServiceManagementPath = find_library("ServiceManagement")
ServiceManagement = CDLL(ServiceManagementPath)

kSMRightBlessPrivilegedHelper = "com.apple.ServiceManagement.blesshelper"
kSMRightModifySystemDaemons = "com.apple.ServiceManagement.daemons.modify"

# pylint: disable=E1101
# c_void_p has no "in_dll" member:
kSMDomainSystemLaunchd = c_void_p.in_dll(ServiceManagement,
                                         "kSMDomainSystemLaunchd")
# pylint: enable=E1101

SMJobBless = ServiceManagement.SMJobBless
SMJobBless.restype = c_bool
SMJobBless.argtypes = [c_void_p, c_void_p, c_void_p, POINTER(c_void_p)]

SMJobRemove = ServiceManagement.SMJobRemove
SMJobRemove.restype = c_bool
SMJobRemove.argtypes = [c_void_p, c_void_p, c_void_p, c_bool,
                        POINTER(c_void_p)]

SMJobCopyDictionary = ServiceManagement.SMJobCopyDictionary
SMJobCopyDictionary.restype = c_void_p
SMJobCopyDictionary.argtypes = [c_void_p, c_void_p]


class AuthorizationItem(Structure):
    """AuthorizationItem Struct"""
    _fields_ = [("name", c_char_p),
                ("valueLength", c_uint32),
                ("value", c_void_p),
                ("flags", c_uint32)]


class AuthorizationRights(Structure):
    """AuthorizationRights Struct"""
    _fields_ = [("count", c_uint32),
                # * 1 here is specific to our use below
                ("items", POINTER(AuthorizationItem))]


class DaemonInstallException(Exception):
    """Error securely installing daemon."""


class AuthUserCanceledException(Exception):
    """The user canceled the authorization."""


class AuthFailedException(Exception):
    """The authorization faild for some reason."""


class DaemonRemoveException(Exception):
    """Error removing existing daemon."""


class DaemonVersionMismatchException(Exception):
    """Incompatible version of the daemon found."""


def create_cfstr(s):
    """Creates a CFString from a python string.

    Note - because this is a "create" function, you have to CFRelease
    the returned string.
    """
    return CFStringCreateWithCString(kCFAllocatorDefault,
                                     s.encode('utf8'),
                                     kCFStringEncodingUTF8)


def get_bundle_version(path_cfstr):
    """Returns a float version from the plist of the given bundle.

    path_cfstr must be a CFStringRef.
    """
    cfurl = CFURLCreateWithFileSystemPath(None,  # use default allocator
                                          path_cfstr,
                                          0,  # POSIX style path
                                          False)
    plist = CFBundleCopyInfoDictionaryForURL(cfurl)
    ver_key_cfstr = create_cfstr("CFBundleVersion")
    version_cfstr = CFDictionaryGetValue(plist, ver_key_cfstr)

    version = CFStringGetDoubleValue(version_cfstr)

    CFRelease(cfurl)
    CFRelease(plist)
    CFRelease(ver_key_cfstr)

    return version


def get_fsevents_daemon_installed_version():
    """Returns helper version or None if helper is not installed."""

    label_cfstr = create_cfstr(FSEVENTSD_JOB_LABEL)
    job_data_cfdict = SMJobCopyDictionary(kSMDomainSystemLaunchd,
                                          label_cfstr)
    CFRelease(label_cfstr)

    if job_data_cfdict is not None:
        key_cfstr = create_cfstr("ProgramArguments")
        args_cfarray = CFDictionaryGetValue(job_data_cfdict, key_cfstr)

        path_cfstr = CFArrayGetValueAtIndex(args_cfarray, 0)
        version = get_bundle_version(path_cfstr)

        # only release things "copied" or "created", not "got".
        CFRelease(job_data_cfdict)
        CFRelease(key_cfstr)
        return version

    return None


def get_authorization():
    """Get authorization to remove and/or install daemons."""

    # pylint: disable=W0201
    authItemBless = AuthorizationItem()
    authItemBless.name = kSMRightBlessPrivilegedHelper
    authItemBless.valueLength = 0
    authItemBless.value = None
    authItemBless.flags = 0

    authRights = AuthorizationRights()
    authRights.count = 1
    authRights.items = (AuthorizationItem * 1)(authItemBless)

    flags = (kAuthorizationFlagDefaults |
             kAuthorizationFlagInteractionAllowed |
             kAuthorizationFlagPreAuthorize |
             kAuthorizationFlagExtendRights)

    authRef = c_void_p()

    status = AuthorizationCreate(byref(authRights),
                                 kAuthorizationEmptyEnvironment,
                                 flags,
                                 byref(authRef))

    if status != errAuthorizationSuccess:

        if status == errAuthorizationInteractionNotAllowed:
            raise AuthFailedException("Authorization failed: "
                                      "interaction not allowed.")

        elif status == errAuthorizationDenied:
            raise AuthFailedException("Authorization failed: auth denied.")

        else:
            raise AuthUserCanceledException()

    if authRef is None:
        raise AuthFailedException("No authRef from AuthorizationCreate: %r"
                                  % status)
    return authRef


def install_fsevents_daemon(authRef):
    """Call SMJobBless to install daemon.

    No return, raises on error.
    """

    desc_cfstr = None

    try:
        error = c_void_p()

        label_cfstr = create_cfstr(FSEVENTSD_JOB_LABEL)
        ok = SMJobBless(kSMDomainSystemLaunchd,
                        label_cfstr,
                        authRef,
                        byref(error))
        CFRelease(label_cfstr)

        if not ok:
            desc_cfstr = CFErrorCopyDescription(error)
            CFShow(desc_cfstr)
            raise DaemonInstallException("SMJobBless error (see above)")

    finally:
        if desc_cfstr:
            CFRelease(desc_cfstr)


def remove_fsevents_daemon(authRef):
    """Call SMJobRemove to remove daemon.

    No return, raises on error.
    """
    desc_cfstr = None
    try:
        error = c_void_p()

        label_cfstr = create_cfstr(FSEVENTSD_JOB_LABEL)
        ok = SMJobRemove(kSMDomainSystemLaunchd,
                         label_cfstr,
                         authRef,
                         True,
                         byref(error))

        CFRelease(label_cfstr)

        if not ok:
            desc_cfstr = CFErrorCopyDescription(error)
            CFShow(desc_cfstr)
            raise DaemonRemoveException("SMJobRemove error (see above)")
    finally:
        if desc_cfstr:
            CFRelease(desc_cfstr)


def add_to_autostart():
    """Add syncdaemon to the session's autostart."""
    # TODO


def _get_userdefaults():
    """Testable wrapper for NSUserDefaults"""
    # HACK: This is required to avoid trial complaining on tear down
    # that assigning to native selectors is not supported.
    return NSUserDefaults.standardUserDefaults()


def add_u1_folder_to_favorites():
    """Add the u1 folder to the favorites sidebar in Finder."""
    sud = _get_userdefaults()
    if sud.boolForKey_(ADDED_TO_FINDER_SIDEBAR_ONCE):
        return
    else:
        sud.setBool_forKey_(True, ADDED_TO_FINDER_SIDEBAR_ONCE)

    u1_path = os.path.expanduser("~/Ubuntu One/")
    u1_url = NSURL.fileURLWithPath_isDirectory_(u1_path, True)

    if u1_url is None:
        logger.error("Error creating URL for favorites")
        return

    lst = LSSharedFileListCreate(None,
                                 kLSSharedFileListFavoriteItems,
                                 None)
    # LSSharedFileListCreate is wrapped by pyobjc, so we do not need
    # to CFRelease(lst) here.

    if lst is None:
        logger.error("Error creating shared file list")
        return

    item = LSSharedFileListInsertItemURL(lst, kLSSharedFileListItemBeforeFirst,
                                         None, None, u1_url, {}, [])
    if not isinstance(item, LSSharedFileListItemRef):
        logger.debug("Error adding ~/Ubuntu One to finder favorites")


@defer.inlineCallbacks
def are_updates_present():
    """Return if there are updates for Ubuntu One."""
    result = yield call_updater(install=False)
    defer.returnValue(result)


def default_folders(user_home=None):
    """Return a list of the folders to add by default."""
    folders = ["~/Desktop",
               "~/Downloads",
               "~/Documents",
               "~/Music",
               "~/Pictures",
               "~/Movies"]
    folders = map(expand_user, folders)
    return folders


def check_and_install_fsevents_daemon(main_app_dir):
    """Checks version of running daemon, maybe installs.

    'main_app_dir' is the path to the running app.

    This will securely install the daemon bundled with the running app
    if there is no currently installed one, or upgrade it if the
    installed one is old. If the installed one is newer, it raises
    a DaemonVersionMismatchException.
    """

    daemon_path = os.path.join(main_app_dir, 'Contents',
                               'Library', 'LaunchServices',
                               FSEVENTSD_JOB_LABEL)
    bundled_version = get_bundle_version(create_cfstr(daemon_path))

    installed_version = get_fsevents_daemon_installed_version()

    if installed_version == bundled_version:
        logger.info("Current fsevents daemon already installed: version %r" %
                    installed_version)
        return

    if installed_version > bundled_version:
        desc = ("Found newer fsevents daemon:"
                " installed %r > bundled version %r." %
                (installed_version, bundled_version))
        logger.error(desc)
        raise DaemonVersionMismatchException(desc)

    authRef = get_authorization()

    if installed_version is not None and \
            installed_version < bundled_version:
        logger.info("Found installed daemon version %r < %r, removing." %
                    (installed_version, bundled_version))

        try:
            remove_fsevents_daemon(authRef)
        except DaemonRemoveException, e:
            logger.exception("Problem removing running daemon: %r" % e)

        AuthorizationFree(authRef, kAuthorizationFlagDestroyRights)

    logger.info("Installing daemon version %r" % bundled_version)

    try:
        authRef = get_authorization()
        install_fsevents_daemon(authRef)

        installed_version = get_fsevents_daemon_installed_version()

        if installed_version:
            logger.info("Installed fsevents daemon successfully: version %r" %
                        installed_version)
        else:
            logger.error("Error installing fsevents daemon, see system.log.")

    except DaemonInstallException, e:
        logger.exception("Exception in fsevents daemon installation: %r" % e)
        raise e

    finally:
        AuthorizationFree(authRef, kAuthorizationFlagDestroyRights)


def install_config_and_daemons():
    """Install required data files and fsevents daemon.

    This function is a replacement for an installer. As such it is
    required on first-run, but it's also called every time we start
    up, in case anything has moved or been deleted.
    """

    # Do nothing if we are running from source:
    if getattr(sys, 'frozen', None) is None:
        return

    main_app_dir = ''.join(__file__.partition('.app')[:-1])
    main_app_resources_dir = os.path.join(main_app_dir,
                                          'Contents',
                                          'Resources')

    config_path = save_config_path('ubuntuone')

    # Always copy missing files, and for update.conf, overwrite
    # existing file to reflect current version ID, but only if bundled
    # version is newer, so we can change the file for testing.
    conf_filenames = [('syncdaemon.conf', False),
                      ('logging.conf', False),
                      ('update.conf', True)]
    for conf_filename, overwrite_old in conf_filenames:
        src_path = os.path.join(main_app_resources_dir,
                                conf_filename)
        dest_path = os.path.join(config_path,
                                 conf_filename)
        do_copy = False
        if not os.path.exists(dest_path):
            do_copy = True
        else:
            src_mtime = os.stat(src_path).st_mtime
            dest_mtime = os.stat(dest_path).st_mtime
            if overwrite_old and src_mtime >= dest_mtime:
                do_copy = True

        if do_copy:
            shutil.copyfile(src_path, dest_path)

    check_and_install_fsevents_daemon(main_app_dir)


def perform_update():
    """Spawn the autoupdate process, which will eventually kill us."""
    call_updater(install=True)


def uninstall_application():
    """Uninstall Ubuntu One."""
    # TODO
