/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Pawel Stolowski <pawel.stolowski@canonical.com>
 *
 */

using Unity.ShoppingLens;

public class Main
{
  public static int main (string[] args)
  {
    Test.init (ref args);

    Test.add_data_func ("/Unit/NoMarkup", test_no_markup);
    Test.add_data_func ("/Unit/BrTagSupport", test_br_tag_support);
    Test.add_data_func ("/Unit/BTagSupport", test_b_tag_support);
    Test.add_data_func ("/Unit/ITagSupport", test_i_tag_support);
    Test.add_data_func ("/Unit/UTagSupport", test_u_tag_support);
    Test.add_data_func ("/Unit/TtTagSupport", test_tt_tag_support);
    Test.add_data_func ("/Unit/STagSupport", test_s_tag_support);
    Test.add_data_func ("/Unit/StrikeTagSupport", test_strike_tag_support);
    Test.add_data_func ("/Unit/BigTagSupport", test_big_tag_support);
    Test.add_data_func ("/Unit/SmallTagSupport", test_small_tag_support);
    Test.add_data_func ("/Unit/SubTagSupport", test_sub_tag_support);
    Test.add_data_func ("/Unit/SupTagSupport", test_sup_tag_support);
    Test.add_data_func ("/Unit/UnsupportedTags", test_unsupported_tags);
    Test.add_data_func ("/Unit/NestedTags", test_nested_tags);
    Test.add_data_func ("/Unit/AmpEntitySupport", test_amp_entity);
    Test.add_data_func ("/Unit/NbspEntitySupport", test_nbsp_entity);
    Test.add_data_func ("/Unit/EntitiesArePreserved", test_basic_entities_are_preserved);
    Test.add_data_func ("/Unit/UnsupportedEntitiesAreRaw", test_unsupported_entities_are_raw);
    Test.add_data_func ("/Unit/NumericEntitiesArePreserved", test_num_entities_are_preserved);

    Test.run ();
    return 0;
  }

  internal static void test_no_markup ()
  {
    string input = "This is a\ntest";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "This is a\ntest");
  }

  internal static void test_br_tag_support ()
  {
    string input = "This is<br/> a<br />test<br>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "This is\n a\ntest\n");
  }

  internal static void test_b_tag_support ()
  {
    string input = "<B>T</b>his <b>is</b> a <B>test</B>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<b>T</b>his <b>is</b> a <b>test</b>");
  }

  internal static void test_i_tag_support ()
  {
    string input = "<I>T</i>his <i>is</i> a <I>test</I>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<i>T</i>his <i>is</i> a <i>test</i>");
  }

  internal static void test_u_tag_support ()
  {
    string input = "<U>T</u>his <u>is</u> a <U>test</U>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<u>T</u>his <u>is</u> a <u>test</u>");
  }

  internal static void test_tt_tag_support ()
  {
    string input = "<TT>T</TT>his <tt>is</tt> a <tT>test</Tt>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<tt>T</tt>his <tt>is</tt> a <tt>test</tt>");
  }

  internal static void test_s_tag_support ()
  {
    string input = "<S>T</s>his <s>is</s> a <S>test</S>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<s>T</s>his <s>is</s> a <s>test</s>");
  }

  internal static void test_strike_tag_support ()
  {
    string input = "<STRIKE>T</STRIKE>his <strike>is</strike> a <STRike>test</Strike>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<s>T</s>his <s>is</s> a <s>test</s>");
  }

  internal static void test_small_tag_support ()
  {
    string input = "<SMALL>T</SMALL>his <small>is</small> a <SMall>test</SmaLL>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<small>T</small>his <small>is</small> a <small>test</small>");
  }

  internal static void test_big_tag_support ()
  {
    string input = "<BIG>T></BIG>his <big>is</big> a <bIG>test</BiG>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<big>T></big>his <big>is</big> a <big>test</big>");
  }

  internal static void test_sub_tag_support ()
  {
    string input = "<SUB>T</SUB>his <sub>is</sub> a <suB>test</SuB>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<sub>T</sub>his <sub>is</sub> a <sub>test</sub>");
  }

  internal static void test_sup_tag_support ()
  {
    string input = "<SUP>T</SUP>his <sup>is</sup> a <suP>test</SuP>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<sup>T</sup>his <sup>is</sup> a <sup>test</sup>");
  }

  internal static void test_unsupported_tags ()
  {
    string input = "<foo>This</bar> is a <a href=\"http://foo.com\">test</a>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "This is a test");
  }

  internal static void test_nested_tags ()
  {
    string input = "<a href=\"wooo\"><small>Click me!</small></a>";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "<small>Click me!</small>");
  }

  internal static void test_amp_entity ()
  {
    string input = "Foo & Bar";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "Foo &amp; Bar");
  }

  internal static void test_nbsp_entity ()
  {
    string input = "Foo&nbsp;Bar";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "Foo Bar");
  }

  internal static void test_basic_entities_are_preserved ()
  {
    string input = "Foo &amp; bar &lt; &GT; &Quot; &apos;";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "Foo &amp; bar &lt; &gt; &quot; &apos;");
  }

  internal static void test_unsupported_entities_are_raw ()
  {
    string input = "Foo &frac14; bar &not;";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "Foo &amp;frac14 bar &amp;not");
  }

  internal static void test_num_entities_are_preserved ()
  {
    string input = "Foo&#160;bar &#8364;";
    string result = MarkupCleaner.html_to_pango_markup (input);
    assert (result == "Foo&#160;bar &#8364;");
  }
}
