/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Pawel Stolowski <pawel.stolowski@canonical.com>
 * based on python code by David Calle <davidc@framli.eu>
 *
 */

namespace Unity.VideoLens
{
  public class RemoteVideoScope : Unity.Scope
  {
    private static int REFRESH_INTERVAL = 3600; // fetch sources & recommendations once an hour
    private static int RETRY_INTERVAL = 60;     // retry sources/recommendations after a minute

    private Soup.Session session;
    private PreferencesManager preferences = PreferencesManager.get_default ();
    Gee.ArrayList<RemoteVideoFile?> recommendations;
    int64 recommendations_last_update = 0;
    Zeitgeist.DataSourceRegistry zg_sources;
    bool use_zeitgeist;

    public RemoteVideoScope ()
    {
      Object (dbus_path: "/net/launchpad/scope/remotevideos");
    }

    protected override void constructed ()
    {
      recommendations = new Gee.ArrayList<RemoteVideoFile?> ();

      use_zeitgeist = false;
      try
      {
        zeitgeist_init ();
        use_zeitgeist = true;
      }
      catch (Error e)
      {
        warning ("Failed to initialize Zeitgeist, won't store events");
      }

      session = new Soup.SessionAsync ();
      session.ssl_use_system_ca_file = true;
      session.ssl_strict = true;
      session.user_agent = "Unity Video Lens Remote Scope v" + Config.VERSION;
      session.add_feature_by_type (typeof (SoupGNOME.ProxyResolverGNOME));

      search_in_global = false;
      search_changed.connect ((search, search_type, cancellable) =>
      {
        update_search_async.begin (search, search_type, cancellable);
      });
      filters_changed.connect (on_filters_changed);
      sources.notify["filtering"].connect (on_filters_changed);

      generate_search_key.connect ((scope, search) =>
      {
        return search.search_string.strip ();
      });

      preferences.notify["remote-content-search"].connect ((obj, pspec) =>
      {
        queue_search_changed (SearchType.DEFAULT);
      });

      activate_uri.connect (on_activate_uri);
      preview_uri.connect (on_preview_uri);

      query_list_of_sources ();

      // refresh the at least once every 30 minutes
      GLib.Timeout.add_seconds (REFRESH_INTERVAL/2, () =>
      {
        queue_search_changed (SearchType.DEFAULT); return true;
      });

      try
      {
        export ();
      }
      catch (Error e)
      {
        error ("Failed to export scope: %s", e.message);
      }
    }

    /* Query the server for a list of sources that will be used
     * to build sources filter options and search queries.
     */
    private void query_list_of_sources ()
    {
      var msg = new Soup.Message ("GET", UbuntuVideoSearch.sources_uri ());
      session.queue_message (msg, sources_cb);
    }

    private Gee.ArrayList<RemoteVideoFile?>? handle_search_response (Soup.Message msg, bool is_treat_yourself = false)
    {
      if (msg.status_code != 200)
      {
        warning ("Unable to get results from the server: %u, %s", msg.status_code, msg.reason_phrase);
      }
      else
      {
        try
        {
          return UbuntuVideoSearch.process_search_results ((string)msg.response_body.data, is_treat_yourself);
        }
        catch (Error e)
        {
          warning ("Error processing search results: %s", e.message);
        }
      }
      return null;
    }

    private void sources_cb (Soup.Session session, Soup.Message msg)
    {
      uint interval = RETRY_INTERVAL;
      if (msg.status_code != 200)
      {
        warning ("Unable to query the server for a list of sources, %u: %s", msg.status_code, msg.reason_phrase);
      }
      else
      {
        try
        {
          var sources_array = UbuntuVideoSearch.process_sources_results ((string) msg.response_body.data);

          // remove all existing sources
          var to_remove = new SList<string> ();
          foreach (var opt in sources.options)
            to_remove.append (opt.id);
          foreach (var id in to_remove)
            sources.remove_option (id);

          // add sources
          foreach (var src in sources_array)
          {
            sources.add_option (src, src, null);
          }
          interval = REFRESH_INTERVAL;
        }
        catch (Error e)
        {
          warning ("Got invalid json from the server");
        }
      }
      GLib.Timeout.add_seconds (interval, () =>
      {
        query_list_of_sources (); return false;
      });
    }

    private Unity.ActivationResponse on_activate_uri (string rawuri)
    {
      var fakeuri = RemoteUri.from_rawuri (rawuri);
      if (fakeuri != null)
      {
        if (use_zeitgeist)
          zeitgeist_insert_event (fakeuri.uri, fakeuri.title, fakeuri.icon);
        try
        {
          GLib.AppInfo.launch_default_for_uri (fakeuri.uri, null);
          return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
        }
        catch (GLib.Error e)
        {
          warning ("Failed to launch default application for '%s': %s", fakeuri.uri, e.message);
        }
      }
      else
      {
        warning ("Invalid raw uri: '%s'", rawuri);
      }
      return new Unity.ActivationResponse (Unity.HandledType.NOT_HANDLED);
    }

    private Unity.ActivationResponse on_play_video (string rawuri)
    {
      return on_activate_uri (rawuri);
    }

    private Unity.Preview? build_preview (RemoteUri uri, RemoteVideoDetails? details)
    {
      string title = uri.title;
      string subtitle = "";
      string description = "";

      if (details != null)
      {
        title = details.title;
        description = details.description;

        if (details.release_date != null && details.release_date != "")
          subtitle = details.release_date;

        if (details.duration > 0)
        {
          string duration = ngettext ("%d min", "%d mins", details.duration).printf (details.duration);
          if (subtitle != "")
            subtitle += ", " + duration;
          else
            subtitle = duration;
        }
      }

      GLib.Icon thumbnail = new GLib.FileIcon (GLib.File.new_for_uri (details != null ? details.image : uri.icon));

      var real_preview = new Unity.MoviePreview (title, subtitle, description, thumbnail);
      var play_video = new Unity.PreviewAction ("play", _("Play"), null);
      play_video.activated.connect (on_play_video);
      real_preview.add_action (play_video);

      // For now, rating == -1 and num_ratings == 0 hides the rating widget from the preview
      real_preview.set_rating (-1, 0);

      if (details != null)
      {
        //TODO: For details of future source types, factor out common detail key/value pairs
        if (details.directors.length > 0)
          real_preview.add_info (new Unity.InfoHint ("directors", ngettext ("Director", "Directors", details.directors.length), null, string.joinv (", ", details.directors)));

        if (details.starring != null && details.starring != "")
          real_preview.add_info (new Unity.InfoHint ("cast", _("Cast"), null, details.starring));

        if (details.genres != null && details.genres.length > 0)
          real_preview.add_info (new Unity.InfoHint ("genres", ngettext("Genre", "Genres", details.genres.length), null, string.joinv (", ", details.genres)));

        // TODO: Add Vimeo & YouTube details for v1 of JSON API
        if (details.uploaded_by != null && details.uploaded_by != "")
          real_preview.add_info (new Unity.InfoHint ("uploaded-by", _("Uploaded by"), null, details.uploaded_by));

        if (details.date_uploaded != null && details.date_uploaded != "")
          real_preview.add_info (new Unity.InfoHint ("uploaded-on", _("Uploaded on"), null, details.date_uploaded));
      }

      return real_preview;
    }

    private Unity.Preview? on_preview_uri (string rawuri)
    {
      var fakeuri = RemoteUri.from_rawuri (rawuri);
      if (fakeuri != null)
      {
        if (fakeuri.details_uri != null && fakeuri.details_uri != "")
        {
          var preview = new AsyncPreview ();
          get_details.begin (fakeuri.details_uri, (obj, res) =>
          {
            try
            {
              var details = get_details.end (res);
              preview.preview_ready (build_preview (fakeuri, details));
            }
            catch (Error e)
            {
              warning ("Failed to fetch video details: %s", e.message);
              preview.preview_ready (build_preview (fakeuri, null));
            }
          });
          return preview;
        }
        else
        {
          return build_preview (fakeuri, null);
        }
      }
      else
      {
        warning ("Invalid raw uri: '%s'", rawuri);
      }

      return null;
    }

    private async RemoteVideoDetails? get_details (string url) throws Error
    {
      var msg = new Soup.Message ("GET", url);
      session.queue_message (msg, (session_, msg_) =>
      {
        msg = msg_;
        get_details.callback ();
      });

      yield;

      if (msg.status_code != 200)
      {
        warning ("Unable to get details from the server: %u, %s", msg.status_code, msg.reason_phrase);
        return null;
      }
      else
      {
        var details = UbuntuVideoSearch.process_details_results ((string) msg.response_body.data);
        return details;
      }
    }

    private async void update_search_async (LensSearch search, SearchType search_type, Cancellable? cancellable)
    {
      var search_string = search.search_string.strip ();
      debug ("Remote search string changed to: %s", search_string);

      var model = search.results_model;
      model.clear ();

      // only perform the request if the user has not disabled
      // online/commercial suggestions. That will hide the category as well.
      if (preferences.remote_content_search != Unity.PreferencesManager.RemoteContent.ALL)
      {
        search.finished();
        return;
      }

      // create a list of activated sources
      var active_sources = new Gee.ArrayList<string> (null);
      foreach (var opt in sources.options)
      {
        if (source_activated (opt.id))
          active_sources.add (opt.id);
      }

      // If all the sources are activated, don't bother passing them as arguments
      if (active_sources.size == sources.options.length ())
      {
        active_sources.clear ();
      }

      if (search_type == Unity.SearchType.DEFAULT)
      {
        if (at_least_one_source_is_on (active_sources))
        {
          yield perform_search (search_string, search, active_sources, cancellable);
        }
      }

      search.finished ();
    }

    private bool source_activated (string id)
    {
      bool active = sources.get_option (id).active;
      bool filtering = sources.filtering;

      if ((active && filtering) || (!active && !filtering))
        return true;
      return false;
    }

    /* Return a general activation state of all sources of this scope.
     * This is needed, because we don't want to show recommends if an option
     * from another scope is the only one activated
     */
    private bool at_least_one_source_is_on (Gee.ArrayList<string> active_sources)
    {
      return (sources.filtering && active_sources.size > 0 || !sources.filtering);
    }

    private void on_filters_changed ()
    {
      queue_search_changed (SearchType.DEFAULT);
    }

    /* Query the server with the search string and the list of sources.
     */
    private async void perform_search (string search_string, LensSearch search, Gee.ArrayList<string> active_sources, Cancellable? cancellable)
    {
      search.results_model.clear ();

      if ((search_string == null || search_string == "") && (active_sources.size == 0) && (recommendations.size > 0))
      {
        var time = new DateTime.now_utc ();
        if (time.to_unix () - recommendations_last_update < REFRESH_INTERVAL)
        {
          debug ("Updating search results with recommendations");
          update_results_model (search.results_model, recommendations);
          return;
        }
      }

      var url = UbuntuVideoSearch.build_search_uri (search_string, active_sources);
      debug ("Querying the server: %s", url);

      bool is_treat_yourself = (search_string == null || search_string == "" || active_sources.size == 0);
      var msg = new Soup.Message ("GET", url);

      session.queue_message (msg, (session_, msg_) =>
      {
        msg = msg_;
        perform_search.callback ();
      });

      var cancelled = false;
      ulong cancel_id = 0;
      if (cancellable != null)
      {
        cancel_id = cancellable.connect (() =>
        {
          cancelled = true;
          session.cancel_message (msg, Soup.KnownStatusCode.CANCELLED);
        });
      }

      yield;

      if (cancelled)
      {
        // we can't disconnect right away, as that would deadlock (cause
        // cancel_message doesn't return before invoking the callback)
        Idle.add (perform_search.callback);
        yield;
        cancellable.disconnect (cancel_id);
        throw new IOError.CANCELLED ("Cancelled");
      }

      if (cancellable != null)
      {
        // clean up
        cancellable.disconnect (cancel_id);
      }

      var results = handle_search_response (msg, is_treat_yourself);
      if (results != null)
      {
        if (search_string == null || search_string.strip () == "" && active_sources.size == 0)
        {
          debug ("Empty search, updating recommendations");
          var time = new DateTime.now_utc ();
          recommendations = results;
          recommendations_last_update = time.to_unix ();
        }
        update_results_model (search.results_model, results);
      }
    }

    private void update_results_model (Dee.Model model, Gee.ArrayList<RemoteVideoFile?> results)
    {
      foreach (var video in results)
      {
        if (video.uri.has_prefix ("http"))
        {
          var fake_uri = new RemoteUri (video.uri, video.title, video.icon, video.details_uri);
          var result_icon = video.icon;

          if (video.category == CAT_INDEX_MORE && video.price != null && video.price != "")
          {
            var anno_icon = new Unity.AnnotatedIcon (new FileIcon (File.new_for_uri (result_icon)));
            anno_icon.category = Unity.CategoryType.MOVIE;
            anno_icon.ribbon = video.price;
            result_icon = anno_icon.to_string ();
          }

          model.append (fake_uri.to_rawuri (), result_icon, video.category, "text/html", video.title, video.comment, video.uri);
        }
      }
    }

    private void zeitgeist_init () throws Error
    {
      zg_sources = new Zeitgeist.DataSourceRegistry ();
      var templates = new PtrArray.sized(1);
      var ev = new Zeitgeist.Event.full (Zeitgeist.ZG_ACCESS_EVENT, Zeitgeist.ZG_USER_ACTIVITY, "lens://unity-lens-video");
      templates.add ((ev as GLib.Object).ref());
      var data_source = new Zeitgeist.DataSource.full ("98898", "Unity Video Lens", "", templates);
      zg_sources.register_data_source (data_source, null);
    }

    private void zeitgeist_insert_event (string uri, string title, string icon)
    {
      var subject = new Zeitgeist.Subject.full (uri, Zeitgeist.NFO_VIDEO, Zeitgeist.NFO_REMOTE_DATA_OBJECT, "", uri, title, icon);
      var event = new Zeitgeist.Event.full (Zeitgeist.ZG_ACCESS_EVENT, Zeitgeist.ZG_USER_ACTIVITY, "lens://unity-lens-video");
      event.add_subject (subject);

      var ev_array = new PtrArray.sized(1);
      ev_array.add ((event as GLib.Object).ref ());
      Zeitgeist.Log.get_default ().insert_events_from_ptrarray (ev_array, null);
    }
  }
}
