/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Pawel Stolowski <pawel.stolowski@canonical.com>
 * based on python code by David Calle <davidc@framli.eu>
 *
 */

namespace Unity.VideoLens
{
  public class VideoScope : Unity.Scope
  {
    private static const int MAX_ZG_EVENTS = 24;
    private static const int CAT_INDEX_MY_VIDEOS = 0;
    private static const int CAT_INDEX_ONLINE = 1;
    private static const int CAT_INDEX_MORE = 2;
    private static const int REFRESH_TIMEOUT = 30;

    private static string cache_directory;

    private string videos_folder;
    private Unity.Extras.PreviewPlayer preview_player;
    private Thumbnailer thumbnailer;
    private Locate locate;
    private BlacklistTracker blacklist_tracker;

    public VideoScope ()
    {
      Object (dbus_path: "/net/launchpad/lens/video/main");

      videos_folder = GLib.Environment.get_user_special_dir (GLib.UserDirectory.VIDEOS);
      cache_directory = GLib.Environment.get_user_cache_dir () + "/unity-lens-video";

      // create cache directory
      try
      {
        var cache_dir = GLib.File.new_for_path (cache_directory);
        if (! cache_dir.query_exists (null))
          cache_dir.make_directory (null);
      }
      catch (Error e)
      {
        error ("Failed to create cache directory: %s", e.message);
      }

      blacklist_tracker = new BlacklistTracker ();
      thumbnailer = new Thumbnailer (cache_directory);
      locate = new Locate (cache_directory, videos_folder);

      search_in_global = true;
      sources.add_option ("local", _("My Videos"), null);
      provides_personal_content = true;

      GLib.Timeout.add_seconds (REFRESH_TIMEOUT, refresh_results);

      search_changed.connect ((search, search_type, cancellable) =>
      {
        dispatch_search (search, search_type, cancellable);
      });

      filters_changed.connect (on_filters_changed);
      sources.notify["filtering"].connect (on_filters_changed);
      preview_uri.connect ((uri) =>
      {
        return generate_preview_for_uri (uri);
      });
    }

    private bool refresh_results ()
    {
      debug ("Queuing new search because of timeout");
      queue_search_changed (Unity.SearchType.DEFAULT);
      return true;
    }

    private void on_filters_changed ()
    {
      queue_search_changed (Unity.SearchType.DEFAULT);
    }

    private async void dispatch_search (LensSearch search, SearchType search_type, Cancellable cancellable)
    {
      var search_string = search.search_string.strip ();
      var search_status = search;
      var model = search.results_model;
      debug ("Search changed to '%s'", search_string);

      if (source_activated ("local"))
      {
        if (search_type == Unity.SearchType.GLOBAL)
        {
          if (search_string == "")
          {
            model.clear ();
            if (search != null)
              search_status.finished ();
            debug ("Global view without search string : hide");
          }
          else
          {
            update_results_model (search_string, model, "global", cancellable, search_status);
          }
        }
        else
        {
          if (search_string == null || search_string == "")
          {
            try
            {
              zg_call (cancellable, search_status);
            }
            catch (GLib.Error e)
            {
              warning ("Failed to call zeitgeist: %s", e.message);
            }
          }
          else
          {
            update_results_model (search_string, model, "lens", cancellable, search_status);
          }
        }
      }
    }

    private void update_results_model (string search_string, Dee.Model model, string cat, Cancellable? cancellable, LensSearch search, bool clear_model = true)
    {
      var home_folder = GLib.Environment.get_home_dir ();

      if (videos_folder != home_folder)
      {
        locate.updatedb ();
      }

      var result_list = locate.run_locate (search_string, thumbnailer, video_filter);
      if (result_list != null)
      {
        GLib.Idle.add (() =>
        {
          result_list.sort ((GLib.CompareFunc?)sort_alpha);
          add_results (search, model, cat, cancellable, result_list, search_string, clear_model);
          return false;
        });
      }
    }

    internal void add_results (LensSearch search_status, Dee.Model model, string cat, Cancellable? cancellable, Gee.ArrayList<VideoFile?> result_list, string search, bool clear_model)
    {
      if (cancellable != null && !cancellable.is_cancelled ())
      {
        if (clear_model)
          search_status.results_model.clear ();

        foreach (var video in result_list)
        {
          results_model.append (video.uri, video.icon, video.category, "text/html", video.title, video.comment, video.uri);
        }

        if (search_status != null)
        {
          debug ("Search finished");
          search_status.finished ();
        }
      }
    }

    internal static int sort_alpha (VideoFile a, VideoFile b)
    {
      return a.lc_title.collate (b.lc_title);
    }

    private async void zg_call (Cancellable? cancellable, LensSearch search_status) throws Error
    {
      bool active = sources.get_option ("local").active;
      bool filtering = sources.filtering;
      string uri = active && filtering ? "file:*" : "*";

      var time_range = new Zeitgeist.TimeRange.to_now ();
      var event_template = new Zeitgeist.Event ();
      var subject = new Zeitgeist.Subject.full (uri, Zeitgeist.NFO_VIDEO, "", "", "", "", "");
      event_template.add_subject (subject);

      var templates = new PtrArray.sized (1);
      templates.add ((event_template as GLib.Object).ref());
      var results = yield Zeitgeist.Log.get_default ().find_events (time_range, templates, Zeitgeist.StorageState.ANY, MAX_ZG_EVENTS, Zeitgeist.ResultType.MOST_RECENT_SUBJECTS, cancellable);
      process_zg_events (results, cancellable, search_status);
    }

    internal bool video_filter (string path)
    {
      try
      {
        if (Utils.is_video (path) && !Utils.is_hidden (path))
        {
          var file = File.new_for_path (path);
          if (!is_blacklisted(file.get_uri ()))
            return true;
        }
      }
      catch (Error e)
      {
        warning ("Failed to get properties of '%s': %s", path, e.message);
      }
      return false;
    }

    internal void process_zg_events (Zeitgeist.ResultSet events, Cancellable cancellable, LensSearch search_status)
    {
      var result_list = new Gee.ArrayList<VideoFile?> ();

      foreach (var event in events)
      {
        if (cancellable.is_cancelled ())
          return;

        var event_uri = event.get_subject (0).get_uri ();
        if (event_uri.has_prefix ("file://"))
        {
          try
          {
            // If the file is local, we use the same methods
            // as other result items.
            var file = GLib.File.new_for_uri (event_uri);
            var path = file.get_path ();
            if (video_filter (path))
            {
              var finfo = file.query_info (GLib.FileAttribute.STANDARD_DISPLAY_NAME, GLib.FileQueryInfoFlags.NONE, null);
              VideoFile video = VideoFile ()
              {
                title = finfo.get_display_name (),
                lc_title = finfo.get_display_name ().down (),
                comment = "",
                uri = event_uri,
                icon = thumbnailer.get_icon (path),
                category = CAT_INDEX_MY_VIDEOS
              };
              result_list.add (video);
            }
          }
          catch (GLib.Error e)
          {
            warning ("Failed to access file '%s': %s", event_uri, e.message);
          }
        }
        else if (event_uri.has_prefix ("http"))
        {
          // If the file is distant, we take
          // all we need from Zeitgeist
          // this one can be any unicode string:
          VideoFile video = VideoFile ()
          {
            title = event.get_subject (0).get_text (),
            comment = "",
            uri = event_uri,
            icon = event.get_subject (0).get_storage (),
            category = CAT_INDEX_ONLINE
          };
          result_list.add (video);
        }
      }

      search_status.results_model.clear ();
      foreach (var video in result_list)
      {
        results_model.append (video.uri, video.icon, video.category, "text/html", video.title, video.comment, video.uri);
      }

      update_results_model ("", search_status.results_model, "lens", cancellable, search_status, false);
    }

    private Unity.ActivationResponse show_in_folder (string uri)
    {
      try
      {
        Unity.Extras.show_in_folder (uri);
        return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
      }
      catch (GLib.Error e)
      {
        warning ("Failed to show in folder '%s': %s", uri, e.message);
      }
      return new Unity.ActivationResponse (Unity.HandledType.NOT_HANDLED);
    }

    private Preview? generate_preview_for_uri (string uri)
    {
      debug ("Preview uri: %s", uri);

      Unity.Preview preview = null;
      var model = results_model;
      var iter = model.get_first_iter ();
      while (!model.is_last (iter))
      {
        if (model.get_string (iter, 0) == uri)
        {
          var title = model.get_string (iter, 4);
          string subtitle = "";
          int64 file_size = 0;
          bool local_video = uri.has_prefix ("file://");

          if (local_video)
          {
            var file = GLib.File.new_for_uri (uri);
            try
            {
              var finfo = file.query_info (GLib.FileAttribute.TIME_MODIFIED + "," + GLib.FileAttribute.STANDARD_SIZE, GLib.FileQueryInfoFlags.NONE, null);
              file_size = finfo.get_size ();
              var tval = finfo.get_modification_time ();
              var dt = new DateTime.from_timeval_local (tval);
              subtitle = dt.format ("%x, %X");
            }
            catch (Error e)
            {
              // empty subtitle
            }
          }
          var desc = model.get_string (iter, 5);

          preview = new Unity.MoviePreview (title, subtitle, desc, null);
          preview.closed.connect (on_preview_closed);
          var play_video = new Unity.PreviewAction ("play", _("Play"), null);
          preview.add_action (play_video);
          preview.image_source_uri = model.get_string (iter, 1);

          if (local_video)
          {
            var show_folder = new Unity.PreviewAction ("show-in-folder", _("Show in Folder"), null);
            show_folder.activated.connect (show_in_folder);
            preview.add_action (show_folder);
          }

          // we may get remote uris from zeitgeist - fetch details for local files only
          if (local_video)
          {
            var async_preview = new Unity.AsyncPreview ();
            preview.image_source_uri = uri;

            if (preview_player == null)
              preview_player = new Unity.Extras.PreviewPlayer ();

            preview_player.video_properties.begin (uri, (obj, res) =>
            {
              try
              {
                var props = preview_player.video_properties.end (res);
                if ("width" in props && "height" in props && "codec" in props)
                {
                  var width = props["width"].get_uint32 ();
                  var height = props["height"].get_uint32 ();
                  var codec = props["codec"].get_string ();
                  string dimensions = "%u*%u".printf (width, height);
                  if (width > 0 && height > 0)
                  {
                    var gcd = Utils.gcd (width, height);
                    dimensions += ", %u:%u".printf (width / gcd, height / gcd);
                  }

                  preview.add_info (new Unity.InfoHint ("format", _("Format"), null, codec));
                  preview.add_info (new Unity.InfoHint ("dimensions", _("Dimensions"), null, dimensions));
                  preview.add_info (new Unity.InfoHint ("size", _("Size"), null, GLib.format_size (file_size)));
                }
              }
              catch (Error e)
              {
                warning ("Couldn't get video details: %s", e.message);
              }
              async_preview.preview_ready (preview);
            });
            return async_preview;
          }
          break;
        }
        iter = model.next (iter);
      }

      if (preview == null)
        warning ("Couldn't find model row for requested preview uri: %s", uri);

      return preview;
    }

    private bool is_blacklisted (string uri)
    {
      foreach (var blacklisted_uri in blacklist_tracker.get_blacklisted_uris ())
      {
        if (uri.has_prefix (blacklisted_uri))
          return true;
      }
      return false;
    }

    private void on_preview_closed (Unity.Preview preview)
    {
      if (preview_player != null)
      {
        try
        {
          preview_player.close ();
        }
        catch (Error e)
        {
          warning ("Failed to close preview player: %s", e.message);
        }
      }
    }

    private bool source_activated (string source)
    {
      // Return the state of a sources filter option
      var active = sources.get_option (source).active;
      var filtering = sources.filtering;
      return (active && filtering) || (!active && !filtering);
    }

  }
}
