/*
 * Copyright (C) 2010 Canonical Ltd
   *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

/*
 * This file contains C bindings written in C++ to hook up with the
 * Xapian indexes generated by Ubuntu Software Center.
 *
 * It also hooks up with libgnome-menu and creates an attached in-memory
 * Xapian index of the menu contents.
 */

using namespace std;

#include <xapian.h>
#include <iostream>
#include <gmenu-tree.h>
#include <unity.h>
#include <gee.h>
#include <string.h>

#define SOFTWARE_CENTER_INDEX "/var/cache/software-center/xapian"
#define QUERY_PARSER_FLAGS Xapian::QueryParser::FLAG_BOOLEAN|Xapian::QueryParser::FLAG_PHRASE|Xapian::QueryParser::FLAG_LOVEHATE|Xapian::QueryParser::FLAG_WILDCARD

// values used in the database
#define XAPIAN_VALUE_APPNAME 170
#define XAPIAN_VALUE_PKGNAME 171
#define XAPIAN_VALUE_ICON 172
#define XAPIAN_VALUE_GETTEXT_DOMAIN 173
#define XAPIAN_VALUE_ARCHIVE_SECTION 174
#define XAPIAN_VALUE_ARCHIVE_ARCH 175
#define XAPIAN_VALUE_POPCON 176
#define XAPIAN_VALUE_SUMMARY 177
#define XAPIAN_VALUE_ARCHIVE_CHANNEL 178
#define XAPIAN_VALUE_DESKTOP_FILE 179

#include "unity-package-search.h"

struct _UnityPackageSearcher
{
  Xapian::Database *db;
  Xapian::Enquire *enquire;
  Xapian::QueryParser *query_parser;
};

/* Do generic searcher setup */
static void
init_searcher (UnityPackageSearcher *searcher)
{
	Xapian::Database db = *searcher->db;

    // Start an enquire session
    Xapian::Enquire *enquire = new Xapian::Enquire (db);
    enquire->set_sort_by_value (XAPIAN_VALUE_APPNAME, FALSE);
    searcher->enquire = enquire;

    // Create query parser
    Xapian::QueryParser *query_parser = new Xapian::QueryParser ();
    query_parser->add_prefix ("section", "AE");
    query_parser->add_prefix ("type", "AT");
    query_parser->add_prefix ("category", "AC");
    query_parser->add_prefix ("name", "AA");
    query_parser->add_prefix ("pkgname", "AP");
    query_parser->set_default_op (Xapian::Query::OP_AND);
    query_parser->set_database (db);
    searcher->query_parser = query_parser;
}

/* Recursively traverse a menu tree and add it to the index */
static void
index_menu_item (Xapian::WritableDatabase *db,
                 Xapian::TermGenerator    *indexer,
                 GMenuTreeItem            *item)
{
  Xapian::Document     doc;
  GSList              *iter;
  GMenuTreeEntry      *entry;
  UnityAppInfoManager *appman;
  GeeList             *cats;
  GeeIterator         *cats_iter;
  gchar               *dum1, *dum2, *dum3;

	g_return_if_fail (db != NULL);
	g_return_if_fail (indexer != NULL);
	g_return_if_fail (item != NULL);
	
	switch (gmenu_tree_item_get_type (item))
	{
	  case GMENU_TREE_ITEM_INVALID:
	    return;
	  case GMENU_TREE_ITEM_DIRECTORY:
	    /* Recurse into directory */
	    iter = gmenu_tree_directory_get_contents (GMENU_TREE_DIRECTORY (item));
	    for (; iter != NULL; iter = iter->next)
	      {
	        index_menu_item (db, indexer, GMENU_TREE_ITEM (iter->data));
	      }
	    break;
	  case GMENU_TREE_ITEM_ENTRY:
	    /* Add this entry to the index */
	    entry = GMENU_TREE_ENTRY (item);
	    
	    /* Store relevant values */
	    if (gmenu_tree_entry_get_display_name (entry))
	      doc.add_value (XAPIAN_VALUE_APPNAME,
  	                   gmenu_tree_entry_get_display_name (entry));
  	  if (gmenu_tree_entry_get_icon (entry))
        doc.add_value (XAPIAN_VALUE_ICON,
                       gmenu_tree_entry_get_icon (entry));
      if (gmenu_tree_entry_get_desktop_file_id (entry))
        doc.add_value (XAPIAN_VALUE_DESKTOP_FILE,
                       gmenu_tree_entry_get_desktop_file_id (entry));
      //doc.add_value (XAPIAN_VALUE_PKGNAME, "");
	    
	    /* Index full text data */
	    indexer->set_document(doc);
      if (gmenu_tree_entry_get_display_name (entry))
        indexer->index_text(gmenu_tree_entry_get_display_name (entry));
      if (gmenu_tree_entry_get_name (entry))
        indexer->index_text(gmenu_tree_entry_get_name (entry));
      if (gmenu_tree_entry_get_comment (entry))
        indexer->index_text(gmenu_tree_entry_get_comment (entry));
	    
	    /* Index the XDG categories */
	    appman = unity_app_info_manager_get_instance ();
	    cats = unity_app_info_manager_get_categories (appman,
	                                                  gmenu_tree_entry_get_desktop_file_id (entry));
	    
	    /* For unknown desktop ids the app info manager returns NULL.
	     * Also happens for many NoDisplay app infos, so it's not an error
	     * case as such */
	    if (cats == NULL)
	      {
	        //g_debug ("Unable to probe XDG categories for %s",
	        //         gmenu_tree_entry_get_desktop_file_id (entry));
	        g_object_unref (appman);
	        break;
	      }
	    
	    cats_iter = GEE_ITERATOR (gee_list_list_iterator (cats));
	    while (gee_iterator_next (cats_iter))
	      {
	        // FIXME: Free dum1?
	        dum1 = (gchar*) gee_iterator_get (cats_iter);
	        dum2 = g_ascii_strdown (dum1, -1);
	        dum3 = g_strconcat ("AC", dum2, NULL);
	        doc.add_term (dum3);
	        g_free (dum1);
	        g_free (dum2);
	        g_free (dum3);
	      }
	    g_object_unref (cats_iter);
	    g_object_unref (cats);
	    g_object_unref (appman);
	    
	    /* Always assume Type=Application for items in a menu... */
	    doc.add_term ("ATapplication");
	    
	    /* Index application names */
	    dum1 = (gchar *) gmenu_tree_entry_get_display_name (entry); // const
	    dum2 = g_strconcat ("AA", dum1, NULL);
      doc.add_term (dum2);
      g_free (dum2);
      
      dum1 = (gchar *) gmenu_tree_entry_get_name (entry); // const
	    dum2 = g_strconcat ("AA", dum1, NULL);
      doc.add_term (dum2);
      g_free (dum2);
      
      /* Index executable name */
      dum1 = g_strdup (gmenu_tree_entry_get_exec (entry)); // alloc
      if (dum1) {
	      dum2 = strstr (dum1, " "); // const
	      dum2 == NULL ? : *dum2 = '\0'; // const
	      dum2 = g_path_get_basename (dum1); // alloc
	      g_free (dum1);
	      dum1 = g_strconcat ("AA", dum2, NULL); // alloc
        doc.add_term (dum1);
        indexer->index_text (dum2);
        g_free (dum2);
        g_free (dum1);
      }
      
      db->add_document(doc);
	    break;
	  case GMENU_TREE_ITEM_SEPARATOR:
	  case GMENU_TREE_ITEM_HEADER:
	  case GMENU_TREE_ITEM_ALIAS:
	    break;
	  default:
	    g_warning ("Unexpected GMenuTreeItemType %u",
	               gmenu_tree_item_get_type (item));
	    return;
  }

  // Add the document to the database.
  
}

/* Create a searcher that searches in a menu tree. The menu tree
 * will be indexed into an in-memory Xapian index */
UnityPackageSearcher*
unity_package_searcher_new_for_menu (GMenuTree *menu)
{
  UnityPackageSearcher *searcher;
  Xapian::WritableDatabase *db;

  searcher = g_new0 (UnityPackageSearcher, 1);
  db = new Xapian::WritableDatabase ();
  searcher->db = db;
  searcher->db->add_database (Xapian::InMemory::open ());
  
  init_searcher (searcher);
  
  /* Index the menu recursively */
  Xapian::TermGenerator *indexer = new Xapian::TermGenerator ();
  index_menu_item (db, indexer,
                   GMENU_TREE_ITEM (gmenu_tree_get_root_directory (menu)));
  delete indexer;
  db->flush ();
  
  return searcher;
}

/* Create a new searcher that searches into the Xapian index
 * provided by the Software Center */
UnityPackageSearcher*
unity_package_searcher_new ()
{
  UnityPackageSearcher *searcher;

  searcher = g_new0 (UnityPackageSearcher, 1);

  // Xapian initialization
  try {
    searcher->db = new Xapian::Database (SOFTWARE_CENTER_INDEX);
    init_searcher (searcher);
  } catch(const Xapian::Error &error) {
    cerr << "Error loading package indexes: "  << error.get_msg() << endl;
    return NULL;
  }
  
  return searcher;
}

void
unity_package_searcher_free (UnityPackageSearcher *searcher)
{
  g_return_if_fail (searcher != NULL);

  delete searcher->db;
  delete searcher->enquire;
  delete searcher->query_parser;
  g_free (searcher);
}

UnityPackageSearchResult*
unity_package_searcher_search (UnityPackageSearcher *searcher,
                               const gchar          *search_string)
{
  UnityPackageSearchResult* result;

  g_return_val_if_fail (searcher != NULL, NULL);
  g_return_val_if_fail (search_string != NULL, NULL);

  string _search_string (search_string);
  Xapian::Query query = searcher->query_parser->parse_query (_search_string, QUERY_PARSER_FLAGS);
  //cout << "Performing query `" << query.get_description() << "'" << endl;

  // Perform search
  searcher->enquire->set_query(query);
  Xapian::MSet matches =
    searcher->enquire->get_mset(0, searcher->db->get_doccount ());	

  // Retrieve the results, note that we build the result->results
  // list in reverse order and then reverse it before we return it
  result = g_slice_new0 (UnityPackageSearchResult);
  result->num_hits = matches.get_matches_estimated ();
  for (Xapian::MSetIterator i = matches.begin();
       i != matches.end();
       ++i)
    {
      UnityPackageInfo *pkginfo = g_slice_new0 (UnityPackageInfo);
      Xapian::Document doc = i.get_document();

      string pkgname = doc.get_value (XAPIAN_VALUE_PKGNAME);
      pkginfo->package_name = g_strdup (pkgname.c_str ());

      string appname = doc.get_value (XAPIAN_VALUE_APPNAME);
      pkginfo->application_name = g_strdup (appname.c_str ());

      string desktop_file = doc.get_value (XAPIAN_VALUE_DESKTOP_FILE);
      pkginfo->desktop_file = g_strdup (desktop_file.c_str ());

      string icon = doc.get_value (XAPIAN_VALUE_ICON);
      pkginfo->icon = g_strdup (icon.c_str ());

      result->results = g_slist_prepend (result->results, pkginfo);
    }

  result->results = g_slist_reverse (result->results);
  return result;
}

// extern "C"
static void _free_package_info (gpointer pkg, gpointer data)
{
  UnityPackageInfo *pkginfo = (UnityPackageInfo*) pkg;

  g_free (pkginfo->package_name);
  g_free (pkginfo->application_name);
  g_free (pkginfo->desktop_file);
  g_free (pkginfo->icon);

  g_slice_free (UnityPackageInfo, pkg);
}

void
unity_package_search_result_free (UnityPackageSearchResult *result)
{
  g_return_if_fail (result != NULL);

  g_slist_foreach (result->results, _free_package_info, NULL);
  g_slist_free (result->results);
  g_slice_free (UnityPackageSearchResult, result);
}

