/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
using Dee;
using DBus;
using Zeitgeist;
using Zeitgeist.Timestamp;
using Unity.Place;
using Config;
using Gee;

namespace Unity.FilesPlace {
  
  const string ACTIVE_SECTION_HINT = "UnityActiveSection";
  const string EXTRA_ACTION_HINT = "UnityExtraAction";
  
  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";
  
  /* Helper class used to encapsulate a state in a navigation sequence,
   * used as generic for our Unity.Place.Browser instance */
  private class BrowsingState
  {
    public Section section;
    public Search? search;
    public string uri;    
  }

  public class Daemon : GLib.Object, Unity.Place.Activation
  {
    private Zeitgeist.Log log;
    private Zeitgeist.Index index;
    private Zeitgeist.Monitor monitor;

    private Unity.Place.Controller control;
    private Unity.Place.EntryInfo files;
    private Unity.Place.Browser<BrowsingState> browser;
    
    /* We use the sections_model in normal operation, but when changing
     * to browsing mode we switch over to use pathbar_model as the sections
     * model. Magic in libunity will ensure us that the Unity Shell is notified
     * (over dbus) that we changed the sections model*/
    private Dee.SharedModel sections_model;
    private Dee.SharedModel pathbar_model;
    private string? browsing_uri = null;
    private Section browsing_root;
    
    /* For each section we have a set of Zeitgeist.Event templates that
     * we use to query Zeitgeist */
    private Gee.List<PtrArray> section_templates;

    /* Store a maping of DateMonth to Dee.ModelIter. We map to the iter and
     * not simply the offset, because in theory anyone on the bus could
     * update the Dee.SharedModel we use for the groups changing the row
     * offsets*/
    private Gee.List<unowned Dee.ModelIter?> months;
    
    /* Keep track of the previous search, so we can determine when to
     * filter down the result set instead of rebuilding it */
    private Search? previous_search;
    
    /** Keep track of the previously active section and don't update
     * state if the section is in fact unchanged */
    private uint previous_active_section;
    
    private bool is_dirty;
    
    private Dee.Index entry_results_by_group;
    private Dee.Index global_results_by_group;

    construct
    {
      sections_model = new Dee.SharedModel("com.canonical.Unity.FilesPlace.SectionsModel",
                                           2, typeof (string), typeof (string));

      pathbar_model = new Dee.SharedModel("com.canonical.Unity.FilesPlace.PathBarModel",
                                          2, typeof (string), typeof (string));

      var groups_model = new Dee.SharedModel(
                                 "com.canonical.Unity.FilesPlace.GroupsModel",
                                 3, typeof (string), typeof (string),
                                 typeof (string));

      var global_groups_model = new Dee.SharedModel(
                                 "com.canonical.Unity.FilesPlace.GlobalGroupsModel",
                                 3, typeof (string), typeof (string),
                                 typeof (string));

      var results_model = new Dee.SharedModel(
                                 "com.canonical.Unity.FilesPlace.ResultsModel",
                                 6, typeof (string), typeof (string),
                                 typeof (uint), typeof (string),
                                 typeof (string), typeof (string));

      var global_results_model = new Dee.SharedModel(
                                 "com.canonical.Unity.FilesPlace.GlobalResultsModel",
                                 6, typeof (string), typeof (string),
                                 typeof (uint), typeof (string),
                                 typeof (string), typeof (string));

      section_templates = new Gee.ArrayList<PtrArray> ();
      prepare_section_templates();

      files = new EntryInfo ("/com/canonical/unity/filesplace/files");
      files.sections_model = sections_model;
      files.entry_renderer_info.groups_model = groups_model;
      files.entry_renderer_info.results_model = results_model;
      files.global_renderer_info.groups_model = global_groups_model;
      files.global_renderer_info.results_model = global_results_model;

      populate_sections ();
      populate_groups ();

      files.icon = @"$(Config.PREFIX)/share/unity/files.png";
      
      previous_search = null;
      previous_active_section = Section.LAST_SECTION; /* Must be an illegal section! */
      
      is_dirty = true;
      
      var analyzer = new Dee.Analyzer.for_uint_column (ResultsColumn.GROUP_ID);
      entry_results_by_group = new Dee.HashIndex (results_model,
                                                  analyzer);
      global_results_by_group = new Dee.HashIndex (global_results_model,
                                                   analyzer);


      // FIXME: We monitor on all events, restrict templates to file events
      var templates = new PtrArray();
      var event = new Zeitgeist.Event ();
      templates.add (event);
      monitor = new Zeitgeist.Monitor (new Zeitgeist.TimeRange.from_now (),
                                       (owned) templates);

      log = new Zeitgeist.Log();
      index = new Zeitgeist.Index();

      /* Listen for section changes */
      files.notify["active-section"].connect (
        (obj, pspec) => {
          if (!files.active)
            return;
          
          if (!(is_dirty || previous_active_section != files.active_section))
            return;
          
          is_dirty = false;
          
          var section = (Section)files.active_section;
          var _results_model = files.entry_renderer_info.results_model;
          var _groups_model = files.entry_renderer_info.groups_model;          
          if (search_is_invalid (files.active_search))
            {
              int group_override = section == Section.ALL_FILES ?
                                                           Group.RECENT : -1;
              update_without_search_async.begin(section,
                                                _results_model, _groups_model,
                                                group_override);
            }
          else
            {
              update_search_async.begin (files.active_search, section,
                                         _results_model, _groups_model,
                                         false, entry_results_by_group);
            }
          
          previous_active_section = files.active_section;
        }
      );

      /* Listen for changes to the place entry search */
      files.notify["active-search"].connect (
        (obj, pspec) => {
          var search = files.active_search;
          
          if (!files.active)
            return;
          
          if (!(Utils.search_has_really_changed (previous_search, search) || is_dirty))
            return;
          
          is_dirty = false;
            
          var _results_model = files.entry_renderer_info.results_model;
          var _groups_model = files.entry_renderer_info.groups_model;
          var section = (Section) files.active_section;
          if (search_is_invalid (files.active_search))
            {
              int group_override = section == Section.ALL_FILES ?
                                                           Group.RECENT : -1;
              update_without_search_async.begin(section,
                                                _results_model, _groups_model,
                                                group_override);
            }
          else
            {
              update_search_async.begin (search, section,
                                         _results_model, _groups_model,
                                         Utils.check_is_filter_search(search,
                                                                      previous_search),
                                         entry_results_by_group);
             }
          previous_search = search;
        }
      );

      /* Listen for changes to the global search */
      files.notify["active-global-search"].connect (
        (obj, pspec) => {
          var search = files.active_global_search;
          
          if (!Utils.search_has_really_changed (previous_search, search))
            return;
            
          var _results_model = files.global_renderer_info.results_model;
          var _groups_model = files.global_renderer_info.groups_model;
          update_search_async.begin (search,
                                     Section.ALL_FILES,
                                     _results_model, _groups_model,
                                     Utils.check_is_filter_search(search,
                                                                  previous_search),
                                     global_results_by_group,
                                     Group.FILES);
          previous_search = search;
        }
      );
      
      /* Listen for when the place is hidden by the Unity Shell, and reset
       * all state when we are deactivated */
      files.notify["active"].connect (
        (obj, pspec) => {
          debug ("Activated: %i", (int)files.active);
          is_dirty = true;
          if (!files.active)
            reset_browsing_state ();
        }
      );

      /* We should not do anything with the results model
       * until we receieve the 'ready' signal */
      results_model.ready.connect (this.on_results_model_ready);

      sections_model.connect ();
      pathbar_model.connect ();
      groups_model.connect ();
      global_groups_model.connect ();
      results_model.connect ();
      global_results_model.connect ();            

      /* The last thing we do is export the controller. Once that is up,
       * clients will expect the SharedModels to work */
      control = new Unity.Place.Controller ("/com/canonical/unity/filesplace");
      control.add_entry (files);
      control.activation = this;

      try {
        control.export ();
      } catch (DBus.Error error) {
        critical ("Failed to export DBus service for '%s': %s",
                  control.dbus_path, error.message);
      }
      
      /* The browser will automagically be exported/unexported on the bus
       * when we set/unset the 'browser' property on the 'files' EntryInfo.
       * Likewise, setting the browser also sets the UnityPlaceBrowserPath
       * and UnitySectionStyle hints accordingly.
       * Such works the magic of the libunity API :-) */
      browser = new Unity.Place.Browser<BrowsingState> (
                                     "/com/canonical/unity/filesplace/browser");
      browser.back.connect (
        (browser, state, comment) => {
          debug ("Go back to: %s", (state as BrowsingState).uri);
          var f = File.new_for_uri ((state as BrowsingState).uri);
          browse_folder.begin (f);
        }
      );
      
      browser.forward.connect (
        (browser, state, comment) => {
          debug ("Go forward to: %s", (state as BrowsingState).uri);
          var f = File.new_for_uri ((state as BrowsingState).uri);
          browse_folder.begin (f);
        }
      );
    }

    private void populate_sections ()
    {
      var sections = files.sections_model;

      if (sections.get_n_rows() != 0)
        {
          critical ("The sections model should be empty before initial population");
          sections.clear ();
        }

      sections.append (SectionsColumn.DISPLAY_NAME, _("All Files"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Documents"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Folders"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Images"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Audio"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Videos"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Presentations"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Other"),
                       SectionsColumn.ICON_HINT, "", -1);
    }

    private void populate_groups ()
    {
      var groups = files.entry_renderer_info.groups_model;

      if (groups.get_n_rows() != 0)
        {
          critical ("The groups model should be empty before initial population");
          groups.clear ();
        }

      /* Always expand the Favorite Folders group */
      files.entry_renderer_info.set_hint ("ExpandedGroups",
                                          @"$((uint)Group.FAVORITE_FOLDERS)");

      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Top Results"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-mostused.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityDefaultRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Recent"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-recent.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityDefaultRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Downloads"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-downloads.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityDefaultRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Favorite Folders"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-favoritefolders.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityDefaultRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Files"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-recent.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityEmptySearchRenderer",
                     GroupsColumn.DISPLAY_NAME, "No search results", /* No i18n. Should never be rendered */
                     GroupsColumn.ICON_HINT, "", -1);
      groups.append (GroupsColumn.RENDERER, "UnityEmptySectionRenderer",
                     GroupsColumn.DISPLAY_NAME, "Empty section", /* No i18n. Should never be rendered */
                     GroupsColumn.ICON_HINT, "", -1);
      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Today"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Yesterday"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("This week"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Last Week"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("This Month"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Past Six Months"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("This Year"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Last Year"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
      // FIXME: For prehistoric items use actual year, eg "2009"


      months = new Gee.ArrayList<unowned Dee.ModelIter?>();
      months.add(null);
      for (uint i = 1; i <= DateMonth.DECEMBER; i++)
      {
        unowned Dee.ModelIter iter = groups.append (
                                  GroupsColumn.RENDERER, "UnityFileInfoRenderer",
                                  GroupsColumn.DISPLAY_NAME, Utils.get_month_name ((DateMonth)i),
                                  GroupsColumn.ICON_HINT, ICON_PATH + "group-daterange.svg", -1);
        months.add(iter);

      }
    }

    private void prepare_section_templates ()
    {
      PtrArray templates;
      Event event;

      /* HACK ALERT: All the (event as GLib.Object).ref() are needed because
       *             GPtrArray doesn't grab a ref to the event objects */

      /* Section.ALL_FILES */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               "", "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.DOCUMENTS
       * FIXME: Filter out presentations: https://bugs.launchpad.net/zeitgeist/+bug/592599 */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_DOCUMENT,
                                               "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.FOLDERS.
       * FIXME: We probably need to be clever here and use something
       *       like subject.origin in stead of NFO_FOLDER */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               "", "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.IMAGES */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_IMAGE, "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);
      
      /* Section.AUDIO */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_AUDIO, "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.VIDEOS */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_VIDEO, "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.PRESENTATIONS
       * FIXME: Zeitgeist logger needs to user finer granularity
       *        on classification as I am not sure it uses
       *        NFO_PRESENTATION yet */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_PRESENTATION, "", "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);

      /* Section.OTHER 
       * Note that subject templates are joined with logical AND */
      templates = new PtrArray.sized(1);
      event = new Event.full("", ZG_USER_ACTIVITY, "");
      event.add_subject (new Subject.full ("file:*",
                                           "!"+NFO_DOCUMENT, "", "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_IMAGE,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_AUDIO,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_VIDEO,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_PRESENTATION,
                                           "",
                                           "", "", "", ""));
      templates.add ((event as GLib.Object).ref());
      section_templates.add (templates);
    }

    private bool search_is_invalid (Search? search)
    {
      /* This boolean expression is unfolded as we seem to get
       * some null dereference if we join them in a big || expression */
      if (search == null)
        return true;
      else if (search.get_search_string () == null)
        return true;
      
      return search.get_search_string () == "";
    }

    public void on_results_model_ready (Dee.SharedModel model)
    {
      /*var section = (Section)files.active_section;
      var _results_model = files.entry_renderer_info.results_model;
      var _groups_model = files.entry_renderer_info.groups_model;
      update_without_search_async.begin(section, _results_model, _groups_model);*/
      monitor.events_inserted.connect (on_events_inserted);

      // FIXME: Use the section queries as monitor templates as well
      //        and make sure we de-dupe the existing results as updates
      //        trickle in.
      //        See https://bugs.launchpad.net/anjali/+bug/598078
      //log.install_monitor.begin (monitor, null);
    }

    private string prepare_search_string (Search? search)
    {
      var s = search.get_search_string ();

      if (s.has_suffix (" "))
        s = s.strip ();

      if (!s.has_suffix ("*"))
        s = s + "*";

      return s;
    }

    private async void update_search_async  (Search search,
                                             Section section,
                                             Dee.Model results_model,
                                             Dee.Model groups_model,
                                             bool is_filter_search,
                                             Dee.Index results_by_group,
                                             int group_override = -1)
    {
      // FIXME: Implement in-folder searching
      if (files.browser != null)
        {
          warning ("In-folder searching not implemented yet");
          return;
        }
      
      if (search_is_invalid (search))
        {
          update_without_search_async.begin (section,
                                             results_model,
                                             groups_model);
          return;
        }

      var search_string = prepare_search_string (search);
      var templates = section_templates.get((int)section);  
      
      try {
        /* Get relevancy ranked results for the "Top Results" group */
        var timer = new Timer ();
        var results = yield index.search (search_string,
                                          new Zeitgeist.TimeRange.anytime(),
                                          templates,
                                          Zeitgeist.StorageState.ANY,
                                          50,
                                          Zeitgeist.ResultType.RELEVANCY,
                                          null);
        
        timer.stop ();
        debug ("Found %u/%u Top Results for search '%s' in %fms",
               results.size (), results.estimated_matches (),
               search_string, timer.elapsed()*1000);

        if (!is_filter_search)
          {
            /* Don't clear the results before the first ones are ready */
            results_model.clear ();            
          }
        else
          {
            /* Remove everything in the Top Results group. We can not do filter
             * searches here as it's sorted by relevancy alone */
            uint group = Group.TOP_RESULTS;
            var top_results = results_by_group.lookup (@"$group",
                                                       TermMatchFlag.EXACT);
            foreach (var row in top_results)
              results_model.remove (row);
          }
        
        Unity.FilesPlace.append_events_sorted (results,
                                               results_model,
                                               groups_model,
                                               section,
                                               Group.TOP_RESULTS);        
        
        /* Get time-grouped results */
        var result_type = section == Section.FOLDERS ?
                               ResultType.MOST_RECENT_ORIGIN : 
                               ResultType.MOST_RECENT_SUBJECTS;
        timer = new Timer ();
        results = yield index.search (search_string,
                                      new Zeitgeist.TimeRange.anytime(),
                                      templates,
                                      Zeitgeist.StorageState.ANY,
                                      100,
                                      result_type,
                                      null);
        
        timer.stop ();
        debug ("Found %u/%u time grouped results for search '%s' in %fms",
               results.size (), results.estimated_matches (),
               search_string, timer.elapsed()*1000);
        
        if (!is_filter_search)
          {
            Unity.FilesPlace.append_events_sorted (results,
                                                   results_model, groups_model,
                                                   section,
                                                   group_override);
          }
        else
          {
            /* We must filter all groups except Top Results.
             * IMPORTANT: This code assumes Group.TOP_RESULTS is the very first
             *            and that Group.RECENT is the second group! */            
            Set<string> uris_ = Utils.get_uri_set (results);
            uint n_groups = groups_model.get_n_rows ();
            for (uint group_ = Group.RECENT; group_ < n_groups; group_++)
              Utils.apply_uri_filter (uris_,
                                      results_by_group.lookup (@"$group_",
                                                               TermMatchFlag.EXACT));
          }

      } catch (GLib.Error e) {
        warning ("Error performing search '%s': %s",
                 search.get_search_string (), e.message);
      }
      
      check_empty_search (search, results_model);
    }

    private async void update_without_search_async (Section section,
                                                    Dee.Model results_model,
                                                    Dee.Model groups_model,
                                                    int group_override = -1)
    {
      bool active_section_hint_new = false;
    
      if (files.browser != null)
        {
          if (section != 0)
            {
              File folder = get_folder_for_pathbar_section (section);
              try {	            
                yield activate (folder.get_uri ());
              } catch (DBus.Error ee) {
	            warning ("Failed to activate URI '%s': %s",
	                     folder.get_uri (), ee.message);
              }
              debug ("Browsed %s from path bar", folder.get_uri ());
              return;
            }
          else
            {
              /* The root section of the pathbar was clicked.
               * Leave folder browsing mode */
              debug ("Root section of pathbar activated. Leaving browsing mode");
              reset_browsing_state ();
              files.active_section = browsing_root;
              files.set_hint (ACTIVE_SECTION_HINT, @"$browsing_root");
              section = browsing_root;
              active_section_hint_new = true;
            }          
        }
      
      /* If we have the UnityActiveSection hint and we didn't just set it,
       * then clear it */
      if (files.get_hint (ACTIVE_SECTION_HINT) != null &&
          !active_section_hint_new)
        {
          // Hack alert: We do the notify() because we need to trigger a DBus
          //             signal that the entry has changed
          files.clear_hint (ACTIVE_SECTION_HINT);
          files.notify_property ("active-section");
          debug ("Clearing active section hint");
        }
      
      var sections_model = files.sections_model;

      if (Section.LAST_SECTION != sections_model.get_n_rows())
        {
          critical ("Section model malformed");
          return;
        }

      if (section > Section.LAST_SECTION)
        {
          critical ("Active section out of bounds: %u", section);
          return;
        }

      var templates = section_templates.get((int)section);
      var result_type = section == Section.FOLDERS ?
                               ResultType.MOST_RECENT_ORIGIN : 
                               ResultType.MOST_RECENT_SUBJECTS;

      try {
        var timer = new Timer();
        var events = yield log.find_events (
                                        new Zeitgeist.TimeRange.anytime(),
                                        templates,
                                        Zeitgeist.StorageState.ANY,
                                        100,
                                        result_type,
                                        null);
        timer.stop ();
        debug ("Got %u events for section %u in %fms",
               events.size(), section, timer.elapsed()*1000);

        /* Don't clear the model before we have the results ready */
        results_model.clear();
        
        if (section == Section.ALL_FILES)
          {
            yield update_favorites_async (results_model, groups_model);
            yield update_downloads_async (results_model, groups_model);
          }

        Unity.FilesPlace.append_events_sorted (events,
                                               results_model, groups_model,
                                               section,
                                               group_override);
      } catch (GLib.Error e) {
        warning ("Error fetching recetnly used files: %s", e.message);
      }
      
      check_empty_section (section, results_model);
    }

    private async void update_favorites_async (Dee.Model results_model,
                                               Dee.Model groups_model)
    {
      // FIXME: Monitor bookmarks file and update on changes
      string contents;
      
      try {
        var favorites_file = @"$(Environment.get_home_dir())/.gtk-bookmarks";
        FileUtils.get_contents (favorites_file, out contents);
        
      } catch (FileError e) {
        warning ("Failed to read favorites: %s", e.message);
        return;
      }
      
      string[] favorites = contents.split ("\n");
      string mimetype = "inode/directory";
      
      foreach (var uri in favorites)
      {
        if (uri == "")
          continue;
              
        string[] parts = uri.split (" ", 2);
        string display_name;
        
        if (parts.length == 1)
          {
            display_name = Uri.unescape_string (uri);
            display_name = Filename.display_basename (display_name);
          }
        else if (parts.length == 2)
          {
            uri = parts[0];
            display_name = parts[1];
          }
        else
          {
            warning ("Internal error computing display name for favorite '%s'",
                     uri);
            display_name = uri;
          }
        
        var icon = Utils.get_icon_for_uri (uri, mimetype);
        
        results_model.append (ResultsColumn.URI, uri,
                              ResultsColumn.ICON_HINT, icon,
                              ResultsColumn.GROUP_ID, Group.FAVORITE_FOLDERS,
                              ResultsColumn.MIMETYPE, mimetype,
                              ResultsColumn.DISPLAY_NAME, display_name,
                              ResultsColumn.COMMENT, uri,
                              -1);
      }
    }

    private async void update_downloads_async (Dee.Model results_model,
                                               Dee.Model groups_model)
    {
      // FIXME: The Downloads folder and update on changes
      unowned string download_path =
                 Environment.get_user_special_dir (UserDirectory.DOWNLOAD);
      var download_dir = File.new_for_path (download_path);
      SList<FileInfo> downloads;
      
      try {
        downloads = yield Utils.list_dir (download_dir);
      } catch (GLib.Error e) {
        warning ("Failed to list downloads from directory '%s': %s",
                 download_path, e.message);
        return;
      }
      
      /* Sort files by mtime, we do an ugly nested ternary
       * to avoid potential long/int overflow */
      downloads.sort ((CompareFunc) Utils.cmp_file_info_by_mtime);
      
      foreach (var info in downloads)
      {
        var uri = download_dir.get_child (info.get_name ()).get_uri ();
        var mimetype = info.get_content_type ();
        var icon_hint = Utils.check_icon_string (uri, mimetype, info);
        results_model.append (ResultsColumn.URI, uri,
                              ResultsColumn.ICON_HINT, icon_hint,
                              ResultsColumn.GROUP_ID, Group.DOWNLOADS,
                              ResultsColumn.MIMETYPE, mimetype,
                              ResultsColumn.DISPLAY_NAME, info.get_display_name (),
                              ResultsColumn.COMMENT, uri,
                              -1);
      }
    }

    private void on_events_inserted (Zeitgeist.Monitor mon,
                                     Zeitgeist.TimeRange time_range,
                                     Zeitgeist.ResultSet events)
    {
      // FIXME: Since we don't really have the timestamps for the evens in
      //        our model, we can't insert the events in the correct place
      //        although it's likely fine to just prepend them

      var results_model = files.entry_renderer_info.results_model;
      var groups_model = files.entry_renderer_info.groups_model;

      foreach (var ev in events)
        {
          if (ev.num_subjects() > 0)
            {
              // FIXME: We only use the first subject...
              Zeitgeist.Subject su = ev.get_subject(0);
              if (File.new_for_uri (su.get_uri ()).query_exists ( null ) == true)
                {
                  string icon = Utils.get_icon_for_uri (su.get_uri (),
                                                        su.get_mimetype ());
                  uint group_id = Utils.get_time_group (ev, groups_model);

                  debug ("Notify %s, %s, %u", su.get_uri(), su.get_mimetype(), group_id);

                  results_model.prepend (ResultsColumn.URI, su.get_uri(),
                                         ResultsColumn.ICON_HINT, icon,
                                         ResultsColumn.GROUP_ID, group_id,
                                         ResultsColumn.MIMETYPE, su.get_mimetype(),
                                         ResultsColumn.DISPLAY_NAME, su.get_text(),
                                         ResultsColumn.COMMENT, su.get_uri (),
                                         -1);
                }
            }
        }
    }

    private void reset_browsing_state ()
    {
      /* We check for actual changes before resetting the properties, in order
       * to avoid spurious PlaceEntryInfoChanged signals which Unity doesn't
       * like that much */
      if (files.browser != null)
        {
          files.browser = null;
        }
      
      if (files.sections_model != sections_model)
        {
          files.sections_model = sections_model;
        }
      
      files.clear_hints ();
      is_dirty = true;
    }

    /**
     * Override of the default activation handler. The files place daemon
     * can handle activation of folders which puts it into "folder browsing mode"
     */
    public async uint32 activate (string uri)
    {
      /* When Unity asks us to activate "." it's a special request for
       * activating our UnityExtraAction hint. Which for the files' place
       * is  "browse current folder in Nautilus" */
      if ("." == uri)
        {
          browse_current_folder_in_nautilus ();
          return ActivationStatus.ACTIVATED_HIDE_DASH;
        }
      
      var f = File.new_for_uri (uri);
      
      if (f.query_file_type (0, null) != FileType.DIRECTORY)
        {
          debug ("Declined activation of URI '%s': Not a directory", uri);
          return ActivationStatus.NOT_ACTIVATED;
        }
      
      debug ("Browsing folder: %s", uri);

      /* Record the URI in the browser */
      var state = new BrowsingState ();
      state.search = files.active_search;
      state.section = (Section)files.active_section;
      state.uri = uri;
      browser.record_state (state, uri); // FIXME: Nicer comment than just the URI
      browsing_uri = uri;
      
      /* We need to mark us as dirty - otherwise we'll discard the section
       * change signal when changed back to the section of the pathbar root */
      is_dirty = true;
      
      /* If we are entering browsing mode record the active section,
       * and set the UnityExtraAction hint to "Browse current folder
       * in Nautilus" */
  	  if (files.browser == null)    
        {
          browsing_root = (Section) files.active_section;
          
          /* The UnityExtraIcon hint must contain a serialized, GIcon,
           * but a path to an image qualifies as such */
          files.set_hint (EXTRA_ACTION_HINT, ICON_PATH + "open-folder.svg");
        }      

      /* Setting the browser property does all the necessary dbus magic,
       * it is also used to indicate to our selves that we are in browsing
       * mode */          
      files.browser = browser;      
      
      /* Change our files.sections_model over to a special model
       * we use to render the breadcrumbs/path bar. Changing the
       * files.sections_model property will automatically notify Unity
       * over DBus with the PlaceEntryInfoChanged signal */
      files.sections_model = pathbar_model;
       
      browse_folder.begin (f);

      return ActivationStatus.ACTIVATED_SHOW_DASH;
    }
    
    private async void browse_folder (File folder)
    {         
      var results_model = files.entry_renderer_info.results_model;
      SList<FileInfo> file_infos;
      
      // FIXME: Alphabetic sorting of folder contents
      /* Populate the results_model with folder contents */
      try {
        file_infos = yield Utils.list_dir (folder);
      } catch (GLib.Error err) {
        warning ("Failed to browse folder '%s': %s",
                 folder.get_uri (), err.message);
        return;
      }

      results_model.clear ();
      foreach (var info in file_infos) {
	    if (info.get_is_hidden() || info.get_is_backup ())
	      continue;
	    
        var uri = folder.get_child (info.get_name ()).get_uri ();
        var mimetype = info.get_content_type ();
        var icon_hint = Utils.check_icon_string (uri, mimetype, info);
        debug ("Found child: %s", uri);
        results_model.append (ResultsColumn.URI, uri,
                              ResultsColumn.ICON_HINT, icon_hint,
                              ResultsColumn.GROUP_ID, 0, // FIXME: which group for folder browsing?
                              ResultsColumn.MIMETYPE, mimetype,
                              ResultsColumn.DISPLAY_NAME, info.get_display_name (),
                              ResultsColumn.COMMENT, uri, // FIXME: Relative path from $HOME
                              -1);
      }
      
      yield update_pathbar_model (folder);
    }
    
    private async void update_pathbar_model (File folder)
    {
      /* Update the pathbar model with path relative to the home dir
       * Unity should be showing the pathbar instead of the normal sections */
      // FIXME: Don't .clear() the model, but compute the diff and update in stead
      pathbar_model.clear ();
      string home_path = Environment.get_home_dir ();
      File parent = folder;
      FileInfo finfo;      
      do {
	    try {
		  if (parent.get_path () == home_path)
		  	break;
		  
	      finfo = parent.query_info (Utils.file_attribs,
	                                 FileQueryInfoFlags.NONE);
        } catch (GLib.Error e) {
	      warning ("Failed to compute breadcrumb path: %s", e.message);  
	      break;
        }
        pathbar_model.prepend (0, finfo.get_display_name (), 1, "", -1);
      } while ((parent = parent.get_parent ()) != null);
      string section_name = sections_model.get_string (sections_model.get_iter_at_row (browsing_root), 0);
      pathbar_model.prepend (0, section_name, -1);
    }
    
    /* Calculates the file URI of the path currently being browsed,
     * up to the path element number @path_element */
    private File get_folder_for_pathbar_section (uint path_element)
    {
      uint n_path_sections = pathbar_model.get_n_rows ();
      if (path_element >= n_path_sections - 1)
        {
		      warning ("Path section out of bounds: %u (of %u)",
		               path_element, n_path_sections);
		      return File.new_for_path (Environment.get_home_dir ());
        }
	  	  
      uint to_remove = n_path_sections - path_element - 1;
      File uri = File.new_for_uri (browsing_uri);
      
      while (to_remove > 0)
        {
          uri = uri.get_parent ();
          to_remove--;
        }
	  
      debug ("For section %u found: %s", path_element, uri.get_uri ());
      return uri;
    }
    
    /* Launch default file manager (Nautilus) on the currently browsed folder */
    private void browse_current_folder_in_nautilus ()
    {
      if (browsing_uri == null || browsing_uri == "")
        {
        warning ("Unable to open current folder in file manager. " +
                 "We don't have a current folder!");
          return;
        }
      
      debug ("Opening folder current folder '%s' in file manager",
             browsing_uri);
      
      try {
        AppInfo.launch_default_for_uri (browsing_uri, null);
      } catch (GLib.Error e) {
        warning ("Failed to open current folder '%s' in file manager: %s",
                 browsing_uri, e.message);
      }
    }
    
    public void check_empty_search (Search? search,
                                    Dee.Model results_model)
    {
      if (results_model.get_n_rows () > 0)
        return;
      
      if (search_is_invalid(search))
        return;
      
      results_model.append (ResultsColumn.URI, "",
                            ResultsColumn.ICON_HINT, "",
                            ResultsColumn.GROUP_ID, Group.EMPTY_SEARCH,
                            ResultsColumn.MIMETYPE, "",
                            ResultsColumn.DISPLAY_NAME, _("Your search did not match any files"),
                            ResultsColumn.COMMENT, "",
                            -1);      
      
      // FIXME: Use prefered browser
      // FIXME: URL escape search string
      results_model.append (ResultsColumn.URI, @"http://google.com/#q=$(search.get_search_string())",
                            ResultsColumn.ICON_HINT, "",
                            ResultsColumn.GROUP_ID, Group.EMPTY_SEARCH,
                            ResultsColumn.MIMETYPE, "",
                            ResultsColumn.DISPLAY_NAME, _("Search the web"),
                            ResultsColumn.COMMENT, "",
                            -1);
    }
    
    public void check_empty_section (Section section,
                                     Dee.Model results_model)
    {
      if (results_model.get_n_rows () > 0)
        return;
            
      string section_name;
      
      switch (section)
      {
        case Section.ALL_FILES:
          section_name = _("files");
          break;
        case Section.DOCUMENTS:
          section_name = _("documents");
          break;
        case Section.FOLDERS:
          section_name = _("folders");
          break;
        case Section.IMAGES:
          section_name = _("images");
          break;
        case Section.AUDIO:
          section_name = _("audio files");
          break;
        case Section.VIDEOS:
          section_name = _("videos");
          break;
        case Section.PRESENTATIONS:
          section_name = _("presentations");
          break;
        case Section.OTHER:
            section_name = _("other files");
          break;
        default:
          warning ("Unknown section number: %u", section);
          section_name = _("files");
          break;
      }
      
      results_model.append (ResultsColumn.URI, "",
                            ResultsColumn.ICON_HINT, "",
                            ResultsColumn.GROUP_ID, Group.EMPTY_SECTION,
                            ResultsColumn.MIMETYPE, "",
                            /* TRANSLATORS: The %s is plural. Eg. "files" */
                            ResultsColumn.DISPLAY_NAME, _("There are no %s in your Home folder").printf (section_name),
                            ResultsColumn.COMMENT, "",
                            -1);
    }
    
  } /* End: Daemon class */
  
  /* Appends a set of Zeitgeist.Events to our Dee.Model assuming that
   * these events are already sorted with descending timestamps */
  public void append_events_sorted (Zeitgeist.ResultSet events,
                                    Dee.Model results,
                                    Dee.Model groups,
                                    Section section,
                                    int group_override = -1)
    {
      Gee.Set<string> seen_uris = new Gee.HashSet<string>(str_hash);
    
      foreach (var ev in events)
        {
          if (ev.num_subjects() > 0)
            {
              // FIXME: We only use the first subject...
              Zeitgeist.Subject su = ev.get_subject(0);
              
              string uri = su.get_uri ();
           	  string display_name = su.get_text ();
           	  string mimetype = su.get_mimetype ();
           	  
           	  /* De-dup the results keyed on the subject URIs */
              if (uri in seen_uris)
                continue;
              seen_uris.add (uri);
              
              if (File.new_for_uri (su.get_uri ()).query_exists ( null ) == true)
               {
               	  
                  
                  if (section == Section.FOLDERS)
                    {
                      File dir = File.new_for_uri(uri).get_parent();
                      uri = dir.get_uri ();
                      try{
                        FileInfo info = dir.query_info (FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME,
	                                                      FileQueryInfoFlags.NONE);
 	                      display_name = info.get_display_name ();
 	                    } catch (GLib.Error e) {
 	                      /* Bugger, we fall back to basename, which might not be
 	                       * valid UTF8... */
 	                      warning ("Unable to get display name for %s", uri);
 	                      display_name = dir.get_basename ();
 	                    }
	                    mimetype = "inode/directory";
                    }
                  
                  string icon = Utils.get_icon_for_uri (uri, mimetype);
                  
                  uint group_id;
                  string comment = "";
                  
                  /* We call this as we want the comment string */
                  group_id= Utils.get_time_group (ev, groups, out comment);

                  if (group_override >= 0)
                    {
                      group_id = (uint) group_override;
                    }
                  
                  //debug ("Got %s, %s, %u, %s", su.get_uri(), su.get_mimetype(), group_id, icon);
                  results.append (ResultsColumn.URI, uri,
                                  ResultsColumn.ICON_HINT, icon,
                                  ResultsColumn.GROUP_ID, group_id,
                                  ResultsColumn.MIMETYPE, mimetype,
                                  ResultsColumn.DISPLAY_NAME, display_name,
                                  ResultsColumn.COMMENT, comment,
                                  -1);
                }
            }
        }
    }

} /* namespace */
