/*
 * Copyright (C) 2013 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#include <unity/scopes/PreviewWidget.h>
#include <unity/scopes/internal/PreviewWidgetImpl.h>

namespace unity
{

namespace scopes
{

/*!
\class PreviewWidget

\brief A widget for a preview.

This class describes an individual widget used when constructing a preview for a result item.
Note that the data that applies to particular widget types is likely to change with different major
versions of Unity; therefore, attributes are of type Variant, that is, loosely typed.

When Unity requests a preview for a particular result, the scope is expected to construct the preview
by instantiating a PreviewWidget. Each widget has a free-form id, a type, and a number of attributes whose
names and types depend on the specific widget type (see \link previewwidgets Preview Widgets\endlink).

The attribute values can either be filled in directly before pushing the widget to Unity
(using add_attribute_value()), or they can be mapped from a result field in a
similar fashion to the components mapping when specifying a CategoryRenderer
(see add_attribute_mapping()). When using add_attribute_mapping(), the corresponding
attribute need not be present in the result; instead, its value can be pushed later
using the PreviewReply::push() method, which accepts the name of the field and its value as a Variant.

Preview widget can also be created entirely from a JSON string. See the documentation of unity::scopes::PreviewWidget::PreviewWidget(std::string const&) constructor for details.

Here is an example that creates a preview and illustrates three ways to associate a preview widget attribute
with its value:

\code{.cpp}
void MyPreview::run(PreviewReplyProxy const& reply)
{
    PreviewWidget w1("img", "image");
    // directly specify source URI for the image widget
    w1.add_attribute_value("source", Variant("http://www.example.org/graphics.png"));

    PreviewWidget w2("hdr", "header");
    // the result associated with this preview already had a title specified, so we'll just map it to the preview widget
    w2.add_attribute_mapping("title", "title");
    // description is not present in the result, but we'll push it later
    w2.add_attribute_mapping("summary", "description");

    PreviewWidgetList widgets;
    widgets.push_back(w1);
    widgets.push_back(w2);
    reply->push(widgets);

    // do a costly database lookup for the description
    std::string description = fetch_description(result().uri());

    reply->push("description", Variant(description));
}
\endcode
*/

/**
\page previewwidgets Preview widget types
\section previewtypes Recognized preview widget types

The following widget types are recognized by Unity:
\arg \c audio
\arg \c video
\arg \c image
\arg \c gallery
\arg \c header
\arg \c actions
\arg \c icon-actions
\arg \c progress
\arg \c text
\arg \c rating-input
\arg \c rating-edit
\arg \c reviews
\arg \c comment-input
\arg \c comment
\arg \c expandable
\arg \c table

\subsection audio audio widget

The audio widget displays a list of tracks with play/pause controls.

List of attributes:

\arg \c tracks A composite attribute containing an array of tuples, where each tuple has up to four
fields: \c title (mandatory string), \c subtitle (optional string), \c source (mandatory URI),
and \c length (optional integer specifying the track length in seconds)

You can construct composite attributes with unity::scopes::VariantBuilder:

\code{.cpp}
{
    PreviewWidget w1("tracks", "audio");

    VariantBuilder builder;
    builder.add_tuple({
        {"title", Variant("Track #1")},
        {"source", Variant("file:///tmp/song1.mp3")},
        {"length", Variant(194)}
    });
    builder.add_tuple({
        {"title", Variant("Track #2")},
        {"source", Variant("file:///tmp/song2.mp3")},
        {"length", Variant(207)}
    });

    w1.add_attribute_value("tracks", builder.end());
    ...
}
\endcode

\subsection video video widget

The video widget displays a still from a video and allows playing the video.

List of attributes:

\arg \c source A URI pointing to the video
\arg \c screenshot A URI pointing to a screenshot of the video (optional)
\arg \c share-data An optional dictionary for making this video shareable with Content Hub. See \link contentsharing content sharing\endlink below.

\code{.cpp}
{
    PreviewWidget w1("video1", "video");

    w1.add_attribute_value("source", Variant("file:///tmp/video1.mp4"));
    ...
}
\endcode

\subsection image image widget

The image widget displays a single image.

List of attributes:

\arg \c source A URI pointing to the image
\arg \c zoomable A boolean specifying whether the image is zoomable (default: \c false)
\arg \c fallback A fallback image to be used if the image URI is empty or cannot be retrieved (default: none)
\arg \c share-data An optional dictionary for making this image shareable with Content Hub. See \link contentsharing content sharing\endlink below.

\code{.cpp}
{
    PreviewWidget w1("img1", "image");

    w1.add_attribute_value("source", Variant("http://example.com/image.jpg"));
    w1.add_attribute_value("fallback", Variant("file:///tmp/image.jpg"));
    ...
}
\endcode

\subsection gallery gallery widget

The gallery widget displays a set of images.

List of attributes:

\arg \c sources An array of image URIs
\arg \c fallback A fallback image to be used if an image URI is empty or cannot be retrieved (default: none)

\code{.cpp}
{
    PreviewWidget w1("gal", "gallery");

    VariantArray arr;
    arr.push_back(Variant("http://example.com/image1.jpg"));
    arr.push_back(Variant("file:///tmp/image2.jpg"));
    arr.push_back(Variant("file:///tmp/image3.jpg"));
    w1.add_attribute_value("sources", Variant(arr));
    w1.add_attribute_value("fallback", Variant("file:///tmp/fallback.png"));
    ...
}
\endcode

\subsection header header widget

The header widget displays basic infomation about the result.

List of attributes:

\arg \c title A string specifying the title
\arg \c subtitle A string specifying the subtitle
\arg \c mascot A URI specifying the mascot
\arg \c fallback A fallback image to be used if the mascot URI is empty or cannot be retrieved (default: none)
\arg \c emblem A URI specifying the emblem

\code{.cpp}
{
    PreviewWidget w1("hdr", "header");

    w1.add_attribute_value("title", Variant("Result item"));
    w1.add_attribute_value("mascot", Variant("http://example.com/image.png"));
    w1.add_attribute_value("fallback", Variant("file:///tmp/fallback_mascot.png"));
    ...
}
\endcode

\subsection actions actions widget

The actions widget displays one or more buttons.

List of attributes:

\arg \c actions A composite attribute containing an array of tuples, where each tuple has at least
these fields: \c id (a mandatory string that is passed to unity::scopes::ScopeBase::activate_preview_action()),
\c label (mandatory string), and \c icon (optional URI).

You can construct composite attributes with unity::scopes::VariantBuilder:

\code{.cpp}
{
    PreviewWidget w1("buttons", "actions");

    VariantBuilder builder;
    builder.add_tuple({
        {"id", Variant("open")},
        {"label", Variant("Open")}
    });
    builder.add_tuple({
        {"id", Variant("download")},
        {"label", Variant("Download")}
    });

    w1.add_attribute_value("actions", builder.end());
    ...
}
\endcode

\subsection icon-actions icon-actions widget

The icon-actions widget displays one or more buttons represented by icons and/or labels. It's similar to actions widget, but uses different layout.
Every button can provide an optional temporary icon to be displayed immediately after user taps it.

List of attributes:

\arg \c actions A composite attribute containing an array of tuples, where each tuple has at least
these fields: \c id (a mandatory string that is passed to unity::scopes::ScopeBase::activate_preview_action()),
\c label (optional string), \c icon (optional URI, required if label is missing), \c temporaryIcon (optional URI).

\subsection progress progress widget

The progress widget displays the progress of an action, such as download progress.

On completion, the scope receives a preview action activation with the id \c "finished" or \c "failed",
depending on the outcome of the operation.

List of attributes:

\arg \c source A tuple with keys understood by a progress provider

\code{.cpp}
{
    PreviewWidget w1("download", "progress");

    VariantMap tuple;
    tuple["dbus-name"] = "com.canonical.DownloadManager";
    tuple["dbus-object"] = "/com/canonical/download/obj1";
    w1.add_attribute_value("source", Variant(tuple));
    ...
}
\endcode

\subsection text text widget

A text widget can be used for text of any length (without formatting).

List of attributes:

\arg \c title Optional string
\arg \c text A string containing the text

\code{.cpp}
{
    PreviewWidget w1("summary", "text");

    w1.add_attribute_value("text", Variant("Lorem Ipsum ..."));
    ...
}
\endcode

\subsection rating-input rating-input widget

The rating-input widget allows users to rate content. It consists of two types of widget:
a star-based rating and an input field for the user to enter his/her review.
It is possible to hide each widget as well as to require them to be filled in.

When a user presses the "Send" button, the scope receives a preview
action activation with the id \c "rated".  The actual rating and/or
review can be accessed via unity::scopes::ActionMetadata::scope_data.
The scope data will be a VariantMap with the following keys:

  - \c "rating" - a double holding the number of stars the user
    selected (1 to 5)
  - \c "review" - a string holding the free text review

List of attributes:

\arg \c rating-label String for the star-based rating (default: "Rate this")
\arg \c review-label String for the review input (default: "Add a review")
\arg \c submit-label String for the confirmation button (default: "Send")
\arg \c rating-icon-empty URI for an empty rating icon
\arg \c rating-icon-full URI for a full rating icon
\arg \c visible String specifying which of the two widgets are visible (\c "rating", \c "review" or default:\c "both")
\arg \c required String specifying which of the two widgets are required to be filled in (\c "rating", \c "review" or default:\c "both")

\code{.cpp}
{
    PreviewWidget w1("rating", "rating-input");

    w1.add_attribute_value("visible", Variant("rating"));
    w1.add_attribute_value("required", Variant("rating"));
    w1.add_attribute_value("rating-icon-empty", Variant("file:///tmp/star-empty.svg"));
    w1.add_attribute_value("rating-icon-full", Variant("file:///tmp/star-full.svg"));
    ...
}
\endcode

\subsection rating-edit rating-edit widget

The rating-edit widget allows users to edit an existing review and rating. When used in a preview, the widget displays an existing
review and a small "pen" icon; user can update the review and/or rating after tapping the "pen" icon.

This widget supports all the attributes of rating-input widget, plus three
extra attributes (\c "review", \c "rating", \c "author") to pre-fill the widget with data of an existing review.

List of attributes:

\arg \c rating-label String for the star-based rating (default: "Rate this")
\arg \c review-label String for the review input (default: "Add a review")
\arg \c submit-label String for the confirmation button (default: "Send")
\arg \c rating-icon-empty URI for an empty rating icon
\arg \c rating-icon-full URI for a full rating icon
\arg \c visible String specifying which of the two widgets are visible (\c "rating", \c "review" or default:\c "both")
\arg \c required String specifying which of the two widgets are required to be filled in (\c "rating", \c "review" or default:\c "both")
\arg \c author String for the name of the reviewer (optional)
\arg \c review String for the text of existing review (optional)
\arg \c rating Number for the rating value (optional)

Note: The rating-edit widget may not be supported by older versions of unity8 shell.

\subsection reviews reviews widget

The reviews widget is used to display previously-rated content.

List of attributes:

\arg \c rating-icon-empty URI for an empty rating icon
\arg \c rating-icon-half URI for an half-full rating icon
\arg \c rating-icon-full URI for a full rating icon
\arg \c reviews A composite attribute containing an array of tuples, where each tuple has up to three fields: \c rating (optional integer specifying the number of stars), \c author (mandatory string) and \c review (optional string).

You can construct composite attributes with unity::scopes::VariantBuilder:

\code{.cpp}
{
    PreviewWidget w1("summary", "reviews");

    w1.add_attribute_value("rating-icon-empty", Variant("file:///tmp/star-empty.svg"));
    w1.add_attribute_value("rating-icon-full", Variant("file:///tmp/star-full.svg"));

    VariantBuilder builder;
    builder.add_tuple({
        {"author", Variant("John Doe")},
        {"rating", Variant(3)}
    });
    builder.add_tuple({
        {"author", Variant("Mr. Smith")},
        {"rating", Variant(5)}
    });

    w1.add_attribute_value("reviews", builder.end());
    ...
}
\endcode

\subsection comment-input comment-input widget

The comment-input widget allows users to add comments. It displays an input box along with "Send" button.

When a user presses the "Send" button, the scope receives a preview
action activation with the id \c "commented".  The actual comment
can be accessed via unity::scopes::ActionMetadata::scope_data.
The scope data will be a VariantMap with the \c "comment" field holding
the text entered by the user.

List of attributes:

\arg \c submit-label String for the label of the submit button (optional, uses "Submit" by default).

\code{.cpp}
{
    PreviewWidget w1("cmt", "comment-input");

    w1.add_attribute_value("submit-label", Variant("Comment it!"));
    ...
}
\endcode

\subsection comment comment widget

The comment widget shows an avatar, author name, subtitle and a comment text.

List of attributes:
\arg \c source URI for an avatar icon (optional)
\arg \c author A string specifying the author of the comment (mandatory)
\arg \c subtitle A string for the subtitle (optional)
\arg \c comment A string for the comment text (mandatory)

\subsection expandable expandable widget

The expandable widget is used to group several widgets into an expandable pane.
The expandable widget can be collapsed or uncollapsed. When it's uncollapsed then
all the contained widgets are shown. When collapsed, only the first few widgets determined
by collapsed-widgets attribute are shown.

List of attributes:
\arg \c title A string specifying the title
\arg \c collapsed-widgets A number of widgets to show when the expandable widget is collapsed (optional).

\code
    PreviewWidget expandable("exp", "expandable");
    expandable.add_attribute_value("title", Variant("This is an expandable widget"));
    expandable.add_attribute_value("collapsed-widgets", Variant(2));

    PreviewWidget w1("w1", "text");
    w1.add_attribute_value("title", Variant("Subwidget 1"));
    w1.add_attribute_value("text", Variant("A text"));
    PreviewWidget w2("w2", "text");
    w2.add_attribute_value("title", Variant("Subwidget 2"));
    w2.add_attribute_value("text", Variant("A text"));
    PreviewWidget w3("w3", "text");
    w3.add_attribute_value("title", Variant("Subwidget 3"));
    w3.add_attribute_value("text", Variant("A text"));
    expandable.add_widget(w1);
    expandable.add_widget(w2);
    expandable.add_widget(w3);
    ...
\endcode

\subsection table table widget

The table widget is used to show a table with labels and values.
When used inside an Expandable widget, the topmost 3 rows are shown until it's expanded.

List of attributes:
\arg \c title A string specifying the title to be shown on top
\arg \c values An array with one element per row. Each element is an array of two strings: label and value

\code
    PreviewWidget table("details", "table");
    table.add_attribute_value("title", Variant("This is a table widget"));

    VariantArray values {
        Variant{VariantArray{Variant{_("Version number")}, Variant{"0.83b"}}},
        Variant{VariantArray{Variant{_("Last updated")}, Variant{"2015-01-15"}}},
        Variant{VariantArray{Variant{_("First released")}, Variant{"2013-12-16"}}},
        Variant{VariantArray{Variant{_("Size")}, Variant{"11.3 MiB"}}},
    };
    table.add_attribute_value("values", Variant(values));
    ...
\endcode

\subsection contentsharing Content sharing

Some widgets support content sharing with the special share-data attribute. When the widget is tapped (clicked), data (image, video etc.)
can be shared with Content Hub.

The share-data attribute is a dictionary (VariantMap) that needs to contain the following keys:

\arg \c uri A single URI to share or an array of URIs.
\arg \c content-type A name of the content type known to Content Hub, e.g. "links", "pictures", "videos". Please refer to Content Hub documentation
for information on supported content types.

Here is an example of a shareable image:

\code
    PreviewWidget image("img", "image");
    image.add_attribute_value("source", Variant("http://www.example.org/graphics.png"));

    VariantMap share_data;
    share_data["uri"] = Variant("http://www.example.org/graphics_big.png");
    share_data["content-type"] = Variant("pictures");
    image.add_attribute_value("share-data", share_data);
\endcode

*/

//! @cond

PreviewWidget::PreviewWidget(std::string const& definition)
    : p(new internal::PreviewWidgetImpl(internal::PreviewWidgetImpl::from_json(definition)))
{
}

PreviewWidget::PreviewWidget(std::string const& id, std::string const &widget_type)
    : p(new internal::PreviewWidgetImpl(id, widget_type))
{
}

PreviewWidget::PreviewWidget(internal::PreviewWidgetImpl *impl)
    : p(impl)
{
}

PreviewWidget::PreviewWidget(PreviewWidget const& other)
    : p(new internal::PreviewWidgetImpl(*(other.p)))
{
}

PreviewWidget::PreviewWidget(PreviewWidget&&) = default;
PreviewWidget& PreviewWidget::operator=(PreviewWidget&&) = default;
PreviewWidget::~PreviewWidget() = default;

PreviewWidget& PreviewWidget::operator=(PreviewWidget const& other)
{
    if (this != &other)
    {
        p.reset(new internal::PreviewWidgetImpl(*(other.p)));
    }
    return *this;
}

VariantMap PreviewWidget::serialize() const
{
    return p->serialize();
}

void PreviewWidget::add_attribute_value(std::string const& key, Variant const& value)
{
    p->add_attribute_value(key, value);
}

void PreviewWidget::add_attribute_mapping(std::string const& key, std::string const& field_name)
{
    p->add_attribute_mapping(key, field_name);
}

void PreviewWidget::add_widget(PreviewWidget const& widget)
{
    p->add_widget(widget);
}

std::string PreviewWidget::id() const
{
    return p->id();
}

std::string PreviewWidget::widget_type() const
{
    return p->widget_type();
}

std::map<std::string, std::string> PreviewWidget::attribute_mappings() const
{
    return p->attribute_mappings();
}

VariantMap PreviewWidget::attribute_values() const
{
    return p->attribute_values();
}

PreviewWidgetList PreviewWidget::widgets() const
{
    return p->widgets();
}

std::string PreviewWidget::data() const
{
    return p->data();
}

//! @endcond

} // namespace scopes

} // namespace unity
