#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/wait.h>


#include "update-notifier.h"
#include "hal.h"

#define CDROM_CHECKER PACKAGE_LIB_DIR"/update-notifier/apt-cdrom-check"

/* reposonses for the dialog */
#define RES_START_PM 1
#define RES_UPGRADE_AUTOMATIC 2

static LibHalContext *hal_ctx;
static DBusConnection *dbus_ctx;

void distro_cd_detected(UpgradeNotifier *un, char *mount_point)
{
   GtkWidget *dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL,
					      GTK_MESSAGE_QUESTION, 
					      GTK_BUTTONS_NONE,
					      NULL );
   gtk_window_set_title(GTK_WINDOW(dialog),_("Ubuntu CD detected"));
   gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);
   gtk_message_dialog_set_markup(GTK_MESSAGE_DIALOG(dialog),
				 _("<span weight=\"bold\" size=\"larger\">"
				   "Ubuntu CD detected</span> \n\n"
				   "You can automatically upgrade from it, "
				   "start the package manager application "
				   "or cancel"));
   gtk_dialog_add_buttons(GTK_DIALOG(dialog),
			  GTK_STOCK_CANCEL,
			  GTK_RESPONSE_REJECT,
			  _("Start package manager"), 
			  RES_START_PM,
			  _("Automatically upgrade"), 
			  RES_UPGRADE_AUTOMATIC,
			  NULL);


   int res = gtk_dialog_run (GTK_DIALOG (dialog));
   char *cmd;
   switch(res) {
   case RES_START_PM:
      cmd = g_strdup_printf("synaptic --add-cdrom %s",mount_point);
      invoke_with_gksu(cmd, _("Package manager"));
      break;
   case RES_UPGRADE_AUTOMATIC:
      cmd = g_strdup_printf("synaptic --add-cdrom %s --dist-upgrade-mode --non-interactive",mount_point);
      invoke_with_gksu(cmd, _("Automatic upgrade"));
      break;
   }
   gtk_widget_destroy (dialog);
}


LibHalFunctions *
up_return_hal_functions ()
{
       static LibHalFunctions hal_functions = { hal_mainloop_integration,
					  NULL /*hal_device_added*/,
					  NULL /*hal_device_removed*/,
					  NULL /*hal_device_new_capability*/,
					  NULL /*hal_device_lost_capability*/,
					  hal_property_modified,
					  NULL /*hal_device_condition*/};

	return &hal_functions;
}

/** Internal UP initialization function
 *
 * @functions			The LibHalFunctions to register as callbacks.
 * @return			The LibHalContext of the HAL connection or
 *				NULL on error.
 */
LibHalContext *
up_do_hal_init (LibHalFunctions *functions)
{
	char **devices;
	int nr;

	hal_ctx = hal_initialize (functions, FALSE);
	if (!hal_ctx) {
	        g_warning ("failed to initialize HAL!\n");
		return NULL;
	}

	if (hal_device_property_watch_all (hal_ctx)) {
	        g_warning ("failed to watch all HAL properties!\n");
		hal_shutdown (hal_ctx);
		return NULL;
	}

	/*
	 * Do something to ping the HAL daemon - the above functions will
	 * succeed even if hald is not running, so long as DBUS is.  But we
	 * want to exit silently if hald is not running, to behave on
	 * pre-2.6 systems.
	 */
	devices = hal_get_all_devices (hal_ctx, &nr);
	if (!devices) {
		g_warning ("seems that HAL is not running\n");
		hal_shutdown (hal_ctx);
		return NULL;
	}
	hal_free_string_array (devices);

	return hal_ctx;
}

/** Invoked by GLib in response to a D-BUS disconnect event.
 *
 * @param  data                 Context pointer
 */
static gboolean
up_reconnect_to_hal (gpointer data __attribute__((__unused__)))
{
	static unsigned int retries = 0;

	g_message ("Trying a reconnect ...");
	hal_ctx = up_do_hal_init (up_return_hal_functions ());
	if (hal_ctx != NULL) {
		g_message ("Reconnected OK.");
		retries = 0;
		return FALSE;
	}

	/* Retry later if it failed. */
	if (retries++ < 100)
		return TRUE;

	/* Too many retries; clean up and bail. */
	gtk_main_quit ();
	return FALSE;
}

/** Invoked by D-BUS to filter messages.
 *
 * @param  connection  D-BUS connection
 * @param  message     D-BUS message
 * @param  user_data   Context pointer
 */
static DBusHandlerResult
up_do_filter_dbus_msg (DBusConnection *connection __attribute__((__unused__)),
                        DBusMessage *message,
                        void *user_data __attribute__((__unused__)))
{
	if (dbus_message_is_signal (message,
	                            DBUS_INTERFACE_ORG_FREEDESKTOP_LOCAL,
	                            "Disconnected")) {
	        g_timeout_add(500, up_reconnect_to_hal, NULL);
		hal_shutdown (hal_ctx);
		hal_ctx = NULL;
		dbus_connection_unref (dbus_ctx);
		dbus_ctx = NULL;
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	else
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}


/** Invoked by libhal for integration with our mainloop. 
 *
 *  @param  ctx                 LibHal context
 *  @param  dbus_connection     D-BUS connection to integrate
 */
void
hal_mainloop_integration (LibHalContext *ctx __attribute__((__unused__)),
			  DBusConnection * dbus_connection)
{
        dbus_connection_set_exit_on_disconnect (dbus_connection, FALSE);
	dbus_connection_setup_with_g_main (dbus_connection, NULL);
	dbus_ctx = dbus_connection;
	dbus_connection_add_filter (dbus_connection, up_do_filter_dbus_msg,
	                            NULL, NULL);
}

/** Type for callback when a property of a device changes. 
 *
 *  @param  udi                 Unique Device Id
 *  @param  key                 Name of the property that has changed
 *  @param  is_removed          Property removed
 *  @param  is_added            Property added
 */
void 
hal_property_modified(LibHalContext *ctx, const char *udi, const char *key,
		      dbus_bool_t is_removed, dbus_bool_t is_added)
{
   char *mount_point;
   //g_print("hal_device_changed()\n");

   /* we are only interessted in mount events from the cdrom*/
   if (g_strcasecmp (key, "volume.is_mounted") != 0)
      return;
   if(!hal_device_get_property_bool(ctx, udi, key))
      return;

   char* storage_device = hal_device_get_property_string (ctx, udi,
						    "block.storage_device");
   if (!storage_device) {
      g_warning("cannot get storage_device\n");
      return;
   }
   char* media_type = hal_device_get_property_string (ctx, storage_device, 
						      "storage.drive_type");
   if(!media_type) {
      g_warning("cannot get storage.drive_type\n");
      return;
   }
   if (g_ascii_strcasecmp(media_type, "cdrom") != 0) {
      g_warning("no cdrom: %s\n",media_type);
      return;
   }
      
   mount_point = hal_device_get_property_string (ctx, udi, 
						 "volume.mount_point");
   //g_print("mounted: %s on %s\n", udi, mount_point);

   char *ubuntu_dir = g_strdup_printf("%s/ubuntu",mount_point);
   if(!g_file_test (ubuntu_dir, G_FILE_TEST_IS_SYMLINK)) {
      g_free(ubuntu_dir);
      return;
   }

   //g_print("this looks like a ubuntu-cdrom\n");
   char *cmd = g_strdup_printf(CDROM_CHECKER" %s",mount_point);
   int retval=-1;
   g_spawn_command_line_sync(cmd, NULL, NULL,  &retval, NULL);
   
   //g_print("retval: %i \n", WEXITSTATUS(retval));
   if(WEXITSTATUS(retval) == 1) {
      distro_cd_detected(hal_ctx_get_user_data(ctx), mount_point);
   }

   g_free(cmd);
   g_free(ubuntu_dir);
}

