#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include "hooks.h"
#include "rfc822.h"
#include "assert.h"
#include <glib.h>
#include <glib/gstdio.h>
#include <locale.h>

/* relative to the home dir */
#define HOOKS_SEEN ".update-notifier/hooks_seen"

void hooks_trayicon_update_tooltip (TrayApplet *un, int num_hooks)
{
   //g_print("update_tooltip: %x \n", un);
   gchar *updates;
   gchar *explanation;

   updates = g_strdup_printf(ngettext("There is %i item of post-update informations available!", "There are %i items of post-update information available!", num_hooks),  num_hooks);
   
   explanation = _("Press this icon to show the information.");

   gtk_tooltips_set_tip(GTK_TOOLTIPS(un->tooltip), 
			GTK_WIDGET (un->eventbox), 
			updates, explanation);

   g_free (updates);
}

gint compare_hook_func(gconstpointer a, gconstpointer b)
{
   //g_print("compare: %s %s\n",(char*)(((HookFileSeen*)a)->filename),(char*)b);
   return strcmp(((HookFileSeen*)a)->filename, b);
}

gboolean hook_file_is_new(HookDialog *h, const gchar *filename)
{
   if(g_list_find_custom(h->hook_files_seen, 
			 filename, compare_hook_func) == NULL)
      return TRUE;
   else 
      return FALSE;
}

gboolean hook_file_mark_as_seen(HookDialog *hook_dialog, 
				const gchar *hook_file)
{
   //g_print("mark_hook_file_as_seen: %s\n",hook_file);

   gchar *filename = g_strdup_printf("%s/"HOOKS_SEEN,g_get_home_dir());
   FILE *f = fopen(filename, "a");
   if(f==NULL)
      return FALSE;

   struct stat buf;
   gchar *file = g_strdup_printf("%s%s",HOOKS_DIR,hook_file);
   g_stat(file, &buf);

   HookFileSeen *t = g_new0(HookFileSeen,1);
   t->filename = g_strdup(hook_file);
   t->mtime = buf.st_mtime;
   t->cmd_run = 0;
   fprintf(f,"%s %i %x\n", t->filename, t->mtime, (int)(t->cmd_run));

   fclose(f);
   g_free(filename);
   g_free(file);

   hook_dialog->hook_files_seen= g_list_append(hook_dialog->hook_files_seen,t);

   return TRUE;
}

gboolean mark_hook_file_as_run(HookDialog *hook_dialog, const gchar *filename)
{
   // IMPLEMENT ME
   //g_print("mark_hook_file_as_run: %s\n",filename);
   return TRUE;
}


/* get the language code in a static allocated buffer
 * short_form: only return the languagecode if true, otherwise
 *             languagecode_countrycode
 */
char* get_lang_code(gboolean short_form)
{
   /* make own private copy */
   static gchar locale[51];
   strncpy(locale, setlocale(LC_MESSAGES, NULL), 50);
   
   // FIXME: we need to be more inteligent here
   // _and_ we probably want to look into the "LANGUAGE" enviroment too
   if(short_form) {
      locale[2] = 0;
      return locale;
   } else {
      locale[5] = 0;
      return locale;
   }
}

/*
 * get a i18n field of the rfc822 header
 * Return Value: a pointer that must not be freed (part of the rfc822 struct)
 */
char *hook_file_lookup_i18n(struct rfc822_header *rfc822, char *field)
{
   gchar *s, *entry;

   /* try $field-$languagecode_$countrycode first */
   s = g_strdup_printf("%s-%s", field, get_lang_code(FALSE));
   entry = rfc822_header_lookup(rfc822, s);
   //g_print("Looking for: %s ; found: %s\n",s,entry);
   g_free(s);
   if(entry != NULL)
      return entry;

   /* try $field-$languagecode next */
   s = g_strdup_printf("%s-%s", field, get_lang_code(TRUE));
   //g_print("Looking for: %s ; found: %s\n",s,entry);
   entry = rfc822_header_lookup(rfc822, s);
   g_free(s);
   if(entry != NULL)
      return entry;

   /* if everything else fails ... return untranslated */
   return rfc822_header_lookup(rfc822, field);
}

/*
 * show the given hook file
 */
void show_one_hook(TrayApplet *ta, gchar *hook_file)
{
   HookDialog *hook_dialog = (HookDialog *)ta->user_data;
   GladeXML *xml = hook_dialog->glade_xml;

   GtkWidget *text = glade_xml_get_widget(xml, "textview_hook");
   assert(text);
   GtkWidget *button_next = glade_xml_get_widget(xml, "button_next");
   assert(button_next);
   GtkWidget *button_run = glade_xml_get_widget(xml, "button_run");
   assert(button_run);

   char *filename = g_strdup_printf("%s%s",HOOKS_DIR,hook_file);

   /* setup the message */
   GtkTextBuffer *buf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
   FILE *f = fopen(filename, "r");
   if(f == NULL) {
      g_critical("can't open %s", filename);
      g_free(filename);
      return;
   }
   struct rfc822_header *rfc822 = rfc822_parse_stanza(f);

   char *cmd = rfc822_header_lookup(rfc822, "Command");
   char *term = g_strstrip(rfc822_header_lookup(rfc822, "Terminal"));
   g_object_set_data(G_OBJECT(button_run),"cmd", g_strdup(cmd));
   g_object_set_data(G_OBJECT(button_run),"term", g_strdup(term));
   g_object_set_data(G_OBJECT(button_run),"hook_file", g_strdup(hook_file));
   if(cmd != NULL) {
      gtk_widget_show(button_run);
   } else {
      gtk_widget_hide(button_run);
   }

   char *name = hook_file_lookup_i18n(rfc822, "Name");
   char *description = hook_file_lookup_i18n(rfc822, "Description");
   char *s = g_strdup_printf("%s\n\n%s\n",name, description);
   gtk_text_buffer_set_text(buf, s, -1);

   fclose(f);
   g_free(s);
   g_free(filename);
   rfc822_header_free_all(rfc822);

   /* mark the current hook file as seen */
   hook_file_mark_as_seen(hook_dialog, hook_file);
   check_update_hooks(ta);
}


void cb_button_run(GtkWidget *self, void *data)
{
   //g_print("cb_button_run()\n");
   gchar *argv[4];

   TrayApplet *ta = (TrayApplet *)data;
   HookDialog *hook_dialog = (HookDialog *)ta->user_data;

   /* mark the current hook file as run */
   gchar *hook_file = g_object_get_data(G_OBJECT(self), "hook_file");
   mark_hook_file_as_run(hook_dialog, (const char*)hook_file);

   gchar *cmd = g_object_get_data(G_OBJECT(self), "cmd");
   gchar *term = g_object_get_data(G_OBJECT(self), "term");

   if(cmd == NULL) {
      g_warning("cmd is NULL\n");
      return;
   }

   if(term != NULL && !g_ascii_strncasecmp(term, "true",-1)) {
      argv[0] = "gnome-terminal";
      argv[1] = "-e";
      argv[2] = cmd;
      argv[3] = NULL;
   } else {
      argv[0] = cmd;
      argv[1] = NULL;
   }
   g_spawn_async(NULL, argv, NULL, G_SPAWN_SEARCH_PATH , 
		 NULL, NULL, NULL, NULL);
}

void cb_button_next(GtkWidget *self, void *data)
{
   //g_print("cb_button_next()\n");
   TrayApplet *ta = (TrayApplet *)data;
   HookDialog *hook_dialog = (HookDialog *)ta->user_data;

   if(hook_dialog->hook_files != NULL)
      show_one_hook(ta, hook_dialog->hook_files->data);
}




void show_hooks(TrayApplet *ta)
{
   //g_print("show_hooks()\n");
   HookDialog* hook_dialog = (HookDialog *)ta->user_data;

   GladeXML *xml = hook_dialog->glade_xml;
   GtkWidget *dia = glade_xml_get_widget(xml, "dialog_hooks");
   assert(dia);
   GtkWidget *button_run = glade_xml_get_widget(xml, "button_run");
   assert(button_run);

   show_one_hook(ta, hook_dialog->hook_files->data);
  
   /* close dialog */
   int res = gtk_dialog_run(GTK_DIALOG(dia));
   if(res == GTK_RESPONSE_CLOSE) {
      /* mark question as seen */
   }
   gtk_widget_hide(dia);

}


gboolean check_update_hooks(TrayApplet *ta)
{
   //g_print("check_update_hooks()\n");

   HookDialog *hook_dialog = (HookDialog*)ta->user_data;

   GDir* dir;
   const gchar *hook_file;

   dir=g_dir_open(HOOKS_DIR, 0, NULL);
   if(dir == NULL)
      g_critical("can't read %s directory\n",HOOKS_DIR);

   while((hook_file=g_dir_read_name(dir)) != NULL) {
      GList *elm = g_list_find_custom(hook_dialog->hook_files,hook_file,
				      strcmp);
      if(hook_file_is_new(hook_dialog, hook_file)) {
	 if(elm == NULL) {
	    //g_print("check_update_hooks: appending: %s\n",hook_file);
	    hook_dialog->hook_files = g_list_append(hook_dialog->hook_files, 
						    g_strdup(hook_file));
	 }
      } else {
	 if(elm != NULL) {
	    //g_print("check_update_hooks: removing (seen): %s\n", hook_file);
	    hook_dialog->hook_files = g_list_remove(hook_dialog->hook_files,
						    elm->data);
	 }
      }
   }
   g_dir_close(dir);

   int num = g_list_length(hook_dialog->hook_files);
   //    g_print("check_update_hook() hook_files: %p %i \n", 
   // 	   hook_dialog->hook_files, num); 

   GladeXML *xml = hook_dialog->glade_xml;
   GtkWidget *button_next = glade_xml_get_widget(xml, "button_next");

   if (num == 0) {
      //g_print("gtk_widget_hide() called on tray_icon\n");
      gtk_widget_hide(GTK_WIDGET(ta->tray_icon));
      gtk_widget_hide(button_next);
   } else {
      //g_print("gtk_widget_show() called on tray_icon\n");
      hooks_trayicon_update_tooltip (ta, num);
      gtk_widget_show(GTK_WIDGET(ta->tray_icon));
      gtk_widget_show(button_next);
   }

   return TRUE;
}


static gboolean
button_release_cb (GtkWidget *widget, 
		   GdkEventButton *event, 
		   TrayApplet *ta)
{
	if (event->button == 1 && event->type == GDK_BUTTON_RELEASE) {
	   //g_print("left click on hook applet\n");
	   show_hooks(ta);
	}
	return TRUE;
}

gboolean init_already_seen_hooks(TrayApplet *ta)
{
   HookDialog* hook_dialog = (HookDialog*)ta->user_data;

   GList *seen = hook_dialog->hook_files_seen;
   char *filename = g_strdup_printf("%s/"HOOKS_SEEN,g_get_home_dir());
   char buf[512];
   int time, was_run;
   FILE *f = fopen(filename, "r");
   if(f==NULL)
      return TRUE;
   while(fscanf(f, "%s %i %i",buf,&time,&was_run) == 3) {
      HookFileSeen *t = g_new0(HookFileSeen, 1);
      t->filename = strdup(buf);
      t->mtime = time;
      t->cmd_run = was_run;
      seen = g_list_append(seen, t);
      
      //g_print("got: %s %i %i \n",buf,time,was_run);
   }
   g_free(filename);
   fclose(f);

   hook_dialog->hook_files_seen = seen;

   return TRUE;
}


void hook_tray_icon_init(TrayApplet *ta)
{
   HookDialog *hook_dialog = g_new0(HookDialog, 1);
   ta->user_data = hook_dialog;
   
   GladeXML *xml = glade_xml_new(GLADEDIR"hooks-dialog.glade",  
				 NULL, NULL);
   hook_dialog->glade_xml = xml;
   glade_xml_signal_connect_data(xml, "on_button_next_clicked", 
				 G_CALLBACK(cb_button_next), 
				 (gpointer)ta);
   glade_xml_signal_connect_data(xml, "on_button_run_clicked", 
				 G_CALLBACK(cb_button_run), 
				 (gpointer)ta);

   /* show dialog on click */
   g_signal_connect (G_OBJECT(ta->tray_icon),
		     "button-release-event",
		     G_CALLBACK (button_release_cb),
		     ta);

   /* read already seen hooks */
   init_already_seen_hooks(ta);
   
   /* Check for hooks */
   check_update_hooks(ta);
}
