#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glade/glade.h>   
#include <libintl.h>
#include <sys/wait.h>
#include <libnotify/notify.h>

#include "update-notifier.h"
#include "update.h"

#define UPGRADE_CHECKER PACKAGE_LIB_DIR"/update-notifier/apt-check"

char* actions[][2] = {
   { "/usr/bin/update-manager", N_("Show updates") },
   { "/usr/sbin/synaptic --upgrade-mode --non-interactive --hide-main-window", N_("Install all updates") },
   { "/usr/sbin/synaptic", N_("Package Manager") },
   { "/usr/sbin/synaptic --update-at-startup --non-interactive --hide-main-window", N_("Update package list now") },
   { NULL, NULL }
};


enum { NOTIFICATION_DEFAULT, NOTIFICATION_IGNORE, NOTIFICATION_SHOW_UPDATES };

static gboolean
button_release_cb (GtkWidget *widget, 
		   GdkEventButton *event, 
		   TrayApplet *ta)
{
        int index;
	if (event->button == 1 && event->type == GDK_BUTTON_RELEASE) {
   	        index = gconf_client_get_int((GConfClient*)(ta->user_data),
					     GCONF_KEY_DEFAULT_ACTION, NULL);
		//g_print("%s %s\n",actions[index][0],actions[index][1]);
		invoke_with_gksu (actions[index][0], _(actions[index][1]));
	}
	return TRUE;
}

static gboolean
button_press_cb (GtkWidget *widget,
		 GdkEventButton *event,
		 TrayApplet *un)
{
	if (event->button == 3) {
		gtk_menu_popup (GTK_MENU (un->menu), NULL, NULL, NULL, NULL,
				event->button, event->time);
	}
	return TRUE;
}

void
update_trayicon_update_tooltip (TrayApplet *ta, int num_upgrades)
{
   //g_print("update_tooltip: %p %p %p\n", ta, ta->tooltip, ta->eventbox);
   gchar *updates;
   gchar *explanation;

   updates=g_strdup_printf(ngettext("There is %i update available",
				    "There are %i updates available",
				    num_upgrades),
			   num_upgrades);
   explanation = ngettext(
           "Press this icon to show the update.",
           "Press this icon to show the updates.", num_upgrades);

   gtk_tooltips_set_tip(GTK_TOOLTIPS(ta->tooltip), 
			GTK_WIDGET (ta->eventbox), 
			updates, explanation);
   g_free(updates);
}


void 
cb_action(GObject *self, void *user_data)
{
   TrayApplet *un = user_data;
	
   int i = (int)g_object_get_data(G_OBJECT(self), "action");
   invoke_with_gksu (actions[i][0], _(actions[i][1]));

   gconf_client_set_int(un->user_data, GCONF_KEY_DEFAULT_ACTION, i, NULL);
}

void 
cb_preferences(GObject *self, void *user_data)
{
   invoke_with_gksu("/usr/bin/gnome-software-properties",
		    _("Software preferences"));    
}

void 
cb_toggled_show_notifications(GObject *self, void *data)
{
      TrayApplet *ta = (TrayApplet*)data;
      GConfClient *gconf = gconf_client_get_default();
      gboolean b = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(self));
      gconf_client_set_bool(gconf, GCONF_KEY_NO_UPDATE_NOTIFICATIONS, !b,NULL);
      
      NotifyHandle *n = (NotifyHandle*)g_object_get_data(G_OBJECT(ta->tray_icon),"notification");
      if(n != NULL) {
	 notify_close(n);
	 g_object_set_data(G_OBJECT(ta->tray_icon), "notification", NULL);
      }
}


void
update_trayicon_create_menu(TrayApplet *ta)
{
	GtkWidget *menuitem;
	GtkAccelGroup *accelgroup;
	int i;

	ta->menu = gtk_menu_new ();

	for(i=0;actions[i][0]!=NULL;i++) {
	   menuitem = gtk_menu_item_new_with_label (_(actions[i][1]));
	   gtk_menu_shell_append (GTK_MENU_SHELL (ta->menu), menuitem);
	   g_object_set_data(G_OBJECT(menuitem), "action", (void*)i);
	   g_signal_connect(G_OBJECT(menuitem), "activate", 
			    G_CALLBACK(cb_action), ta);
	}

	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append (GTK_MENU_SHELL (ta->menu), menuitem);

	menuitem = gtk_check_menu_item_new_with_label(_("Show notifications"));
	gboolean b = gconf_client_get_bool((GConfClient*)(ta->user_data),
					   GCONF_KEY_NO_UPDATE_NOTIFICATIONS, 
					   NULL);
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem), !b);
	gtk_menu_shell_append (GTK_MENU_SHELL (ta->menu), menuitem);
	g_signal_connect(G_OBJECT(menuitem), "toggled", 
			 G_CALLBACK(cb_toggled_show_notifications), ta);
	

	menuitem = gtk_image_menu_item_new_from_stock (GTK_STOCK_PREFERENCES, accelgroup);
	gtk_menu_shell_append (GTK_MENU_SHELL (ta->menu), menuitem);
	g_signal_connect(G_OBJECT(menuitem), "activate", 
			 G_CALLBACK(cb_preferences), (void*)ta);

	gtk_widget_show_all (ta->menu);
}

/* this tells the trayicon that apt is downloading something */
void 
update_apt_is_running(TrayApplet *ta, gboolean is_running)
{
   //  we make the icon insensitive, because we can't make the
   //  EggTrayIcon insensitive (otherwise we don't get a tooltip)
   // FIXME: we need block menu events
   gtk_widget_set_sensitive(GTK_WIDGET(ta->icon), !is_running);
   if(is_running) {
      gtk_tooltips_set_tip(GTK_TOOLTIPS(ta->tooltip), 
			   GTK_WIDGET (ta->eventbox),
			   _("Package Manager is working"),"");
   }
}

/*
 * this function is called when no changes are detected in
 * /var/lib/apt/lists or /var/lib/apt/lists/partial for TIMEOUT_APT_GET_UPDATE
*/
gboolean 
update_timer_finished(gpointer data)
{
   //g_print("update_timer_finished()\n");

   UpgradeNotifier *un = (UpgradeNotifier*)data;
   update_apt_is_running(un->update, FALSE);
   update_check(un->update);
   un->update_finished_timer = 0;

   /* return FALSE to remove source */
   return FALSE; 
}

/*
 * this function is called when the notification handle is clicked
 */
static 
void notify_callback(NotifyHandle *handle, guint32 uid, void *user_data)
{
   GMainLoop *loop = (GMainLoop*)user_data;

   if(uid == NOTIFICATION_SHOW_UPDATES) {
      invoke_with_gksu (actions[0][0], _(actions[0][1]));
   }

   notify_close(handle);
   g_main_loop_quit(loop);
}

// actually show the notification 
gint 
show_notification(gpointer user_data)
{
   TrayApplet *ta = (TrayApplet *)user_data;

   int x,y;
   gdk_window_get_origin(GTK_WIDGET(ta->tray_icon)->window, &x, &y);

   // no usefull coordiante yet, do another timeout
   if(x <= 0 || y <= 0)
      return TRUE;

   // check if the update-icon is still visible (in the delay time a 
   // update may already have been performed)
   if(!GTK_WIDGET_VISIBLE(ta->tray_icon))
      return FALSE;

   // now show a notification handle 
   NotifyHandle *n;
   NotifyHints *hints;

   //g_print("Placing notification at: %i %i", x, y);
   GtkRequisition size;
   gtk_widget_size_request(GTK_WIDGET(ta->tray_icon), &size);
   x = x + size.width/2;
   y = y + size.height;

   GMainLoop *loop;
   loop = g_main_loop_new(NULL, FALSE);
     
   hints = notify_hints_new();
   notify_hints_set_int(hints, "x", x);
   notify_hints_set_int(hints, "y", y);
   
   n = notify_send_notification(NULL,  // replaces nothing
				NULL,  // type
				NOTIFY_URGENCY_NORMAL,
				_("New updates available"), 
				_("Click on the update icon or the link "
				  "to see the available Updates"),
				NULL,  // no icon
				TRUE,  // expires
				60,    // expire timout (big!)
				hints, // hints
				loop,  // user data
				3,    //  actions
				NOTIFICATION_DEFAULT, 
				"default", 
				notify_callback,
				NOTIFICATION_SHOW_UPDATES, 
				_("Show Updates"), 
				notify_callback,
				NOTIFICATION_IGNORE, 
				_("Tell me about this later"), 
				notify_callback
				);
   
   // save the notification handle
   g_object_set_data(G_OBJECT(ta->tray_icon), "notification", n);

   // remove this from the timeout now
   return FALSE;
}

gboolean
update_check (TrayApplet *ta)
{
   //g_print("update_check()\n");

   int num_upgrades = 0;
   int exit_status = 0;
   gboolean res = FALSE;
   GError error;

   char *cmd[] = { UPGRADE_CHECKER, NULL };
   char *ret;

   if(g_spawn_sync(NULL, cmd, NULL,  
		   G_SPAWN_STDOUT_TO_DEV_NULL, 
		   NULL, NULL, NULL, &ret,
		   &exit_status, &error)) {
      // read the value from childs stderr, E: indicates a problem
      if(ret[0] == 'E') {
	 gtk_tooltips_set_tip(GTK_TOOLTIPS(ta->tooltip), 
			      GTK_WIDGET (ta->eventbox), 
			      _("A error occured, please run Package Manager "
				"from the right-click menu or "
				"apt-get on a terminal to see what is wrong"),
			      _("This usually means that your installed "
				"packages have unmet dependencies"));
	 gtk_widget_show(ta->tray_icon);
	 g_free(ret);
	 return TRUE;
	 
      } 
      num_upgrades = atoi(ret);
      g_free(ret);
   } else
      g_warning("Error launching %s", UPGRADE_CHECKER);
   
   if (num_upgrades == 0) {
      gtk_widget_hide(GTK_WIDGET(ta->tray_icon));
      NotifyHandle *n = (NotifyHandle*)g_object_get_data(G_OBJECT(ta->tray_icon),"notification");
      if(n != NULL) {
	 notify_close(n);
	 g_object_set_data(G_OBJECT(ta->tray_icon), "notification", NULL);
      }
      return TRUE;
   } 

   // update the tooltip
   update_trayicon_update_tooltip(ta, num_upgrades);
      
   // if we are already visible, skip the rest
   if(GTK_WIDGET_VISIBLE(GTK_WIDGET(ta->tray_icon))) 
      return TRUE;

   // show the icon
   gtk_widget_show(GTK_WIDGET(ta->tray_icon));

   // the user does not no notification messages
   if(gconf_client_get_bool((GConfClient*)(ta->user_data),
			    GCONF_KEY_NO_UPDATE_NOTIFICATIONS, NULL))
      return TRUE;

   // show the notification with some delay. otherwise on a login
   // the origin of the window is 0,0 and that looks ugly
   g_timeout_add(5000, show_notification, ta);
      

   //g_print("upgrades: %i\n", un->num_upgrades);

   return TRUE;
}


void 
update_tray_icon_init(TrayApplet *ta)
{
	// get gconf
	ta->user_data = gconf_client_get_default();

        g_signal_connect (G_OBJECT(ta->tray_icon),
			  "button-release-event",
			  G_CALLBACK (button_release_cb),
			  ta);
	g_signal_connect (G_OBJECT(ta->tray_icon),
			  "button-press-event",
			  G_CALLBACK (button_press_cb),
			  ta);

	/* Menu initialization */
	update_trayicon_create_menu (ta);

	/* Check for updates for the first time */
	update_check (ta);
}
