/* update-notifier.c
 * Copyright (C) 2004 Lukas Lipka <lukas@pmad.net>
 *           (C) 2004 Michael Vogt <mvo@debian.org>
 *           (C) 2004 Michiel Sikkes <michiel@eyesopened.nl>
 *           (C) 2004-2006 Canonical
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <signal.h>
#include <gnome.h>
#include <grp.h>
#include <pwd.h>
#include <limits.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-monitor.h>

#include "update-notifier.h"
#include "eggtrayicon.h"
#include "hal.h"
#include "update.h"
#include "hooks.h"

/* some prototypes */
extern gboolean up_get_clipboard (void);
gboolean update_timer_finished(gpointer data);
static void tray_destroyed_cb(GtkWidget *widget, void *data);

#define UPDATE_ICON_FILE PACKAGE_DATA_DIR"/pixmaps/update-icon.png"
#define HOOK_ICON_FILE PACKAGE_DATA_DIR"/pixmaps/hook-notifier.png"
#define REBOOT_ICON_FILE PACKAGE_DATA_DIR"/pixmaps/reboot-icon.png"

// the time when we check for fam events
#define TIMEOUT_FAM 1000*5 /* 5 sec */

// the timeout (in msec) for apt-get update (changes in 
// /var/lib/apt/lists{/partial})
#define TIMEOUT_APT_GET_UPDATE 1000*30 /* 30 sec */

// the timeout (in sec) when a further activity from dpkg/apt
// causes the applet to "ungray"
#define TIMEOUT_APT_RUN 120 /* 120 sec */


void
invoke_with_gksu(gchar *cmd, gchar *desktop)
{
        //g_print("invoke_update_manager ()\n");
	gchar *argv[5];
	argv[0] = "/usr/bin/gksu";
	argv[1] = "--desktop";
	argv[2] = desktop;
	argv[3] = cmd;
	argv[4] = NULL;

	g_spawn_async (NULL, argv, NULL, 0, NULL, NULL, NULL, NULL);
}



gboolean
trayapplet_create (TrayApplet *un, char *name, char *iconfile)
{
        //g_print("trayicon_create()\n");
	GdkPixbuf *pixbuf;

	/* setup widgets */
        un->tooltip = gtk_tooltips_new ();
	un->tray_icon = egg_tray_icon_new (name);
	un->eventbox = gtk_event_box_new ();
	gtk_widget_show(un->eventbox);
	un->icon = gtk_image_new ();
	gtk_widget_show(un->icon);

	/* load icon */
	pixbuf = gtk_image_get_pixbuf (GTK_IMAGE(un->icon));
	if (pixbuf)
		g_object_unref (G_OBJECT (pixbuf));
	pixbuf = gdk_pixbuf_new_from_file (iconfile, NULL);
	gtk_image_set_from_pixbuf (GTK_IMAGE (un->icon), pixbuf);

	/* build container */
	gtk_container_add (GTK_CONTAINER (un->eventbox), un->icon);
	gtk_container_add (GTK_CONTAINER (un->tray_icon), un->eventbox);

	return TRUE;
}


/* 
 the following files change:
 on "install":
  - /var/lib/dpkg/lock
  - /var/lib/dpkg/ *
  - /var/lib/update-notifier/dpkg-run-stamp
 on "update":
  - /var/lib/apt/lists/lock
  - /var/lib/apt/lists/ *
  - /var/lib/dpkg/lock
*/
void
monitor_cb(GnomeVFSMonitorHandle *handle,
	   const gchar *monitor_uri,
	   const gchar *info_uri,
	   GnomeVFSMonitorEventType event_type,
	   gpointer user_data)
{
   UpgradeNotifier *un = (UpgradeNotifier*)user_data;
   int i;

#if 0
   g_print("monitor_uri: %s\n", monitor_uri);
   g_print("info_uri: %s\n", info_uri);
   g_print("event_type: %i\n",event_type);
#endif

   // we ignore lock file events because we can only get 
   // when a lock was taken, but not when it was removed
   if(g_str_has_suffix(info_uri, "lock"))
      return;

   // look for apt-get install/update
   if(g_str_has_prefix(info_uri,"file:///var/lib/apt/") 
      || g_str_has_prefix(info_uri,"file:///var/cache/apt/")
      || strcmp(info_uri,"file:///var/lib/dpkg/status") == 0) {
      un->apt_get_runing=TRUE;
   } 
   if(strstr(info_uri, "/var/lib/update-notifier/dpkg-run-stamp")) {
      un->dpkg_was_run = TRUE;
   } 
   if(strstr(info_uri, REBOOT_FILE)) {
      //g_print("reboot required\n");
      un->reboot_pending = TRUE;
   }
   if(strstr(info_uri, HOOKS_DIR)) {
      //g_print("new hook!\n");
      un->hook_pending = TRUE;
   }
}

/*
 * We periodically check here what actions happend in this "time-slice". 
 * This can be:
 * - dpkg_was_run=TRUE: set when apt wrote the "dpkg-run-stamp" file
 * - apt_get_runing: set when apt/dpkg activity is detected (in the 
 *                   lists-dir, archive-cache, or /var/lib/dpkg/status
 * - hook_pending: we have new upgrade hoook information
 * - reboot_pending: we need to reboot
 *
 */
gboolean file_monitor_periodic_check(gpointer data)

{
   UpgradeNotifier *un = (UpgradeNotifier *)data;

   // DPkg::Post-Invoke has written a stamp file, that means a install/remove
   // operation finished, we can show hooks/reboot notifications then
   if(un->dpkg_was_run) {

      // check updates
      update_check(un->update);

      // any apt-get update  must be finished, otherwise 
      // apt-get install wouldn't be finished
      update_apt_is_running(un->update, FALSE);
      if(un->update_finished_timer > 0) 
	 g_source_remove(un->update_finished_timer);
      
      // show pending hooks/reboots
      if(un->hook_pending) {
	 //g_print("checking hooks now\n");
	 check_update_hooks(un->hook);
	 un->hook_pending = FALSE;
      }
      if(un->reboot_pending) {
	 //g_print("checking reboot now\n");
	 reboot_check (un->reboot);
	 un->reboot_pending = FALSE;
      }

      // apt must be finished when a new stamp file was writen, so we
      // reset the apt_get_runing time-slice field because its not 
      // important anymore (it finished runing)
      //
      // This may leave a 5s race condition when a apt-get install finished
      // and something new (apt-get) was started
      un->apt_get_runing = FALSE;
      un->last_apt_action = 0;
   }

   // apt-get update/install or dpkg is runing (and updates files in 
   // it's list/cache dir) or in /var/lib/dpkg/status
   if(un->apt_get_runing) 
      update_apt_is_running(un->update, TRUE);

   // update time information for apt/dpkg
   time_t now = time(NULL);
   if(un->apt_get_runing) 
      un->last_apt_action = now;

   // no apt operation for a long time
   if(un->last_apt_action > 0 &&
      (now - un->last_apt_action) >  TIMEOUT_APT_RUN) {
      update_apt_is_running(un->update, FALSE);
      update_check(un->update);
      un->last_apt_action = 0;
   }

   // reset the bitfields (for the next "time-slice")
   un->dpkg_was_run = FALSE;
   un->apt_get_runing = FALSE;

   return TRUE;
}




/* u_abort seems like an internal error notification.
 * End user might not understand the message at all */
void u_abort(gchar *msg)
{
   msg = g_strdup_printf("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s\n", _("Internal error"), msg);
   gtk_dialog_run(GTK_DIALOG(gtk_message_dialog_new_with_markup(NULL,0,
						     GTK_MESSAGE_ERROR,
						     GTK_BUTTONS_CLOSE,
						     msg)));
   g_free(msg);
   exit(1);
}

// FIXME: get the apt directories with apt-config or something
gboolean monitor_init(UpgradeNotifier *un)
{
   int i;
   GnomeVFSResult res;
   GnomeVFSMonitorHandle *monitor_handle;

   static const char *monitor_dirs[] = { 
      "/var/lib/apt/lists/", "/var/lib/apt/list/partial", 
      "/var/cache/apt/archives/", "/var/cache/apt/archives/partial", 
      HOOKS_DIR,
      NULL};
      
   for(i=0;monitor_dirs[i] != NULL;i++) {
      res = gnome_vfs_monitor_add(&monitor_handle,
				  monitor_dirs[i],
				  GNOME_VFS_MONITOR_DIRECTORY,
				  monitor_cb, un);
   }
   res = gnome_vfs_monitor_add(&monitor_handle,
			       "/var/lib/dpkg/status", 
			       GNOME_VFS_MONITOR_FILE,
			       monitor_cb,
			       un);
   res = gnome_vfs_monitor_add(&monitor_handle,
			       "/var/lib/update-notifier/dpkg-run-stamp", 
			       GNOME_VFS_MONITOR_FILE,
			       monitor_cb,
			       un);
   res = gnome_vfs_monitor_add(&monitor_handle,
			       "/var/run/reboot-required", 
			       GNOME_VFS_MONITOR_FILE,
			       monitor_cb,
			       un);

   g_timeout_add (TIMEOUT_FAM, (GSourceFunc)file_monitor_periodic_check, un);


   return TRUE;
}




static gboolean
tray_icons_init(UpgradeNotifier *un)
{
   //g_debug("tray_icons_init");

   /* new upates tray icon */
   un->update = g_new0 (TrayApplet, 1);
   trayapplet_create(un->update, "update-notifier", UPDATE_ICON_FILE);
   update_tray_icon_init(un->update);

   /* a destroy signal handler (e.g. when the gnome-panel is restarted */
   g_signal_connect(G_OBJECT(un->update->tray_icon), "destroy", 
		    G_CALLBACK(tray_destroyed_cb), un);

   /* update hook icon*/
   un->hook = g_new0 (TrayApplet, 1);
   trayapplet_create(un->hook, "hook-notifier", HOOK_ICON_FILE);
   hook_tray_icon_init(un->hook);

   /* reboot required icon */
   un->reboot = g_new0 (TrayApplet, 1);
   trayapplet_create(un->reboot, "reboot-notifier", REBOOT_ICON_FILE);
   reboot_tray_icon_init(un->reboot);

   return FALSE; // for the tray_destroyed_cb
}

static void
tray_destroyed_cb(GtkWidget *widget, void *data)
{
   g_debug("tray_destoryed_cb");
   // try re-init on destroy
   g_timeout_add(5000, (GSourceFunc)(tray_icons_init), data);
}

// this function checks if the user is in the admin group
// if there is no admin group, we return true becuase there
// is no way to figure if the user is a admin or not
static gboolean
in_admin_group()
{
   int i, ng = 0;
   gid_t *groups = NULL;
   struct passwd *pw = getpwuid(getuid());
   if (pw == NULL)
      return TRUE;

   struct group *grp = getgrnam("admin");
   if(grp == NULL) 
      return TRUE;
   
   if (getgrouplist(pw->pw_name, pw->pw_gid, NULL, &ng) < 0) {
      groups = (gid_t *) malloc(ng * sizeof (gid_t));
      getgrouplist(pw->pw_name, pw->pw_gid, groups, &ng);
   }
   for(i=0;i<ng;i++) {
      if(groups[i] == grp->gr_gid) {
	 return TRUE;
      }
   }
   
   if(groups != NULL)
      free(groups);

   return FALSE;
}

static GOptionEntry entries[] = 
{
   { "debug-hooks", 0, 0, G_OPTION_ARG_INT, &HOOK_DEBUG, "Enable hooks debugging"},
   { NULL }
};

int 
main (int argc, char *argv[])
{
	GnomeClient *client;
	UpgradeNotifier *un;

	// option parsing
	GOptionContext *context;
	context = g_option_context_new (" - inform about updates");
	g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);

	gnome_program_init (PACKAGE, PACKAGE_VERSION, 
			    LIBGNOMEUI_MODULE,
			    argc, argv, 
			    GNOME_PARAM_GOPTION_CONTEXT, context,
			    GNOME_PARAM_NONE);

	notify_init("update-notifier");
        bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
        bind_textdomain_codeset(PACKAGE, "UTF-8");
        textdomain(PACKAGE);

	//g_print("starting update-notifier\n");
	
	// check if we should run at all (see HideAdminTools spec
	// on https://wiki.ubuntu.com/HideAdminToolsToUsers)
	if(!in_admin_group()) {
	   g_warning("not starting because user is not in admin group\n");
	   exit(0);
	}

	client = gnome_master_client ();
	if (up_get_clipboard ())
	  gnome_client_set_restart_style (client, GNOME_RESTART_ANYWAY);
	else {
	   gnome_client_set_restart_style (client, GNOME_RESTART_NEVER);
	   g_warning ("already running?\n");
	   return 1;
	}

	/* Make sure we die when the session dies */
	g_signal_connect (G_OBJECT (client), "die",
			  G_CALLBACK (gtk_main_quit), NULL);
	
	/* Create the UpgradeNotifier object */
	un = g_new0 (UpgradeNotifier, 1);

	// check for .update-notifier dir and create if needed
	gchar *dirname = g_strdup_printf("%s/.update-notifier",
					 g_get_home_dir());
	if(!g_file_test(dirname, G_FILE_TEST_IS_DIR))
	   g_mkdir(dirname, 0700);
	g_free(dirname);

	// create the icons with a timeout
	g_timeout_add(5000, (GSourceFunc)(tray_icons_init), un);

	// init hal (needs to be done _after_ the icons are created)
	/* setup hal so that inserted cdroms can be checked */
	LibHalContext *hal_ctx = up_do_hal_init();
	if (!hal_ctx) {
	   u_abort("failed to initialize HAL!\n");
	   return 1;
	}
	libhal_ctx_set_user_data(hal_ctx, un);
	
	// init fam
	monitor_init(un);
	
	/* Start the main gtk loop */
	gtk_main ();

	return 0;
}
