#include <fcntl.h>
#include <unistd.h>

#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <stdexcept>

#include <gtest/gtest.h>

#include "grail.h"

/**
 * @brief Fixture for check-mapping test suite.
 */
class CheckMapping : public testing::Test {
 protected:

  grail ge_;
  int fd_;
  grail_coord pos_;

  CheckMapping()
      : fd_(-1) {
    std::memset(&ge_, 0, sizeof(ge_));
    std::memset(&pos_, 0, sizeof(pos_));
  }

  /**
   * @brief Setup procedure is called before every test.
   */
  void SetUp() {
    ge_.get_clients = TpGetClients;
    ge_.event = TpEvent;
    ge_.gesture = TpGesture;
    ge_.priv = this;

    fd_ = open(TEST_ROOT_DIR "io/evemu/one-tap.evemu", O_RDONLY | O_NONBLOCK);

    if (0 != grail_open(&ge_, fd_))
      throw std::runtime_error(
          "Problem initializing grail for test-suite CheckMapping.");
  }

  /**
   * @brief Cleanup procedure is called before every test.
   */
  virtual void TearDown() {
    grail_close(&ge_, fd_);
    close(fd_);
  }

  static int TpGetClients(struct grail* ge,
                          grail_client_info* clients,
                          int max_clients, const grail_coord* coords,
                          int num_coords, const grail_mask_t *types,
                          int type_bytes) {
    memset(&clients[0], 0, sizeof(clients[0]));
    memset(clients[0].mask, ~0, sizeof(clients[0].mask));
    clients[0].id.client = 345;
    clients[0].id.root = 1;
    clients[0].id.event = 2;
    clients[0].id.child = 3;
    return 1;
  }

  static void TpEvent(grail* ge, const input_event* ev) {
  }

  static void TpGesture(grail* ge, const grail_event* ev) {
    CheckMapping* data = static_cast<CheckMapping*>(ge->priv);
    if (ev->type != GRAIL_TYPE_TAP1)
      return;
    data->pos_.x = ev->prop[GRAIL_PROP_TAP_X];
    data->pos_.y = ev->prop[GRAIL_PROP_TAP_Y];
  }

  static float Distance(const grail_coord* p, float x, float y) {
    double dx = p->x - x;
    double dy = p->y - y;
    return std::sqrt(dx * dx + dy * dy);
  }
};

TEST_F(CheckMapping,device_coordinates) {
  grail_pull(&ge_, fd_);

  EXPECT_NEAR(0, CheckMapping::Distance(&pos_, 17086, 14789), 1E-8);
}

TEST_F(CheckMapping,mapped_coordinates) {
  static const grail_coord min = { 0, 0 }, max = { 1.6, 1 };
  grail_set_bbox(&ge_, &min, &max);

  grail_pull(&ge_, fd_);

  EXPECT_NEAR(0., CheckMapping::Distance(&pos_, 0.834303, 0.451338), 1E-6);
}
