#include <fcntl.h>
#include <unistd.h>

#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <stdexcept>

#include <gtest/gtest.h>
#include <utouch/frame-mtdev.h>

#include "grail.h"

/**
 * @brief Fixture for check-transform test suite.
 */
class CheckTransform : public testing::Test {
 protected:
  evemu_device* evemu_;
  utouch_frame_handle fh_;
  grail_handle ge_;
  FILE* fp_;
  timeval evtime_;

  static float Distance(const grail_coord * p, float x, float y) {
    double dx = p->x - x;
    double dy = p->y - y;
    return std::sqrt(dx * dx + dy * dy);
  }

  CheckTransform()
      : evemu_(NULL),
        fp_(NULL) {
    std::memset(&fh_, 0, sizeof(fh_));
    std::memset(&ge_, 0, sizeof(ge_));
    std::memset(&evtime_, 0, sizeof(evtime_));
  }

  void SetUp() {
    fp_ = fopen(TEST_ROOT_DIR "io/evemu/one-tap.evemu", "r");
    if (!fp_)
      throw std::runtime_error(
          "Error opening pre-recorded events file io/evemu/one-tap.evemu.");

    evemu_ = evemu_new(NULL);
    if (!evemu_ || evemu_read(evemu_, fp_) <= 0)
      throw std::runtime_error("Error reading emulated event.");

    fh_ = utouch_frame_new_engine(100, 32, 100);
    if (!fh_ || utouch_frame_init_mtdev(fh_, evemu_))
      throw std::runtime_error("Error initializing utouch frame.");

    ge_ = grail_new(fh_, 10, 0);
    if (!ge_)
      throw std::runtime_error(
          "Error allocating and initializing grail context.");
  }

  void TearDown(struct test_data *data) {
    grail_delete(ge_);
    utouch_frame_delete_engine(fh_);
    evemu_delete(evemu_);
    fclose(fp_);
  }

  int GetEvent(input_event* ev) {
    return evemu_read_event_realtime(fp_, ev, &evtime_) > 0;
  }

  const grail_element* GetElement(const input_event* ev) {
    const utouch_frame* touch;
    const grail_frame* frame;

    touch = utouch_frame_pump_mtdev(fh_, ev);
    if (!touch)
      return NULL;
    frame = grail_pump_frame(ge_, touch);
    if (!frame || !frame->num_ongoing)
      return NULL;

    return frame->ongoing[frame->num_ongoing - 1];
  }
};

TEST_F(CheckTransform, device_coordinates) {
  struct input_event ev;
  const struct grail_element *slot;
  struct grail_coord pos = { 0, 0 };

  while (GetEvent(&ev)) {
    slot = GetElement(&ev);
    if (slot)
      pos = slot->center;
  }

  EXPECT_NEAR(0., CheckTransform::Distance(&pos, 17086, 14789), 1E-6);
}

TEST_F(CheckTransform, mapped_coordinates) {
  static const struct grail_coord min = { 0, 0 }, max = { 1.6, 1 };
  struct input_event ev;
  const grail_element* slot;
  struct grail_coord pos = { 0, 0 };

  grail_set_bbox(ge_, &min, &max);

  while (GetEvent(&ev)) {
    slot = GetElement(&ev);
    if (slot)
      pos = slot->center;
  }

  EXPECT_NEAR(0., CheckTransform::Distance(&pos, 0.834303, 0.451338), 1E-6);
}
