/*****************************************************************************
 *
 * grail-eventprinter - pretty prints events as produced by Grail
 *
 * Copyright (C) 2010-2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

/* Regular applications must include <grail.h>. Since
 * this application needs to access Grail's internals
 * to access the private fast_fileread struct element,
 * it needs this header.
 *
 * Only Grail's internal tools are allowed to include
 * this header.
 */
#include "../src/v2/grail-impl.h"

#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/stat.h>

static grail_mask_t flag_mask[DIM_GRAIL_TYPE_BYTES];

/* Order is the same as with defines in grail-types.h */

static const char* const grail_event_names[] = {
	"Drag1",
	"Pinch1",
	"Rotate1",
	"Drag2",
	"Pinch2",
	"Rotate2",
	"Drag3",
	"Pinch3",
	"Rotate3",
	"Drag4",
	"Pinch4",
	"Rotate4",
	"Drag5",
	"Pinch5",
	"Rotate5",
	"Tap1",
	"Tap2",
	"Tap3",
	"Tap4",
	"Tap5",
	"Edrag",
	"Epinch",
	"Erotate",
	"Mdrag",
	"Mpinch",
	"Mrotate",
	"Sysflag1",
	"Unknown27",
	"Unknown28",
	"Unknown29",
	"Unknown30",
	"Unknown31",
	"Touch1",
	"Touch2",
	"Touch3",
	"Touch4",
	"Touch5",
	"Etouch",
	"Mtouch"
};

static int tp_get_clients(struct grail *ge,
			  struct grail_client_info *clients, int max_clients,
			  const struct grail_coord *coords, int num_coords,
			  const grail_mask_t *types, int type_bytes)
{
	printf("Client query.\n");

	memset(&clients[0], 0, sizeof(clients[0]));
	clients[0].id.client = 345;
	clients[0].id.root = 1;
	clients[0].id.event = 2;
	clients[0].id.child = 3;
	memcpy(clients[0].mask, flag_mask, sizeof(flag_mask));
	return 1;
}

static void tp_event(struct grail *ge, const struct input_event *ev)
{
	printf("Input event.\n");
}

static int is_drag_event(const struct grail_event *ev) {
	return ev->type == GRAIL_TYPE_DRAG1 ||
		ev->type == GRAIL_TYPE_DRAG2 ||
		ev->type == GRAIL_TYPE_DRAG3 ||
		ev->type == GRAIL_TYPE_DRAG4 ||
		ev->type == GRAIL_TYPE_DRAG5;
}

static void print_drag_stats(const struct grail_event *ev) {
	printf(" c = (%f, %f), delta = (%f, %f)\n",
		ev->prop[GRAIL_PROP_DRAG_X],
		ev->prop[GRAIL_PROP_DRAG_Y],
		ev->prop[GRAIL_PROP_DRAG_VX],
		ev->prop[GRAIL_PROP_DRAG_VY]);
}

static int is_rotate_event(const struct grail_event *ev) {
	return ev->type == GRAIL_TYPE_ROTATE1 ||
		ev->type == GRAIL_TYPE_ROTATE2 ||
		ev->type == GRAIL_TYPE_ROTATE3 ||
		ev->type == GRAIL_TYPE_ROTATE4 ||
		ev->type == GRAIL_TYPE_ROTATE5;
}

static void print_rotate_stats(const struct grail_event *ev) {
	printf(" a = %f, va = %f\n",
		ev->prop[GRAIL_PROP_ROTATE_A],
		ev->prop[GRAIL_PROP_ROTATE_VA]);
}

static int is_pinch_event(const struct grail_event *ev) {
	return ev->type == GRAIL_TYPE_PINCH1 ||
		ev->type == GRAIL_TYPE_PINCH2 ||
		ev->type == GRAIL_TYPE_PINCH3 ||
		ev->type == GRAIL_TYPE_PINCH4 ||
		ev->type == GRAIL_TYPE_PINCH5;
}

static void print_pinch_stats(const struct grail_event *ev) {
	printf(" radius = %f, delta r = %f\n",
		ev->prop[GRAIL_PROP_PINCH_R],
		ev->prop[GRAIL_PROP_PINCH_VR]);
}

const char *event_name(const struct grail_event *ev) {
	return grail_event_names[ev->type];
}

static void tp_gesture(struct grail *ge, const struct grail_event *ev)
{
	if (ev->status == GRAIL_STATUS_BEGIN) {
		printf("BEGIN gesture %s\n", event_name(ev));
		if (is_drag_event(ev)){
			print_drag_stats(ev);
		}
		if (is_rotate_event(ev)) {
			print_rotate_stats(ev);
		}
		if (is_pinch_event(ev)) {
			print_pinch_stats(ev);
		}
	}
	if (ev->status == GRAIL_STATUS_UPDATE) {
		printf("UPDATE gesture %s.\n", event_name(ev));
	}
	if (ev->status == GRAIL_STATUS_END) {
		printf("END gesture %s.\n", event_name(ev));
		if (is_drag_event(ev)){
			print_drag_stats(ev);
		}
		if (is_rotate_event(ev)) {
			print_rotate_stats(ev);
		}
		if (is_pinch_event(ev)) {
			print_pinch_stats(ev);
		}
	}
}

static void loop_device(struct grail *ge, int fd)
{
	while (!grail_idle(ge, fd, 5000))
		grail_pull(ge, fd);
}

int main(int argc, char *argv[])
{
	struct grail ge;
	struct stat fs;
	int i, fd;

	if (argc < 2) {
		fprintf(stderr, "Usage: %s <device or trace file>\n", argv[0]);
		return -1;
	}

	memset(&ge, 0, sizeof(ge));
	ge.get_clients = tp_get_clients;
	ge.event = tp_event;
	ge.gesture = tp_gesture;

	unsigned long mask = 0xffffffff; /* Get every gesture type. */
	for (i = 0; i < DIM_GRAIL_TYPE; i++)
		if ((mask >> i) & 1)
			grail_mask_set(flag_mask, i);

	fd = open(argv[1], O_RDONLY | O_NONBLOCK);
	if (fd < 0) {
		fprintf(stderr, "error: could not open device\n");
		return -1;
	}
	if (fstat(fd, &fs)) {
		fprintf(stderr, "error: could not stat the device\n");
		return -1;
	}
	if (fs.st_rdev && ioctl(fd, EVIOCGRAB, 1)) {
		fprintf(stderr, "error: could not grab the device\n");
		return -1;
	}

	if (grail_open(&ge, fd)) {
		fprintf(stderr, "error: could not open touch device\n");
		return -1;
	}
	ge.impl->fast_fileread = 1;

	loop_device(&ge, fd);
	grail_close(&ge, fd);

	if (fs.st_rdev)
		ioctl(fd, EVIOCGRAB, 0);
	close(fd);
	return 0;
}
