/*      $Id: shortcuts_plugin.c 21631 2006-05-11 21:24:25Z benny $
 *
 * Copyright (c) 2006 Jean-Francois Wauthy <pollux@xfce.org>
 *                    Olivier Fourdan <fourdan@xfce.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <libxfce4mcs/mcs-manager.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>
#include <xfce-mcs-manager/manager-plugin.h>

#include "keys_management.h"
#include "shortcuts_plugin.h"

/* */
#define DEFAULT_SHORTCUT_THEME "Default"
#define DEFAULT_SHORTCUTS_PATH PACKAGE_DATADIR"/shortcuts/default.xml"
#define BORDER 5

enum
{
    COLUMN_THEME_NAME,
    COLUMN_FILE_NAME,
    THEMES_NUM_COLUMNS
};

enum
{
    COLUMN_COMMAND,
    COLUMN_SHORTCUT,
    SHORTCUTS_NUM_COLUMNS
};

typedef struct _shortcut_tree_foreach_struct shortcut_tree_foreach_struct;
struct _shortcut_tree_foreach_struct
{
    gchar *shortcut;
    gchar *path;
    GtkTreeSelection *selection;
    gboolean found;
};

typedef struct _launcher launcher;
struct _launcher
{
    MyKey *key;
    gchar *command;
};

/* globals */
static gboolean wait_release = FALSE;
static GSList *shortcut_list = NULL;
static gchar *theme_name = NULL;
static gchar *theme_path = NULL;

/* static prototypes */
gboolean shortcuts_plugin_save_settings (McsPlugin *);
static void cb_browse_command (GtkWidget *, GtkEntry *);
static gboolean command_exists (const gchar *);

/****************************/
/* Low-level events filter  */
/****************************/

static void
handleKeyPress (XKeyPressedEvent * ev)
{
    GSList *element;
    int state;

    state = ev->state & MODIFIER_MASK;

    if (wait_release)
    {
        return;
    }

    for (element = shortcut_list; element != NULL; element = g_slist_next (element))
    {
        launcher *shortcut = (launcher *) element->data;
        if ((shortcut->key->keycode == ev->keycode) && (shortcut->key->modifier == state))
        {
            GError *error = NULL;
            GdkDisplay *gdisplay;
            GdkScreen *gscr;
            gint monitor;
            
            /* We must wait for a release event before te trigger another key press */
            wait_release = TRUE;
            gdisplay = gdk_display_get_default ();
            gscr = xfce_gdk_display_locate_monitor_with_pointer (gdisplay, &monitor);
            
            if (!xfce_gdk_spawn_command_line_on_screen (gscr, shortcut->command, &error))
            {
                if (error)
                {
                    g_warning ("%s", error->message);
                    g_error_free (error);
                }
            }

            return;
        }
    }
}

static void
handleKeyRelease (XKeyReleasedEvent * ev)
{
    wait_release = FALSE;
}

static void
handleMappingNotify (XMappingEvent * ev)
{
    GSList *element;

    /* Clear up the flag as the keyboard mapping changes, we may never receive the release event */
    wait_release = FALSE;

    /* Refreshes the stored modifier and keymap information */
    XRefreshKeyboardMapping (ev);

    /* Update internal modifiers masks if necessary */
    if (ev->request == MappingModifier)
    {
        init_modifiers ();
    }

    /* Regrab all keys if the notify is for keyboard (ie not pointer) */
    gdk_error_trap_push ();
    if (ev->request != MappingPointer)
    {
        for (element = shortcut_list; element != NULL; element = g_slist_next (element))
        {
            launcher *shortcut = (launcher *) element->data;
            ungrab_key (shortcut->key);
            grab_key (shortcut->key);
        }
    }
    gdk_flush();
    gdk_error_trap_pop ();
}

static void
handle_event (XEvent * ev)
{
    switch (ev->type)
    {
        case KeyPress:
            handleKeyPress ((XKeyPressedEvent *) ev);
            break;
        case KeyRelease:
            handleKeyRelease ((XKeyReleasedEvent *) ev);
            break;
        case MappingNotify:
            handleMappingNotify ((XMappingEvent *) ev);
            break;
        default:
            break;
    }
}

static GdkFilterReturn
event_filter (GdkXEvent *gdk_xevent, GdkEvent *event, gpointer data)
{
    handle_event ((XEvent *) gdk_xevent);

    return GDK_FILTER_CONTINUE;
}

static void
add_event_listener (void)
{
    GdkDisplay *gdisplay;

    gdisplay = gdk_display_get_default ();
    XAllowEvents (GDK_DISPLAY_XDISPLAY (gdisplay), AsyncBoth, CurrentTime);
    gdk_window_add_filter (NULL, event_filter, (gpointer) NULL);
}

static void
free_launcher_data (launcher * shortcut)
{
    ungrab_key (shortcut->key);
    g_free (shortcut->key);
    g_free (shortcut->command);
}

static void
free_shortcut_list (void)
{
    GSList *element;

    for (element = shortcut_list; element != NULL; element = g_slist_next (element))
    {
        launcher *launcher_data = (launcher *) element->data;
        free_launcher_data (launcher_data);
        g_free (launcher_data);
    }

    g_slist_free (shortcut_list);
    shortcut_list = NULL;
}

static GSList *
search_key_in_shortcut_list (gchar * key_string)
{
    GSList *element;
    MyKey *key;

    key = parseKeyString (key_string);

    for (element = shortcut_list; element != NULL; element = g_slist_next (element))
    {
        launcher *shortcut = (launcher *) element->data;

        if ((shortcut->key->keycode == key->keycode) && (shortcut->key->modifier == key->modifier))
        {
            g_free (key);
            return (element);
        }
    }
    g_free (key);

    return (NULL);
}

/*********************/
/* Parsing functions */
/*********************/
struct ShortcutsFileParserState
{
    gboolean started;
    GtkWidget *treeview;
    GQueue *parents;
    gchar *theme_name;
};

static gint
_find_attribute (const gchar ** attribute_names, const gchar * attr)
{
    gint i;

    for (i = 0; attribute_names[i]; i++)
    {
        if (!strcmp (attribute_names[i], attr))
            return i;
    }

    return -1;
}

/* functions used to extract the theme name */
static void
get_theme_name_start (GMarkupParseContext * context, const gchar * element_name,
    const gchar ** attribute_names, const gchar ** attribute_values, gpointer user_data, GError ** error)
{
    struct ShortcutsFileParserState *state = user_data;

    if (!state->started && !strcmp (element_name, "shortcuts-theme"))
    {
        int i;

        state->started = TRUE;
        if ((i = _find_attribute (attribute_names, "name")) != -1)
            state->theme_name = g_strdup (attribute_values[i]);

    }
}

static void
get_theme_name_end (GMarkupParseContext * context, const gchar * element_name, gpointer user_data, GError ** error)
{
    struct ShortcutsFileParserState *state = user_data;

    if (!strcmp (element_name, "shortcuts-theme"))
        state->started = FALSE;
}

static gchar *
get_theme_name (const gchar * filename)
{
    gchar *theme_name = NULL;
    gchar *file_contents = NULL;
    GMarkupParseContext *gpcontext = NULL;
    struct stat st;
    struct ShortcutsFileParserState state = { FALSE, NULL, NULL, NULL };
    GMarkupParser gmparser = { get_theme_name_start, get_theme_name_end, NULL, NULL, NULL };
    GError *err = NULL;
#ifdef HAVE_MMAP
    gint fd = -1;
    void *maddr = NULL;
#endif

    g_return_val_if_fail ( filename != NULL, NULL);

    if (stat (filename, &st) < 0)
    {
        g_warning ("Unable to open the shortcuts definitions file \"%s\"\n", filename);
        goto cleanup;
    }

#ifdef HAVE_MMAP
    fd = open (filename, O_RDONLY, 0);
    if (fd < 0)
        goto cleanup;

    maddr = mmap (NULL, st.st_size, PROT_READ, MAP_FILE | MAP_SHARED, fd, 0);
    if (maddr)
        file_contents = maddr;
#endif

    if (!file_contents && !g_file_get_contents (filename, &file_contents, NULL, &err))
    {
        if (err)
        {
            g_warning ("Unable to read file '%s' (%d): %s\n", filename, err->code, err->message);
            g_error_free (err);
        }

        goto cleanup;
    }

    state.started = FALSE;
    state.parents = g_queue_new ();

    gpcontext = g_markup_parse_context_new (&gmparser, 0, &state, NULL);

    if (!g_markup_parse_context_parse (gpcontext, file_contents, st.st_size, &err))
    {
        g_warning ("Error parsing shortcuts definitions (%d): %s\n", err->code, err->message);
        g_error_free (err);

        goto cleanup;
    }

    if (g_markup_parse_context_end_parse (gpcontext, NULL))
        theme_name = g_strdup (state.theme_name);

  cleanup:
    if (gpcontext)
        g_markup_parse_context_free (gpcontext);
#ifdef HAVE_MMAP
    if (maddr)
    {
        munmap (maddr, st.st_size);
        file_contents = NULL;
    }
    if (fd > -1)
        close (fd);
#endif
    if (file_contents)
        free (file_contents);
    if (state.parents)
    {
        g_queue_foreach (state.parents, (GFunc) g_free, NULL);
        g_queue_free (state.parents);
    }
    if (state.theme_name)
        g_free (state.theme_name);

    return theme_name;
}

/* functions used to extract the theme name */
static void
parse_theme_start (GMarkupParseContext * context, const gchar * element_name,
    const gchar ** attribute_names, const gchar ** attribute_values, gpointer user_data, GError ** error)
{
    struct ShortcutsFileParserState *state = user_data;
    int i, j;

    if (!state->started && !strcmp (element_name, "shortcuts-theme"))
    {
        state->started = TRUE;
    }
    else if (!state->started)
        return;

    i = _find_attribute (attribute_names, "keys");
    j = _find_attribute (attribute_names, "command");

    if (!strcmp (element_name, "shortcut") && (i != -1) && (j != -1))
    {

        if (state->treeview)
        {
            GtkTreeIter iter;
            GtkTreeModel *model;

            model = gtk_tree_view_get_model (GTK_TREE_VIEW (state->treeview));
            gtk_list_store_prepend (GTK_LIST_STORE (model), &iter);
            gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                                COLUMN_COMMAND, attribute_values[j], 
                                COLUMN_SHORTCUT, attribute_values[i] ? attribute_values[i] : "", -1);
        }
        
        if (attribute_values[i])
        {
            launcher *shortcut;

            shortcut = g_new (launcher, 1);
            shortcut->key = parseKeyString ((gchar *) attribute_values[i]);
            shortcut->command = g_strdup ((gchar *) attribute_values[j]);
            grab_key (shortcut->key);

            shortcut_list = g_slist_append (shortcut_list, shortcut);
        }
    }
}

static void
parse_theme_end (GMarkupParseContext * context, const gchar * element_name, gpointer user_data, GError ** error)
{
    struct ShortcutsFileParserState *state = user_data;

    if (!strcmp (element_name, "shortcuts-theme"))
        state->started = FALSE;
}

static gboolean
parse_theme (const gchar * filename, KeyboardMcsDialog * dialog)
{
    gboolean ret = FALSE;
    GtkTreeModel *model;
    gchar *file_contents = NULL;
    GMarkupParseContext *gpcontext = NULL;
    struct stat st;
    struct ShortcutsFileParserState state = { FALSE, NULL, NULL, NULL };
    GMarkupParser gmparser = { parse_theme_start, parse_theme_end, NULL, NULL, NULL };
    GError *err = NULL;
#ifdef HAVE_MMAP
    gint fd = -1;
    void *maddr = NULL;
#endif

    g_return_val_if_fail ((filename != NULL), FALSE);

    if (stat (filename, &st) < 0)
    {
        g_warning ("Unable to open the shortcuts definitions file \"%s\"\n", filename);
        goto cleanup;
    }

#ifdef HAVE_MMAP
    fd = open (filename, O_RDONLY, 0);
    if (fd < 0)
    {
        goto cleanup;
    }

    maddr = mmap (NULL, st.st_size, PROT_READ, MAP_FILE | MAP_SHARED, fd, 0);
    if (maddr)
    {
        file_contents = maddr;
    }
#endif

    if (!file_contents && !g_file_get_contents (filename, &file_contents, NULL, &err))
    {
        if (err)
        {
            g_warning ("Unable to read file '%s' (%d): %s\n", filename, err->code, err->message);
            g_error_free (err);
        }
        goto cleanup;
    }

    state.started = FALSE;
    state.parents = g_queue_new ();
    if (dialog)
    {
        state.treeview = dialog->treeview_shortcuts;
        model = gtk_tree_view_get_model (GTK_TREE_VIEW (state.treeview));
        gtk_list_store_clear (GTK_LIST_STORE (model));
    }
    else
    {
        state.treeview = NULL;
    }

    if (shortcut_list)
    {
        free_shortcut_list ();
    }

    gpcontext = g_markup_parse_context_new (&gmparser, 0, &state, NULL);

    gdk_error_trap_push ();
    if (!g_markup_parse_context_parse (gpcontext, file_contents, st.st_size, &err))
    {
        g_warning ("Error parsing shortcuts definitions (%d): %s\n", err->code, err->message);
        g_error_free (err);

        goto cleanup;
    }

    if (g_markup_parse_context_end_parse (gpcontext, NULL))
    {
        if (dialog)
        {
            dialog->theme_modified = FALSE;
        }
        ret = TRUE;
    }

  cleanup:
    gdk_flush();
    gdk_error_trap_pop ();
    if (gpcontext)
    {
        g_markup_parse_context_free (gpcontext);
    }
#ifdef HAVE_MMAP
    if (maddr)
    {
        munmap (maddr, st.st_size);
        file_contents = NULL;
    }
    if (fd > -1)
    {
        close (fd);
    }
#endif
    if (file_contents)
    {
        free (file_contents);
    }
    if (state.parents)
    {
        g_queue_foreach (state.parents, (GFunc) g_free, NULL);
        g_queue_free (state.parents);
    }
    if (state.theme_name)
    {
        g_free (state.theme_name);
    }

    return ret;
}

static gboolean
save_theme_foreach_func (GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    gchar *command;
    gchar *shortcut;
    FILE *file_theme;

    gtk_tree_model_get (model, iter, COLUMN_COMMAND, &command, COLUMN_SHORTCUT, &shortcut, -1);

    file_theme = (FILE *) data;

    fprintf (file_theme, "\t<shortcut command=\"%s\" keys=\"%s\"/>\n", command, shortcut ? shortcut : "");

    g_free (command);
    g_free (shortcut);

    return FALSE;
}

void
shortcuts_plugin_save_theme (KeyboardMcsDialog * dialog)
{
    GtkTreeModel *model;
    FILE *file_theme;

    g_return_if_fail (dialog != NULL);

    file_theme = fopen (theme_path, "w+");

    fprintf (file_theme, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
    fprintf (file_theme, "<shortcuts-theme name=\"%s\">\n", theme_name);

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->treeview_shortcuts));
    gtk_tree_model_foreach (model, &save_theme_foreach_func, file_theme);

    fprintf (file_theme, "</shortcuts-theme>\n");

    dialog->theme_modified = FALSE;

    fclose (file_theme);
}

/********************/
/* Plugin internals */
/********************/
static gint
themes_sort_func (GtkTreeModel * model, GtkTreeIter * a, GtkTreeIter * b, gpointer user_data)
{
    gchar *a_str = NULL;
    gchar *b_str = NULL;
    gint result;

    gtk_tree_model_get (model, a, 0, &a_str, -1);
    gtk_tree_model_get (model, b, 0, &b_str, -1);

    if (a_str == NULL)
        a_str = g_strdup ("");
    if (b_str == NULL)
        b_str = g_strdup ("");

    result = g_utf8_collate (a_str, b_str);

    g_free (a_str);
    g_free (b_str);

    return result;
}

static gint
shortcuts_sort_func (GtkTreeModel * model, GtkTreeIter * a, GtkTreeIter * b, gpointer user_data)
{
    gchar *a_str = NULL;
    gchar *b_str = NULL;
    gint result;

    gtk_tree_model_get (model, a, 0, &a_str, -1);
    gtk_tree_model_get (model, b, 0, &b_str, -1);

    if (a_str == NULL)
        a_str = g_strdup ("");
    if (b_str == NULL)
        b_str = g_strdup ("");

    result = g_utf8_collate (a_str, b_str);

    g_free (a_str);
    g_free (b_str);

    return result;
}

void shortcuts_plugin_response (gint response_id, KeyboardMcsDialog* status)
{
    if(response_id == GTK_RESPONSE_HELP)
    {
        g_message("HELP: TBD");
    }
}

static void
update_themes_list (KeyboardMcsDialog * dialog)
{
    GtkTreeModel *model;
    gchar *path;
    const gchar *filename;
    GtkTreeIter iter;
    GtkTreeRowReference *row_ref = NULL;
    GtkTreePath *tree_path;

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->treeview_shortcuts));
    gtk_list_store_clear (GTK_LIST_STORE (model));
    model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->treeview_themes));
    gtk_list_store_clear (GTK_LIST_STORE (model));

    /* Default system theme first */
    gtk_list_store_prepend (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_THEME_NAME, DEFAULT_SHORTCUT_THEME, 
                                                       COLUMN_FILE_NAME, DEFAULT_SHORTCUTS_PATH, -1);
    tree_path = gtk_tree_model_get_path(model, &iter);
    row_ref = gtk_tree_row_reference_new(model, tree_path);
    gtk_tree_path_free(tree_path);

    path = xfce_resource_lookup (XFCE_RESOURCE_CONFIG, "xfce4/shortcuts/");

    if (path)
    {
        GDir *dir_shortcuts;

        dir_shortcuts = g_dir_open (path, 0, NULL);

        while ((filename = g_dir_read_name (dir_shortcuts)))
        {
            if (g_str_has_suffix (filename, ".xml"))
            {
                gchar *full_path;
                gchar *theme;

                full_path = g_build_filename (path, filename, NULL);

                theme = get_theme_name (full_path);

                if (theme)
                {
                    gtk_list_store_prepend (GTK_LIST_STORE (model), &iter);
                    gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_THEME_NAME, theme, 
                                                                       COLUMN_FILE_NAME, full_path, -1);

                    if(strcmp(theme, theme_name) == 0)
                    {
                        GtkTreePath *tree_path = gtk_tree_model_get_path(model, &iter);
                        row_ref = gtk_tree_row_reference_new(model, tree_path);
                        gtk_tree_path_free(tree_path);
                    }
                }
                else
                {
                    g_warning ("Error while loading themes : %s has no name", full_path);
                }

                g_free (full_path);
                g_free (theme);
            }
        }
    }

    if(row_ref)
    {
        GtkTreePath *tree_path;

        tree_path = gtk_tree_row_reference_get_path(row_ref);
        gtk_tree_view_set_cursor(GTK_TREE_VIEW (dialog->treeview_themes), tree_path, NULL, FALSE);

        gtk_tree_path_free(tree_path);
        gtk_tree_row_reference_free(row_ref);
    }

    g_free (path);
}

static void
dialog_set_sensitive (KeyboardMcsDialog * dialog, gboolean active)
{
    gtk_widget_set_sensitive (dialog->button_del_theme, active);
    gtk_widget_set_sensitive (dialog->button_add_shortcut, active);
    gtk_widget_set_sensitive (dialog->button_del_shortcut, active);
    gtk_widget_set_sensitive (dialog->treeview_shortcuts, active);
    gtk_widget_set_sensitive (dialog->menuitem_popup_rename_theme, active);
    gtk_widget_set_sensitive (dialog->menuitem_popup_del_theme, active);
}

static void
cb_treeview_themes_selection_changed (GtkTreeSelection * selection, gpointer data)
{
    KeyboardMcsDialog *dialog;
    GtkTreeIter iter;
    GtkTreeModel *model;
    McsPlugin *mcs_plugin;
    gchar *new_theme, *new_path;

    dialog = (KeyboardMcsDialog *) data;
    mcs_plugin = dialog->mcs_plugin;

    if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
        gtk_tree_model_get (model, &iter, COLUMN_THEME_NAME, &new_theme, COLUMN_FILE_NAME, &new_path, -1);
        dialog_set_sensitive (dialog, strcmp (DEFAULT_SHORTCUT_THEME, new_theme));

        if (theme_name && strcmp (theme_name, new_theme))
        {
            if (g_file_test (new_path, G_FILE_TEST_EXISTS))
            {
                /* save the current theme */
                if (dialog->theme_modified)
                    shortcuts_plugin_save_theme (dialog);

                /* update and notify the channel value */
                g_free (theme_name);
                g_free (theme_path);
                theme_name = new_theme;
                theme_path = new_path;
                mcs_manager_set_string (mcs_plugin->manager, "Xfce4/ShortcutThemeName", CHANNEL3, new_theme);
                mcs_manager_set_string (mcs_plugin->manager, "Xfce4/ShortcutThemeFile", CHANNEL3, new_path);
                mcs_manager_notify (mcs_plugin->manager, CHANNEL3);
                shortcuts_plugin_save_settings (mcs_plugin);

                /* load the new theme in the treeview */
                parse_theme (new_path, dialog);
            }
            else
            {
                g_warning ("The shortcut theme file doesn't exist !");

                g_free (new_theme);
                g_free (new_path);
                update_themes_list (dialog);
            }
        }
    }
    else
    {
        dialog_set_sensitive (dialog, FALSE);
    }
}

/* theme treeview : popup menu */
static void
show_rename_theme_dialog (KeyboardMcsDialog * dialog)
{
    GtkWidget *dialog_input;
    GtkWidget *hbox;
    GtkWidget *label;
    GtkWidget *entry;

    dialog_input = gtk_dialog_new_with_buttons (_("Rename theme"), 
                                                GTK_WINDOW (gtk_widget_get_toplevel(dialog->dialog_keyboard)),
                                                GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, 
                                                GTK_STOCK_CANCEL, 
                                                GTK_RESPONSE_REJECT, 
                                                GTK_STOCK_OK, 
                                                GTK_RESPONSE_ACCEPT, 
                                                NULL);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog_input),
                                     GTK_RESPONSE_ACCEPT);

    hbox = gtk_hbox_new (FALSE, BORDER);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_input)->vbox), hbox, FALSE, TRUE, BORDER);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("New name:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
    gtk_widget_show (label);

    entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (hbox), entry, FALSE, TRUE, 0);
    gtk_entry_set_text (GTK_ENTRY (entry), theme_name);
    gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
    gtk_widget_show (entry);

    if (gtk_dialog_run (GTK_DIALOG (dialog_input)) == GTK_RESPONSE_ACCEPT)
    {
        g_free (theme_name);
        theme_name = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));

        shortcuts_plugin_save_theme (dialog);
        update_themes_list (dialog);
    }

    gtk_widget_destroy (dialog_input);
}

static void
cb_treeview_themes_activate (GtkWidget * treeview, GtkTreePath * path, GtkTreeViewColumn * column, gpointer data)
{
    KeyboardMcsDialog * dialog = (KeyboardMcsDialog *) data;
    
    if (strcmp(DEFAULT_SHORTCUT_THEME, theme_name))
    {
        show_rename_theme_dialog (dialog);
    }
}

static gboolean
is_modifier (guint keycode)
{
    XModifierKeymap *keymap;
    gboolean result = FALSE;
    gint n;

    keymap = XGetModifierMapping (gdk_display);
    for (n = 0; n < keymap->max_keypermod * 8; ++n)
    {
        if (keycode == keymap->modifiermap[n])
        {
            result = TRUE;
            break;
        }
    }

    XFreeModifiermap (keymap);

    return result;
}

static gboolean
shortcut_tree_foreach_func (GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    shortcut_tree_foreach_struct *stfs = (shortcut_tree_foreach_struct *) data;
    gchar *shortcut_key = stfs->shortcut;
    gchar *current_shortcut = NULL;

    gtk_tree_model_get (model, iter, COLUMN_SHORTCUT, &current_shortcut, -1);

    if (!gtk_tree_selection_path_is_selected (stfs->selection, path) && (strcmp (shortcut_key, current_shortcut) == 0))
    {
        stfs->found = TRUE;
        stfs->path = gtk_tree_path_to_string (path);
    }

    g_free (current_shortcut);

    return stfs->found;
}

static gboolean
cb_compose_shortcut (GtkWidget * widget, GdkEventKey * event, gpointer data)
{
    KeyboardMcsDialog *dialog = (KeyboardMcsDialog *) data;
    GdkModifierType consumed_modifiers = 0;
    gchar shortcut_string[80] = "";
    guint keyval;
    guint modifiers;
    gchar *accelerator;
    gint i;
    gchar **shortcuts;
    gchar **current_shortcut;
    shortcut_tree_foreach_struct stfs;
    GtkTreeModel *model;
    GtkTreeSelection *selection;
    GtkTreeIter iter;
    GSList *element;
    gchar *command;

    if (is_modifier (event->hardware_keycode))
        return TRUE;

    gdk_keymap_translate_keyboard_state (gdk_keymap_get_default (), 
        event->hardware_keycode, event->state, event->group, NULL, NULL, NULL, &consumed_modifiers);

    keyval = gdk_keyval_to_lower (event->keyval);

    switch (keyval)
    {
        case GDK_ISO_Left_Tab:
            keyval = GDK_Tab;
            break;
        case GDK_ISO_Level3_Latch:
        case GDK_ISO_Level3_Lock:
        case GDK_ISO_Level3_Shift:
        case GDK_Scroll_Lock:
            return TRUE;
            break;
        default:
            break;
    }

    /* Release keyboard */
    gdk_keyboard_ungrab (GDK_CURRENT_TIME);

    /*
     * gtk_accelerator_get_default_mod_mask () limits the number of modifiers.
     * That's a good thing because we don't want the "fake" modifiers such as
     * Scroll/Num/Caps Lock to be taken into account.
     *
     * Unfortunately, at this time of writing, gdk doesn't know about all of the
     * modifiers, like META, SUPER or HYPER, but just SHIFT, ALT and CONTROL...
     *
     * It means that ppl won't be able to use the "Windows" key as a modifier...
     * Too bad, that's life, you may gently ask the gtk+ maintainers about it.
     *
     * Things may change in the future? See this thread:
     *
     * http://mail.gnome.org/archives/gtk-devel-list/2005-September/msg00024.html
     *
     * (Olivier)
     */

    modifiers = event->state & (~consumed_modifiers | GDK_MODIFIER_MASK);
    modifiers = modifiers & gtk_accelerator_get_default_mod_mask ();
    accelerator = gtk_accelerator_name (keyval, modifiers);

    for (i = 0; i < strlen (accelerator); i++)
    {
        if (accelerator[i] == '>')
            accelerator[i] = '<';
    }

    shortcuts = g_strsplit (accelerator, "<", 0);

    current_shortcut = shortcuts;
    while (*current_shortcut)
    {
        if (strlen (*current_shortcut))
        {
            strcat (shortcut_string, *current_shortcut);
            strcat (shortcut_string, "+");
        }
        *current_shortcut++;
    }

    shortcut_string[strlen (shortcut_string) - 1] = '\0';       /* replace the trailing '+' */
    g_free (accelerator);
    g_strfreev (shortcuts);

    /* check if shorcut already exists */
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview_shortcuts));
    gtk_tree_selection_get_selected (selection, &model, &iter);

    stfs.shortcut = shortcut_string;
    stfs.found = FALSE;
    stfs.selection = selection;
    gtk_tree_model_foreach (model, &shortcut_tree_foreach_func, &stfs);

    if (stfs.found)
    {
        GtkTreePath *path_old;
        GtkTreeIter iter_old;
        GtkWidget *dialog;

        dialog = gtk_message_dialog_new (GTK_WINDOW (widget),
            GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, _("Shortcut already in use !\nAre you sure you want to use it ?"));

        if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_NO)
        {
            gtk_dialog_response (GTK_DIALOG (widget), GTK_RESPONSE_OK);
            return TRUE;
        }

        path_old = gtk_tree_path_new_from_string (stfs.path);
        gtk_tree_model_get_iter (model, &iter_old, path_old);
        g_free (stfs.path);
        gtk_tree_path_free (path_old);

        gtk_list_store_set (GTK_LIST_STORE (model), &iter_old, COLUMN_SHORTCUT, "", -1);
    }

    gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_SHORTCUT, shortcut_string, -1);

    gtk_tree_model_get (model, &iter, COLUMN_COMMAND, &command, -1);

    gdk_error_trap_push ();
    if ((element = search_key_in_shortcut_list (shortcut_string)) != NULL)
    {
        /* Change the command associated with the shortcut */
        launcher *launcher_data = (launcher *) element->data;

        free_launcher_data (launcher_data);
        launcher_data->key = parseKeyString (shortcut_string);
        launcher_data->command = g_strdup (command);
        grab_key (launcher_data->key);
    }
    else
    {
        /* Add the shortcut to the list */
        launcher *launcher_data;

        launcher_data = g_new (launcher, 1);
        launcher_data->key = parseKeyString (shortcut_string);
        launcher_data->command = g_strdup (command);
        grab_key (launcher_data->key);
        shortcut_list = g_slist_append (shortcut_list, launcher_data);
    }
    gdk_flush();
    gdk_error_trap_pop ();
    g_free (command);

    /* End operations */
    dialog->theme_modified = TRUE;
    gtk_dialog_response (GTK_DIALOG (widget), GTK_RESPONSE_OK);

    return TRUE;
}

static void
cb_treeview_shortcuts_activate (GtkWidget * treeview, GtkTreePath * path, GtkTreeViewColumn * column, gpointer data)
{
    KeyboardMcsDialog *dialog = (KeyboardMcsDialog *) data;

    if (column == gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), COLUMN_COMMAND))
    {
        GtkTreeSelection *selection;
        GtkTreeModel *model;
        GtkTreeIter iter;

        gchar *shortcut = NULL;
        gchar *command = NULL;

        GtkWidget *dialog_command;
        GtkWidget *label;
        GtkWidget *entry;
        GtkWidget *img;
        GtkWidget *button;
        GtkWidget *hbox;
        GtkWidget *hbox_entry;

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
        gtk_tree_selection_get_selected (selection, &model, &iter);
        gtk_tree_model_get (model, &iter, COLUMN_SHORTCUT, &shortcut, COLUMN_COMMAND, &command, -1);

        /* Create dialog */
        dialog_command = gtk_dialog_new_with_buttons (_("Choose command"), 
            GTK_WINDOW (gtk_widget_get_toplevel(dialog->dialog_keyboard)),
            GTK_DIALOG_MODAL, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
        gtk_dialog_set_default_response (GTK_DIALOG (dialog_command),
            GTK_RESPONSE_OK);
        label = gtk_label_new (_("Command:"));
        entry = gtk_entry_new_with_max_length (255);
        gtk_entry_set_text (GTK_ENTRY (entry), command);
        gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);

        hbox_entry = gtk_hbox_new (FALSE, BORDER);
        gtk_box_pack_start (GTK_BOX (hbox_entry), entry, FALSE, FALSE, 0);
        img = gtk_image_new_from_stock (GTK_STOCK_OPEN, GTK_ICON_SIZE_BUTTON);
        button = gtk_button_new ();
        gtk_container_add (GTK_CONTAINER (button), img);
        g_signal_connect (G_CALLBACK (button), "clicked", G_CALLBACK (cb_browse_command), entry);
        gtk_box_pack_start (GTK_BOX (hbox_entry), button, FALSE, FALSE, 0);

        hbox = gtk_hbox_new (FALSE, BORDER);
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), hbox_entry, FALSE, TRUE, 0);
        gtk_container_set_border_width (GTK_CONTAINER (hbox), BORDER);

        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_command)->vbox), hbox, FALSE, TRUE, 0);
        gtk_widget_show_all (dialog_command);

        if (gtk_dialog_run (GTK_DIALOG (dialog_command)) == GTK_RESPONSE_OK)
        {

            if (strlen (gtk_entry_get_text (GTK_ENTRY (entry))) > 0 && command_exists (gtk_entry_get_text (GTK_ENTRY (entry))))
            {
                gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_COMMAND, gtk_entry_get_text (GTK_ENTRY (entry)), -1);
                dialog->theme_modified = TRUE;
            }
            else
                xfce_err (_("The command doesn't exist or the file is not executable !"));
        }

        gtk_widget_destroy (dialog_command);
        g_free (shortcut);
        g_free (command);
    }
    else if (column == gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), COLUMN_SHORTCUT))
    {
        GtkTreeSelection *selection;
        GtkTreeModel *model;
        GtkTreeIter iter;
        gchar *command = NULL;
        gchar *shortcut = NULL;
        GtkWidget *dialog_compose;
        GtkWidget *hbox;
        GdkPixbuf *icon = NULL;
        GtkWidget *image;
        GtkWidget *label;
        GtkWidget *button;
        GSList *element;
        gint response;
        gchar *dialog_text = NULL;

        /* Get shortcut name */
        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
        gtk_tree_selection_get_selected (selection, &model, &iter);
        gtk_tree_model_get (model, &iter, COLUMN_COMMAND, &command, COLUMN_SHORTCUT, &shortcut, -1);

        if ((element = search_key_in_shortcut_list (shortcut)))
        {
            launcher *launcher_data = (launcher *) element->data;

            free_launcher_data (launcher_data);
            shortcut_list = g_slist_remove (shortcut_list, launcher_data);
            g_free (launcher_data);
        }

        dialog_text = g_strdup_printf ("<i>%s</i>\n<b>%s</b>", _("Set shortcut for command:"), command);

        /* Create dialog */
        dialog_compose = gtk_dialog_new_with_buttons (_("Set shortcut"), 
            GTK_WINDOW (gtk_widget_get_toplevel(dialog->dialog_keyboard)),
            GTK_DIALOG_MODAL, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);

        button = xfce_create_mixed_button (GTK_STOCK_CLEAR, _("No shortcut"));
        gtk_widget_show (button);
        gtk_dialog_add_action_widget (GTK_DIALOG (dialog_compose), button, GTK_RESPONSE_NO);

        hbox = gtk_hbox_new (FALSE, BORDER);
        gtk_container_set_border_width (GTK_CONTAINER (hbox), BORDER);
        gtk_widget_show (hbox);

        icon = xfce_themed_icon_load ("xfce4-keys.png", 48);
        if (icon)
        {
            image = gtk_image_new_from_pixbuf (icon);
            gtk_widget_show (image);
            gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, TRUE, 0);
        }

        label = gtk_label_new (dialog_text);
        gtk_label_set_markup (GTK_LABEL (label), dialog_text);
        gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);
        gtk_widget_show (label);
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_compose)->vbox), hbox, FALSE, TRUE, 0);

        /* Center cancel button */
        gtk_button_box_set_layout (GTK_BUTTON_BOX (GTK_DIALOG (dialog_compose)->action_area), GTK_BUTTONBOX_SPREAD);

        /* Connect signals */
        g_signal_connect (G_OBJECT (dialog_compose), "key-release-event", G_CALLBACK (cb_compose_shortcut), dialog);

        /* Take control on the keyboard */
        if (gdk_keyboard_grab (gtk_widget_get_root_window (dialog_compose), TRUE, GDK_CURRENT_TIME) != GDK_GRAB_SUCCESS)
        {
            g_warning ("Cannot grab the keyboard");
            g_free (dialog_text);
            g_free (shortcut);
            g_free (command);
            return;
        }

        /* Stop our own event filter so that it doesn's interfere with gtk+ */
        gdk_window_remove_filter (NULL, event_filter, (gpointer) NULL);

        /* Show dialog */
        response = gtk_dialog_run (GTK_DIALOG (dialog_compose));

        /* Put back our event filter */
        gdk_window_add_filter (NULL, event_filter, (gpointer) NULL);

        if (response == GTK_RESPONSE_NO)
        {
            gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_SHORTCUT, "", -1);
            dialog->theme_modified = TRUE;
        }

        /* Release keyboard if not yet done */
        gdk_keyboard_ungrab (GDK_CURRENT_TIME);

        gtk_widget_destroy (dialog_compose);
        g_free (dialog_text);
        g_free (command);
        g_free (shortcut);
    }
}

static gboolean
cb_popup_menu_themes (GtkTreeView * treeview, GdkEventButton * event, gpointer data)
{
    KeyboardMcsDialog *dialog;

    dialog = (KeyboardMcsDialog *) data;

    /* Right click draws the context menu */
    if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS))
    {
        GtkTreePath *path;
        GdkScreen *screen;

        if (gtk_tree_view_get_path_at_pos (treeview, event->x, event->y, &path, NULL, NULL, NULL))
        {
            GtkTreeSelection *selection;
            GtkTreeModel *model;
            GtkTreeIter iter;
            gchar *theme_name;

            selection = gtk_tree_view_get_selection (treeview);
            gtk_tree_selection_unselect_all (selection);
            gtk_tree_selection_select_path (selection, path);
            gtk_tree_selection_get_selected (selection, &model, &iter);
            gtk_tree_model_get (model, &iter, COLUMN_THEME_NAME, &theme_name, -1);
            
            if (strcmp (DEFAULT_SHORTCUT_THEME, theme_name))
            {
                gtk_widget_set_sensitive (dialog->menuitem_popup_rename_theme, TRUE);
                gtk_widget_set_sensitive (dialog->menuitem_popup_del_theme, TRUE);
            }
            else
            {
                gtk_widget_set_sensitive (dialog->menuitem_popup_rename_theme, FALSE);
                gtk_widget_set_sensitive (dialog->menuitem_popup_del_theme, FALSE);
            }
            g_free (theme_name);

        }
        else
        {
            gtk_widget_set_sensitive (dialog->menuitem_popup_rename_theme, FALSE);
            gtk_widget_set_sensitive (dialog->menuitem_popup_del_theme, FALSE);
        }

        screen = xfce_gdk_display_locate_monitor_with_pointer (NULL, NULL);
        gtk_menu_set_screen (GTK_MENU (dialog->menu_popup_themes), screen ? screen : gdk_screen_get_default ());
        gtk_menu_popup (GTK_MENU (dialog->menu_popup_themes), NULL, NULL, NULL, NULL, event->button, gtk_get_current_event_time ());

        return TRUE;
    }

    return FALSE;
}

/* shortcuts treeview : popup menu */
static gboolean
cb_popup_menu_shortcuts (GtkTreeView * treeview, GdkEventButton * event, gpointer data)
{
    KeyboardMcsDialog *dialog;

    dialog = (KeyboardMcsDialog *) data;

    /* Right click draws the context menu */
    if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS))
    {
        GtkTreePath *path;
        GdkScreen *screen;

        if (gtk_tree_view_get_path_at_pos (treeview, event->x, event->y, &path, NULL, NULL, NULL))
        {
            GtkTreeSelection *selection;

            selection = gtk_tree_view_get_selection (treeview);
            gtk_tree_selection_unselect_all (selection);
            gtk_tree_selection_select_path (selection, path);
            gtk_widget_set_sensitive (dialog->menuitem_popup_edit_shortcut, TRUE);
            gtk_widget_set_sensitive (dialog->menuitem_popup_del_shortcut, TRUE);
        }
        else
        {
            gtk_widget_set_sensitive (dialog->menuitem_popup_edit_shortcut, FALSE);
            gtk_widget_set_sensitive (dialog->menuitem_popup_del_shortcut, FALSE);
        }

        screen = xfce_gdk_display_locate_monitor_with_pointer (NULL, NULL);
        gtk_menu_set_screen (GTK_MENU (dialog->menu_popup_shortcuts), screen ? screen : gdk_screen_get_default ());
        gtk_menu_popup (GTK_MENU (dialog->menu_popup_shortcuts), NULL, NULL, NULL, NULL, event->button, gtk_get_current_event_time ());

        return TRUE;
    }

    return FALSE;
}

static void
add_theme (KeyboardMcsDialog *dialog)
{
    GtkWidget *dialog_input;
    GtkWidget *hbox;
    GtkWidget *label;
    GtkWidget *entry;
    GtkTreeSelection *selection;

    dialog_input = gtk_dialog_new_with_buttons (_("New theme"), 
                                                GTK_WINDOW (gtk_widget_get_toplevel(dialog->dialog_keyboard)),
                                                GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, 
                                                GTK_STOCK_CANCEL, 
                                                GTK_RESPONSE_REJECT, 
                                                GTK_STOCK_OK, 
                                                GTK_RESPONSE_ACCEPT, 
                                                NULL);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog_input),
                                     GTK_RESPONSE_ACCEPT);
    hbox = gtk_hbox_new (FALSE, BORDER);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_input)->vbox), hbox, FALSE, TRUE, BORDER);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("Name:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
    gtk_widget_show (label);

    entry = gtk_entry_new ();
    gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), entry, FALSE, TRUE, 0);
    gtk_widget_show (entry);

    if ((gtk_dialog_run (GTK_DIALOG (dialog_input)) == GTK_RESPONSE_ACCEPT) &&
        strcmp(DEFAULT_SHORTCUT_THEME, gtk_entry_get_text (GTK_ENTRY (entry))))
    {
        gchar *path;
        gchar *temp;
        gchar *full_path;

        FILE *file_theme;
        GtkTreeIter iter;
        GtkTreeModel *model;

        path = xfce_resource_lookup (XFCE_RESOURCE_CONFIG, "xfce4/shortcuts/");
        temp = g_build_filename (path, gtk_entry_get_text (GTK_ENTRY (entry)), NULL);
        full_path = g_strconcat (temp, ".xml", NULL);
        g_free (temp);

        while (g_file_test (full_path, G_FILE_TEST_EXISTS))
        {
            GtkWidget *dialog_filename;
            GtkWidget *entry_new_filename;

            xfce_warn ("The file %s already exists, please select an other name.", full_path);
            g_free (full_path);

            dialog_filename = gtk_dialog_new_with_buttons (_("File already exists"), 
                                                           GTK_WINDOW (gtk_widget_get_toplevel(dialog->dialog_keyboard)),
                                                           GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, 
                                                           GTK_STOCK_CANCEL, 
                                                           GTK_RESPONSE_REJECT, 
                                                           GTK_STOCK_OK, 
                                                           GTK_RESPONSE_ACCEPT, 
                                                           NULL);

            hbox = gtk_hbox_new (FALSE, BORDER);
            gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_filename)->vbox), hbox, FALSE, TRUE, BORDER);
            gtk_widget_show (hbox);

            label = gtk_label_new (_("Filename:"));
            gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
            gtk_widget_show (label);

            entry_new_filename = gtk_entry_new ();
            gtk_box_pack_start (GTK_BOX (hbox), entry_new_filename, FALSE, TRUE, 0);
            gtk_widget_show (entry_new_filename);

            if (gtk_dialog_run (GTK_DIALOG (dialog_filename)) == GTK_RESPONSE_ACCEPT)
            {
                temp = g_build_filename (path, gtk_entry_get_text (GTK_ENTRY (entry_new_filename)), NULL);

                if (g_str_has_suffix (temp, ".xml"))
                    full_path = g_strdup (temp);
                else
                    full_path = g_strconcat (temp, ".xml", NULL);

                g_free (temp);
            }
            else
            {
                gtk_widget_destroy (dialog_filename);
                gtk_widget_destroy (dialog_input);
                g_free (path);
                return;
            }
            gtk_widget_destroy (dialog_filename);
        }

        if (dialog->theme_modified)
            shortcuts_plugin_save_theme (dialog);

        file_theme = fopen (full_path, "w");
        fprintf (file_theme, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        fprintf (file_theme, "<shortcuts-theme name=\"%s\">\n", gtk_entry_get_text (GTK_ENTRY (entry)));
        model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->treeview_shortcuts));
        gtk_tree_model_foreach (model, &save_theme_foreach_func, file_theme);
        fprintf (file_theme, "</shortcuts-theme>\n");
        fclose (file_theme);

        model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->treeview_themes));
        gtk_list_store_prepend (GTK_LIST_STORE (model), &iter);
        gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_THEME_NAME, gtk_entry_get_text (GTK_ENTRY (entry)), COLUMN_FILE_NAME, full_path, -1);

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview_themes));
        gtk_tree_selection_select_iter (selection, &iter);

        g_free (path);
        g_free (full_path);
    }

    gtk_widget_destroy (dialog_input);
}

static void
del_theme (KeyboardMcsDialog *dialog)
{
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreeIter iter;
    gchar *theme_name;
    gchar *theme_path;
    gchar *message;
    McsPlugin *mcs_plugin;

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview_themes));

    if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    {
        return;
    }

    gtk_tree_model_get (model, &iter, COLUMN_THEME_NAME, &theme_name, COLUMN_FILE_NAME, &theme_path, -1);

    message = g_strdup_printf (_("Do you really want to delete the '%s' theme ?"), theme_name);
    mcs_plugin = dialog->mcs_plugin;

    if (xfce_confirm (message, GTK_STOCK_YES, NULL))
    {
        unlink (theme_path);
        g_free (theme_name);
        g_free (theme_path);

        theme_name = g_strdup (DEFAULT_SHORTCUT_THEME);
        theme_path = g_strdup (DEFAULT_SHORTCUTS_PATH);
        update_themes_list (dialog);
    }

    g_free (message);
}

static void
cb_button_add_theme_clicked (GtkWidget * widget, gpointer data)
{
    add_theme ((KeyboardMcsDialog *) data);
}

static void
cb_button_del_theme_clicked (GtkWidget * widget, gpointer data)
{
    del_theme ((KeyboardMcsDialog *) data);
}

static void
cb_button_del_shortcut_clicked (GtkWidget * widget, gpointer data)
{
    KeyboardMcsDialog *dialog;
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreeIter iter;
    gchar *command;
    gchar *shortcut;
    gchar *message;

    dialog = (KeyboardMcsDialog *) data;

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview_shortcuts));

    if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    {
        return;
    }

    gtk_tree_model_get (model, &iter, COLUMN_COMMAND, &command, COLUMN_SHORTCUT, &shortcut, -1);

    message = g_strdup_printf (_("Do you really want to delete the shorcut entry for the '%s' command ?"), command);

    if (xfce_confirm (message, GTK_STOCK_YES, NULL))
    {
        GSList *element;

        dialog->theme_modified = TRUE;

        if ((element = search_key_in_shortcut_list (shortcut)) != NULL)
        {
            launcher *launcher_data = (launcher *) element->data;

            free_launcher_data (launcher_data);
            shortcut_list = g_slist_remove (shortcut_list, launcher_data);
            g_free (launcher_data);
        }
        gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
    }

    g_free (shortcut);
    g_free (command);
    g_free (message);
}

static void
cb_button_add_shortcut_clicked (GtkWidget * widget, gpointer data)
{
    KeyboardMcsDialog *dialog;
    GtkWidget *dialog_command;
    GtkWidget *label;
    GtkWidget *entry;
    GtkWidget *img;
    GtkWidget *button;
    GtkWidget *hbox;
    GtkWidget *hbox_entry;

    dialog = (KeyboardMcsDialog *) data;

    /* Create dialog */
    dialog_command = gtk_dialog_new_with_buttons (_("Choose command"), 
                                                  GTK_WINDOW (gtk_widget_get_toplevel(dialog->dialog_keyboard)),
                                                  GTK_DIALOG_MODAL, 
                                                  GTK_STOCK_CANCEL, 
                                                  GTK_RESPONSE_CANCEL, 
                                                  GTK_STOCK_OK, 
                                                  GTK_RESPONSE_OK,
                                                  NULL);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog_command),
                                     GTK_RESPONSE_OK);

    label = gtk_label_new (_("Command:"));
    entry = gtk_entry_new_with_max_length (255);
    gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);

    hbox_entry = gtk_hbox_new (FALSE, BORDER);
    gtk_box_pack_start (GTK_BOX (hbox_entry), entry, FALSE, FALSE, 0);
    img = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_BUTTON);
    button = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (button), img);
    g_signal_connect (G_CALLBACK (button), "clicked", G_CALLBACK (cb_browse_command), entry);
    gtk_box_pack_start (GTK_BOX (hbox_entry), button, FALSE, FALSE, 0);

    hbox = gtk_hbox_new (FALSE, BORDER);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), hbox_entry, FALSE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), BORDER);

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_command)->vbox), hbox, FALSE, TRUE, 0);
    gtk_widget_show_all (dialog_command);

    if (gtk_dialog_run (GTK_DIALOG (dialog_command)) == GTK_RESPONSE_OK)
    {

        if (strlen (gtk_entry_get_text (GTK_ENTRY (entry))) > 0 && command_exists (gtk_entry_get_text (GTK_ENTRY (entry))))
        {
            GtkTreeModel *model;
            GtkTreeIter iter;
            GtkTreeViewColumn *column;
            GtkTreePath *tree_path;
            
            model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->treeview_shortcuts));
            gtk_list_store_prepend (GTK_LIST_STORE (model), &iter);
            gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_COMMAND, 
                                gtk_entry_get_text (GTK_ENTRY (entry)), COLUMN_SHORTCUT, "", -1);
            column = gtk_tree_view_get_column (GTK_TREE_VIEW (dialog->treeview_shortcuts), COLUMN_SHORTCUT);
            tree_path = gtk_tree_model_get_path(model, &iter);
            gtk_tree_view_set_cursor(GTK_TREE_VIEW (dialog->treeview_shortcuts), tree_path, column, FALSE);
            cb_treeview_shortcuts_activate (dialog->treeview_shortcuts, tree_path, column, data);
            gtk_tree_path_free(tree_path);

            dialog->theme_modified = TRUE;
        }
        else
        {
            xfce_err (_("The command doesn't exist or the file is not executable !"));
        }
    }

    gtk_widget_destroy (dialog_command);
}

static void
cb_menuitem_add_theme_activate (GtkWidget * widget, gpointer data)
{
    add_theme ((KeyboardMcsDialog *) data);
}

static void
cb_menuitem_del_theme_activate (GtkWidget * widget, gpointer data)
{
    del_theme ((KeyboardMcsDialog *) data);
}

static void
cb_menuitem_rename_theme_activate (GtkWidget * widget, gpointer data)
{
    KeyboardMcsDialog * dialog = (KeyboardMcsDialog *) data;
    
    if (strcmp(DEFAULT_SHORTCUT_THEME, theme_name))
    {
        show_rename_theme_dialog (dialog);
    }
}

static void
cb_menuitem_edit_shortcut_activate (GtkWidget * widget, gpointer data)
{
    KeyboardMcsDialog * dialog = (KeyboardMcsDialog *) data;

    GtkTreeModel *model;
    GtkTreeIter iter;
    GtkTreeViewColumn *column;
    GtkTreePath *tree_path;
    GtkTreeSelection *selection;

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview_shortcuts));
    if (!selection)
    {
        return;
    }

    gtk_tree_selection_get_selected (selection, &model, &iter);
    tree_path = gtk_tree_model_get_path(model, &iter);
    column = gtk_tree_view_get_column (GTK_TREE_VIEW (dialog->treeview_shortcuts), COLUMN_COMMAND);
    cb_treeview_shortcuts_activate (dialog->treeview_shortcuts, tree_path, column, data);
    gtk_tree_path_free(tree_path);
}

static void
cb_menuitem_add_shortcut_activate (GtkWidget * widget, gpointer data)
{
    KeyboardMcsDialog * dialog = (KeyboardMcsDialog *) data;

    cb_button_add_shortcut_clicked (dialog->button_add_shortcut, data);
}

static void
cb_menuitem_del_shortcut_activate (GtkWidget * widget, gpointer data)
{
    KeyboardMcsDialog * dialog = (KeyboardMcsDialog *) data;

    cb_button_del_shortcut_clicked (dialog->button_del_shortcut, data);
}

/********************/
/* Shortcut edition */
/********************/
static gboolean
command_exists (const gchar * command)
{
    gchar *cmd_buf = NULL;
    gchar *cmd_tok = NULL;
    gboolean result = FALSE;

    cmd_buf = g_strdup (command);
    cmd_tok = strtok (cmd_buf, " ");

    if (g_find_program_in_path (cmd_buf) != NULL)
        result = TRUE;

    g_free (cmd_buf);

    return result;
}

static void
cb_browse_command (GtkWidget * widget, GtkEntry * entry_command)
{
    GtkWidget *filesel_dialog;

    filesel_dialog = xfce_file_chooser_new (_("Select command"), NULL,
                                            XFCE_FILE_CHOOSER_ACTION_OPEN, 
                                            GTK_STOCK_CANCEL, 
                                            GTK_RESPONSE_CANCEL, 
                                            GTK_STOCK_OPEN, 
                                            GTK_RESPONSE_ACCEPT, 
                                            NULL);

    xfce_file_chooser_set_filename (XFCE_FILE_CHOOSER (filesel_dialog), 
                                    gtk_entry_get_text (entry_command));
    xfce_gtk_window_center_on_monitor_with_pointer (GTK_WINDOW (filesel_dialog));

    if (gtk_dialog_run (GTK_DIALOG (filesel_dialog)) == GTK_RESPONSE_ACCEPT)
    {
        gchar *filename;

        filename = xfce_file_chooser_get_filename (XFCE_FILE_CHOOSER (filesel_dialog));
        gtk_entry_set_text (entry_command, filename);

        g_free (filename);
    }

    gtk_widget_destroy (filesel_dialog);
}

GtkWidget* shortcuts_plugin_create_dialog (KeyboardMcsDialog *dialog)
{
    GtkWidget *hbox, *hbox2;
    GtkWidget *vbox;
    GtkWidget *frame;

    GtkListStore *model;

    hbox = gtk_hbox_new (FALSE, BORDER);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), BORDER);
    gtk_widget_show (hbox);

    /* */
    frame = xfce_framebox_new (_("Themes"), FALSE);
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, TRUE, 0);

    vbox = gtk_vbox_new (FALSE, BORDER);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), BORDER);
    gtk_widget_show (vbox);
    xfce_framebox_add (XFCE_FRAMEBOX (frame), vbox);

    dialog->scrolledwindow_themes = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (dialog->scrolledwindow_themes), 
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_set_border_width (GTK_CONTAINER (dialog->scrolledwindow_themes), BORDER);
    gtk_widget_show (dialog->scrolledwindow_themes);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->scrolledwindow_themes, TRUE, TRUE, 0);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (dialog->scrolledwindow_themes), GTK_SHADOW_IN);

    dialog->treeview_themes = gtk_tree_view_new ();
    gtk_widget_show (dialog->treeview_themes);
    gtk_container_add (GTK_CONTAINER (dialog->scrolledwindow_themes), dialog->treeview_themes);
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (dialog->treeview_themes), FALSE);

    dialog->button_add_theme = gtk_button_new_from_stock (GTK_STOCK_ADD);
    gtk_widget_show (dialog->button_add_theme);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->button_add_theme, FALSE, FALSE, 0);
    dialog->button_del_theme = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
    gtk_widget_show (dialog->button_del_theme);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->button_del_theme, FALSE, FALSE, 0);
    gtk_widget_set_sensitive (dialog->button_del_theme, FALSE);

    /* */
    frame = xfce_framebox_new (_("Shortcuts"), FALSE);
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);

    vbox = gtk_vbox_new (FALSE, BORDER);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), BORDER);
    gtk_widget_show (vbox);
    xfce_framebox_add (XFCE_FRAMEBOX (frame), vbox);

    dialog->scrolledwindow_shortcuts = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (dialog->scrolledwindow_shortcuts), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_set_border_width (GTK_CONTAINER (dialog->scrolledwindow_shortcuts), BORDER);
    gtk_widget_show (dialog->scrolledwindow_shortcuts);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->scrolledwindow_shortcuts, TRUE, TRUE, 0);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (dialog->scrolledwindow_shortcuts), GTK_SHADOW_IN);

    model = gtk_list_store_new (SHORTCUTS_NUM_COLUMNS, G_TYPE_STRING, G_TYPE_STRING);
    dialog->treeview_shortcuts = gtk_tree_view_new_with_model (GTK_TREE_MODEL (model));
    gtk_widget_show (dialog->treeview_shortcuts);
    gtk_container_add (GTK_CONTAINER (dialog->scrolledwindow_shortcuts), dialog->treeview_shortcuts);
    gtk_widget_set_size_request (dialog->treeview_shortcuts, 350, 200);

    hbox2 = gtk_hbox_new (FALSE, BORDER);
    gtk_container_set_border_width (GTK_CONTAINER (hbox2), BORDER);
    gtk_box_pack_start (GTK_BOX (vbox), hbox2, FALSE, FALSE, 0);
    gtk_widget_show (hbox2);

    dialog->button_add_shortcut = gtk_button_new_from_stock (GTK_STOCK_ADD);
    gtk_widget_show (dialog->button_add_shortcut);
    gtk_box_pack_start (GTK_BOX (hbox2), dialog->button_add_shortcut, TRUE, FALSE, 0);
    gtk_widget_set_sensitive (dialog->button_add_shortcut, FALSE);
    dialog->button_del_shortcut = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
    gtk_widget_show (dialog->button_del_shortcut);
    gtk_box_pack_start (GTK_BOX (hbox2), dialog->button_del_shortcut, TRUE, FALSE, 0);
    gtk_widget_set_sensitive (dialog->button_del_shortcut, FALSE);

    /* popup menus */
    dialog->menu_popup_themes = gtk_menu_new ();
    dialog->menuitem_popup_rename_theme = gtk_image_menu_item_new_from_stock (GTK_STOCK_EDIT, NULL);
    gtk_container_add (GTK_CONTAINER (dialog->menu_popup_themes), dialog->menuitem_popup_rename_theme);
    dialog->menuitem_popup_add_theme = gtk_image_menu_item_new_from_stock (GTK_STOCK_ADD, NULL);
    gtk_container_add (GTK_CONTAINER (dialog->menu_popup_themes), dialog->menuitem_popup_add_theme);
    dialog->menuitem_popup_del_theme = gtk_image_menu_item_new_from_stock (GTK_STOCK_REMOVE, NULL);
    gtk_container_add (GTK_CONTAINER (dialog->menu_popup_themes), dialog->menuitem_popup_del_theme);
    gtk_widget_show_all (dialog->menu_popup_themes);

    dialog->menu_popup_shortcuts = gtk_menu_new ();
    dialog->menuitem_popup_edit_shortcut = gtk_image_menu_item_new_from_stock (GTK_STOCK_EDIT, NULL);
    gtk_container_add (GTK_CONTAINER (dialog->menu_popup_shortcuts), dialog->menuitem_popup_edit_shortcut);
    dialog->menuitem_popup_add_shortcut = gtk_image_menu_item_new_from_stock (GTK_STOCK_ADD, NULL);
    gtk_container_add (GTK_CONTAINER (dialog->menu_popup_shortcuts), dialog->menuitem_popup_add_shortcut);
    dialog->menuitem_popup_del_shortcut = gtk_image_menu_item_new_from_stock (GTK_STOCK_REMOVE, NULL);
    gtk_container_add (GTK_CONTAINER (dialog->menu_popup_shortcuts), dialog->menuitem_popup_del_shortcut);
    gtk_widget_show_all (dialog->menu_popup_shortcuts);

    return hbox;
}

void shortcuts_plugin_setup_dialog (KeyboardMcsDialog * dialog)
{
    GtkTreeModel *model;
    GtkTreeSelection *selection;

    dialog->theme_modified = FALSE;

    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (dialog->treeview_themes),
                                                 -1, 
                                                 NULL, 
                                                 gtk_cell_renderer_text_new (), 
                                                 "text", 
                                                 COLUMN_THEME_NAME, 
                                                 NULL);

    model = (GtkTreeModel *) gtk_list_store_new (THEMES_NUM_COLUMNS, G_TYPE_STRING, G_TYPE_STRING);
    gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (model), 0, themes_sort_func, NULL, NULL);
    gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (model), 0, GTK_SORT_ASCENDING);
    gtk_tree_view_set_model (GTK_TREE_VIEW (dialog->treeview_themes), model);

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview_themes));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_BROWSE);
    g_signal_connect (G_OBJECT (selection), "changed", (GCallback) cb_treeview_themes_selection_changed, dialog);

    g_signal_connect (G_OBJECT (dialog->treeview_themes), "button-press-event", G_CALLBACK (cb_popup_menu_themes), dialog);
    g_signal_connect (G_OBJECT (dialog->button_add_theme), "clicked", G_CALLBACK (cb_button_add_theme_clicked), dialog);
    g_signal_connect (G_OBJECT (dialog->button_del_theme), "clicked", G_CALLBACK (cb_button_del_theme_clicked), dialog);

    /* shortcuts treeview */
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (dialog->treeview_shortcuts), 
                                                 -1, 
                                                 _("Command"),
                                                 gtk_cell_renderer_text_new (), 
                                                 "text", 
                                                 COLUMN_COMMAND, 
                                                 NULL);
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (dialog->treeview_shortcuts), 
                                                 -1, 
                                                 _("Shortcut"),
                                                 gtk_cell_renderer_text_new (), 
                                                 "text", 
                                                 COLUMN_SHORTCUT, 
                                                 NULL);

    model = (GtkTreeModel *) gtk_list_store_new (SHORTCUTS_NUM_COLUMNS, G_TYPE_STRING, G_TYPE_STRING);
    gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (model), 0, shortcuts_sort_func, NULL, NULL);
    gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (model), 0, GTK_SORT_ASCENDING);
    gtk_tree_view_set_model (GTK_TREE_VIEW (dialog->treeview_shortcuts), model);
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (dialog->treeview_shortcuts), TRUE);

    g_signal_connect (G_OBJECT (dialog->treeview_shortcuts), "button-press-event", G_CALLBACK (cb_popup_menu_shortcuts), dialog);
    g_signal_connect (G_OBJECT (dialog->button_add_shortcut), "clicked", G_CALLBACK (cb_button_add_shortcut_clicked), dialog);
    g_signal_connect (G_OBJECT (dialog->button_del_shortcut), "clicked", G_CALLBACK (cb_button_del_shortcut_clicked), dialog);

    /* popup menus */
    g_signal_connect (G_OBJECT (dialog->menuitem_popup_add_theme), "activate", G_CALLBACK (cb_menuitem_add_theme_activate), dialog);
    g_signal_connect (G_OBJECT (dialog->menuitem_popup_del_theme), "activate", G_CALLBACK (cb_menuitem_del_theme_activate), dialog);
    g_signal_connect (G_OBJECT (dialog->menuitem_popup_rename_theme), "activate", G_CALLBACK (cb_menuitem_rename_theme_activate), dialog);
    g_signal_connect (G_OBJECT (dialog->treeview_themes), "row-activated", G_CALLBACK (cb_treeview_themes_activate), dialog);

    g_signal_connect (G_OBJECT (dialog->menuitem_popup_add_shortcut), "activate", G_CALLBACK (cb_menuitem_add_shortcut_activate), dialog);
    g_signal_connect (G_OBJECT (dialog->menuitem_popup_del_shortcut), "activate", G_CALLBACK (cb_menuitem_del_shortcut_activate), dialog);
    g_signal_connect (G_OBJECT (dialog->menuitem_popup_edit_shortcut), "activate", G_CALLBACK (cb_menuitem_edit_shortcut_activate), dialog);

    g_signal_connect (G_OBJECT (dialog->treeview_shortcuts), "row-activated", G_CALLBACK (cb_treeview_shortcuts_activate), dialog);

    update_themes_list (dialog);

    xfce_gtk_window_center_on_monitor_with_pointer (GTK_WINDOW (dialog->dialog_keyboard));
    gdk_x11_window_set_user_time(GTK_WIDGET (dialog->dialog_keyboard)->window, 
            gdk_x11_get_server_time (GTK_WIDGET (dialog->dialog_keyboard)->window));
            
    gtk_widget_show (dialog->dialog_keyboard);
}

gboolean shortcuts_plugin_save_settings (McsPlugin * plugin)
{
    gboolean result;
    gchar *file, *path;

    path = g_build_filename ("xfce4", RCDIR, RCFILE3, NULL);
    file = xfce_resource_save_location (XFCE_RESOURCE_CONFIG, path, TRUE);

    result = mcs_manager_save_channel_to_file (plugin->manager, CHANNEL3, file);
    g_free (path);
    g_free (file);

    return (result);
}

void shortcuts_plugin_init(McsPlugin * plugin)
{
    gchar *file, *path;
    McsSetting *setting;

    /* read settings channel from file */
    path = g_build_filename ("xfce4", RCDIR, RCFILE3, NULL);
    file = xfce_resource_lookup (XFCE_RESOURCE_CONFIG, path);
    g_free (path);

    if (!file)
    {
        file = xfce_get_userfile (OLD_RCDIR, RCFILE3, NULL);
    }

    if (g_file_test (file, G_FILE_TEST_EXISTS))
    {
        mcs_manager_add_channel_from_file (plugin->manager, CHANNEL3, file);
    }
    else
    {
        mcs_manager_add_channel (plugin->manager, CHANNEL3);
    }
    g_free (file);

    mcs_manager_notify (plugin->manager, CHANNEL3);

    setting = mcs_manager_setting_lookup(plugin->manager, "Xfce4/ShortcutThemeName", CHANNEL3);
    if (setting) 
    {
        theme_name = g_strdup(setting->data.v_string);
    }
    else
    {
        theme_name = g_strdup(DEFAULT_SHORTCUT_THEME);
    }

    setting = mcs_manager_setting_lookup(plugin->manager, "Xfce4/ShortcutThemeFile", CHANNEL3);
    if (setting) 
    {
        theme_path = g_strdup(setting->data.v_string);
    }
    else
    {
        theme_path = g_strdup(DEFAULT_SHORTCUTS_PATH);
    }

    init_modifiers ();
    parse_theme (theme_path, NULL);
    add_event_listener ();
}

void shortcuts_plugin_load_theme (KeyboardMcsDialog *dialog)
{
    gchar *path = NULL;

    /* check XDG_CONFIG_HOME/xfce4/shortcuts/ exists; if not it creates it */
    path = xfce_resource_save_location (XFCE_RESOURCE_CONFIG, "xfce4/shortcuts/", TRUE);
    if (G_UNLIKELY (path == NULL))
    {
        g_warning ("failed to create the shorcuts directory");
        return;
    }
    g_free (path);

    parse_theme (theme_path, dialog);
}
