/* vim: set expandtab ts=8 sw=4: */

/*  $Id$
 *
 *  Copyright © 2005 Nick Schermer <nickschermer@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gtk/gtk.h>


#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4panel/xfce-panel-plugin.h>

#include "clipman.h"
#include "clipman-dialogs.h"

/**
 * Dialog response
 **/
static void
clipman_configure_response (GtkWidget *dialog, int response, ClipmanOptions *options)
{
	if(response == GTK_RESPONSE_HELP)
	{
		DBG("This will open a the clipman wiki... in the future...");
	}
	else
	{
		DBG("Save the dialog settings");
		
		options->clipman->ExitSave		= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (options->GW_ExitSave));
		options->clipman->IgnoreSelect		= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (options->GW_IgnoreSelection));
		options->clipman->PreventEmpty		= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (options->GW_PreventEmpty));
		
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (options->GW_Behaviour)))
			options->clipman->Behaviour = RESTORING;
		else
			options->clipman->Behaviour = STRICTLY;
	
		options->clipman->ItemNumbers		= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (options->GW_ItemNumbers));
		options->clipman->SeparateMenu		= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (options->GW_SeparateMenu));
		options->clipman->ColoredItems		= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (options->GW_ColoredItems));
		
		if (options->clipman->HistoryItems != gtk_range_get_value (GTK_RANGE (options->GW_HistorySize)))
			clipman_check_array_len (options->clipman);
		
		if (options->clipman->MenuCharacters != gtk_range_get_value (GTK_RANGE (options->GW_ItemChars)))
			clipman_regenerate_titles (options->clipman);
		
		options->clipman->HistoryItems		= gtk_range_get_value (GTK_RANGE (options->GW_HistorySize));
		options->clipman->MenuCharacters	= gtk_range_get_value (GTK_RANGE (options->GW_ItemChars));
		
		clipman_save (options->clipman->plugin, options->clipman);
		
		xfce_panel_plugin_unblock_menu (options->clipman->plugin);
		
		gtk_widget_destroy (dialog);
		g_free (options);
	}
	
}

/**
 * Sync Spin button when slider is dragged
 **/
static void
set_scale_to_spin (GtkWidget *scalewidget, GtkWidget *spinwidget)
{
	gint value;
	value = gtk_range_get_value (GTK_RANGE (scalewidget));
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(spinwidget), value);
}

/**
 * Sync slider button when spin button is changed
 **/
static void
set_spin_to_scale(GtkWidget *spinwidget, GtkWidget *scalewidget)
{
	gint value;
	value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(spinwidget)); 
	gtk_range_set_value (GTK_RANGE (scalewidget), value);
}

/**
 * Disables some options if selection is ignored
 **/
static void
toggle_ignore_selection (GtkWidget *button, ClipmanOptions *options)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (options->GW_IgnoreSelection)))
	{
		gtk_widget_set_sensitive (options->GW_SeparateMenu, FALSE);
	}
	else
	{
		gtk_widget_set_sensitive (options->GW_SeparateMenu, TRUE);
	}
}

/**
 * Configure dialog
 **/
void
clipman_configure (XfcePanelPlugin *plugin, ClipmanPlugin *clipman)
{
	DBG("Show the properties dialog");
	
	if (clipman->block)
		return;
	
	GtkWidget *dialog, *dialog_vbox, *header, *frame, *button, *label;
	GtkWidget *vbox, *hbox;
	
	ClipmanOptions *options;
	options = g_new0 (ClipmanOptions, 1);
	options->clipman = clipman;
	
	xfce_panel_plugin_block_menu (clipman->plugin);
	
	dialog = gtk_dialog_new_with_buttons (_("Properties"), 
			GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (plugin))),
			GTK_DIALOG_DESTROY_WITH_PARENT |
			GTK_DIALOG_NO_SEPARATOR,
			/* GTK_STOCK_HELP, GTK_RESPONSE_HELP, */
			GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
			NULL);
		gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
		gtk_window_set_icon_name (GTK_WINDOW (dialog), "gtk-properties");
		gtk_window_set_keep_above(GTK_WINDOW (dialog), TRUE);
		gtk_window_stick (GTK_WINDOW (dialog));
	
	dialog_vbox = GTK_DIALOG (dialog)->vbox;
	
	frame = xfce_create_header (NULL, _("Clipman"));
		gtk_widget_set_size_request (GTK_BIN (frame)->child, -1, 32);
		gtk_container_set_border_width (GTK_CONTAINER (frame), BORDER-3);
		gtk_box_pack_start (GTK_BOX (dialog_vbox), frame, FALSE, TRUE, 0);
	
	/**
	 * The general frame
	 **/
	frame = gtk_frame_new (NULL);
		gtk_box_pack_start (GTK_BOX (dialog_vbox), frame, FALSE, TRUE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (frame), BORDER-3);
		
	vbox = gtk_vbox_new (FALSE, 2);
		gtk_container_add (GTK_CONTAINER (frame), vbox);
		gtk_container_set_border_width (GTK_CONTAINER (vbox), BORDER);
	
	button = options->GW_ExitSave = gtk_check_button_new_with_mnemonic (_("Save clipboard contents on _exit"));
		gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), clipman->ExitSave);
	
	button = options->GW_IgnoreSelection = gtk_check_button_new_with_mnemonic (_("_Ignore selections"));
		gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), clipman->IgnoreSelect);
		
		g_signal_connect (G_OBJECT (button), "toggled",
				G_CALLBACK (toggle_ignore_selection), options);
	
	button = options->GW_PreventEmpty = gtk_check_button_new_with_mnemonic (_("Pre_vent empty clipboard"));
		gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), clipman->PreventEmpty);
			
	label = gtk_label_new (_("<b>General</b>"));
		gtk_frame_set_label_widget (GTK_FRAME (frame), label);
		gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
		gtk_misc_set_padding (GTK_MISC (label), 2, 0);
	
	/**
	 * separate clipboards frame
	 **/
	frame = gtk_frame_new (NULL);
		gtk_box_pack_start (GTK_BOX (dialog_vbox), frame, FALSE, TRUE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (frame), BORDER-3);
		
	vbox = gtk_vbox_new (FALSE, 2);
		gtk_container_add (GTK_CONTAINER (frame), vbox);
		gtk_container_set_border_width (GTK_CONTAINER (vbox), BORDER);
	
	GSList *group = NULL;
		
	button = options->GW_Behaviour = gtk_radio_button_new_with_mnemonic (group, _("Normal clipboard _management"));
		gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);

		gtk_radio_button_set_group (GTK_RADIO_BUTTON (button), group);
		group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (button));
		
		if(clipman->Behaviour == RESTORING)
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);
	
	button = gtk_radio_button_new_with_mnemonic (group, _("Strictly separate _both clipboards"));
		gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
		
		gtk_radio_button_set_group (GTK_RADIO_BUTTON (button), group);
		group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (button));
		
		if(clipman->Behaviour == STRICTLY)
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);
		
	label = gtk_label_new (_("<b>Clipboard behaviour</b>"));
		gtk_frame_set_label_widget (GTK_FRAME (frame), label);
		gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
		gtk_misc_set_padding (GTK_MISC (label), 2, 0);
	
	/**
	 * Menu appearance frame
	 **/
	frame = gtk_frame_new (NULL);
		gtk_box_pack_start (GTK_BOX (dialog_vbox), frame, FALSE, TRUE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (frame), BORDER-3);
	
	vbox = gtk_vbox_new (FALSE, 2);
		gtk_container_add (GTK_CONTAINER (frame), vbox);
		gtk_container_set_border_width (GTK_CONTAINER (vbox), BORDER);
	
	button = options->GW_ItemNumbers = gtk_check_button_new_with_mnemonic (_("_Show item numbers"));
		gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), clipman->ItemNumbers);
	
	button = options->GW_SeparateMenu = gtk_check_button_new_with_mnemonic (_("Se_parate clipboards"));
		gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), clipman->SeparateMenu);
	
	button = options->GW_ColoredItems = gtk_check_button_new_with_mnemonic (_("Use _colored menu items"));
		gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), clipman->ColoredItems);
	
	label = gtk_label_new (_("<b>Menu appearance</b>"));
		gtk_frame_set_label_widget (GTK_FRAME (frame), label);
		gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
		gtk_misc_set_padding (GTK_MISC (label), 2, 0);
	
	/**
	 * Call some functions
	 **/
	
	toggle_ignore_selection (options->GW_IgnoreSelection, options);
	
	/**
	 * Numbers frame
	 **/
	frame = gtk_frame_new (NULL);
		gtk_box_pack_start (GTK_BOX (dialog_vbox), frame, FALSE, TRUE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (frame), BORDER-3);
	
	vbox = gtk_vbox_new (FALSE, 2);
		gtk_container_add (GTK_CONTAINER (frame), vbox);
		gtk_container_set_border_width (GTK_CONTAINER (vbox), BORDER);
	 
	label = gtk_label_new (_("Clipboard history items:"));
		gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	
	hbox = gtk_hbox_new (FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
	
	button = options->GW_HistorySize = gtk_hscale_new (GTK_ADJUSTMENT (gtk_adjustment_new (clipman->HistoryItems, MINHISTORY, MAXHISTORY, 1, 5, 0)));
		gtk_widget_show (button);
		gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 10);
		gtk_scale_set_draw_value (GTK_SCALE (button), FALSE);
		gtk_scale_set_digits (GTK_SCALE (button), 0);
	
	button = gtk_spin_button_new_with_range(MINHISTORY, MAXHISTORY, 1);
		gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(button), clipman->HistoryItems);
		
		/* Sync history widgets */
		g_signal_connect (G_OBJECT (options->GW_HistorySize), "value_changed",
				G_CALLBACK (set_scale_to_spin), button);
		
		g_signal_connect (G_OBJECT (button), "value_changed",
				G_CALLBACK (set_spin_to_scale), options->GW_HistorySize);
	
	label = gtk_label_new (_("Menu item characters:"));
		gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	
	hbox = gtk_hbox_new (FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
	
	button = options->GW_ItemChars = gtk_hscale_new (GTK_ADJUSTMENT (gtk_adjustment_new (clipman->MenuCharacters, MINCHARS, MAXCHARS, 1, 5, 0)));
		gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 10);
		gtk_scale_set_draw_value (GTK_SCALE (button), FALSE);
		gtk_scale_set_digits (GTK_SCALE (button), 0);
	
	button = gtk_spin_button_new_with_range(MINCHARS, MAXCHARS, 1);
		gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(button), clipman->MenuCharacters);
		
		g_signal_connect (G_OBJECT (options->GW_ItemChars), "value_changed",
				G_CALLBACK (set_scale_to_spin), button);
	
		g_signal_connect (G_OBJECT (button), "value_changed",
				G_CALLBACK (set_spin_to_scale), options->GW_ItemChars);
	
	label = gtk_label_new (_("<b>Numbers</b>"));
		gtk_frame_set_label_widget (GTK_FRAME (frame), label);
		gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
		gtk_misc_set_padding (GTK_MISC (label), 4, 0);
	
	g_signal_connect(dialog, "response",
			G_CALLBACK(clipman_configure_response), options);
	
	gtk_widget_show_all (dialog);
}

/**
 * This shows the about dialog
 **/
void
clipman_about (XfcePanelPlugin *plugin)
{
	DBG("Show the about dialog");
	
	XfceAboutInfo	*about;
	GtkWidget	*dialog;
	GdkPixbuf	*image;
	
	about = xfce_about_info_new (_("Clipman"), "",
		_("Clipboard manager for the Xfce desktop"), 
		XFCE_COPYRIGHT_TEXT ("2005", "Nick Schermer"),
		XFCE_LICENSE_GPL);
	xfce_about_info_set_homepage (about, "http://www.xfce.org");
	xfce_about_info_add_credit (about, "Nick Schermer", "nickschermer@gmail.com", _("Developer"));
	xfce_about_info_add_credit (about, "Eduard Roccatello", "eduard@xfce.org", _("Developer"));
	
	image = xfce_themed_icon_load ("gtk-paste", 32);
	dialog = xfce_about_dialog_new (NULL, about, image);
	g_object_unref (image);
	
	gtk_widget_set_size_request (dialog, 400, 300);
	xfce_gtk_window_center_on_monitor_with_pointer (GTK_WINDOW (dialog));
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	xfce_about_info_free (about);
	
}

/**
 * The response function for the edit dialog
 **/
static void
clipman_edit_response (GtkWidget *dialog, int response, ClipmanAction *action)
{
	DBG("...");
	
	if (response == GTK_RESPONSE_OK)
	{
		DBG("Question: Save new content");
		
		gchar *newtext;
		GtkTextIter start, end;

		GtkTextBuffer *buffer =
			gtk_text_view_get_buffer (GTK_TEXT_VIEW(action->text));

		gtk_text_buffer_get_start_iter(buffer, &start);
		gtk_text_buffer_get_end_iter(buffer, &end);
		
		newtext = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

		if ( newtext != NULL &&
		     strcmp(newtext, "")
		   )
		{
			clipman_replace_text (action->clip->text, newtext);
			
			g_free (action->clip->text);
			g_free (action->clip->title);
			
			action->clip->text	= g_strdup (newtext);
			action->clip->title	= clipman_create_title (newtext, action->clipman->MenuCharacters);
		}
		else
		{
			DBG ("new text was empty, old value not changed");
		}
		
		gtk_text_buffer_delete(buffer, &start, &end);
		g_free (newtext);
		
		/* Unlock */
		action->clipman->block = FALSE;
		
		gtk_widget_destroy (action->text);
		gtk_widget_destroy (dialog);
	}
	else if (response == GTK_RESPONSE_REJECT)
	{
		gtk_text_buffer_set_text (	gtk_text_view_get_buffer (GTK_TEXT_VIEW (action->text)),
						action->clip->text,
						-1
					 );
	}
	else
	{
		DBG("Question: Close");
		
		/* Unlock */
		action->clipman->block = FALSE;
		
		gtk_widget_destroy (action->text);
		gtk_widget_destroy (dialog);
	}
}

/**
 * The edit dialog
 **/
static void
clipman_edit (ClipmanAction *action)
{
	DBG("...");
	
	GtkWidget *dialog;
	GtkWidget *scroll;
	GtkWidget *dialog_vbox;
	
	/* Block some function */
	action->clipman->block = TRUE;
	
	dialog = gtk_dialog_new_with_buttons (_("Edit content"), 
			GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (action->clipman->plugin))),
			GTK_DIALOG_DESTROY_WITH_PARENT |
			GTK_DIALOG_NO_SEPARATOR,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			GTK_STOCK_UNDO, GTK_RESPONSE_REJECT,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CLOSE,
			NULL);
		gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
		gtk_window_set_icon_name (GTK_WINDOW (dialog), "gtk-edit");
		gtk_window_set_keep_above(GTK_WINDOW (dialog), TRUE);
		gtk_widget_set_size_request (dialog, 400, 250);
		gtk_window_stick (GTK_WINDOW (dialog));
	
	dialog_vbox = GTK_DIALOG (dialog)->vbox;
	
	scroll = gtk_scrolled_window_new (NULL, NULL);
		gtk_box_pack_start (GTK_BOX (dialog_vbox), scroll, TRUE, TRUE, 0);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll), GTK_SHADOW_IN);
		
	action->text = gtk_text_view_new ();
		gtk_text_view_set_pixels_above_lines (GTK_TEXT_VIEW (action->text), 2);
		gtk_text_view_set_pixels_below_lines (GTK_TEXT_VIEW (action->text), 2);
		gtk_text_view_set_left_margin (GTK_TEXT_VIEW (action->text), 2);
		gtk_text_view_set_right_margin (GTK_TEXT_VIEW (action->text), 2);
		gtk_container_add (GTK_CONTAINER (scroll), action->text);
		
		gtk_text_buffer_set_text (	gtk_text_view_get_buffer (GTK_TEXT_VIEW (action->text)),
						action->clip->text,
						-1
					 );
	
	g_signal_connect(dialog, "response",
			G_CALLBACK(clipman_edit_response), action);
	
	gtk_widget_show_all (dialog);
}

/**
 * Response to the question dialog
 * There the edit dialog will be called or a clip deleted
 **/
static void
clipman_question_response (GtkWidget *dialog, int response, ClipmanAction *action)
{
	DBG("...");
	
	if (response == GTK_RESPONSE_OK)
	{
		DBG("Question: Edit Content");
		
		clipman_edit (action);
	}
	else if (response == GTK_RESPONSE_REJECT)
	{
		DBG("Question: Delete");
		
		/* Check if this clip exists in the clipboard and if so remove it */
		clipman_replace_text (action->clip->text, "");
		
		/* Remove the clip from the array */
		g_ptr_array_remove (action->clipman->clips, action->clip);
		clipman_free_clip (action->clip);
		
		action->clipman->block = FALSE;
	}
	else
	{
		DBG("Question: Close");
		
		action->clipman->block = FALSE;
	}
	
	gtk_widget_destroy (dialog);
}

/**
 * The 3rd button question dialog
 * I know the GTK_RESPONSE_* values are a bit strange, but who cares
 **/
void
clipman_question (ClipmanAction *action)
{
	DBG("...");
	
	/* Check if blocked */
	if (action->clipman->block)
		return;
	
	/* Block the plugin because we're deleting something (maybe) */
	action->clipman->block = TRUE;
	
	GtkWidget *dialog, *image, *label, *hbox, *dialog_vbox;
	
	dialog = gtk_dialog_new_with_buttons (_("Question"), 
			GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (action->clipman->plugin))),
			GTK_DIALOG_DESTROY_WITH_PARENT |
			GTK_DIALOG_NO_SEPARATOR,
			GTK_STOCK_EDIT, GTK_RESPONSE_OK,
			GTK_STOCK_DELETE, GTK_RESPONSE_REJECT,
			GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
			NULL);
		gtk_widget_set_size_request (dialog, 400, -1);
		gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
		gtk_window_set_icon_name (GTK_WINDOW (dialog), "gtk-dialog-question");
		gtk_window_set_keep_above(GTK_WINDOW (dialog), TRUE);
		gtk_window_stick (GTK_WINDOW (dialog));
	
	dialog_vbox = GTK_DIALOG (dialog)->vbox;
	
	hbox = gtk_hbox_new (FALSE, BORDER);
		gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox, TRUE, TRUE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (hbox), BORDER);
	
	image = gtk_image_new_from_icon_name ("gtk-dialog-question", GTK_ICON_SIZE_DIALOG);
		gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, TRUE, 0);
	
	label = gtk_label_new (_("What to do with this clipboard item?"));
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	
	gtk_widget_show_all (dialog);
	
	g_signal_connect(dialog, "response",
			G_CALLBACK(clipman_question_response), action);
}
