/*
 * log.c - log utils
 *
 * Copyright 2009 Canonical Ltd.
 *
 * Authors:
 *    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
 *    David Barth <david.barth@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"
#include <sys/time.h>
#include <time.h>
#include <stdio.h>

#include <glib-object.h>

static FILE *logfile = NULL;

static void
log_logger_null (const char     *domain,
                 GLogLevelFlags  log_level,
                 const char     *message,
                 gpointer        user_data)
{
  return;
}

char*
log_create_timestamp (void)
{
  struct timeval tv;
  struct tm     *tm;

  /* FIXME: deal with tz offsets */
  gettimeofday (&tv, NULL);
  tm = localtime (&tv.tv_sec);

  return g_strdup_printf ("%.2d:%.2d:%.2d:%.6ld",
                          tm->tm_hour, tm->tm_min, tm->tm_sec,
                          tv.tv_usec - tv.tv_sec * G_USEC_PER_SEC);
}

void
log_logger_file (const gchar    *log_domain,
                 GLogLevelFlags  log_level,
                 const gchar    *message,
                 gpointer        data)
{
  if (logfile == NULL)
    return;

  char *ts = log_create_timestamp ();
  char *level_str = NULL;

  if (log_level & G_LOG_LEVEL_DEBUG)
    level_str = "DBG";

  if (log_level & G_LOG_LEVEL_INFO)
    level_str = "INF";

  if (log_level & G_LOG_LEVEL_MESSAGE)
    level_str = "MSG";

  if (log_level & G_LOG_LEVEL_WARNING)
    level_str = "WRN";

  if (log_level & G_LOG_LEVEL_CRITICAL)
    level_str = "CRI";

  if (log_level & G_LOG_LEVEL_ERROR)
    level_str = "ERR";

  fprintf (logfile, "[%s - %s] %s\n",
           ts, level_str, message);

  fflush (logfile);

  g_free (ts);
}

void
log_init (const char *filename)
{
  /* the file is truncated at every run */
  logfile = fopen (filename, "w");

  if (logfile == NULL)
    g_warning ("could not open/append to %s; logging disabled", filename);

  g_log_set_handler (G_LOG_DOMAIN,
                     G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
                     log_logger_file,
                     NULL);

  /* discard all debug messages unless DEBUG is set */
  if (! g_getenv ("DEBUG"))
    {
      g_log_set_handler (NULL,
                         G_LOG_LEVEL_DEBUG,
                         log_logger_null, NULL);
    }
  else
    {
      g_message ("DEBUG mode enabled");
    }
}

