// $Id: Facade.java,v 1.18 2005/09/09 13:18:52 mkl Exp $
// Copyright (c) 2005 The Regents of the University of California. All
// Rights Reserved. Permission to use, copy, modify, and distribute this
// software and its documentation without fee, and without a written
// agreement is hereby granted, provided that the above copyright notice
// and this paragraph appear in all copies. This software program and
// documentation are copyrighted by The Regents of the University of
// California. The software program and documentation are supplied "AS
// IS", without any accompanying services from The Regents. The Regents
// does not warrant that the operation of the program will be
// uninterrupted or error-free. The end-user understands that the program
// was developed for research purposes and is advised not to rely
// exclusively on the program for any reason. IN NO EVENT SHALL THE
// UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
// SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
// ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
// THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY
// WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
// PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
// CALIFORNIA HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
// UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

package org.argouml.model;

import java.util.Collection;
import java.util.Iterator;

/**
 * Facade object for the Model subsystem in ArgoUML.<p>
 *
 * The purpose of this Facade object is to allow for decoupling other
 * subsystems from the insides of the model. For this purpose all of
 * the methods in this class give away and accept handles (of type
 * {@link java.lang.Object}) to the objects within the model.<p>
 *
 * This is just recognizers and getters.<p>
 *
 * To manipulate the objects of the model there is a set of factories
 * and helpers within the Model subsystem.<p>
 *
 * Signatures for all recognizers are:
 * <ul>
 * <li>public boolean isATYPE(Object handle)
 * <li>public boolean isPROPERTY(Object handle)
 * <li>public boolean hasPROPERTY(Object handle)
 * </ul>
 *
 * Signatures for all getters are:
 * <ul>
 * <li>public Object getROLENAME(Object handle) - 1..1
 * <li>public Iterator/Collection getROLENAMEs(Object handle) - 0..*
 * <li>public String getName(Object handle) - Name
 * </ul>
 *
 * @author Linus Tolke
 */
public interface Facade {
    /**
     * This tag is set on elements that are generated by reference when
     * importing.
     * If it is set, then the critics could ignore those objects if they want.
     */
    String GENERATED_TAG = "GeneratedFromImport";

    ////////////////////////////////////////////////////////////////
    /**
     * Recognizer for Abstraction.
     *
     * @param handle candidate
     * @return true if handle is an Abstraction.
     */
    boolean isAAbstraction(Object handle);

    /**
     * Recognizer for Action.
     *
     * @param handle candidate
     * @return true if handle is an Action
     */
    boolean isAAction(Object handle);

    /**
     * Recognizer for ActionExpression.
     *
     * @param handle candidate
     * @return true if handle is an ActionExpression
     */
    boolean isAActionExpression(Object handle);

    /**
     * Recognizer for ActionSequence.
     *
     * @param handle candidate
     * @return true if handle is an action sequence
     */
    boolean isAActionSequence(Object handle);

    /**
     * Recognizer for Action state.
     *
     * @param handle candidate
     * @return true if handle is an Action state
     */
    boolean isAActionState(Object handle);

    /**
     * Recognizer for CallState.
     *
     * @param handle candidate
     * @return true if handle is an call state
     */
    boolean isACallState(Object handle);

    /**
     * Recognizer for ObjectFlowState.
     *
     * @param handle candidate
     * @return true if handle is an objectflow state
     */
    boolean isAObjectFlowState(Object handle);

    /**
     * Recognizer for SubactivityState.
     *
     * @param handle candidate
     * @return true if handle is an subactivity state
     */
    boolean isASubactivityState(Object handle);

    /**
     * Recognizer for Actor.
     *
     * @param handle candidate
     * @return true if handle is an Actor
     */
    boolean isAActor(Object handle);

    /**
     * Recognizer for AggregationKind.
     *
     * @param handle candidate
     * @return true if handle is an AggregationKind
     */
    boolean isAAggregationKind(Object handle);

    /**
     * Recognizer for Association.
     *
     * @param handle candidate
     * @return true if handle is an Association
     */
    boolean isAAssociation(Object handle);

    /**
     * Recognizer for AssociationEnd.
     *
     * @param handle candidate
     * @return true if handle is an AssociationEnd
     */
    boolean isAAssociationEnd(Object handle);

    /**
     * Recognizer for AssociationRole.
     *
     * @param handle candidate
     * @return true if handle is an AssociationRole
     */
    boolean isAAssociationRole(Object handle);

    /**
     * Recognizer for AssociationEndRole.
     *
     * @param handle candidate
     * @return true if handle is an AssociationEndRole
     */
    boolean isAAssociationEndRole(Object handle);

    /**
     * Recognizer for Attribute.
     *
     * @param handle candidate
     * @return true if handle is an Attribute
     */
    boolean isAAttribute(Object handle);

    /**
     * Recognizer for asynchronisity of an action.
     *
     * @param handle candidate
     * @return true if the argument is asynchronous
     */
    boolean isAsynchronous(Object handle);

    /**
     * Recognizer for abstract classes and operations.
     *
     * @param handle candidate
     * @return true if handle is abstract.
     */
    boolean isAbstract(Object handle);

    /**
     * Recognizer for ActivityGraph.
     *
     * @param handle candidate
     * @return true if handle is ActivityGraph.
     */
    boolean isAActivityGraph(Object handle);

    /**
     * Recognizer for bases. A base is an object that is some form of
     * an element in the model. MBase in Novosoft terms.
     *
     * TODO: Does this have a real UML meaning? Isn't it
     * the ModelElement that is the base? - Bob.
     *
     * @param handle candidate
     * @return true if handle is a base.
     */
    boolean isABase(Object handle);

    /**
     * Recognizer for behavioral features.
     *
     * @param handle candidate
     * @return true if handle is a behavioral feature
     */
    boolean isABehavioralFeature(Object handle);

    /**
     * Recognizer for CallAction.
     *
     * @param handle candidate
     * @return true if handle is a CallAction
     */
    boolean isACallAction(Object handle);

    /**
     * Recognizer for CallEvent.
     *
     * @param handle candidate
     * @return true if handle is a CallEvent
     */
    boolean isACallEvent(Object handle);

    /**
     * Recognizer for ChangeEvent.
     *
     * @param handle candidate
     * @return true if handle is a ChangeEvent
     */
    boolean isAChangeEvent(Object handle);

    /**
     * Recognizer for Class.
     *
     * @param handle candidate
     * @return true if handle is a Class
     */
    boolean isAClass(Object handle);

    /**
     * Recognizer for AssociationClass.
     *
     * @param handle candidate
     * @return true if handle is an AssociationClass
     */
    boolean isAAssociationClass(Object handle);

    /**
     * Recognizer for a Element that is Classifier and RelationShip.
     *
     * TODO: Does this really belong here? Is it commonly used? If it really
     * belongs here it is not a recognizer so it should be renamed to
     * isClassifierAndRelationship. (comment by Bob from MDR impl - tfm)
     *
     * @param handle candidate
     * @return true if handle is a Classifier and a Relationship
     */
    boolean isAClassifierAndARelationship(Object handle);

    /**
     * Recognizer for Classifier.
     *
     * @param handle candidate
     * @return true if handle is a Classifier
     */
    boolean isAClassifier(Object handle);

    /**
     * Recognizer for ClassifierInState.
     *
     * @param handle candidate
     * @return true if handle is a ClassifierInState
     */
    boolean isAClassifierInState(Object handle);

    /**
     * Recognizer for ClassifierRole.
     *
     * @param handle candidate
     * @return true if handle is a ClassifierRole
     */
    boolean isAClassifierRole(Object handle);

    /**
     * Recognizer for Comment.
     *
     * @param handle candidate
     * @return true if handle is a Comment
     */
    boolean isAComment(Object handle);

    /**
     * Recognizer for Collaboration.
     *
     * @param handle candidate
     * @return true if handle is a Collaboration
     */
    boolean isACollaboration(Object handle);

    /**
     * Recognizer for Component.
     *
     * @param handle candidate
     * @return true if handle is a Component
     */
    boolean isAComponent(Object handle);

    /**
     * Recognizer for ComponentInstance.
     *
     * @param handle candidate
     * @return true if handle is a ComponentInstance
     */
    boolean isAComponentInstance(Object handle);

    /**
     * Recognizer for Constraint.
     *
     * @param handle candidate
     * @return true if handle is a Constraint
     */
    boolean isAConstraint(Object handle);

    /**
     * Recognizer for CreateAction.
     *
     * @param handle candidate
     * @return true if handle is a CreateAction
     */
    boolean isACreateAction(Object handle);

    /**
     * Recognizer for DataType.
     *
     * @param handle candidate
     * @return true if handle is a DataType
     */
    boolean isADataType(Object handle);

    /**
     * Recognizer for DataValue.
     *
     * @param handle candidate
     * @return true if handle is a DataValue
     */
    boolean isADataValue(Object handle);

    /**
     * Recognizer for Dependency.
     *
     * @param handle candidate
     * @return true if handle is a Dependency
     */
    boolean isADependency(Object handle);

    /**
     * Recognizer for DestroyAction.
     *
     * @param handle candidate
     * @return true if handle is a DestroyAction
     */
    boolean isADestroyAction(Object handle);

    /**
     * Recognizer for CompositeState.
     *
     * @param handle candidate
     * @return true if handle is a CompositeState
     */
    boolean isACompositeState(Object handle);

    /**
     * Recognizer for Element.
     *
     * @param handle candidate
     * @return true if handle is an Element
     */
    boolean isAElement(Object handle);

    /**
     * Recognizer for ElementImport.
     *
     * @param handle candidate
     * @return true if handle is an ElementImport
     */
    boolean isAElementImport(Object handle);

    /**
     * Recognizer for ElementResidence.
     *
     * @param handle candidate
     * @return true if handle is an ElementResidence
     */
    boolean isAElementResidence(Object handle);

    /**
     * Recognizer for Event.
     *
     * @param handle candidate
     * @return true if handle is an Event
     */
    boolean isAEvent(Object handle);

    /**
     * Recognizer for Exception.
     *
     * @param handle candidate
     * @return true if handle is an Exception
     */
    boolean isAException(Object handle);

    /**
     * Recognizer for Expression.
     *
     * @param handle candidate
     * @return true if handle is an Expression
     */
    boolean isAExpression(Object handle);

    /**
     * Recognizer for Extend.
     *
     * @param handle candidate
     * @return true if handle is an Extend
     */
    boolean isAExtend(Object handle);

    /**
     * Recognizer for ExtensionPoint.
     *
     * @param handle candidate
     * @return true if handle is an ExtensionPoint
     */
    boolean isAExtensionPoint(Object handle);

    /**
     * Recognizer for Feature.
     *
     * @param handle candidate
     * @return true if handle is a Feature
     */
    boolean isAFeature(Object handle);

    /**
     * Recognizer for FinalState.
     *
     * @param handle candidate
     * @return true if handle is a FinalState
     */
    boolean isAFinalState(Object handle);

    /**
     * Recognizer for Flow.
     *
     * @param handle candidate
     * @return true if handle is a Flow
     */
    boolean isAFlow(Object handle);

    /**
     * Recognizer for Guard.
     *
     * @param handle candidate
     * @return true if handle is a Guard
     */
    boolean isAGuard(Object handle);

    /**
     * Recognizer for GeneralizableElement.
     *
     * @param handle candidate
     * @return true if handle is a GeneralizableElement
     */
    boolean isAGeneralizableElement(Object handle);

    /**
     * Recognizer for GeneralizableElement.
     *
     * @param handle candidate
     * @return true if handle is a GeneralizableElement
     */
    boolean isAGeneralization(Object handle);

    /**
     * Recognizer for Include.
     *
     * @param handle candidate
     * @return true if handle is an Include
     */
    boolean isAInclude(Object handle);

    /**
     * Recognizer for Instance.
     *
     * @param handle candidate
     * @return true if handle is a Instance
     */
    boolean isAInstance(Object handle);

    /**
     * Recognizer for Interaction.
     *
     * @param handle candidate
     * @return true if handle is a Interaction
     */
    boolean isAInteraction(Object handle);

    /**
     * Recognizer for Interface.
     *
     * @param handle candidate
     * @return true if handle is a Interface
     */
    boolean isAInterface(Object handle);

    /**
     * Recognizer for Link.
     *
     * @param handle candidate
     * @return true if handle is a Link
     */
    boolean isALink(Object handle);

    /**
     * Recognizer for LinkEnd.
     *
     * @param handle candidate
     * @return true if handle is a LinkEnd
     */
    boolean isALinkEnd(Object handle);

    /**
     * Recognizer for Message.
     *
     * @param handle candidate
     * @return true if handle is a Method
     */
    boolean isAMessage(Object handle);

    /**
     * Recognizer for Method.
     *
     * @param handle candidate
     * @return true if handle is a Method
     */
    boolean isAMethod(Object handle);

    /**
     * Recognizer for Model.
     *
     * @param handle candidate
     * @return true if handle is a Model
     */
    boolean isAModel(Object handle);

    /**
     * Recognizer for ModelElement.
     *
     * @param handle candidate
     * @return true if handle is a ModelElement
     */
    boolean isAModelElement(Object handle);

    /**
     * Recognizer for Multiplicity.
     *
     * @param handle candidate
     * @return true if handle is a Multiplicity
     */
    boolean isAMultiplicity(Object handle);

    /**
     * Recognizer for MultiplicityRange.
     *
     * @param handle candidate
     * @return true if handle is a MultiplicityRange
     */
    boolean isAMultiplicityRange(Object handle);

    /**
     * Recognizer for Namespace.
     *
     * @param handle candidate
     * @return true if handle is a Namespace
     */
    boolean isANamespace(Object handle);

    /**
     * Recognizer for N-ary Association.
     *
     * TODO: This is not a recognizer for some type. Rename to
     * isNaryAssociation?
     *
     * @param handle candidate
     * @return true if handle is an Association
     */
    boolean isANaryAssociation(Object handle);

    /**
     * Recognizer for a Node.
     *
     * @param handle candidate
     * @return true if handle is a Node
     */
    boolean isANode(Object handle);

    /**
     * Recognizer for a NodeInstance.
     *
     * @param handle candidate
     * @return true if handle is a NodeInstance
     */
    boolean isANodeInstance(Object handle);

    /**
     * Recognizer for Operation.
     *
     * @param handle candidate
     * @return true if handle is an Operation
     */
    boolean isAOperation(Object handle);

    /**
     * Recognizer for Object.
     *
     * @param handle candidate
     * @return true if handle is an Object
     */
    boolean isAObject(Object handle);

    /**
     * Recognizer for Parameter.
     *
     * @param handle candidate
     * @return true if handle is a Parameter
     */
    boolean isAParameter(Object handle);

    /**
     * Recognizer for Partition.
     *
     * @param handle candidate
     * @return true if handle is a Partition
     */
    boolean isAPartition(Object handle);

    /**
     * Recognizer for Permission.
     *
     * @param handle candidate
     * @return true if handle is an Permission
     */
    boolean isAPermission(Object handle);

    /**
     * Recognizer for Package.
     *
     * @param handle candidate
     * @return true if handle is a Package
     */
    boolean isAPackage(Object handle);

    /**
     * Recognizer for Pseudostate.
     *
     * @param handle candidate
     * @return true if handle is a Pseudostate
     */
    boolean isAPseudostate(Object handle);

    /**
     * Recognizer for PseudostateKind.
     *
     * @param handle candidate
     * @return true if handle is a PseudostateKind
     */
    boolean isAPseudostateKind(Object handle);

    /**
     * Returns the Kind of a Pseudostate.
     *
     * TODO: - Do we need this as well as getKind - I think not
     *
     * @param handle the Pseudostate
     * @return the Kind
     */
    Object getPseudostateKind(Object handle);

    /**
     * Returns the Kind of a Pseudostate or Parameter.
     *
     * @param handle the Pseudostate or Parameter
     * @return the Kind
     */
    Object getKind(Object handle);

    /**
     * Returns the receiver object of a message or stimulus.
     *
     * @param handle candidate
     * @return receiver
     */
    Object getReceiver(Object handle);

    /**
     * Returns the Link belonging to the given LinkEnd.
     *
     * @param handle the LinkEnd
     * @return the Link
     */
    Object getLink(Object handle);

    /**
     * Check whether two pseudostatekinds are equal/of the same type.
     *
     * @return true if the are the same type
     * @param ps1 one kind
     * @param ps2 one kind
     */
    boolean equalsPseudostateKind(Object ps1, Object ps2);

    /**
     * Recognizer for Reception.
     *
     * @param handle candidate
     * @return true if handle is a Reception
     */
    boolean isAReception(Object handle);

    /**
     * Recognizer for Returnaction.
     *
     * @param handle candidate
     * @return true if handle is a returnaction
     */
    boolean isAReturnAction(Object handle);

    /**
     * Recognizer for Relationship.
     *
     * @param handle candidate
     * @return true if handle is a Relationship
     */
    boolean isARelationship(Object handle);

    /**
     * Recognizer for SendAction.
     *
     * @param handle candidate
     * @return true if handle is a SendAction
     */
    boolean isASendAction(Object handle);

    /**
     * Recognizer for Signal.
     *
     * @param handle candidate
     * @return true if handle is a Signal
     */
    boolean isASignal(Object handle);

    /**
     * Recognizer for SignalEvent.
     *
     * @param handle candidate
     * @return true if handle is a SignalEvent
     */
    boolean isASignalEvent(Object handle);

    /**
     * Recognizer for SimpleState.
     *
     * @param handle candidate
     * @return true if handle is a StateMachine
     */
    boolean isASimpleState(Object handle);

    /**
     * Recognizer for StateMachine.
     *
     * @param handle candidate
     * @return true if handle is a StateMachine
     */
    boolean isAStateMachine(Object handle);

    /**
     * Recognizer for stimulus.
     *
     * @param handle candidate
     * @return true if handle is a stimulus
     */
    boolean isAStimulus(Object handle);

    /**
     * Recognizer for StateVertex.
     *
     * @param handle candidate
     * @return true if handle is a StateVertex
     */
    boolean isAStateVertex(Object handle);

    /**
     * Recognizer for Stereotype.
     *
     * @param handle candidate
     * @return true if handle is a Stereotype
     */
    boolean isAStereotype(Object handle);

    /**
     * Recognizer for StructuralFeature.
     *
     * @param handle candidate
     * @return true if handle is a StructuralFeature
     */
    boolean isAStructuralFeature(Object handle);

    /**
     * Recognizer for State.
     *
     * @param handle candidate
     * @return true if handle is a State
     */
    boolean isAState(Object handle);

    /**
     * Recognizer for StubState.
     *
     * @param handle candidate
     * @return true if handle is a StubState
     */
    boolean isAStubState(Object handle);

    /**
     * Recognizer for SubmachineState.
     *
     * @param handle candidate
     * @return true if handle is a SubmachineState
     */
    boolean isASubmachineState(Object handle);

    /**
     * Recognizer for Subsystem.
     *
     * @param handle candidate
     * @return true if handle is a Subsystem
     */
    boolean isASubsystem(Object handle);

    /**
     * Recognizer for SynchState.
     *
     * @param handle candidate
     * @return true if handle is a SynchState
     */
    boolean isASynchState(Object handle);

    /**
     * Recognizer for TaggedValue.
     *
     * @param handle candidate
     * @return true if handle is a TaggedValue
     */
    boolean isATaggedValue(Object handle);

    /**
     * Recognizer for TerminateAction.
     *
     * @param handle candidate
     * @return true if handle is a TerminateAction
     */
    boolean isATerminateAction(Object handle);

    /**
     * Recognizer for Transition.
     *
     * @param handle candidate
     * @return true if handle is a Transition
     */
    boolean isATransition(Object handle);

    /**
     * Recognizer for TimeEvent.
     *
     * @param handle candidate
     * @return true if handle is a TimeEvent
     */
    boolean isATimeEvent(Object handle);

    /**
     * Recognizer for UninterpretedAction.
     *
     * @param handle candidate
     * @return true if handle is a UninterpretedAction
     */
    boolean isAUninterpretedAction(Object handle);

    /**
     * Recognizer for Usage.
     *
     * @param handle candidate
     * @return true if handle is a Usage
     */
    boolean isAUsage(Object handle);

    /**
     * Recognizer for a Use Case.
     *
     * @param handle candidate
     * @return true if handle is a Transition
     */
    boolean isAUseCase(Object handle);

    /**
     * Recognizer for VisibilityKind.
     *
     * @param handle candidate
     * @return true if handle is a VisibilityKind
     */
    boolean isAVisibilityKind(Object handle);

    /**
     * Recognizer for Classes that are Active.
     *
     * @param handle candidate
     * @return true if Class is Active
     */
    boolean isActive(Object handle);

    /**
     * Recognizer for attributes that are changeable.
     *
     * @param handle candidate
     * @return true if handle is changeable
     */
    boolean isChangeable(Object handle);

    /**
     * Recognizer for attributes with classifier scope.
     *
     * @param handle candidate
     * @return true if handle has classifier scope.
     */
    boolean isClassifierScope(Object handle);

    /**
     * Recognizer for concurrent composite state.
     *
     * @param handle composite state
     * @return true if concurent.
     */
    boolean isConcurrent(Object handle);

    /**
     * Recognizer for ConcurrentRegion.
     *
     * @param handle candidate
     * @return true if handle is a ConcurrentRegion
     */
    boolean isAConcurrentRegion(Object handle);

    /**
     * Recognizer for constructor.
     *
     * A constructor is an operation that has a stereotype named create or
     * a method that is a specification of an operator that is a constructor.
     *
     * @param handle candidate
     * @return true if handle is a constructor.
     */
    boolean isConstructor(Object handle);

    /**
     * Returns true if the given element is Frozen.
     *
     * @param handle candidate
     * @return boolean true if Frozen
     */
    boolean isFrozen(Object handle);

    /**
     * Returns true if a given associationend is a composite.
     * @param handle candidate
     * @return boolean
     */
    boolean isComposite(Object handle);

    /**
     * Returns true if a given associationend is a composite.
     * @param handle candidate
     * @return boolean
     */
    boolean isAggregate(Object handle);

    /**
     * Recognizer for attributes that are initialized.
     *
     * @param handle candidate
     * @return true if the attribute is initialized.
     */
    boolean isInitialized(Object handle);

    /**
     * Recognizer for attributes with instance scope.
     *
     * @param handle candidate
     * @return true if handle has instance scope.
     */
    boolean isInstanceScope(Object handle);

    /**
     * Recognizer for internal transitions.
     *
     * @author mvw
     * @param handle candidate
     * @return true if handle is an internal transition.
     */
    boolean isInternal(Object handle);

    /**
     * Recognizer for leafs.
     *
     * @param handle candidate GeneralizableElement
     * @return true if handle is a leaf
     */
    boolean isLeaf(Object handle);

    /**
     * Recognizer for roots.
     *
     * @param handle candidate GeneralizableElement
     * @return true if handle is a leaf
     */
    boolean isRoot(Object handle);

    /**
     * Recognizer for specifications.
     *
     * @param handle candidate ModelElement
     * @return true if handle is a specification
     */
    boolean isSpecification(Object handle);

    /**
     * Recognizer for Navigable elements.
     *
     * @param handle candidate
     * @return true if handle is navigable
     */
    boolean isNavigable(Object handle);

    /**
     * Recognizer for primary objects.<p>
     *
     * A primary object is an object that is created by the parser or
     * by a user.
     * Object that are created when importing some other object are not.<p>
     *
     * @param handle candidate
     * @return true if primary object.
     */
    boolean isPrimaryObject(Object handle);

    /**
     * Recognizer for attributes with private.
     *
     * @param handle candidate
     * @return true if handle has private
     */
    boolean isPrivate(Object handle);

    /**
     * Recognizer for attributes with public.
     *
     * @param handle candidate
     * @return true if handle has public
     */
    boolean isPublic(Object handle);

    /**
     * Recognizer for MBehaviouralFeature's that are queries.
     *
     * @param handle candidate
     * @return true if it is a query
     */
    boolean isQuery(Object handle);

    /**
     * Recognizer for attributes with protected.
     *
     * @param handle candidate
     * @return true if handle has protected
     */
    boolean isProtected(Object handle);

    /**
     * Recognizer for realize.
     *
     * @param handle candidate
     * @return true if handle has a realize stereotype
     */
    boolean isRealize(Object handle);

    /**
     * Recognizer for return.
     *
     * @param handle candidate parameter
     * @return true if handle is a return parameter.
     */
    boolean isReturn(Object handle);

    /**
     * Recognizer for singleton.
     *
     * @param handle candidate
     * @return true if handle is a singleton.
     */
    boolean isSingleton(Object handle);

    /**
     * Recognizer for model elements with a given stereotype.
     *
     * @param handle candidate model element
     * @param stereotypename a string that is the stereotype name.
     * @return true if handle is an object that has the given stereotype.
     */
    boolean isStereotype(Object handle, String stereotypename);

    /**
     * Returns true if the given CompositeState is the top state.
     *
     * @param handle CompositeState
     * @return boolean true if top state
     */
    boolean isTop(Object handle);

    /**
     * Recognizer for type.
     *
     * @param handle candidate
     * @return true if handle is a type.
     */
    boolean isType(Object handle);

    /**
     * Recognizer for utility.
     *
     * @param handle candidate
     * @return true if handle is a utility.
     */
    boolean isUtility(Object handle);

    ////////////////////////////////////////////////////////////////

    /**
     * Returns the association connected to an association end or
     * the association belonging to the given link.
     *
     * @param handle is the link
     * @return association end
     */
    Object getAssociation(Object handle);

    /**
     * Returns the association end between some classifier and some associaton.
     *
     * @param handle is the classifier
     * @param assoc is the association
     * @return association end
     */
    Object getAssociationEnd(Object handle, Object assoc);

    /**
     * The list of Association Ends.
     *
     * @param handle the object that we get the association ends from.
     * @return Collection with association ends.
     */
    Collection getAssociationEnds(Object handle);

    /**
     * The list of association roles.
     *
     * @param handle the object that we get the association roles from.
     * @return Collection of association roles.
     */
    Collection getAssociationRoles(Object handle);

    /**
     * The list of Attributes.
     *
     * @param handle classifier to examine.
     * @return iterator with attributes.
     */
    Collection getAttributes(Object handle);

    /**
     * The baseclass of some stereotype.
     *
     * @param handle the stereotype
     * @return the baseclass
     */
    Object getBaseClass(Object handle);

    /**
     * The base of some model element.<p>
     *
     * There is a bug in NSUML which gets the addition and base
     * relationships back to front for include relationships. Solve
     * by reversing their accessors in the code.
     *
     * @param handle the model element
     * @return the base
     */
    Object getBase(Object handle);

    /**
     * Get the bases of a classifier role.
     *
     *
     * @param handle classifier role.
     * @return the bases.
     */
    Collection getBases(Object handle);

    /**
     * Get the behaviors of a Modelelement.
     *
     *
     * @param handle modelelement to examine.
     * @return the behaviors.
     */
    Collection getBehaviors(Object handle);

    /**
     * Get the behavioral feature of an parameter.
     *
     * @param handle expression.
     * @return the behavioral feature.
     */
    Object getBehavioralFeature(Object handle);

    /**
     * Get the body of an method/constraint/expression/comment.<p>
     *
     * If the argument is an expression, the body returned is a
     * {@link String} (or <code>null</code>). If you want a type-safe
     * version of this version of the call,
     * use {@link DataTypesHelper#getBody(Object)}.
     *
     * @param handle expression.
     * @return the body.
     */
    Object getBody(Object handle);

    /**
     *  Return the Synch State's bound.
     *  @param handle the synch State
     *  @return bound
     */
    int getBound(Object handle);

    /**
     * Return Changeability of a StructuralFeature or a AssociationEnd.
     *
     * @param handle the StructuralFeature or AssociationEnd
     * @return the Changeability
     */
    Object getChangeability(Object handle);

    /**
     * Get the child of a generalization.
     *
     * @param handle generalization.
     * @return the child.
     */
    Object getChild(Object handle);

    /**
     * Get the children of some generalizable element.
     *
     * @param handle to the generalizable element.
     * @return a collection with all children.
     */
    Collection getChildren(Object handle);

    /**
     * Gets the classifiers roles of some model element.
     *
     * @param handle the model element
     * @return the classifiers roles of the instance
     */
    Collection getClassifierRoles(Object handle);

    /**
     * Get the classifier of an Association End.
     *
     * @param handle The Association End to get from.
     * @return The classifier of the Association End.
     */
    Object getClassifier(Object handle);

    /**
     * Gets the classifierss of some instance.
     *
     * @param handle the instance
     * @return the classifierss of the instance
     */
    Collection getClassifiers(Object handle);

    /**
     * Gets the classifiers in state of some model element.
     *
     * @param handle the model element
     * @return the classifierss in state
     */
    Collection getClassifiersInState(Object handle);

    /**
     * Gets the clients of some dependency.
     *
     * @param handle the dependency
     * @return the clients of the dependency
     */
    Collection getClients(Object handle);

    /**
     * Get the client dependencies of some classifier.
     *
     * @param handle to the classifier.
     * @return an iterator with all client dependencies.
     */
    Collection getClientDependencies(Object handle);

    /**
     * Get the condition of an extend.
     *
     * @param handle The extend.
     * @return the condition
     */
    Object getCondition(Object handle);

    /**
     * Get the concurrency of an operation.
     *
     * @param handle The operation.
     * @return the concurrency.
     */
    Object getConcurrency(Object handle);

    /**
     * The list of connections to an association or link.
     *
     * @param handle to the association or link
     * @return a Collection with all connections.
     */
    Collection getConnections(Object handle);

    /**
     * Returns the effect of some transition.
     *
     * @param handle is the transition
     * @return the effect
     */
    Object getEffect(Object handle);

    /**
     * Get the residences of an element.
     *
     * @param handle the model element that we are getting the residences of
     * @return the residence collection
     */
    Collection getElementResidences(Object handle);

    /**
     * Returns the ElementImports of this ModelElement.
     *
     * @param handle the ModelElement
     * @return the collection of ElementImports
     */
    Collection getElementImports2(Object handle);

    /**
     * Returns the entry action to a state.
     *
     * @param handle is the state
     * @return the entry
     */
    Object getEntry(Object handle);

    /**
     * Returns the exit action to a state.
     *
     * @param handle is the state
     * @return the exit action
     */
    Object getExit(Object handle);

    /**
     * Get the Expression belonging to a Guard, ChangeEvent or timeEvent.
     *
     * @param handle the Object to get the Expression from
     * @return Object the Expression
     */
    Object getExpression(Object handle);

    /**
     * Returns all extends of a use case or extension point.
     *
     * @param handle is the use case or the extension point
     * @return the extends
     */
    Collection getExtends(Object handle);

    /**
     * Returns all extends of a use case.
     *
     * @param handle is the use case
     * @return the extends
     */
    Collection getExtends2(Object handle);

    /**
     * Gets the use case extension of an extend.
     *
     * @param handle is the extend
     * @return The extension
     */
    Object getExtension(Object handle);

    /**
     * Returns the Extensionpoint at given index-number.
     *
     * @param handle Extend
     * @param index int
     * @return ExtensionPoint
     */
    Object getExtensionPoint(Object handle, int index);

    /**
     * Returns all extends of a use case.
     *
     * @param handle is the use case or the extend
     * @return the extends
     */
    Collection getExtensionPoints(Object handle);

    /**
     * The list of Features from a Classifier.
     *
     * @param handle Classifier to retrieve from.
     * @return Collection with Features
     */
    Collection getFeatures(Object handle);

    /**
     * Gets the generalization between two generalizable elements.
     * Returns null if there is none.
     *
     * @param handle is the child
     * @param parent is the parent
     * @return The generalization
     */
    Object getGeneralization(Object handle, Object parent);

    /**
     * The list of Generalizations from a GeneralizableElement.
     *
     * @param handle GeneralizableElement to retrieve from.
     * @return Generalizations
     */
    Collection getGeneralizations(Object handle);

    /**
     * Gets the guard for some given transition.
     *
     * @param handle is the transition
     * @return Object
     */
    Object getGuard(Object handle);

    /**
     * Returns the Icon of a Stereotype.
     *
     * @param handle the Stereotype to get the Icon from
     * @return the Icon
     */
    Object getIcon(Object handle);

    /**
     * Gets the component of some element residence.
     *
     * @param handle is an element residence
     * @return component
     * @deprecated see getModelElementContainer
     */
    Object getImplementationLocation(Object handle);

    /**
     * Returns the includers for some use case.<p>
     *
     * <em>Note:</em> Changes from getIncludes in UML 1.3
     * to getIncluders in UML 1.4.
     *
     * @param handle is the use case
     * @return the includes as a Collection
     */
    Collection getIncludes(Object handle);

    /**
     * Returns the includes for some use case.<p>
     *
     * <em>Note:</em>Changes from getIncludes2 in UML 1.3
     * to getIncludes in UML 1.4.
     *
     * @param handle is the use case
     * @return the includes as a Collection
     */
    Collection getIncludes2(Object handle);

    /**
     * Returns the incoming transitions for some statevertex.
     *
     * @param handle is the state vertex
     * @return Collection
     */
    Collection getIncomings(Object handle);

    /**
     * Returns the initial value for some attribute.
     *
     * @param handle is the attribute
     * @return initial value
     */
    Object getInitialValue(Object handle);

    /**
     * Returns the instance of an AttributeLink or LinkEnd.
     *
     * @param handle is the attribute link or link end
     * @return initial value
     */
    Object getInstance(Object handle);

    /**
     * Returns the Instances for some Clasifier.
     *
     * @param handle is the classifier
     * @return Collection
     */
    Collection getInstances(Object handle);

    /**
     * Returns the collection of States for some ClasifierInState.
     *
     * @param handle is the classifierInState
     * @return Collection
     */
    Collection getInStates(Object handle);

    /**
     * Returns the interaction for some message.
     *
     * @param handle is the message
     * @return the interaction
     */
    Object getInteraction(Object handle);

    /**
     * Returns the interactions belonging to a collaboration.
     *
     * @param handle is the collaboration
     * @return Collection
     */
    Collection getInteractions(Object handle);

    /**
     * Returns the internal transitions belonging to a state.
     *
     * @param handle is the state
     * @return Collection
     */
    Collection getInternalTransitions(Object handle);

    /**
     * Returns the messages belonging to some interaction.
     *
     * @param handle candidate
     * @return Collection
     */
    Collection getMessages(Object handle);

    /**
     * Returns the messages belonging to some other message.
     *
     * @param handle is the message
     * @return Collection
     */
    Collection getMessages3(Object handle);

    /**
     * Get the messages that are activated by the given message.
     *
     * @param handle Message
     * @return the Collection of Messages
     */
    Collection getMessages4(Object handle);

    /**
     * Returns the messages received by the given classifier role.
     *
     * @param handle is the classifier role
     * @return Collection
     */
    Collection getMessages1(Object handle);

    /**
     * Returns the messages send by the given classifier role.
     *
     * @param handle is the classifier role
     * @return Collection
     */
    Collection getMessages2(Object handle);

    /**
     * Get the model of some model element.
     *
     * @param handle to the model element.
     * @return model for the model element.
     */
    Object getModel(Object handle);

    /**
     * @param handle an Element Import.
     * @return the model element
     */
    Object getModelElement(Object handle);

    /**
     * Get the Multiplicity from a model element.
     *
     * @param handle model element to retrieve from.
     * @return multiplicity
     */
    Object getMultiplicity(Object handle);

    /**
     * Get the Ranges from a Multiplicity.
     *
     * @param handle multiplicity to retrieve from.
     * @return iterator containing ranges
     */
    Iterator getRanges(Object handle);

    /**
     * Get the comments of an element.
     *
     * @param handle the model element that we are getting the comments of
     * @return the comment (or null)
     */
    Collection getComments(Object handle);

    /**
     * Get the modelelement that were commented.
     *
     * @param handle the comment that we are getting the model elements of
     * @return the modelelements (or null)
     */
    Collection getAnnotatedElements(Object handle);

    /**
     * Get the communication connection of an message.
     *
     * @param handle the message that we are getting the communication
     * connection
     * @return the communication connection
     */
    Object getCommunicationConnection(Object handle);

    /**
     * Get the communication link of a stimulus.
     *
     * @param handle the message that we are getting the communication link
     * @return the communication link
     */
    Object getCommunicationLink(Object handle);

    /**
     * Get the collaborations of an element.
     *
     * @param handle the model element that we are getting the
     * collaborations of.
     * @return the collaborations
     */
    Collection getCollaborations(Object handle);

    /**
     * Get the component instance of an instance.
     *
     * @param handle is the instance
     * @return the component instance
     */
    Object getComponentInstance(Object handle);

    /**
     * Returns the collection of ConstrainingElements of a Collaboration.
     *
     * @param handle the Collaboration
     * @return the collection of ConstrainingElements
     */
    Collection getConstrainingElements(Object handle);

    /**
     * Returns the collection of ConstrainedElements of a constraint.
     *
     * @param handle the Constraint
     * @return the collection of ConstrainedElements
     */
    Collection getConstrainedElements(Object handle);

    /**
     * Get the collection of all constraints of the given ModelElement.
     *
     * @param handle the ModelElement
     * @return the collection of all constraints
     */
    Collection getConstraints(Object handle);

    /**
     * Returns the container for the given modelelement. The container is the
     * owner of the modelelement. It will be null for elements that don't have
     * an owner. All elements except for the root element in a project should
     * have an owner. The root element is allways a model.<p>
     *
     * In  the future, this function could return the container of Figs too.
     *
     * @param handle is the base
     * @return Object
     */
    Object getModelElementContainer(Object handle);

    /**
     * Returns the CompositeState that is the container of
     * the given StateVertex.<p>
     *
     * UML 1.4 adds support for ElementResidence
     *
     * @param handle the StateVertex
     * @return the CompositeState that is the container
     */
    Object getContainer(Object handle);

    /**
     * Returns the collection of ModelElements contained in a Partition.
     *
     * @param handle the Partition
     * @return the contents of the Partition
     */
    Collection getContents(Object handle);

    /**
     * Returns the context of some given statemachine or the context
     * of some given interaction.
     *
     * @param handle the statemachine or the interaction
     * @return the context of the statemachine or interaction or null
     * if the statemachine or interaction doesn't have a context.
     */
    Object getContext(Object handle);

    /**
     * Return the collection of the Contexts of a given Signal.
     *
     * @param handle the Signal
     * @return a collection of the Contexts
     */
    Collection getContexts(Object handle);

    /**
     * Return the collection of Actions that create/instantiate
     * the given Classifier.
     *
     * @param handle the Classifier
     * @return a collection containing all the creating actions
     */
    Collection getCreateActions(Object handle);

    /**
     * Get the default value of a parameter.
     *
     * @param handle the parameter that we are getting the defaultvalue from
     * @return the default value
     */
    Object getDefaultValue(Object handle);

    /**
     * Get deferrable events of a state.
     *
     * @param handle the state that we are getting the deferrable event from
     * @return the deferrable events collection
     */
    Collection getDeferrableEvents(Object handle);

    /**
     * Returns the context of some given statemachine or the context
     * of some given interaction.
     *
     * @param handle the statemachine or the interaction
     * @return the context of the statemachine or interaction or null
     * if the statemachine or interaction doesn't have a context.
     */
    Collection getDeploymentLocations(Object handle);

    /**
     * Get the discriminator.
     *
     * @param handle the Generalization
     * @return the discriminator a String
     */
    Object getDiscriminator(Object handle);

    /**
     * Get the dispatchaction of a stimulus.
     *
     * @param handle the stimulus that we are getting the dispatchaction of
     * @return the dispatchaction (or null)
     */
    Object getDispatchAction(Object handle);

    /**
     * Returns the do activity action of a state.
     *
     * @param handle is the state
     * @return the do activity
     */
    Object getDoActivity(Object handle);

    /**
     * Return the Links of a given Association.
     *
     * @param handle the Association
     * @return the collection of Links
     */
    Collection getLinks(Object handle);

    /**
     * Return the LinkEnds of a given Instance or AssociationEnd.
     *
     * @param handle the candidate
     * @return the collection of LinkEnds
     */
    Collection getLinkEnds(Object handle);

    /**
     * Gets a location of some extension point.
     *
     * @param handle extension point
     * @return the location
     */
    String getLocation(Object handle);

    /**
     * Get the methods of an operation.
     *
     * @param handle the operation that we are getting the methods of
     * @return methods collection (or null)
     */
    Collection getMethods(Object handle);

    /**
     * Get the namespace of an element.
     *
     * @param handle the model element that we are getting the namespace of
     * @return the namespace (or null)
     */
    Object getNamespace(Object handle);

    /**
     * Get the node instance of a component instance.
     *
     * @param handle the model element that we are getting the node instance of
     * @return the node instance
     */
    Object getNodeInstance(Object handle);

    /**
     * The collection of object flow states.
     *
     * @param handle the classifier
     * @return collection of object flow states
     */
    Collection getObjectFlowStates(Object handle);

    /**
     * Get the operation of a Call Action or Call Event.
     *
     * @param handle the model element that we are getting the operation of
     * @return the Operation
     */
    Object getOperation(Object handle);

    /**
     * Get the occurrences of an operation.
     * @param handle the Opration
     * @return the collection of occurrences
     */
    Collection getOccurrences(Object handle);

    /**
     * Get the list of operations.
     *
     * @param handle classifier to examine.
     * @return Collection with operations.
     */
    Collection getOperations(Object handle);

    /**
     * Returns the opposite end of an association end.
     *
     * @param handle is the association end
     * @return Object the opposite end.
     */
    Object getOppositeEnd(Object handle);

    /**
     * Get ordering of an association end.
     *
     * @param handle association end to retrieve from
     * @return ordering
     */
    Object getOrdering(Object handle);

    /**
     * Returns the list of Transitions outgoing from the given stateVertex.
     *
     * @param handle statevertex
     * @return Collection
     */
    Collection getOutgoings(Object handle);

    /**
     * Get the list of Associations Ends connected to this association end.
     *
     * @param handle association end to start from
     * @return A Collection with all connected association ends.
     */
    Collection getOtherAssociationEnds(Object handle);

    /**
     * Get the list of Link Ends connected to this link end.
     *
     * @param handle link end to start from
     * @return A Collection with all connected link ends.
     */
    Collection getOtherLinkEnds(Object handle);
    
    /**
     * The list of owned elements of the the package.
     *
     * @param handle package to retrieve from.
     * @return A Collection with the owned elements.
     */
    Collection getOwnedElements(Object handle);

    /**
     * Get the owner scope of a feature.
     *
     * @param handle feature
     * @return owner scope
     */
    Object getOwnerScope(Object handle);

    /**
     * Get the powertype of a generalization.
     *
     * @param handle generalization
     * @return powertype
     */
    Object getPowertype(Object handle);

    /**
     * Get the powertype ranges of a classifier.
     *
     * @param handle classifier to retrieve from
     * @return collection of poertype ranges
     */
    Collection getPowertypeRanges(Object handle);

    /**
     * Get the predecessors of a message.
     *
     * @param handle message to retrieve from
     * @return collection of predecessors
     */
    Collection getPredecessors(Object handle);

    /**
     * Get the qualified attributes of an association end.
     *
     * @param handle association end to retrieve from
     * @return Collection with qualifiers.
     */
    Collection getQualifiers(Object handle);

    /**
     * Determine if the passed parameter has a RETURN direction kind.
     *
     * @return true if it is a return direction kind
     * @param handle is the parameter
     */
    boolean hasReturnParameterDirectionKind(Object handle);

    /**
     * Returns the Package that is connected by the given ElementImport.
     *
     * @param handle the ElementImport
     * @return the Package
     */
    Object getPackage(Object handle);

    /**
     * Get a parameter of a behavioral feature.
     *
     * @param handle behavioral feature to retrieve from
     * @param n parameter number
     * @return parameter.
     */
    Object getParameter(Object handle, int n);

    /**
     * Get the parameters of a Object Flow State, Behavioral Feature,
     * Classifier or Event.
     *
     * @param handle operation to retrieve from
     * @return A Collection with the parameters.
     */
    Collection getParameters(Object handle);

    /**
     * Get the parent of a generalization.
     *
     * @param handle generalization.
     * @return the parent.
     */
    Object getParent(Object handle);

    /**
     * Returns the raised signals of an operation.
     *
     * @param handle is the operation
     * @return raised signals
     */
    Collection getRaisedSignals(Object handle);

    /**
     * Returns the receptions of a signal.
     *
     * @param handle is the signal
     * @return receptions
     */
    Collection getReceptions(Object handle);

    /**
     * Returns the recurrence iteration expression of an action.
     *
     * @param handle is the action.
     * @return the recurrence
     */
    Object getRecurrence(Object handle);

    /**
     * Returns the represented classifier of a collaboration.
     *
     * @param handle is the collaboration
     * @return represented classifier
     */
    Object getRepresentedClassifier(Object handle);

    /**
     * Returns the represented operation of a collaboration.
     *
     * @param handle is the collaboration
     * @return represented operation
     */
    Object getRepresentedOperation(Object handle);

    /**
     * Returns the script belonging to a given action.
     *
     * @param handle is the action
     * @return the script
     */
    Object getScript(Object handle);

    /**
     * Returns the sender object of a stimulus or a message.
     *
     * @param handle is the stimulus or message
     * @return the sender
     */
    Object getSender(Object handle);

    /**
     * Returns the sender object of a stimulus or a message.
     *
     * TODO: Check if this javadoc comment is really correct?
     *
     * @param handle is the object
     * @return the signal
     */
    Object getSignal(Object handle);

    /**
     * Get the resident element.
     *
     * @param handle is the element residence
     * @return resident element
     */
    Object getResident(Object handle);

    /**
     * Returns the collection of elements in a given component.
     *
     * @param handle the component
     * @return the Collection of ResidentElements
     */
    Collection getResidentElements(Object handle);

    /**
     * Returns a collection with all residents belonging to the given
     * node.
     *
     * @param handle is the node, nodeinstance, componentinstance
     * @return Collection
     */
    Collection getResidents(Object handle);

    /**
     * Gets the source for a given transition.
     *
     * @param handle is the transition
     * @return Object (MStateVertex)
     */
    Object getSource(Object handle);

    /**
     * Gets the source for some given flow.
     *
     * @param handle is the flow
     * @return Collection
     */
    Collection getSources(Object handle);

    /**
     * Returns the sourceflows of a model element.
     *
     * @param handle is the model element
     * @return a collection of sourceflows
     */
    Collection getSourceFlows(Object handle);

    /**
     * The list of Specializations from a GeneralizableElement.
     *
     * @param handle GeneralizableElement to retrieve from.
     * @return Collection of Specializations.
     * @deprecated UML 1.3 only
     */
    Collection getSpecializations(Object handle);

    /**
     * Returns the state machine belonging to some given state or transition
     * If you need to find the StateMachine for an internal transition,
     * or for ANY state,
     * use StateMachinesHelper.getStateMachine() instead.
     *
     * @param handle is the state or transition
     * @return Object
     */
    Object getStateMachine(Object handle);

    /**
     * Returns the state belonging to some given transition.
     *
     * @param handle is the transition
     * @return Object
     */
    Object getState(Object handle);

    /**
     * Returns the states from a deferable event.
     *
     * @param handle is the event
     * @return Object
     */
    Collection getStates(Object handle);

    /**
     * Returns the stereotypes belonging to some given model element.<p>
     *
     * Note! For UML version 1.3 there can only be one stereotype
     * per model element. This means that the returned Collection will
     * only have zero or one elements. Don't write any code that rely
     * on this! Consider it to be a Collection with zero or more
     * elements as it will be in later UML versions.
     *
     * @param handle The model element.
     * @return a Collection with all stereotypes or empty if none.
     */
    Collection getStereotypes(Object handle);

    /**
     * Returns the stimuli belonging to some given link.
     *
     * @param handle is the link
     * @return Object
     */
    Collection getStimuli(Object handle);

    /**
     * Returns the Stimuli that are received by the given Instance.
     *
     * @param handle the Instance
     * @return the collection of stimuli
     */
    Collection getStimuli2(Object handle);

    /**
     * Returns the Stimuli that are send by the given Instance.
     *
     * @param handle the Instance
     * @return the collection of stimuli
     */
    Collection getStimuli3(Object handle);

    /**
     * Returns a collection with all subvertices belonging to the given
     * composite state.
     *
     * @param handle is the composite state
     * @return Collection
     */
    Collection getSubvertices(Object handle);

    /**
     * Returns the submachine of a submachine state.
     *
     * @param handle is the submachine state
     * @return submachine
     */
    Object getSubmachine(Object handle);

    /**
     * Returns the submachine of a submachine state.
     *
     * @param handle is the submachine state
     * @return submachine
     */
    Collection getSubmachineStates(Object handle);

    /**
     * The list of SupplierDependencies from a ModelElement.
     *
     * @param handle model element.
     * @return A Collection with the supplier dependencies.
     */
    Collection getSupplierDependencies(Object handle);

    /**
     * The top of a state machine.
     *
     * @param handle the state machine
     * @return the top
     */
    Object getTop(Object handle);

    /**
     * Get the transition of a guard or action.
     *
     * @param handle the guard or action
     * @return the transition
     */
    Object getTransition(Object handle);

    /**
     * Get the trigger of a transition.
     *
     * @param handle the transition
     * @return the trigger
     */
    Object getTrigger(Object handle);

    /**
     * The type of a StructuralFeature, AssociationEnd, Parameter or
     * ObjectFlowState.
     *
     * @param handle the StructuralFeature, AssociationEnd, Parameter or
     *  ObjectFlowState
     * @return the type
     */
    Object getType(Object handle);

    /**
     * Returns the target of some transition.
     *
     * @param handle is the transition
     * @return Object
     */
    Object getTarget(Object handle);

    /**
     * Returns the target scope of some model element.
     *
     * @param handle is the model element
     * @return Object
     */
    Object getTargetScope(Object handle);

    /**
     * Returns the targetflows of a model element.
     *
     * @param handle is the model element
     * @return a collection of targetflows
     */
    Collection getTargetFlows(Object handle);

    /**
     * Returns the upper bound of the multiplicity of the given handle (an
     * associationend).
     *
     * @param handle is the model element
     * @return int
     */
    int getUpper(Object handle);

    /**
     * Returns the use case of an extension point.
     *
     * @param handle is the extension point
     * @return a use case
     */
    Object getUseCase(Object handle);

    /**
     * Returns the upper bound of the multiplicity of the given handle (an
     * associationend).
     *
     * @param handle is the model element
     * @return int
     */
    int getLower(Object handle);

    /**
     * Returns the transitions belonging to the given handle. The handle can be
     * a statemachine or a composite state or an event.
     * If it's a statemachine the
     * transitions will be given back belonging to that statemachine. If it's a
     * compositestate the internal transitions of that compositestate will be
     * given back.
     * If it's an event, all transitions triggered by this event
     * will be given back.
     *
     * @param handle is the model element
     * @return Collection
     */
    Collection getTransitions(Object handle);

    /**
     * This method returns all attributes of a given Classifier.
     *
     * @param handle is the classifier you want to have the attributes for.
     * @return a collection of the attributes
     */
    Collection getStructuralFeatures(Object handle);

    /**
     * Returns the Specification of a given Reception.
     *
     * @param handle the Reception
     * @return String the Specification
     */
    String getSpecification(Object handle);

    /**
     * Returns all Interfaces of which this class is a realization.
     *
     * @param handle  the class you want to have the interfaces for
     * @return a collection of the Interfaces
     */
    Collection getSpecifications(Object handle);

    /**
     * Returns the suppliers of a dependency.
     *
     * @param handle is the dependency
     * @return a collection of the suppliers
     */
    Collection getSuppliers(Object handle);

    /**
     * Returns the action belonging to some message or argument.
     *
     * @param handle is the message or argument.
     * @return the action
     */
    Object getAction(Object handle);

    /**
     * Returns the activator belonging to some message.
     *
     * @param handle is the message
     * @return the activator
     */
    Object getActivator(Object handle);

    /**
     * Returns the actual arguments for a given action.
     *
     * @param handle is the action
     * @return the actual arguments
     */
    Collection getActualArguments(Object handle);

    /**
     * Returns an addition for a given inlcude.
     * There is a bug in NSUML which gets the addition and base
     * relationships back to front for include relationships. Solve
     * reversing their accessors in the code
     *
     * @param handle is the include
     * @return the addition
     */
    Object getAddition(Object handle);

    /**
     * Returns the AggregationKind of a given AssociationEnd.
     *
     * @param handle the AssociationEnd
     * @return the AggregationKind
     */
    Object getAggregation(Object handle);

    /**
     * Returns all associated classes for some given classifier.
     * Returns an empty collection if the given argument handle is not
     * a classifier.  The given parameter is included in the returned
     * collection if it has a self-referencing association.
     *
     * @param handle is the classifier
     * @return Collection
     */
    Collection getAssociatedClasses(Object handle);

    ////////////////////////////////////////////////////////////////

    /**
     * The name of a model element or some diagram part.
     *
     * @param handle that points out the object.
     * @return the name
     */
    String getName(Object handle);

    /**
     * Return the owner of a feature or its
     * association end if it is a
     * qualified attribute.
     *
     * @param handle is the feature
     * @return classifier
     */
    Object getOwner(Object handle);

    /**
     * Return the tag of a tagged value.
     *
     * @param handle The tagged value belongs to this.
     * @return The found tag as a String.
     */
    String getTag(Object handle);

    /**
     * Return the tagged values iterator of a model element.
     *
     * @param handle The tagged values belong to this.
     * @return The tagged values iterator
     */
    Iterator getTaggedValues(Object handle);

    /**
     * Returns the TaggedValues of a ModelElement.
     *
     * @param handle the ModelElement
     * @return the Collection of TaggedValues
     */
    Collection getTaggedValuesCollection(Object handle);

    /**
     * Return the tagged value with a specific tag.
     *
     * @param handle The model element the tagged value belongs to.
     * @param name The tag name.
     * @return The found tag, null if not found
     */
    Object getTaggedValue(Object handle, String name);

    /**
     * Return the value of a tagged value with a specific tag.
     *
     * @param handle The model element that the tagged value belongs to.
     * @param name The tag name.
     * @return The value of the found tag. "" if not found.
     */
    String getTaggedValueValue(Object handle, String name);

    /**
     * Return the key (tag) of some tagged value.
     *
     * TODO: This does exactly the same as getTag(Object). Remove one of them.
     *
     * @param handle The tagged value.
     * @return The found value as String.
     */
    String getTagOfTag(Object handle);

    /**
     * Returns the Value of some UML Object.
     *
     * @param handle Object
     * @return Object the exact type depends on the handle type
     * (String, Expression, Instance, TaggedValue...)
     */
    Object getValue(Object handle);

    /**
     * Return the value of some tagged value.
     *
     * @param handle The tagged value.
     * @return The found value as String.
     */
    String getValueOfTag(Object handle);

    /**
     * Return the UUID of this element.
     *
     * @param base base element (MBase type)
     * @return UUID
     */
    String getUUID(Object base);

    /**
     * Return the visibility of this element.
     *
     * @param handle an nsuml model element
     * @return visibility
     */
    Object getVisibility(Object handle);

    /**
     * Get the partitions from a container.
     *
     * @param container The container.
     * @return The partitions (a collection).
     */
    Collection getPartitions(Object container);

    /**
     *  Return the Stub State's referenced state.
     *  @param o Stub State
     *  @return referenced state
     */
    String getReferenceState(Object o);

    ////////////////////////////////////////////////////////////////

    /**
     * Returns a named object in the given object by calling it's lookup method.
     *
     * @param handle the object that we search through
     * @param name of the model element
     * @return found object, null otherwise
     */
    Object lookupIn(Object handle, String name);

    ////////////////////////////////////////////////////////////////

    /**
     * Returns the name of the UML Model class.<p>
     *
     * @param handle The object to check.
     * @return classname of modelelement
     */
    String getUMLClassName(Object handle);

    /**
     * Recognizer for Arguments.
     *
     * @param modelElement candidate
     * @return true if an argument.
     */
    boolean isAArgument(Object modelElement);

    /**
     * Returns a tooltip that should be shown for the given model element.<p>
     *
     * @param modelElement The model element for which to gerneate a tip
     * @return the tip
     */
    String getTipString(Object modelElement);

    /**
     * Returns a textual representation of the given model element.<p>
     * All implementations should return the same value as would be expected
     * from the toString() method of the NSUML class MModelElement.
     *
     * @param modelElement The model element for which to convert to text
     * @return the textual representation of the model element
     */
    String toString(Object modelElement);
    
    /**
     * @param target The TimeEvent.
     * @return TimeExpression
     */
    Object getWhen(Object target);
    /**
     * Recognizer for TagDefinition. Implemented only in UML 1.4+ metamodels.
     *
     * @param handle candidate
     * @return true if handle is a TagDefinition
     */
    boolean isATagDefinition(Object handle);

}
