/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included  with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.logger;

import java.util.HashMap;
import java.util.Map;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.LogKitLogger;
import org.apache.avalon.framework.logger.Logger;
import org.apache.log.Hierarchy;
import org.apache.log.LogTarget;
import org.apache.log.Priority;

/**
 * LogKitLoggerManager implementation.  It populates the LoggerManager
 * from a configuration file.
 *
 * @author <a href="mailto:giacomo@apache.org">Giacomo Pati</a>
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.1 $ $Date: 2002/01/17 20:44:59 $
 * @since 4.0
 */
public class LogKitLoggerManager
    implements LoggerManager, Contextualizable, Configurable
{
    /** Map for name to logger mapping */
    final private Map         m_loggers = new HashMap();

    /** The context object */
    private Context           m_context;

    /** The hierarchy private to LogKitManager */
    private Hierarchy         m_hierarchy;

    /** The root logger to configure */
    private String            m_prefix;

    /** The default logger used for this system */
    final private Logger      m_defaultLogger;

    /**
     * Creates a new <code>DefaultLogKitManager</code>. It will use a new <code>Hierarchy</code>.
     */
    public LogKitLoggerManager()
    {
        this( new Hierarchy() );
    }

    /**
     * Creates a new <code>DefaultLogKitManager</code> with an existing <code>Hierarchy</code>.
     */
    public LogKitLoggerManager( final Hierarchy hierarchy )
    {
        this( null, hierarchy );
    }

    /**
     * Creates a new <code>DefaultLogKitManager</code> using
     * specified logger name as root logger.
     */
    public LogKitLoggerManager( final String prefix )
    {
        this( prefix, new Hierarchy() );
    }

    /**
     * Creates a new <code>DefaultLogKitManager</code> with an existing <code>Hierarchy</code> using
     * specified logger name as root logger.
     */
    public LogKitLoggerManager( final String prefix, final Hierarchy hierarchy )
    {
        this( prefix, hierarchy,
              new LogKitLogger( hierarchy.getLoggerFor("") ) );
    }

    /**
     * Creates a new <code>DefaultLogKitManager</code> with an existing <code>Hierarchy</code> using
     * specified logger name as root logger.
     */
    public LogKitLoggerManager( final String prefix, final Hierarchy hierarchy, final Logger defaultLogger )
    {
        m_prefix = prefix;
        m_hierarchy = hierarchy;
        m_defaultLogger = defaultLogger;
    }

    /**
     * Retrieves a Logger from a category name. Usually
     * the category name refers to a configuration attribute name.  If
     * this LogKitManager does not have the match the default Logger will
     * be returned and a warning is issued.
     *
     * @param categoryName  The category name of a configured Logger.
     * @return the Logger.
     */
    public final Logger getLoggerForCategory( final String categoryName )
    {
        final Logger logger = (Logger) m_loggers.get( categoryName );

        if( null != logger )
        {
            if( m_defaultLogger.isDebugEnabled() )
            {
                m_defaultLogger.debug( "Logger for category " + categoryName + " returned" );
            }
            return logger;
        }

        if( m_defaultLogger.isDebugEnabled() )
        {
            m_defaultLogger.debug( "Logger for category " + categoryName
                              + " not defined in configuration. New Logger created and returned" );
        }

        return new LogKitLogger(m_hierarchy.getLoggerFor( categoryName ));
    }

    public final Logger getDefaultLogger()
    {
        return m_defaultLogger;
    }

    /**
     * Reads a context object.
     *
     * @param context The context object.
     * @throws ContextException if the context is malformed
     */
    public final void contextualize( final Context context )
        throws ContextException
    {
        m_context = context;
    }

    /**
     * Reads a configuration object and creates the category mapping.
     *
     * @param configuration  The configuration object.
     * @throws ConfigurationException if the configuration is malformed
     */
    public final void configure( final Configuration configuration )
        throws ConfigurationException
    {
        final Configuration factories = configuration.getChild( "factories" );
        final LogTargetFactoryManager targetFactoryManager = setupTargetFactoryManager( factories );

        final Configuration targets = configuration.getChild( "targets" );
        final LogTargetManager targetManager = setupTargetManager( targets, targetFactoryManager );

        final Configuration categories = configuration.getChild( "categories" );
        final Configuration [] category = categories.getChildren( "category" );
        setupLoggers( targetManager, m_prefix, category );
    }

    /**
     * Setup a LogTargetFactoryManager
     *
     * @param configuration  The configuration object.
     * @throws ConfigurationException if the configuration is malformed
     */
    private final LogTargetFactoryManager setupTargetFactoryManager( final Configuration configuration )
        throws ConfigurationException
    {
        final DefaultLogTargetFactoryManager targetFactoryManager = new DefaultLogTargetFactoryManager();
        if( targetFactoryManager instanceof LogEnabled )
        {
            targetFactoryManager.enableLogging( m_defaultLogger );
        }

        if( targetFactoryManager instanceof Contextualizable )
        {
            try
            {
                targetFactoryManager.contextualize( m_context );
            }
            catch( final ContextException ce )
            {
                throw new ConfigurationException( "cannot contextualize default factory manager", ce );
            }
        }

        targetFactoryManager.configure( configuration );

        return targetFactoryManager;
    }

    /**
     * Setup a LogTargetManager
     *
     * @param configuration  The configuration object.
     * @throws ConfigurationException if the configuration is malformed
     */
    private final LogTargetManager setupTargetManager( final Configuration configuration,
                                                       final LogTargetFactoryManager targetFactoryManager )
        throws ConfigurationException
    {
        final DefaultLogTargetManager targetManager = new DefaultLogTargetManager();

        if (targetManager instanceof LogEnabled)
        {
            targetManager.enableLogging( m_defaultLogger );
        }

        if( targetManager instanceof Contextualizable )
        {
            try
            {
                targetManager.contextualize( m_context );
            }
            catch( final ContextException ce )
            {
                throw new ConfigurationException( "cannot contextualize factory manager", ce );
            }
        }

        if( targetManager instanceof LogTargetFactoryManageable )
        {
            targetManager.setLogTargetFactoryManager( targetFactoryManager );
        }

        if( targetManager instanceof Configurable )
        {
            targetManager.configure( configuration );
        }

        return targetManager;
    }

    /**
     * Setup Loggers
     *
     * @param configuration []  The array object of configurations for categories.
     * @throws ConfigurationException if the configuration is malformed
     */
    private final void setupLoggers( final LogTargetManager targetManager,
                                     final String parentCategory,
                                     final Configuration [] categories )
        throws ConfigurationException
    {
        for( int i = 0; i < categories.length; i++ )
        {
            final String category = categories[i].getAttribute( "name" );
            final String loglevel = categories[i].getAttribute( "log-level" ).toUpperCase();

            final Configuration [] targets = categories[i].getChildren( "log-target" );
            final LogTarget [] log_targets = new LogTarget[ targets.length ];
            for( int j = 0; j < targets.length; j++ )
            {
                final String id = targets[j].getAttribute( "id-ref" );
                log_targets[j] = targetManager.getLogTarget( id );
            }

            if( "".equals( category ) && log_targets.length > 0 )
            {
                m_hierarchy.setDefaultPriority( Priority.getPriorityForName( loglevel ) );
                m_hierarchy.setDefaultLogTargets( log_targets );
            }

            final String full_category;
            if( null == parentCategory )
            {
                full_category = category;
            }
            else
            {
                full_category = parentCategory + org.apache.log.Logger.CATEGORY_SEPARATOR + category;
            }

            final org.apache.log.Logger logger = m_hierarchy.getLoggerFor( full_category );
            m_loggers.put( full_category, new LogKitLogger( logger ) );
            if( m_defaultLogger.isDebugEnabled() )
            {
                m_defaultLogger.debug( "added logger for category " + full_category );
            }
            logger.setPriority( Priority.getPriorityForName( loglevel ) );
            logger.setLogTargets( log_targets );

            final Configuration [] sub_categories = categories[i].getChildren( "category" );
            if( null != sub_categories )
            {
                setupLoggers( targetManager, full_category, sub_categories );
            }
        }
    }
}
