/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Circuit class used to map trees of information
 *
 * @version 0.0.20, 04/07/1998
 * @author  Federico Barbieri <fede@apache.org>
 * @author  Stefano Mazzocchi <mazzocch@mbox.systemy.it>
 */
public class Circuit
{
    protected Map m_map;

    public Circuit()
    {
        m_map = new HashMap();
    }

    public void addNode( final String name )
    {
        if( null == m_map.get( name ) )
        {
            m_map.put( name, new Node( name ) );
        }
    }

    public void removeNode( final String name )
    {
        String tmp = null;
        Iterator e = m_map.keySet().iterator();

        while( e.hasNext() )
        {
            tmp = (String)e.next();

            if( !tmp.equals( name ) )
            {
                try { unlink( tmp, name ); }
                catch( final CircuitException ce) {}
                try { unlink( name, tmp ); }
                catch( final CircuitException ce ) {}
            }

        }

        m_map.remove( name );
    }

    public void link( final String parent, final String child )
        throws CircuitException
    {
        Node tempNode = null;
        final Node pnode = (Node)m_map.get( parent );
        final Node cnode = (Node)m_map.get( child );
        if( null == pnode )
        {
            throw new CircuitException( "Unknown node " + parent );
        }
        else if( null == cnode )
        {
            throw new CircuitException( "Unknown node " + child );
        }
        else if( pnode.isChildOf( cnode ) )
        {
            throw new CircuitException( "Loop! Node " + parent +
                                        " is already child of node " + child );
        }
        else
        {
            final Iterator e = m_map.values().iterator();

            while( e.hasNext() )
            {
                tempNode = (Node)e.next();
                if( tempNode.isChildOf( cnode ) )
                {
                    tempNode.m_parents.addAll( pnode.m_parents );
                }
            }
        }
    }

    public void unlink( final String parent, final String child )
        throws CircuitException
    {
        final Node cnode = (Node)m_map.get( child );
        final Node pnode = (Node)m_map.get( parent );

        if( cnode.m_parents.contains( pnode ) )
        {
            Node tempNode = null;
            final Iterator e = m_map.values().iterator();

            while( e.hasNext() )
            {
                tempNode = (Node)e.next();

                if( tempNode.m_parents.contains( cnode ) )
                {
                    tempNode.m_parents.removeAll( pnode.m_parents );
                }
            }
        }
        else
        {
            throw new CircuitException( "Node " + parent + " is not parent of node " + child );
        }
    }

    public List getAncestors()
    {
        final List ancestors = new ArrayList(5);
        String name = null;
        Node tempNode = null;
        final Iterator e = m_map.keySet().iterator();

        while( e.hasNext() )
        {
            name = (String)e.next();
            tempNode = (Node)m_map.get( name );

            if( 1 == tempNode.m_parents.size() )
            {
                ancestors.add( name );
            }
        }

        return ancestors;
    }

    public String getAncestor()
    {
        String name = null;
        Node tempNode = null;
        final Iterator e = m_map.keySet().iterator();

        while( e.hasNext() )
        {
            name = (String)e.next();
            tempNode = (Node)m_map.get( name );

            if( 1 == tempNode.m_parents.size() )
            {
                return name;
            }
        }

        return null;
    }

    public boolean isEmpty()
    {
        return m_map.isEmpty();
    }

    protected final class Node
    {
        protected final List m_parents;
        protected final String m_name;

        protected Node( final String name )
        {
            m_parents = new ArrayList( 5 );
            m_parents.add( this );
            m_name = name;
        }

        protected final boolean isChildOf( final Node parent )
        {
            return m_parents.contains( parent );
        }

        public final String toString()
        {
            StringBuffer buffer = new StringBuffer();
            Iterator e = m_parents.iterator();
            buffer.append( m_name + "[" );

            while( e.hasNext() )
            {
                buffer.append(((Node) e.next()).m_name + " ");
            }

            buffer.append("]");
            return buffer.toString();
        }
    }

    public final class CircuitException
        extends RuntimeException
    {
        public CircuitException()
        {
        }

        public CircuitException( final String message )
        {
            super( message );
        }
    }

    public String toString()
    {
        StringBuffer buffer = new StringBuffer();
        String name = null;
        Node tempNode = null;
        final Iterator e = m_map.keySet().iterator();

        while( e.hasNext() )
        {
            name = (String)e.next();
            tempNode = (Node)m_map.get( name );
            buffer.append( name ).append( "(" )
                  .append( tempNode.m_parents.size() - 1 )
                  .append( ") " );
        }

        return buffer.toString();
    }
}

