/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.xml;

import org.apache.avalon.excalibur.pool.Poolable;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.parameters.Parameterizable;
import org.apache.avalon.framework.parameters.Parameters;
import org.w3c.dom.Document;
import org.xml.sax.*;
import org.xml.sax.ext.LexicalHandler;

import javax.xml.parsers.*;
import java.io.IOException;

/**
 * An XMLParser that is only dependant on JAXP 1.1 compliant parsers.
 *
 * The configuration can contain the following parameters :
 * <ul>
 * <li>validate (boolean, default = <code>false</code>) : should the parser
 *     validate parsed documents ?
 * </li>
 * <li>namespace-prefixes (boolean, default = <code>false</code>) : do we want
 *     namespaces declarations also as 'xmlns:' attributes ?<br>
 *     <i>Note</i> : setting this to <code>true</code> confuses some XSL
 *     processors (e.g. Saxon).
 * </li>
 * <li>stop-on-warning (boolean, default = <code>true</code>) : should the parser
 *     stop parsing if a warning occurs ?
 * </li>
 * <li>stop-on-recoverable-error (boolean, default = <code>true</code>) : should the parser
 *     stop parsing if a recoverable error occurs ?
 * </li>
 * <li>reuse-parsers (boolean, default = <code>true</code>) : do we want to reuse
 *     parsers or create a new parser for each parse ?<br>
 *     <i>Note</i> : even if this parameter is <code>true</code>, parsers are not
 *     recycled in case of parsing errors : some parsers (e.g. Xerces) don't like
 *     to be reused after failure.
 * </li>
 * </ul>
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:cziegeler@apache.org">Carsten Ziegeler</a>
 * @author <a href="mailto:sylvain@apache.org">Sylvain Wallez</a>
 * @version CVS $Revision: 1.9 $ $Date: 2002/01/03 07:18:28 $
 */
public class JaxpParser
extends AbstractLogEnabled
implements Parser, ErrorHandler, Composable, Parameterizable, Poolable {

    /** the SAX Parser factory */
    protected SAXParserFactory factory;

    /** the Document Builder factory */
    protected DocumentBuilderFactory docFactory;

    /** The SAX reader. It is created lazily by {@link #setupXMLReader()}
        and cleared if a parsing error occurs. */
    protected XMLReader reader;

    /** The DOM builder. It is created lazily by {@link #setupDocumentBuilder()}
        and cleared if a parsing error occurs. */
    protected DocumentBuilder docBuilder;

    /** the component manager */
    protected ComponentManager manager;

    /** the Entity Resolver */
    protected EntityResolver resolver;

    /** do we want namespaces also as attributes ? */
    protected boolean nsPrefixes;

    /** do we want to reuse parsers ? */
    protected boolean reuseParsers;

    /** do we stop on warnings ? */
    protected boolean stopOnWarning;

    /** do we stop on recoverable errors ? */
    protected boolean stopOnRecoverableError;


    /**
     * Get the Entity Resolver from the component manager
     */
    public void compose(ComponentManager manager)
    throws ComponentException
    {
        this.manager = manager;
        if (  this.manager.hasComponent(EntityResolver.ROLE) )
        {
            this.resolver = (EntityResolver)this.manager.lookup(EntityResolver.ROLE);
            if ( this.getLogger().isDebugEnabled() )
            {
                this.getLogger().debug("JaxpParser: Using EntityResolver: " + this.resolver);
            }
        }
    }

    /**
     * Configure
     */
    public void parameterize( Parameters params )
    {
        // Validation and namespace prefixes parameters
        boolean validate = params.getParameterAsBoolean("validate", false);
        this.nsPrefixes = params.getParameterAsBoolean("namespace-prefixes", false);
        this.reuseParsers = params.getParameterAsBoolean("reuse-parsers", true);
        this.stopOnWarning = params.getParameterAsBoolean("stop-on-warning", true);
        this.stopOnRecoverableError = params.getParameterAsBoolean("stop-on-recoverable-error", true);

        this.factory = SAXParserFactory.newInstance();
        this.factory.setNamespaceAware(true);
        this.factory.setValidating(validate);

        this.docFactory = DocumentBuilderFactory.newInstance();
        docFactory.setNamespaceAware(true);
        docFactory.setValidating(validate);
        if ( this.getLogger().isDebugEnabled() )
        {
            this.getLogger().debug("JaxpParser: validating: " + validate +
                                   ", namespace-prefixes: " + this.nsPrefixes +
                                   ", reuse parser: " + this.reuseParsers +
                                   ", stop on warning: " + this.stopOnWarning +
                                   ", stop on recoverable-error: " + this.stopOnRecoverableError);
        }
    }

    public void parse(InputSource in, ContentHandler consumer)
    throws SAXException, IOException
    {
        this.setupXMLReader();

        // Ensure we will use a fresh new parser at next parse in case of failure
        XMLReader tmpReader = this.reader;
        this.reader = null;

        try {
            if (consumer instanceof LexicalHandler)
            {
                tmpReader.setProperty("http://xml.org/sax/properties/lexical-handler",
                        (LexicalHandler)consumer);
            }
        }
        catch (SAXException e)
        {
            this.getLogger().warn("SAX2 driver does not support property: "+
                                  "'http://xml.org/sax/properties/lexical-handler'");
        }

        tmpReader.setErrorHandler( this );
        tmpReader.setContentHandler( consumer );
        if ( null != this.resolver )
        {
            tmpReader.setEntityResolver( this.resolver );
        }


        tmpReader.parse(in);

        // Here, parsing was successful : restore this.reader
        if ( this.reuseParsers )
            this.reader = tmpReader;
    }


    /**
     * Parses a new Document object from the given InputSource.
     */
    public Document parseDocument(InputSource input)
    throws SAXException, IOException
    {
        this.setupDocumentBuilder();

        // Ensure we will use a fresh new parser at next parse in case of failure
        DocumentBuilder tmpBuilder = this.docBuilder;
        this.docBuilder = null;

        if ( null != this.resolver )
        {
            tmpBuilder.setEntityResolver(this.resolver);
        }

        Document result = tmpBuilder.parse(input);

        // Here, parsing was successful : restore this.builder
        if ( this.reuseParsers)
            this.docBuilder = tmpBuilder;

        return result;
    }

    /**
     * Creates a new <code>XMLReader</code> if needed.
     */
    protected void setupXMLReader()
    throws SAXException
    {
        if ( null == this.reader )
        {
            // Create the XMLReader
            try
            {
                this.reader = factory.newSAXParser().getXMLReader();
            }
            catch( ParserConfigurationException pce )
            {
                throw new SAXException( "Cannot produce a valid parser", pce );
            }
            if ( this.nsPrefixes ) {
                try
                {
                    this.reader.setFeature("http://xml.org/sax/features/namespace-prefixes", this.nsPrefixes);
                }
                catch ( SAXException e )
                {
                    this.getLogger().warn("SAX2 XMLReader does not support setting feature: "+
                                      "'http://xml.org/sax/features/namespace-prefixes'");
                }
            }
        }
    }

    /**
     * Creates a new <code>DocumentBuilder</code> if needed.
     */
    protected void setupDocumentBuilder()
    throws SAXException
    {
        if ( null == this.docBuilder )
        {
            try
            {
                this.docBuilder = this.docFactory.newDocumentBuilder();
            }
            catch (ParserConfigurationException pce)
            {
                throw new SAXException( "Could not create DocumentBuilder", pce );
            }
        }
    }

    /**
     * Receive notification of a recoverable error.
     */
    public void error( SAXParseException e )
    throws SAXException
    {
        final String msg = "Error parsing "+e.getSystemId()+" (line "+
                           e.getLineNumber()+" col. "+e.getColumnNumber()+
                           "): "+e.getMessage();
        if ( this.stopOnRecoverableError )
        {
            throw new SAXException( msg, e );
        }
        this.getLogger().error( msg, e );
    }

    /**
     * Receive notification of a fatal error.
     */
    public void fatalError( SAXParseException e )
    throws SAXException
    {
        throw new SAXException("Fatal error parsing "+e.getSystemId()+" (line "+
                               e.getLineNumber()+" col. "+e.getColumnNumber()+
                               "): "+e.getMessage(),e);
    }

    /**
     * Receive notification of a warning.
     */
    public void warning( SAXParseException e )
    throws SAXException
    {
        final String msg = "Warning parsing "+e.getSystemId()+" (line "+
                           e.getLineNumber()+" col. "+e.getColumnNumber()+
                           "): "+e.getMessage();
        if ( this.stopOnWarning )
        {
            throw new SAXException( msg, e );
        }
        this.getLogger().warn( msg, e );
    }
}
