//
// File:        C.java
// Package:     gov.llnl.babel.backend.c
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: C.java,v 1.36 2002/09/30 23:00:58 epperly Exp $
// Description: common C binding routines shared by C code generators
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import java.util.Iterator;
import java.util.List;

/**
 * Class <code>C</code> contains common C language binding routines
 * shared by the C backend code generators.  This class simply collects
 * many common C binding routines into one place.
 */
public class C 
{
  // FIXME!
  private final static String s_types[] = 
  {
    "void", 
    "SIDL_bool",
    "char",
    "struct SIDL_dcomplex",
    "double",
    "struct SIDL_fcomplex", 
    "float",
    "int32_t",
    "int64_t",
    "void*",
    "char*"
  };

  // FIXME!
  //private final static String s_array_types[] = {
  //   null,                         "struct SIDL_int__array*",
  //   "struct SIDL_char__array*",   "struct SIDL_dcomplex__array*",
  //   "struct SIDL_double__array*", "struct SIDL_fcomplex__array*",
  //   "struct SIDL_float__array*",  "struct SIDL_int__array*",
  //   "struct SIDL_long__array*",   "struct SIDL_opaque__array*",
  //   "struct SIDL_string__array*"
  //};

  private final static String s_array_types[] = {
    null,                 
    "struct SIDL_bool__array*",
    "struct SIDL_char__array*",   
    "struct SIDL_dcomplex__array*",
    "struct SIDL_double__array*", 
    "struct SIDL_fcomplex__array*",
    "struct SIDL_float__array*",  
    "struct SIDL_int__array*",
    "struct SIDL_long__array*",   
    "struct SIDL_opaque__array*",
    "struct SIDL_string__array*"
  };

  private static String s_exceptionType = null;

  static {
    SymbolID id = new SymbolID(BabelConfiguration.getBaseException(),
                               new Version());
    s_exceptionType = getObjectName(id);
  }


  /**
   * Generate the header filename associated with a symbol identifier.
   * Replace the "." scope separators in the symbol by underscores and
   * append the suffix ".h".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getHeaderFile(SymbolID id) {
    return id.getFullName().replace('.', '_') + ".h";
  }


  /**
   * Generate the stub filename associated with a symbol identifier.
   * Replace the "." scope separators in the symbol by underscores and
   * append the suffix "_Stub.c".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getStubFile(SymbolID id) {
    return id.getFullName().replace('.', '_') + "_Stub.c";
  }


  /**
   * Generate the skeleton filename associated with a symbol identifier.
   * Replace the "." scope separators in the symbol by underscores and
   * append the suffix "_Skel.c".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getSkelFile(SymbolID id) {
    return id.getFullName().replace('.', '_') + "_Skel.c";
  }


  /**
   * Generate the implementation header filename associated with a
   * symbol identifier.  Replace the "." scope separators in the symbol
   * by underscores and append the suffix "_Impl.h".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getImplHeaderFile(SymbolID id) {
    return id.getFullName().replace('.', '_') + "_Impl.h";
  }


  /**
   * Generate the implementation source filename associated with a
   * symbol identifier.  Replace the "." scope separators in the symbol
   * by underscores and append the suffix "_Impl.c".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getImplSourceFile(SymbolID id) {
    return id.getFullName().replace('.', '_') + "_Impl.c";
  }


  /**
   * Convert a symbol name into its private data structure identifier.
   * Unlike the IOR, though, use the typedef version (i.e., no "struct")
   * where the SIDL name with the "." scope separators replaced by 
   * underscores is followed by "__data".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getDataName(SymbolID id) {
    return "struct " + id.getFullName().replace('.', '_') + "__data";
  }


  /**
   * Convert a symbol name into its private data structure get access
   * function name.  The function name is the SIDL name with the "." scope
   * separators replaced by underscores followed by "__get_data".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getDataGetName(SymbolID id) {
    return id.getFullName().replace('.', '_') + "__get_data";
  }


  /**
   * Convert a symbol name into its private data structure set access
   * function name.  The function name is the SIDL name with the "." scope
   * separators replaced by underscores followed by "__set_data".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getDataSetName(SymbolID id) {
    return id.getFullName().replace('.', '_') + "__set_data";
  }


  /**
   * Convert a symbol name into its private destructor function name provided
   * by the SkelSource.
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getPrivateDestructor(SymbolID id) {
      return id.getFullName().replace('.', '_') + "__delete";
  }


  /**
   * Convert a symbol name into an IOR identifier.  This method replaces
   * the "." scope separators in the symbol by underscores.
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getSymbolName(SymbolID id) {
    return id.getFullName().replace('.', '_');
  }


  /**
   * Convert a SIDL enumerated type into its symbol name, which is
   * "enum " followed by the symbol name followed by "__enum".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getEnumName(SymbolID id) {
    return "enum " + getSymbolName(id) + "__enum";
  }


  /**
   * Convert a SIDL symbol name into its object name -- for the purposes of
   * this package that means convert it into its typedef object name.  The 
   * typedef name is the SIDL symbol name with the "." scope separators 
   * replaced by underscores.
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getObjectName(SymbolID id) {
    return id.getFullName().replace('.', '_');
  }


  /**
   * Convert a SIDL symbol into the name of its associated set EPV
   * method, which is the symbol name appended with "__set_epv".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getSetEPVName(SymbolID id) {
    return getSymbolName(id) + "__set_epv";
  }


  /**
   * Convert a SIDL symbol into the name of its associated set static
   * EPV method, which is the symbol name appended with "__set_sepv".
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code>.
   */
  public static String getSetSEPVName(SymbolID id) {
    return getSymbolName(id) + "__set_sepv";
  }


  /**
   * Generate a C return string for the specified SIDL type.  Most of
   * the SIDL return strings are listed in the static structures defined
   * at the start of the class.  Symbol types and array types require
   * special processing.
   *
   * @param type the <code>Type</code> whose return string is being built.
   */
  public static String getReturnString(Type type)
    throws CodeGenerationException 
  {
    /*
     * If the type is one of the primitive types, then just return
     * its string value from the lookup table.
     */
    int t = type.getType();
    if (t < s_types.length) {
      return s_types[t];
    }

    /*
     * If the type is a symbol, then look up the symbol type and return
     * the associated type name.
     */
    if (t == Type.SYMBOL) 
    {
      Symbol symbol = Utilities.lookupSymbol(type.getSymbolID());

      if (symbol.getSymbolType() == Symbol.ENUM) {
        //FIXME?!  Is this appropriate??
        return getEnumName(symbol.getSymbolID());
      } else {
        return getObjectName(symbol.getSymbolID());
      }
    }

    /*
     * If the type is an array, then either return one of the primitive
     * array types or construct the corresponding array type.
     */
    if (t == Type.ARRAY) 
    {
      Type atype = type.getArrayType();
      int  a     = atype.getType();

      if (a < s_array_types.length) {
        return s_array_types[a];
      } else {
        //FIXME?!  Is this appropriate??
        return IOR.getArrayName(atype.getSymbolID()) + "*";
      }
    }

    return null;
  }


  /**
   * Generate a C argument string for the specified SIDL argument.
   * The formal argument name is not included.
   */
  public static String getArgumentString(Argument arg)
    throws CodeGenerationException 
  {
    String s    = getReturnString(arg.getType());
    int    type = arg.getType().getType();

    if (arg.getMode() == Argument.IN) {
      if (type == Type.STRING) {
        s = "const " + s;
      }
    }
    else {
      s = s + "*";
    }

    return s;
  }

  /**
   * Generate a C argument string with the formal argument name.
   */
  public static String getArgumentWithFormal(Argument arg)
    throws CodeGenerationException 
  {
   return getArgumentString(arg) + " " + arg.getFormalName();
  }


  /**
   * Generate the impl method's name.
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code> 
   *           associated with the method.
   *
   * @param method the <code>String</code> version of the name of the
   *               method whose impl name is being built.
   */
  public static String getMethodImplName(SymbolID id, String methodName) {
    return "impl_" + id.getFullName().replace('.','_') + '_' + methodName;
  }

  /**
   * Generate the skel method's name.  In most cases, the skel name is the
   * impl name except when the method has an array with an ordering
   * specification.
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code> 
   *           associated with the method.
   *
   * @param method the method
   */
  public static String getMethodSkelName(SymbolID id, Method method) {
    return method.hasArrayOrderSpec() 
      ? ("skel_" + id.getFullName().replace('.','_') + '_' +
          method.getLongMethodName())
      : getMethodImplName(id, method.getLongMethodName());
  }

  /**
   * Generate the full method name associated with the symbol id and the
   * specified method.  The returned name prepends the symbol name and
   * only one underbar to the method's name.
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code> 
   *           associated with the method.
   *
   * @param method the <code>Method</code> whose full name is being built.
   */
  public static String getFullMethodName(SymbolID id, Method method) {
    return getSymbolName(id) + "_" + method.getLongMethodName();
  }


  /**
   * Generate the full method name associated with the symbol id and the
   * specified method.  The returned name prepends the symbol name and
   * only one underbar to the method's name.
   *
   * @param id the <code>SymbolID</code> of the <code>Symbol</code> 
   *           associated with the method.
   *
   * @param method the <code>String</code> version of the name of the
   *               method whose full name is being built.
   */
  public static String getFullMethodName(SymbolID id, String methodName) {
    return getSymbolName(id) + "_" + methodName;
  }

  public static String getEnsureArray(Type arrayType) {
    switch(arrayType.getDetailedType()) {
    case Type.CLASS:
    case Type.INTERFACE:
      return ArrayMethods.generateEnsureName(arrayType.getSymbolID());
    case Type.ENUM:
      return "SIDL_int__array_ensure";
    default:
      return "SIDL_" + arrayType.getTypeString() +
        "__array_ensure";
    }
  }

  public static String getDelRefArray(Type arrayType) {
    switch(arrayType.getDetailedType()) {
    case Type.CLASS:
    case Type.INTERFACE:
      return ArrayMethods.generateDelRefName(arrayType.getSymbolID());
    case Type.ENUM:
      return "SIDL_int__array_deleteRef";
    default:
      return "SIDL_" + arrayType.getTypeString() +
        "__array_deleteRef";
    }
  }

  public static String getExceptionType() 
  {
    return s_exceptionType;
  }
}

