//
// File:        CxxSkelSource.java
// Package:     gov.llnl.babel.backend.cxx
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.43 $
// Date:        $Date: 2003/09/02 20:20:27 $
// Description: Write Cxx extension header file for a BABEL extendable
// 
// This is typically directed by GenCxxClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.cxx;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeGenerator;
import gov.llnl.babel.backend.cxx.Cxx;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForCxx;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * Create and write a header for a Cxx C extension class to wrap a 
 * BABEL extendable in a Cxx object. The header has to expose a 
 * function to create a wrapped IOR, a function to check if a 
 * <code>PyObject</code> is an instance of this extension type, and
 * an import macro.
 */
public class CxxSkelSource {

  private static final String s_ensureOrderConstant[] = {
    "SIDL::general_order",
    "SIDL::column_major_order",
    "SIDL::row_major_order"
  };

  private Extendable d_ext = null;
  private LanguageWriterForCxx d_writer = null;

  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public CxxSkelSource(Extendable ext) {
    d_ext = ext;
  }
  
  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    String filename = Cxx.generateFilename( d_ext.getSymbolID(), 
                                            Cxx.FILE_ROLE_SKEL, 
                                            Cxx.FILE_TYPE_CXX_SOURCE );
    //System.out.println("Create " + filename + "...");

    try { 
      d_writer = Cxx.createSource( d_ext, Cxx.FILE_ROLE_SKEL, "SKELSRCS" );
      SymbolID id = d_ext.getSymbolID();
      //writeBanner(cls, writer);
      d_writer.generateInclude( Cxx.generateFilename(d_ext.getSymbolID(), 
                                                     Cxx.FILE_ROLE_IMPL,
                                                     Cxx.FILE_TYPE_CXX_HEADER),
                                true );
      d_writer.generateInclude(IOR.getHeaderFile(id), true);
      if( d_ext.hasExceptionThrowingMethod(true) ) { 
        d_writer.generateInclude( Cxx.generateFilename("SIDL.BaseException",
                                  Cxx.FILE_ROLE_STUB, 
                                  Cxx.FILE_TYPE_CXX_HEADER), 
                                  true );
      }
      // NOTE: the following is not a binding to a SIDL type
      d_writer.generateInclude( "SIDL_String.h", false );

      d_writer.printlnUnformatted("#include <stddef.h>");
      d_writer.println();
      //      Class cls = (Class) d_ext;
      //      if ( ( cls != null ) && ( cls.getParentClass() == null) ) {
      //        writeImplDispatch(C.getPrivateDestructor(id), 
      //                          IOR.getBuiltinMethod(IOR.DELETE, id, false));
      //      }
      Cxx.beginExternCRegion( d_writer );
      writeSkelFunctions();
      writeConstructor();
      writeDestructor();
      //writeGetDataPointer();
      //writeSetDataPointer();
      writeInitializeEPV();
      if (d_ext.hasStaticMethod(true)) {
        writeInitializeSEPV();
      }
      Cxx.endExternCRegion( d_writer );
      //    } catch ( java.io.IOException ex) { 
      //      throw new CodeGenerationException("IOException : " + ex.getMessage() );
    } finally { 
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
    }
  }

  /**
   * Write a function to initialize entries in the entry point vector for
   * a particular class. This will generate an assignment statement for
   * each non-<code>static</code> method defined locally in the class.
   * 
   */
  private void writeInitializeEPV() 
  {
    SymbolID id = d_ext.getSymbolID();
    d_writer.println("void");
    d_writer.print(IOR.getSetEPVName(id));
    d_writer.print("(");
    d_writer.print(IOR.getEPVName(id));
    d_writer.println(" *epv) {");
    d_writer.increaseTabLevel();
    d_writer.writeCommentLine("initialize builtin methods");
    initializeMethodPointer
      (IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id), id); 
    initializeMethodPointer
      (IOR.getBuiltinMethod(IOR.DESTRUCTOR, id), id); 
    d_writer.writeCommentLine("initialize local methods");
    Iterator i = d_ext.getMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      if ( !m.isAbstract() ) { 
	  initializeMethodPointer( m, id);
      }
    }
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Write a function to initialize entries in the static entry point vector
   * for a particular class. This will generate an assignment statement for
   * each <code>static</code> method defined locally in the class.
   * 
   */
  private void writeInitializeSEPV() { 
    SymbolID id = d_ext.getSymbolID();
    d_writer.println("void");
    d_writer.print(IOR.getSetSEPVName(id));
    d_writer.print("(");
    d_writer.print(IOR.getSEPVName(id));
    d_writer.println(" *sepv) {");
    d_writer.increaseTabLevel();
    Iterator i = d_ext.getMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      if (m.isStatic()) {
        d_writer.print("sepv->");
        d_writer.print(IOR.getVectorEntry(m.getLongMethodName()));
        d_writer.print(" = ");
        d_writer.print(Cxx.getMethodSkelName(id, m.getLongMethodName()));
        d_writer.println(";");
      }
    }
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }





  /**
   * For non-<code>static</code> methods, write an assignment statement to
   * initialize a particular membor of the entry point
   * vector. Initialization of <code>static</code> methods appears
   * elsewhere. 
   * 
   * @param m          a description of the method to initialize
   * @param id         the name of class that owns the method
   */
  private void initializeMethodPointer(Method m,
                                       SymbolID id)
  {
    final String methodName = m.getLongMethodName();
    switch (m.getDefinitionModifier()) {
    case Method.FINAL:
    case Method.NORMAL:
      d_writer.print("epv->");
      d_writer.print(IOR.getVectorEntry(methodName));
      d_writer.print(" = ");
      d_writer.print(Cxx.getMethodSkelName(id, methodName));
      d_writer.println(";");
      break;
    case Method.ABSTRACT:
      d_writer.print("epv->");
      d_writer.print(IOR.getVectorEntry(methodName));
      d_writer.println(" = NULL;");
      break;
    default:
      /* do nothing */
      break;
    }
  }

  private void writeConstructor() { 
    SymbolID id = d_ext.getSymbolID();
    d_writer.print("void ");
    d_writer.print( Cxx.getMethodSkelName(id,"_ctor") ); // Package_class
    d_writer.print("(");
    d_writer.print( IOR.getObjectName(id) ); 
    d_writer.println("* self ) { ");
    d_writer.increaseTabLevel();
    d_writer.print("self->d_data = ");
    d_writer.print(Cxx.reinterpretCast("void*","new " + 
                                       Cxx.getSymbolName(id,"impl") + 
                                       "(self)"));
    d_writer.println(";");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  private void writeDestructor() { 
    SymbolID id = d_ext.getSymbolID();
    d_writer.print("void ");
    d_writer.print( Cxx.getMethodSkelName(id,"_dtor") ); // Package_class
    d_writer.print("(");
    d_writer.print( IOR.getObjectName(id) ); 
    d_writer.println("* self ) { ");
    d_writer.increaseTabLevel();
    d_writer.print("delete ( ");
    d_writer.print(Cxx.reinterpretCast(Cxx.getSymbolName(id,"impl")+"*", 
                                         "self->d_data"));
    d_writer.println(" );");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }
    

  private void writeGetDataPointer( ) {
    SymbolID id = d_ext.getSymbolID();
    d_writer.print(C.getDataName(id));
    d_writer.println("*");
    d_writer.print(C.getDataGetName(id));
    d_writer.print("(");
    d_writer.print(IOR.getObjectName(id));
    d_writer.println(" *self) {");
    d_writer.increaseTabLevel();
    d_writer.println("return self ? self->d_data : NULL;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }
  
  private void writeSetDataPointer( ) {
    SymbolID id = d_ext.getSymbolID();
    d_writer.println("void");
    d_writer.print(C.getDataSetName(id));
    d_writer.print("(");
    d_writer.print(IOR.getObjectName(id));
    d_writer.println(" *self,");
    d_writer.increaseTabLevel();
    d_writer.print(C.getDataName(id));
    d_writer.println(" *data) {");
    d_writer.println("if (self) {");
    d_writer.increaseTabLevel();
    d_writer.println("self->d_data = data;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }


  /**
   * write skel functions that dispatch C IOR functionpointers
   * to C++ method calls
   */
  public void writeSkelFunctions() throws CodeGenerationException { 
    Iterator i = d_ext.getMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      if ( !m.isAbstract() ) { 
	  writeImplDispatch( m );
      }
    }
  }

  /**
   * Create a wrapper function in the skeleton for a particular IOR method.
   * 
   * @param method    the description of the IOR method to be wrapped.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void writeImplDispatch( Method m )
    throws CodeGenerationException 
  {
    final ArrayList vArgs = m.getArgumentList();
    final int nargs = vArgs.size();
    final boolean throwsExceptions = (m.getThrows().size()>0);
    StringBuffer func_decl = new StringBuffer( nargs * 32 ); 
    // everything through first { 
    StringBuffer pre_impl = new StringBuffer( nargs * 128 );
    // everything before the impl call
    StringBuffer impl_call = new StringBuffer( nargs * 32 );
    // The actual call to the impl method
    StringBuffer impl_suffix = new StringBuffer( 16 );
    // might be "._get_ior()" iff return type is object
    StringBuffer post_impl = new StringBuffer( nargs * 128 );
    // after the impl returns
    String shortMethodName = m.getShortMethodName();
    String longMethodName = m.getLongMethodName();
    String className = d_ext.getSymbolID().getShortName();

    if ( shortMethodName.equals(className) ) { 
      System.out.println("WARNING: gov.llnl.babel.backend.Cxx.CxxSkelSource: SIDL / C++ conflict!");
      System.out.println("         methodName == className is not allowed in C++");
      System.out.println("         (this is restricted to constructors in C++)");
      System.out.println("         changing to " + className + "::f_" + shortMethodName + "()");
      shortMethodName = "f_" + shortMethodName;
    }
    
    if ( m.isStatic() ) {
      func_decl.append( "static " );
    }

    Type return_type = m.getReturnType();
    func_decl.append( IOR.getReturnString( return_type ) );
    func_decl.append( "\n" );
    if ( return_type.getType() != Type.VOID ) { 
	pre_impl.append( IOR.getReturnString( return_type ) );
	if ( return_type.getDetailedType() == Type.ENUM ) { 
	    pre_impl.append(" _result;\n");
	} else if ( (return_type.getDetailedType() == Type.FCOMPLEX) ||
		    (return_type.getDetailedType() == Type.DCOMPLEX) ) { 
	    pre_impl.append( " _result = {0,0};\n");
	} else {
	    pre_impl.append( " _result = 0;\n");
	}
    }
    switch( return_type.getDetailedType() ) { 
    case Type.VOID:
      break;
    case Type.OPAQUE:
      impl_call.append("_result = ");
      break;
    case Type.BOOLEAN:
      pre_impl.append( Cxx.getReturnString(return_type)+ " _local_result;\n" );
      impl_call.append( "_local_result = " );
      post_impl.append( "_result = ( _local_result ? TRUE : FALSE );\n" );
      break;
    case Type.CHAR:
    case Type.INT:
    case Type.LONG:
    case Type.FLOAT:
    case Type.DOUBLE:
      impl_call.append("_result = ");
      break;
    case Type.ENUM:
      pre_impl.append( Cxx.getEnumName( return_type.getSymbolID()) + " _local_result;\n");
      impl_call.append("_local_result = ");
      post_impl.append("_result = (" + IOR.getEnumName( return_type.getSymbolID()) + ")_local_result;\n");
      break;
    case Type.FCOMPLEX:
      pre_impl.append( Cxx.getReturnString(return_type) +" _local_result;\n" );
      impl_call.append( "_local_result = " );
      post_impl.append( "_result.real = _local_result.real();\n");
      post_impl.append( "_result.imaginary = _local_result.imag();\n" );
      break;
    case Type.DCOMPLEX:
      pre_impl.append( Cxx.getReturnString(return_type) + " _local_result;\n" );
      impl_call.append( "_local_result = " );
      post_impl.append( "_result.real = _local_result.real();\n");
      post_impl.append( "_result.imaginary = _local_result.imag();\n" );
      break;
    case Type.STRING:
      pre_impl.append( Cxx.getReturnString(return_type) + " _local_result;\n" );
      impl_call.append( "_local_result = " );
      post_impl.append( "_result = SIDL_String_strdup( _local_result.c_str() );\n" );
      break;
    case Type.SYMBOL:
      throw new CodeGenerationException( "Type.SYMBOL should have been resolved to\n" + 
                                         "Type.ENUM, Type.CLASS, or Type.INTERFACE");
      //break;
    case Type.CLASS:
    case Type.INTERFACE:
      pre_impl.append(Cxx.getReturnString(return_type) + " _local_result;\n");
      impl_call.append( "_local_result = " );
      post_impl.append( "if ( _local_result._not_nil() ) {\n");
      post_impl.append( "  _local_result.addRef();\n");
      post_impl.append( "}\n" );
      post_impl.append( "_result = _local_result._get_ior();\n");
      break;
    case Type.ARRAY:
      pre_impl.append(Cxx.getReturnString(return_type) + " _local_result;\n");
      impl_call.append("_local_result = " );
      post_impl.append("if ( _local_result._not_nil() ) {\n");
      post_impl.append("  _local_result.ensure(" + 
		       return_type.getArrayDimension() + ", " + 
		       s_ensureOrderConstant[return_type.getArrayOrder()] + ");\n");
      post_impl.append("  _local_result.addRef();\n");
      post_impl.append("}\n" );
      post_impl.append("_result = _local_result._get_ior();\n");
      break;
    default:      
    } // end case( return_type.getType() )

    func_decl.append( Cxx.getMethodSkelName( d_ext.getSymbolID(), 
                                             longMethodName ) );
    func_decl.append( "( " );
    
    

    if ( m.isStatic() ) { 
      impl_call.append( Cxx.getMethodImplName( d_ext.getSymbolID(), shortMethodName) );
      impl_call.append( "( " );
    } else { 
      func_decl.append( IOR.getObjectName( d_ext.getSymbolID() ) + "* self" );
      if ( nargs > 0 ) { func_decl.append( ", " ); }
      pre_impl.append( Cxx.getSymbolName( d_ext.getSymbolID(), "impl" ) );
      pre_impl.append( " *_this = " );
      pre_impl.append(Cxx.reinterpretCast(Cxx.getSymbolName(d_ext.getSymbolID(), "impl" ) + "*", 
                                          "self->d_data") );
      pre_impl.append( ";\n" );
      impl_call.append( "_this->" + shortMethodName + "( ");
    }

    // writeup the argument lists
    for ( Iterator it = vArgs.iterator(); it.hasNext(); ) { 
      Argument arg = (Argument) it.next();
      Type type = arg.getType();
      int typeInt = type.getDetailedType();
      String argName = arg.getFormalName();
      String mode = "/* " + Cxx.argModeToString( arg ) + " */ ";
      int modeInt = arg.getMode();
      func_decl.append( mode );
      func_decl.append( IOR.getArgumentString(arg) + " " + argName );
      if ( it.hasNext() ) { 
        func_decl.append(",");
      }
      impl_call.append( mode );
      switch ( typeInt ) { 
      case Type.OPAQUE:
        switch( modeInt ) { 
        case Argument.IN:
          impl_call.append( argName );
          break;
        case Argument.OUT:
        case Argument.INOUT:
          impl_call.append( "*" + argName );
          break;
        }
        break;
      case Type.BOOLEAN:
        switch ( modeInt ) { 
        case Argument.IN:
          pre_impl.append( "bool _local_" + argName + " = (" + argName + "==TRUE);\n" );
          impl_call.append( "_local_" + argName );
          break;
        case Argument.OUT:
          pre_impl.append( "bool _local_" + argName + ";\n" );
          impl_call.append( "_local_" + argName );
          post_impl.append( "*" + argName + " = (_local_" + argName + " ? TRUE : FALSE);\n");
          break;
        case Argument.INOUT:
          pre_impl.append( "bool _local_" + argName + " = (*" + argName + "==TRUE);\n" );
          impl_call.append( "_local_" + argName );
          post_impl.append( "*" + argName + " = (_local_" + argName + " ? TRUE : FALSE);\n");
          break;
        }
        break;
      case Type.CHAR:
      case Type.INT:
      case Type.LONG:
      case Type.FLOAT:
      case Type.DOUBLE:
        switch ( modeInt ) { 
        case Argument.IN:
          impl_call.append( argName );
          break;
        case Argument.OUT:
        case Argument.INOUT:
          impl_call.append( "*" + argName );
          break;
        }
        break;
      case Type.ENUM:
        switch ( modeInt ) { 
        case Argument.IN:
          impl_call.append( "("+ Cxx.getEnumName(type.getSymbolID()) + "&)" +argName );
          break;
        case Argument.OUT:
        case Argument.INOUT:
          impl_call.append( "("+ Cxx.getEnumName(type.getSymbolID()) + "&)*" + argName );
          break;
        }
        break;
      case Type.FCOMPLEX:
        switch( modeInt ) { 
        case Argument.IN:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + " ( " + 
                           argName + ".real, " + 
                           argName + ".imaginary );\n");
          impl_call.append( "_local_" + argName );
          break;
        case Argument.OUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + ";\n");
          impl_call.append( "_local_" + argName );
          post_impl.append( argName + "->real = _local_" + argName +
                            ".real();\n" + 
                            argName + "->imaginary = _local_" + argName + 
                            ".imag();\n" );
          break;
        case Argument.INOUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + " ( " + 
                           argName + "->real, " + 
                           argName + "->imaginary );\n");
          impl_call.append( "_local_" + argName );
          post_impl.append( argName + "->real = _local_" + argName +
                            ".real();\n" + 
                            argName + "->imaginary = _local_" + argName + 
                            ".imag();\n" );
          break;
        }
        break;
      case Type.DCOMPLEX:
        switch( modeInt ) { 
        case Argument.IN:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + " ( " + 
                           argName + ".real, " + 
                           argName + ".imaginary );\n");
          impl_call.append( "_local_" + argName );
          break;
        case Argument.OUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + ";\n");
          impl_call.append( "_local_" + argName );
          post_impl.append( argName + "->real = _local_" + argName +
                            ".real();\n" + 
                            argName + "->imaginary = _local_" + argName + 
                            ".imag();\n" );
          break;
        case Argument.INOUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + " ( " + 
                           argName + "->real, " + 
                           argName + "->imaginary );\n");
          impl_call.append( "_local_" + argName );
          post_impl.append( argName + "->real = _local_" + argName +
                            ".real();\n" + 
                            argName + "->imaginary = _local_" + argName + 
                            ".imag();\n" );
          break;
        }
        break;
      case Type.STRING:
        switch ( modeInt ) { 
        case Argument.IN:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + 
			   "= ( " + argName + " ? " + argName + ": \"\" );\n");
          impl_call.append( "_local_" + argName );
          break;
        case Argument.OUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + ";\n");
          impl_call.append( "_local_" + argName );
          post_impl.append( "*" + argName + " = SIDL_String_strdup( _local_"
                            + argName + ".c_str() );\n" );
          break;
        case Argument.INOUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + 
			   "= ( *" + argName + " ? *" + argName + ": \"\" );\n");
          impl_call.append( "_local_" + argName );
	  post_impl.append( "if ( *" + argName + "== 0) {\n");
          post_impl.append( "  *" + argName + " = SIDL_String_strdup( _local_" + argName +
                            ".c_str() );\n" );
          post_impl.append( "} else if ( strlen( *" + argName + " ) >= _local_" +
                            argName + ".length() ) {\n");
          post_impl.append( "  _local_" + argName + ".copy( *" + argName + 
                            ", ::std::string::npos );\n");
          post_impl.append( "  (*" + argName + ")[ _local_" + argName +
                            ".length()] = 0;\n");
          post_impl.append( "} else { \n" );
          post_impl.append( "  SIDL_String_free( *" + argName + ");\n");
          post_impl.append( "  *" + argName + " = SIDL_String_strdup( _local_" + argName +
                            ".c_str() );\n" );
          post_impl.append( "}" );
          break;
        }
        break;
      case Type.SYMBOL:
        throw new CodeGenerationException( "Type.SYMBOL should have been resolved to\n" + 
                                           "Type.ENUM, Type.CLASS, or Type.INTERFACE");
        //break;
      case Type.CLASS:
      case Type.INTERFACE:	  
        switch( modeInt ) { 
        case Argument.IN:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + 
                           "(" + argName + ");\n");
          impl_call.append( "_local_" + argName );
          break;
        case Argument.OUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName +
                           ";\n");
          impl_call.append( "_local_" + argName );
          post_impl.append( "if ( _local_" + argName + "._not_nil() ) {\n");
          post_impl.append( "  _local_" + argName + ".addRef();\n");
          post_impl.append( "}\n" );
          post_impl.append( "*" + argName + " = _local_" + argName + 
                            "._get_ior();\n" );
          break;
        case Argument.INOUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + 
                           "((*" + argName + "));\n");
          impl_call.append( "_local_" + argName );
          post_impl.append( "if ( _local_" + argName + "._not_nil() ) {\n");
          post_impl.append( "  _local_" + argName + ".addRef();\n");
          post_impl.append( "}\n" );
          post_impl.append( "if (*" + argName + ") (*((*" + argName + 
			    ")->d_epv->f_deleteRef))(*" +
			    argName + ");\n");
          post_impl.append( "*" + argName + " = _local_" + argName + 
                            "._get_ior();\n" );
          break;
        }
        break;
      case Type.ARRAY:
	
        switch( modeInt ) { 
        case Argument.IN:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName +
                           "(" + argName + ");\n");
          pre_impl.append("if (" + argName + ") { ");
          pre_impl.append("_local_" + argName + ".addRef();");
	  pre_impl.append("_local_" + argName + ".ensure(" + 
			  type.getArrayDimension() + ", " + 
			  s_ensureOrderConstant[type.getArrayOrder()] + ");\n");
          pre_impl.append(" }\n");
          impl_call.append( "_local_" + argName );
          break;
        case Argument.OUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName +
                           ";\n");
          impl_call.append( "_local_" + argName);
          post_impl.append( "if ( _local_" + argName + "._not_nil() ) {\n");
	  post_impl.append("  _local_" + argName + ".ensure(" + 
			   type.getArrayDimension() + ", " + 
			   s_ensureOrderConstant[type.getArrayOrder()] + ");\n");
          post_impl.append( "  _local_" + argName + ".addRef();\n");
          post_impl.append( "}\n" );
          post_impl.append( "*" + argName + " = _local_" + argName + 
                            "._get_ior();\n" );
          break;
        case Argument.INOUT:
          pre_impl.append( Cxx.getReturnString(type) + " _local_" + argName + "(*" + argName + ");\n");
	  pre_impl.append( "*" + argName + " = 0;\n");
	  pre_impl.append( "_local_" + argName + ".ensure(" + 
			  type.getArrayDimension() + ", " + 
			  s_ensureOrderConstant[type.getArrayOrder()] + ");\n");
          impl_call.append( "_local_" + argName );
          post_impl.append( "if ( _local_" + argName + "._not_nil() ) {\n");
	  post_impl.append("  _local_" + argName + ".ensure(" + 
			   type.getArrayDimension() + ", " + 
			   s_ensureOrderConstant[type.getArrayOrder()] + ");\n");
          post_impl.append("_local_" + argName + ".addRef();\n");
          post_impl.append( "}\n" );
          post_impl.append( "*" + argName + " = _local_" + argName + 
                            "._get_ior();\n" );
          break;
        }
        break;
      }
      if ( it.hasNext() ) { 
        impl_call.append(", ");
      }
    }

    // if method throws exceptions...
    if (throwsExceptions ) { 
      if ( nargs > 0 || (!m.isStatic())) { // if other args are listed.
        func_decl.append( ", " );
      }
      pre_impl.append("*_exception = 0;//init just to be safe\n");
      func_decl.append("SIDL_BaseException__object ** _exception");
    }

    // final details
    func_decl.append( " )\n");
    impl_call.append( " )" + impl_suffix.toString() + ";\n" ); 

    // finally dump everything out
    d_writer.println( func_decl.toString().trim() );
    d_writer.println( "{" );
    d_writer.increaseTabLevel();
    d_writer.writeCommentLine( "pack args to dispatch to impl" );
    d_writer.println( pre_impl.toString().trim() );
    d_writer.writeCommentLine( "dispatch to impl" );
    if ( throwsExceptions ) { 
      d_writer.println("try { ");
      d_writer.increaseTabLevel();
    }
    d_writer.println( impl_call.toString().trim() );
    if ( throwsExceptions ) { 
      d_writer.decreaseTabLevel();
      d_writer.println("} catch ( ::SIDL::StubBase& _sb ) { ");
      d_writer.increaseTabLevel();
      d_writer.println("::SIDL::BaseException _ex(_sb);");
      d_writer.println("_ex.addRef();");
      d_writer.println("*_exception = _ex._get_ior();");
      if ( return_type.getType() != Type.VOID ) { 
        d_writer.println( "return _result;" );
      }
      d_writer.decreaseTabLevel();
      d_writer.println("}");
    }
    d_writer.writeCommentLine( "unpack results and cleanup" );
    d_writer.println( post_impl.toString().trim() );
    if ( return_type.getType() != Type.VOID ) { 
      d_writer.println( "return _result;" );
    }
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();


  } // end writeImplDispatch


}
