//
// File:        AbbrevHeader.java
// Package:     gov.llnl.babel.backend.fortran
// Copyright:   (c) 2002 The Regents of the University of California
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.13 $
// Date:        $Date: 2003/09/03 15:08:42 $
// Description: Write #include file to mangle names when necessary
// 
//
// Copyright (c) 2002, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
//
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
//
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.fortran;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.fortran.Fortran;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.mangler.FortranMangler;
import gov.llnl.babel.backend.mangler.NameMangler;
import gov.llnl.babel.backend.mangler.NonMangler;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.Iterator;


public class AbbrevHeader {

  /**
   * The maximum number of characters allowed in a name.
   */
  static final public int MAXNAME = 31;
  static final public int MAXUNMANGLED = 21;

  static private final String s_arrayMethods[] = {
    "access",
    "addRef",
    "borrow",
    "copy",
    "create1d",
    "create2dCol",
    "create2dRow",
    "createCol",
    "createRow",
    "deleteRef",
    "dimen",
    "ensure",
    "get",
    "get1",
    "get2",
    "get3",
    "get4",
    "get5",
    "get6",
    "get7",
    "isColumnOrder",
    "isRowOrder",
    "lower",
    "set",
    "set1",
    "set2",
    "set3",
    "set4",
    "set5",
    "set6",
    "set7",
    "slice",
    "smartCopy",
    "stride",
    "upper"
  };

  static private final String s_skelMethods[] = {
    "_get_data",
    "_set_data"
  };

  private LanguageWriter d_lw;

  public AbbrevHeader(LanguageWriter writer) 
  {
    d_lw = writer;
  }

  private void writeLine(String longName,
                          String shortName)
  {
    d_lw.disableLineBreak();
    d_lw.print("#define ");
    d_lw.print(longName);
    d_lw.print(" ");
    d_lw.println(shortName);
    d_lw.enableLineBreak();
  }

  private void writeEntry(String longName,
                          String shortName)
  {
    String monocase = longName.toLowerCase();
    writeLine(longName, shortName);
    if (!monocase.equals(longName)) {
      writeLine(monocase, shortName.toLowerCase());
    }
    monocase = longName.toUpperCase();
    if (!monocase.equals(longName)) {
      writeLine(monocase, shortName.toUpperCase());
    }
  }

  public void generateMethods(String symName,
                              NameMangler non,
                              NameMangler fort,
                              String [] methodNames)
    throws UnsupportedEncodingException
  {
    final String suffix = Fortran.getMethodSuffix();
    int i;
    for(i = 0; i < methodNames.length; ++i){
      String longName = 
        non.shortArrayName(symName, methodNames[i], suffix);
      if (longName.length() > MAXNAME) {
        String shortName =
          fort.shortArrayName(symName, methodNames[i], suffix);
        writeEntry(longName, shortName);
      }
    } 
  }

  public void generateClassMethods(String symName,
                                   NameMangler non,
                                   NameMangler fort,
                                   String [] methodNames)
    throws UnsupportedEncodingException
  {
    final String suffix = Fortran.getMethodSuffix();
    int i;
    for(i = 0; i < methodNames.length; ++i){
      String longName = 
        non.shortName(symName, methodNames[i], suffix);
      if (longName.length() > MAXNAME) {
        String shortName =
          fort.shortName(symName, methodNames[i], suffix);
        writeEntry(longName, shortName);
      }
    } 
  }

  public void generateNamedMethod(String symName,
                                  String methName,
                                  NameMangler non,
                                  NameMangler fort)
    throws UnsupportedEncodingException
  {
    String longName = non.shortName(symName, methName, 
                                    Fortran.getMethodSuffix());
    if (longName.length() > MAXNAME) {
      String shortName = fort.shortName(symName, methName, 
                                        Fortran.getMethodSuffix());
      writeEntry(longName, shortName);
    }
  }

  private void processMethod(String symName,
                             Method m,
                             NameMangler non,
                             NameMangler fort)
    throws UnsupportedEncodingException
  {
    generateNamedMethod(symName, m.getLongMethodName(), non, fort);
  }

  private void generateType(String symName,
                            NameMangler non,
                            NameMangler fort,
                            String suffix)
    throws UnsupportedEncodingException
  {
    String longName = non.shortName(symName, suffix);
    if (longName.length() > MAXNAME) {
      String shortName = fort.shortName(symName, suffix);
      writeEntry(longName, shortName);
    }
  }

  private void generateExtendable(Extendable ext,
                                  NameMangler non,
                                  NameMangler fort)
    throws UnsupportedEncodingException
  {
    final String symName = ext.getFullName();
    Iterator i = ext.getMethods(true).iterator();
    while (i.hasNext()) {
      processMethod(symName, (Method)i.next(), non, fort);
    }
    processMethod(symName, 
                  StubDoc.createCast(ext.getSymbolID()),
                  non, fort);
    processMethod(symName, 
                  StubDoc.createCastTwo(ext.getSymbolID()),
                  non, fort);
    if (!ext.isAbstract()) {
      if ("f90".equals(BabelConfiguration.getInstance().getTargetLanguage())) {
        generateNamedMethod(symName, "new", non, fort);
      }
      else {
        generateNamedMethod(symName, "_create", non, fort);
      }
    }

    if ("f90".equals(BabelConfiguration.getInstance().getTargetLanguage())) {
      final int maxArray = BabelConfiguration.getInstance().getMaximumArray();
      if (!ext.isInterface()) {
        generateType(symName, non, fort, "_impl");
        generateType(symName, non, fort, "_private");
        generateType(symName, non, fort, "_wrap");
      }
      generateType(symName, non, fort, "");
      for(int d = 1; d <= maxArray ; ++d) {
        generateType(symName, non, fort, "_" + Integer.toString(d) + "d");
      }
      generateType(symName, non, fort, "_array");
      generateType(symName, non, fort, "_t");
      generateType(symName, non, fort, "_type");
    }
  }

  public void generateCode(Symbol sym)
    throws CodeGenerationException
  {
    try {
      final String symName = sym.getFullName();
      NameMangler non = new NonMangler();
      NameMangler fort = new FortranMangler(MAXNAME, MAXUNMANGLED);
      if (sym instanceof Extendable) {
        generateExtendable((Extendable)sym, non, fort);
        generateClassMethods(symName, non, fort, s_skelMethods);
      }
      generateMethods(symName, non, fort, s_arrayMethods);
    }
    catch (NoSuchAlgorithmException nsae) {
      throw new CodeGenerationException("NoSuchAlgorithmException: "
                                        + nsae.getMessage());
    }
    catch (UnsupportedEncodingException uee) {
      throw new CodeGenerationException("UnsupportedEncodingException: " +
                                        uee.getMessage());
    }
  }

  public static void generateCode(Symbol	sym,
                                LanguageWriter writer)
    throws CodeGenerationException
  {
    AbbrevHeader source = new AbbrevHeader(writer);
    source.generateCode(sym);
  }
}
