//
// File:        IORHeader.java
// Package:     gov.llnl.babel.backend.ior
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: IORHeader.java,v 1.23 2003/10/16 21:44:17 epperly Exp $
// Description: generate IOR header to a pretty writer stream
//
// Copyright (c) 2000-2003, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.ior;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.ior.IORSource;
import gov.llnl.babel.backend.SortComparator;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Package;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Class <code>IORHeader</code> generates an IOR header to a language writer
 * output stream.  The constructor takes a language writer stream and method
 * <code>generateCode</code> generates the intermediate object header code
 * for the specified symbol to the output stream.  The language writer output
 * stream is not closed by this object.
 */
public class IORHeader {
   private final static String SIDL_EXCEPTION = 
     BabelConfiguration.getBaseException();

   private LanguageWriterForC d_writer;

   /**
    * This is a convenience utility function that writes the symbol
    * header information into the provided language writer output stream.
    * The output stream is not closed on exit.  A code generation
    * exception is thrown if an error is detected.
    */
   public static void generateCode(Symbol symbol, LanguageWriterForC writer)
         throws CodeGenerationException {
      IORHeader header = new IORHeader(writer);
      header.generateCode(symbol);
   }

   /**
    * Create a <code>IORHeader</code> object that will write symbol information
    * to the provided output language writer stream.
    */
   public IORHeader(LanguageWriterForC writer) {
      d_writer = writer;
   }

   /**
    * Write IOR header information for the provided symbol to the language
    * writer output stream provided in the constructor.  This method does
    * not close the writer output stream and may be called for more than
    * one symbol (although the generated header may not be valid input for
    * the C compiler).  A code generation exception is generated if an error
    * is detected.
    */
   public void generateCode(Symbol symbol) throws CodeGenerationException {
      if (symbol != null) {
         switch (symbol.getSymbolType()) {
         case Symbol.CLASS:
            generateExtendable((Extendable) symbol);
            break;
         case Symbol.ENUM:
            generateEnumeration((Enumeration) symbol);
            break;
         case Symbol.INTERFACE:
            generateExtendable((Extendable) symbol);
            break;
         case Symbol.PACKAGE:
            generatePackage((Package) symbol);
            break;
         }
      }
   }

   /**
    * Generate the IOR header for a SIDL enumerated type.  Enumerated
    * types are mapped to C enumerated types, which are guaranteed to be
    * cast-compatible with integers and of size int.  See "C: A Reference
    * Manual", by Harbison and Steele.
    */
   private void generateEnumeration(Enumeration enm) {
      SymbolID id = enm.getSymbolID();
      String header = IOR.getHeaderFile(id);
      
      d_writer.writeBanner(enm, header, CodeConstants.C_IS_NOT_IMPL, 
         CodeConstants.C_DESC_IOR_PREFIX + id.getFullName());
      d_writer.openHeaderGuard(header);
      d_writer.generateInclude("SIDLType.h", true);
      d_writer.openCxxExtern();
      d_writer.writeComment(enm, false);

      /*
       * Write out the enumeration symbol and increase the tab level.
       */
      d_writer.print(IOR.getEnumName(id));
      d_writer.println(" {");
      d_writer.increaseTabLevel();

      /*
       * Output each of the enumerators (with fill space) along with its
       * assigned value.  For pretty output, find the maximum enumerator
       * length and then space over that much to align the equals signs.
       */
      int maxlength = Utilities.getWidth(enm.getEnumerators());
      final String namespace = id.getFullName().replace('.','_') + "_";
      maxlength += namespace.length();
      for (Iterator e = enm.getIterator(); e.hasNext(); ) {
         String name = (String) e.next();
         Comment cmt = enm.getEnumeratorComment(name);
         d_writer.writeComment(cmt, true);
         d_writer.printAligned(namespace + name, maxlength);
         d_writer.print(" = ");
         d_writer.print(String.valueOf(enm.getEnumeratorValue(name)));
         if (e.hasNext()) {
            d_writer.print(",");
         }
         d_writer.println();
         if (cmt != null) {
           d_writer.println();
         }
      }

      /*
       * Close the enumeration statement and write out the array structure
       * for the enumerated type.
       */
      d_writer.decreaseTabLevel();
      d_writer.println("};");
      d_writer.println();

      /*
       * Close the C++ extern block and include guard.
       */
      d_writer.closeCxxExtern();
      d_writer.closeHeaderGuard();
   }

   /**
    * Generate an IOR header for a SIDL class or interface description.
    * The header file begins with the standard banner and internal include
    * guard.  This is followed by include statements for external header
    * files, exported symbols defined by the source implementation, and
    * forward declarations.  The entry point vector data structures are
    * defined next, followed by the interface or class object and then
    * the array declaration.  The header concludes with close statements
    * for the header guards.
    */
   private void generateExtendable(Extendable ext)
         throws CodeGenerationException {
      /*
       * Generate the file banner and open the header include guard.
       */
      SymbolID id = ext.getSymbolID();
      String header = IOR.getHeaderFile(id);
      
      d_writer.writeBanner(ext, header, CodeConstants.C_IS_NOT_IMPL,
         CodeConstants.C_DESC_IOR_PREFIX + id.getFullName());
         
      d_writer.openHeaderGuard(header);

      /*
       * Include external header files and open the C++ extern block.
       */
      Set defined = generateIncludes(ext);
      d_writer.openCxxExtern();

      /*
       * Declare exported symbols and forward declarations.
       */
      generateExportedSymbols(ext);
      defined.add(id);
      generateForwardDeclarations(ext, defined);

      /*
       * Generate entry point vector(s).
       */
      if (ext.hasStaticMethod(true)) {
         generateEPV(ext, true);
      }
      generateEPV(ext, false);

      /*
       * Generate the class or interface object itself
       */
      if (ext.isInterface()) {
         generateInterfaceObject((Interface) ext);
      } else {
         generateClassObject((Class) ext);
         generateExternalStruct(ext);
         IORSource.generateExternalSignature(d_writer, ext, ";");
         d_writer.println();

      }
      /*
       * Conclude the header file by closing the C++ extern block and
       * header include guard.
       */
      d_writer.closeCxxExtern();
      d_writer.closeHeaderGuard();
   }

   /**
    * Generate the IOR header for a SIDL package description.  The package
    * header file consists of the standard header information along with
    * include statements for all package symbols.
    */
   private void generatePackage(Package p) {
      SymbolID id = p.getSymbolID();
      String header = IOR.getHeaderFile(id);
      
      d_writer.writeBanner(p, header, CodeConstants.C_IS_NOT_IMPL,
         CodeConstants.C_DESC_IOR_PREFIX + id.getFullName());
      d_writer.openHeaderGuard(header);
      d_writer.writeComment(p, false);

      /*
       * Write out the IOR include files for each of the symbols within
       * the package.
       */
      List entries = Utilities.sort(p.getSymbols().keySet());
      for (Iterator i = entries.iterator(); i.hasNext(); ) {
         String include = IOR.getHeaderFile((SymbolID) i.next());
         d_writer.generateInclude(include, true);
      }
      d_writer.println();
      
      d_writer.closeHeaderGuard();
   }

  private void generateExternalStruct(Symbol sym)
  {
    if (sym instanceof Class) {
      Class cls = (Class)sym;
      final SymbolID id = sym.getSymbolID();
      final String symbolType = IOR.getSymbolType(sym);
      final int maxDim = BabelConfiguration.getMaximumArray();
      d_writer.println(IOR.getExternalName(id) + " {");
      d_writer.increaseTabLevel();
      if (!cls.isAbstract()) {
        d_writer.println(symbolType);
        d_writer.println("(*createObject)(void);");
        d_writer.println();
      }
      d_writer.println(symbolType);
      d_writer.println("(*createRemote)(const char *url);");
      d_writer.println();
      if (cls.hasStaticMethod(true)) {
        d_writer.println(IOR.getSEPVName(id) + "*");
        d_writer.println("(*getStaticEPV)(void);");
        d_writer.println();
      }
      d_writer.decreaseTabLevel();
      d_writer.println("};");
      d_writer.println();
    }
  }

   /**
    * Generate the list of include files required to satisfy data
    * dependencies within this header file.  This includes all parents
    * (for classes) as well as any enumerated types.  These dependencies
    * will be placed in a <code>Set</code> which will be sorted and then
    * output.  If any of the symbols do not exist in the symbol table,
    * then throw a code generation exception.
    */
   private Set generateIncludes(Extendable ext)
         throws CodeGenerationException {
      /*
       * Create the set of include symbol identifiers and add inheritance
       * parents from interfaces and (if a class) the superclass.
       */
      Set includes = new HashSet();

      if (!ext.isInterface()) {
         Class cls = (Class) ext;
         includes.addAll(Utilities.getUniqueInterfaceIDs(cls));

         Class parent = cls.getParentClass();
         if (parent != null) {
            includes.add(parent.getSymbolID());
         }
      }

      /*
       * Get the list of dependencies and extract enumerated types.
       */
      Set dependencies = ext.getSymbolReferences();
      if ((dependencies != null) && (!dependencies.isEmpty())) {
         for (Iterator i = dependencies.iterator(); i.hasNext(); ) {
            SymbolID id = (SymbolID) i.next();
            Symbol symbol = Utilities.lookupSymbol(id);
            if (symbol.getSymbolType() == Symbol.ENUM) {
               includes.add(id);
            }
         }
      }

      /*
       * Sort the entries and output the list of include files.
       */
      d_writer.generateInclude("SIDL_header.h", true);
      if (!includes.isEmpty()) {
         List entries = Utilities.sort(includes);
         for (Iterator i = entries.iterator(); i.hasNext(); ) {
            String header = IOR.getHeaderFile((SymbolID) i.next());
            d_writer.generateInclude(header, true);
         }
         d_writer.println();
      }

      return includes;
   }

   /**
    * Generate the exported symbols for an extendable object.  These
    * symbols are the name of the object structure and its constructors
    * and initializers.
    */
   private void generateExportedSymbols(Extendable ext) {
      SymbolID id = ext.getSymbolID();
      String type = IOR.getObjectName(id);
      String sepv = IOR.getSEPVName(id);
  
      d_writer.writeComment(ext, false);

      /*
       * Output the structures that will be declared in this header.
       */
      d_writer.println(IOR.getArrayName(id) + ";");
      d_writer.println(type + ";");
      if (ext.hasStaticMethod(true)) {
         d_writer.println(sepv + ";");
      }
      d_writer.println();

      /*
       * Output the functions that will be defined in the implementation.
       */
      if (!ext.isAbstract()) {
         d_writer.println("extern " + type + "*");
         d_writer.println(IOR.getNewName(id) + "(void);");
         d_writer.println();
      }
      d_writer.println("extern " + type + "*");
      d_writer.println(IOR.getRemoteName(id) + "(const char *url);");
      d_writer.println();
      if (ext.hasStaticMethod(true)) {
         d_writer.println("extern " + sepv + "*");
         d_writer.println(IOR.getStaticsName(id) + "(void);");
         d_writer.println();
      }
      if (!ext.isInterface()) {
         d_writer.println("extern void " + IOR.getInitName(id) + "(");
         d_writer.increaseTabLevel();
         d_writer.println(type + "* self);");
         d_writer.decreaseTabLevel();
         d_writer.println("extern void " + IOR.getFiniName(id) + "(");
         d_writer.increaseTabLevel();
         d_writer.println(type + "* self);");
         d_writer.decreaseTabLevel();
         d_writer.println("extern void " + IOR.getVersionName(id) + 
                          "(int32_t *major, int32_t *minor);");
         d_writer.println();
      }
   }

   /**
    * Generate the necessary forward declarations to satisfy external data
    * dependencies for the methods defined in interfaces and classes.  The
    * data dependencies are generated by extracting type information from
    * the method signatures and removing any symbols that have already been
    * defined.  If any of the symbols do not exist in the symbol table,
    * then throw a code generation exception.
    */
   private void generateForwardDeclarations(Extendable ext, Set defined)
         throws CodeGenerationException {
      /*
       * Generate the set of dependencies by extracting the data
       * dependencies from each of the methods in the methods list.
       */
      Set references = new HashSet();

      for (Iterator i = ext.getMethods(true).iterator(); i.hasNext(); ) {
         Method method = (Method) i.next();
         references.addAll(method.getSymbolReferences());
         if (!method.getThrows().isEmpty()) {
            Symbol symbol = Utilities.lookupSymbol(SIDL_EXCEPTION);
            references.add(symbol.getSymbolID());
         }
      }

      /*
       * Remove all previously defined types and continue if there are
       * any symbols that remain in the dependency set.
       */
      references.removeAll(defined);
      if (!references.isEmpty()) {

         /*
          * Generate the set of referenced symbols that are interfaces
          * or classes.  These will be output as forward declarations.
          */
         Set declare = new HashSet();
         for (Iterator i = references.iterator(); i.hasNext(); ) {
            SymbolID id = (SymbolID) i.next();
            Symbol symbol = Utilities.lookupSymbol(id);
            int type = symbol.getSymbolType();
            if ((type == Symbol.INTERFACE) || (type == Symbol.CLASS)) {
               declare.add(id);
            }
         }

         /*
          * Sort the entries and output the list of forward references.
          */
         if (!declare.isEmpty()) {
            d_writer.writeComment(
               "Forward references for external classes and interfaces.",
               false);
            List entries = Utilities.sort(declare);
            for (Iterator i = entries.iterator(); i.hasNext(); ) {
               SymbolID id = (SymbolID) i.next();
               d_writer.println(IOR.getArrayName(id) + ";");
               d_writer.println(IOR.getObjectName(id) + ";");
            }
            d_writer.println();
         }
      }
   }

  /**
   * Generate a single entry in the definition of an entry
   * point vector.
   * 
   * @param m           a description of the method
   * @param self        a string with the type of the object
   * @param alreadySeen set of method names already generated.
   *                    Each element is a <code>String</code>.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateEPVEntry(Method m, String self, Set alreadySeen)
    throws CodeGenerationException
  {
    String name = m.getLongMethodName();
    if (!alreadySeen.contains(name)) {
      alreadySeen.add(name);
      d_writer.print(IOR.getReturnString(m.getReturnType()));
      d_writer.print(" (*");
      d_writer.print(IOR.getVectorEntry(name));
      d_writer.println(")(");
      d_writer.increaseTabLevel();

      /*
       * If the method is not static, then it will begin with an
       * object reference.
       */
      boolean has_throws = !m.getThrows().isEmpty();
      List args = m.getArgumentList();

      if (!m.isStatic()) {
        d_writer.print(self);
        if ((args.size() > 0) || has_throws) {
          d_writer.println(",");
        }
      } else {
        if ((args.size() == 0) && !has_throws) {
          d_writer.print("void");
        }
      }

      /*
       * Output each argument in turn.
       */
      for (Iterator a = args.iterator(); a.hasNext(); ) {
        Argument arg = (Argument) a.next();
        d_writer.print(IOR.getArgumentWithFormal(arg));
        if (a.hasNext() || has_throws) {
          d_writer.println(",");
        }
      }

      /*
       * If there is a throws clause, then return the exception type
       * as the last in the argument list.
       */
      if (has_throws) {
        d_writer.print(IOR.getExceptionType());
        d_writer.print("*_ex");
      }
      d_writer.println(");");
      d_writer.decreaseTabLevel();
    }
  }

  /**
   * Generate the EPV entries for the methods locally introduced in an
   * extendable (i.e. interface or class). This will create an EPV
   * entry for each method whose name is not in <code>alreadySeen</code>.
   *
   * @param ext                the class/interface whose parents will be
   *                           asked to write their EPV entries.
   * @param self               the type used for the self pointer.
   * @param doStatic           should <code>static</code> or
   *                           non-<code>static</code> entries be listed?
   * @param methodsAlreadySeen set of method names already included.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateLocalEntries(Extendable ext,
                                    String     self,
                                    boolean    doStatic,
                                    Set        methodsAlreadySeen)
    throws CodeGenerationException
  {
    /*
     * Process the methods in sorted order and output each function
     * pointer prototype in order.
     */
    d_writer.writeCommentLine("Methods introduced in " +
                              ext.getSymbolID().getSymbolName());
    Iterator m = null;
    if (doStatic) {
      m = ext.getStaticMethods(false).iterator();
    } else {
      m = ext.getNonstaticMethods(false).iterator();
    }
    while (m.hasNext()) {
      Method method = (Method) m.next();
      generateEPVEntry(method, self, methodsAlreadySeen);
    }
  }

  /**
   * Generate the implicit/builtin methods for this fundamental
   * class or interface. These methods always appear first in
   * entry point vector (EPV).
   *
   * @param ext         the class/interface whose parents will be
   *                    asked to write their EPV entries.
   * @param self        the type used for the self pointer.
   * @param doStatic    should <code>static</code> or
   *                    non-<code>static</code> entries be listed?
   * @param alreadySeen set of method names already included.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateBuiltinEntries(Extendable ext,
                                      String     self,
                                      boolean    doStatic,
                                      Set        alreadySeen)
    throws CodeGenerationException
  {
    /*
     * If this is not a static EPV, then output standard function
     * pointers for destructor, constructor, and virtual destructor.
     */
    if (!doStatic) {
      final SymbolID id = ext.getSymbolID();
      final int numBuiltins = (ext.isInterface() 
                               ? IOR.INTERFACE_BUILTIN_METHODS
                               : IOR.CLASS_BUILTIN_METHODS);
      d_writer.writeCommentLine("Implicit builtin methods");
      for(int i = 0; i < numBuiltins; ++i){
        Method b = IOR.getBuiltinMethod(i, id);
        generateEPVEntry(b, self, alreadySeen); 
      }
    }
  }

  /**
   * Generate the entry point vector entries for the parent class and/or
   * interfaces of <code>ext</code>. This must write the methods in a
   * consistent order.
   *
   * @param ext                the class/interface whose parents will be
   *                           asked to write their EPV entries.
   * @param self               the type used for the self pointer.
   * @param doStatic           should <code>static</code> or
   *                           non-<code>static</code> entries be listed?
   * @param methodsAlreadySeen set of method names already included.
   * @param extAlreadySeen     set of extendables whose methods have
   *                           already been included
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateParentEntries(Extendable ext,
                                     String     self,
                                     boolean    doStatic,
                                     Set        methodsAlreadySeen,
                                     Set        extAlreadySeen) 
    throws CodeGenerationException
  {
    if (ext instanceof gov.llnl.babel.symbols.Class) {
      generateEPVEntries
        (((gov.llnl.babel.symbols.Class)ext).getParentClass(),
         self, doStatic, methodsAlreadySeen, extAlreadySeen);
    }
    /*
     * It is critical that we always visit parent interfaces in a canonical
     * order. The precise order is not important. The fact that the same
     * order is used every time is critical!
     */
    Object[] parents = ext.getParentInterfaces(false).toArray();
    Arrays.sort(parents, new SortComparator());
    for(int i = 0; i < parents.length; ++i){
      generateEPVEntries((Extendable)parents[i], self, doStatic, methodsAlreadySeen,
                         extAlreadySeen);
    }
  }

  private static final boolean isBaseClass(Extendable ext) {
    return 
      ((ext instanceof gov.llnl.babel.symbols.Class) &&
       (((gov.llnl.babel.symbols.Class)ext).getParentClass() == null));
       
  }
  
  private static final boolean hasParents(Extendable ext) {
    return 
      (!isBaseClass(ext)) || (!ext.getParentInterfaces(false).isEmpty());
  }

  /**
   * Generate EPV entries in a top down fashion. Parent entries are listed
   * before local entries. Each method name is listed only once.
   *
   * @param ext                the class/interface whose parents will be
   *                           asked to write their EPV entries.
   * @param self               the type used for the self pointer.
   * @param doStatic           should <code>static</code> or
   *                           non-<code>static</code> entries be listed?
   * @param methodsAlreadySeen set of method names already included.
   * @param extAlreadySeen     a set of extendables whose methodss have
   *                           already been included.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateEPVEntries(Extendable ext, 
                                  String     self,
                                  boolean    doStatic,
                                  Set        methodsAlreadySeen,
                                  Set        extAlreadySeen) 
    throws CodeGenerationException 
  {
    if ((ext != null) && !extAlreadySeen.contains(ext)) {
      if (hasParents(ext)) {
        generateParentEntries(ext, self, doStatic, methodsAlreadySeen, extAlreadySeen);
      }
      generateLocalEntries(ext, self, doStatic, methodsAlreadySeen);
      extAlreadySeen.add(ext);
    }
  }
  

   /**
    * Generate the EPV (entry point vector) data structure for the methods
    * in the interface or class.  If the flag is true, then generate an EPV
    * for the static members of the class.
    *
    * The EPV must be ordered correctly!  The parent entries must appear
    * before child entries.  This is required for runtime features.
    */
   private void generateEPV(Extendable ext, boolean do_static)
         throws CodeGenerationException {
      /*
       * Generate the name for this entry point vector as well as a pointer
       * to "self" for this structure.  For classes, self will be the object
       * structure whereas for interfaces it is void*.
       */
      final SymbolID id = ext.getSymbolID();
      String self = null;
      if (ext.isInterface()) {
         self = "void* self";
      } else {
         self = IOR.getObjectName(id) + "* self";
      }

      /*
       * Write a comment block to document the EPV structure.
       */
      String comment = null;
      if (do_static) {
         comment = "Declare the static entry point vector.";
      } else {
         comment = "Declare the method entry point vector.";
      }
      d_writer.writeComment(comment, false);

      /*
       * Begin definition of the EPV structure and increase tab indent.
       */
      if (do_static) {
         d_writer.print(IOR.getSEPVName(id));
      } else {
         d_writer.print(IOR.getEPVName(id));
      }
      d_writer.println(" {");
      d_writer.increaseTabLevel();
      Set methodsAlreadySeen = new HashSet();
      Set extAlreadySeen = new HashSet();
      generateBuiltinEntries(ext, self, do_static, methodsAlreadySeen);
      generateEPVEntries(ext, self, do_static, methodsAlreadySeen, extAlreadySeen);
      methodsAlreadySeen = null;
      extAlreadySeen = null;
      d_writer.decreaseTabLevel();
      d_writer.println("};");
      d_writer.println();
   }

   /*
    * Generate the object data structure for an interface.  This structure
    * contains an EPV, a pointer to the object implementation, and an integer
    * identifier.
    */
   private void generateInterfaceObject(Interface ifc) {
      SymbolID id = ifc.getSymbolID();
      String epv  = IOR.getEPVName(id);
      String type = IOR.getObjectName(id);

      d_writer.writeComment("Define the interface object structure.", false);

      int width = epv.length() + 2;

      d_writer.println(type + " {");
      d_writer.increaseTabLevel();

      d_writer.println(epv + "* d_epv;");

      d_writer.printAligned("void*", width);
      d_writer.println("d_object;");

      d_writer.decreaseTabLevel();
      d_writer.println("};");
      d_writer.println();
   }

   /**
    * Generate the object data structure for a class.  This structure
    * starts with the parent object (if not null) followed by new interfaces
    * not found in parent objects.  It also contains an EPV, a data pointer,
    * and an integer identifier.
    */
   private void generateClassObject(Class cls) {
      d_writer.writeComment("Define the class object structure.", false);

      /*
       * Extract unique interfaces that will be output in the object
       * data structure.
       */
      List unique = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
      int width = Utilities.getWidth(unique) + "struct __object".length();

      /*
       * Extract information about the parent class (if it exists).
       */
      Class parent = cls.getParentClass();
      SymbolID pid = null;
      String ptype = null;

      if (parent != null) {
         pid = parent.getSymbolID();
         ptype = IOR.getObjectName(pid);
         if (ptype.length() > width) {
            width = ptype.length();
         }
      }

      /*
       * Extract information about the EPV structure and its width.
       */
      SymbolID id = cls.getSymbolID();
      String epv = IOR.getEPVName(id) + "*";
      if (epv.length() > width) {
         width = epv.length();
      }

      /*
       * Output the object data structure.  Start with the parent class
       * (if it exists) followed by the new interface objects.
       */
      d_writer.println(IOR.getObjectName(id) + " {");
      d_writer.increaseTabLevel();

      if (parent != null) {
         d_writer.printAligned(ptype, width);
         d_writer.println(" d_" + IOR.getSymbolName(pid).toLowerCase() + ";");
      }

      for (Iterator i = unique.iterator(); i.hasNext(); ) {
         SymbolID sid = (SymbolID) i.next();
         d_writer.printAligned(IOR.getObjectName(sid), width);
         d_writer.println(" d_" + IOR.getSymbolName(sid).toLowerCase() + ";");
      }

      /*
       * Fill in the remainder of the data members in the structure.
       */
      d_writer.printAligned(epv, width);
      d_writer.println(" d_epv;");
      
      d_writer.printAligned("void*", width);
      d_writer.println(" d_data;");

      d_writer.decreaseTabLevel();
      d_writer.println("};");
      d_writer.println();
   }
}
